/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 */

#include "icewm.h"

int initializing = 1;

YApplication *app = 0;
XContext windowContext;
XContext frameContext;
XContext clientContext;

Atom _XA_WM_PROTOCOLS;
Atom _XA_WM_TAKE_FOCUS;
Atom _XA_WM_DELETE_WINDOW;
Atom _XA_WM_STATE;
Atom _XA_WM_CHANGE_STATE;
Atom _XATOM_MWM_HINTS;
//Atom _XA_MOTIF_WM_INFO;
Atom _XA_WM_COLORMAP_WINDOWS;
Atom _XA_WM_WORKSPACES;
Atom _XA_WM_WORKSPACE;

Colormap defaultColormap;

YColor *black;
YColor *white;

YColor *buttonBg;
YColor *buttonFg;

YColor *scrollBarBg;
YColor *scrollBarArrow;
YColor *scrollBarSlider;

YColor *listBoxBg;
YColor *listBoxFg;
YColor *listBoxSelBg;
YColor *listBoxSelFg;

YColor *menuBg;
YColor *menuItemFg;
YColor *activeMenuItemBg;
YColor *activeMenuItemFg;
YColor *disabledMenuItemFg;

YColor *activeBorderBg;
YColor *inactiveBorderBg;

YColor *activeTitleBarBg;
YColor *activeTitleBarFg;

YColor *inactiveTitleBarBg;
YColor *inactiveTitleBarFg;

YColor *taskBarBg;

YColor *normalTaskBarAppFg;
YColor *normalTaskBarAppBg;

YColor *activeTaskBarAppFg;
YColor *activeTaskBarAppBg;

YColor *statusFg;
YColor *statusBg;

GC blackGC;
GC whiteGC;
GC outlineGC;
GC clipPixmapGC;

YPixmap *closePixmap;
YPixmap *minimizePixmap;
YPixmap *maximizePixmap;
YPixmap *restorePixmap;

YIcon *defaultAppIcon;

YPixmap *startPixmap;
YPixmap *windowsPixmap;
YPixmap *mailPixmap;
YPixmap *unreadMailPixmap;
YPixmap *newMailPixmap;

YPixmap *PixNum[10];
YPixmap *PixNoColon;
YPixmap *PixColon;

YFont *titleFont;
YFont *menuFont;
YFont *statusFont;
YFont *normalTaskBarFont;
YFont *activeTaskBarFont;
YFont *windowListFont;

Cursor leftPointer;
Cursor rightPointer;

Cursor movePointer;

Cursor sizeRightPointer;
Cursor sizeTopRightPointer;
Cursor sizeTopPointer;
Cursor sizeTopLeftPointer;
Cursor sizeLeftPointer;
Cursor sizeBottomLeftPointer;
Cursor sizeBottomPointer;
Cursor sizeBottomRightPointer;

YMenu *windowMenu = 0;
YMenu *occupyMenu = 0;
YMenu *windowListMenu = 0;
YMenu *windowListPopup = 0;
YMenu *rootMenu = 0;

MoveSizeStatus *statusMoveSize = 0;
SwitchWindow *switchWindow = 0;
MailBoxStatus *mailBoxStatus = 0;
TaskBar *taskBar = 0;
WindowList *windowList = 0;
CtrlAltDelete *ctrlAltDelete = 0;

#ifdef SHAPE
int shapesSupported;
int shapeEventBase, shapeErrorBase;
#endif

PhaseType phase = phaseStartup;

static sigset_t oldSignalMask;
static struct sigaction oldSignalCHLD;
void setSignals() {
    //signal(SIGCHLD, SIG_IGN);
    struct sigaction sig;
    sig.sa_handler = SIG_IGN;
    sigemptyset(&sig.sa_mask);
    sig.sa_flags = 0;
    sigaction(SIGCHLD, &sig, &oldSignalCHLD);
    
    sigset_t mask;
    sigemptyset(&mask);
    sigaddset(&mask, SIGINT);
    sigaddset(&mask, SIGTERM);
    sigprocmask(SIG_BLOCK, &mask, &oldSignalMask);
}

void resetSignals() {
    sigset_t mask;
    struct sigaction old_sig;
    
    //signal(SIGCHLD, SIG_DFL);
    sigaction(SIGCHLD, &oldSignalCHLD, &old_sig);
    sigprocmask(SIG_SETMASK, &oldSignalMask, &mask);
}

static void initContexts() {
    windowContext = XUniqueContext();
    frameContext = XUniqueContext();
    clientContext = XUniqueContext();
}

static void initAtoms() {
    _XA_WM_PROTOCOLS = XInternAtom(app->display(), "WM_PROTOCOLS", False);
    _XA_WM_TAKE_FOCUS = XInternAtom(app->display(), "WM_TAKE_FOCUS", False);
    _XA_WM_DELETE_WINDOW = XInternAtom(app->display(), "WM_DELETE_WINDOW", False);
    _XA_WM_STATE = XInternAtom(app->display(), "WM_STATE", False);
    _XA_WM_CHANGE_STATE = XInternAtom(app->display(), "WM_CHANGE_STATE", False);
    _XATOM_MWM_HINTS = XInternAtom(app->display(), _XA_MOTIF_WM_HINTS, False);
    _XA_WM_COLORMAP_WINDOWS = XInternAtom(app->display(), "WM_COLORMAP_WINDOWS", False);
    //_XA_MOTIF_WM_INFO = XInternAtom(app->display(), "_MOTIF_WM_INFO", False);
    _XA_WM_WORKSPACES = XInternAtom(app->display(), "WM_WORKSPACES", False);
    _XA_WM_WORKSPACE = XInternAtom(app->display(), "WM_WORKSPACE", False);
}

static void createGC(GC &gc, YColor fore, YColor back, YFont *font = 0) {
    XGCValues gcv;

    gcv.foreground = fore.pixel();
    gcv.background = back.pixel();

    if (font)
        gcv.font = font->getFontStruct()->fid;
    
    gc = XCreateGC(app->display(),
                   app->root()->handle(),
                   GCForeground | GCBackground | (font ? GCFont : 0),
                   &gcv);
}

static void initGCs() {
    titleFont = new YFont(titleFontName);
    menuFont = new YFont(menuFontName);
    statusFont = new YFont(statusFontName);
    normalTaskBarFont = new YFont(normalTaskBarFontName);
    activeTaskBarFont = new YFont(activeTaskBarFontName);
    windowListFont = new YFont(windowListFontName);

    black = new YColor(clrBlack);
    white = new YColor(clrWhite);

    buttonBg = new YColor(clrNormalButton);
    buttonFg = new YColor(clrNormalButtonText);

    scrollBarBg = new YColor(clrScrollBar);
    scrollBarArrow = new YColor(clrScrollBarArrow);
    scrollBarSlider= new YColor(clrScrollBarSlider);

    listBoxBg = new YColor(clrListBox);
    listBoxFg = new YColor(clrListBoxText);
    listBoxSelBg = new YColor(clrListBoxSelected);
    listBoxSelFg = new YColor(clrListBoxSelectedText);

    menuBg = new YColor(clrNormalMenu);
    menuItemFg = new YColor(clrNormalMenuItemText);
    activeMenuItemBg = new YColor(clrActiveMenuItem);
    activeMenuItemFg = new YColor(clrActiveMenuItemText);
    disabledMenuItemFg = new YColor(clrDisabledMenuItemText);

    activeBorderBg = new YColor(clrActiveBorder);
    inactiveBorderBg = new YColor(clrInactiveBorder);

    activeTitleBarBg = new YColor(clrActiveTitleBar);
    activeTitleBarFg = new YColor(clrActiveTitleBarText);

    inactiveTitleBarBg = new YColor(clrInactiveTitleBar);
    inactiveTitleBarFg = new YColor(clrInactiveTitleBarText);

    taskBarBg = new YColor(clrDefaultTaskBar);

    normalTaskBarAppBg = new YColor(clrNormalTaskBarApp);
    normalTaskBarAppFg = new YColor(clrNormalTaskBarAppText);

    activeTaskBarAppBg = new YColor(clrActiveTaskBarApp);
    activeTaskBarAppFg = new YColor(clrActiveTaskBarAppText);

    statusBg = new YColor(clrMoveSizeStatus);
    statusFg = new YColor(clrMoveSizeStatusText);

    createGC(blackGC, YColor(clrBlack), YColor(clrBlack));
    createGC(whiteGC, YColor(clrWhite), YColor(clrWhite));

    {
        XGCValues gcv;

        gcv.foreground = YColor(clrActiveBorder).pixel();
        gcv.line_width = (wsBorderX + wsBorderY) / 2;
        gcv.subwindow_mode = IncludeInferiors;
        gcv.function = GXxor;

        outlineGC = XCreateGC(app->display(), app->root()->handle(),
                              GCFunction | GCLineWidth |
                              GCForeground | GCSubwindowMode,
                              &gcv);

        clipPixmapGC = XCreateGC(app->display(), app->root()->handle(),
                                 0, &gcv);
    }
}

void loadPixmap(char *path, char *base, const char *name, YPixmap **pixmap) {
    strcpy(base, name);

    *pixmap = new YPixmap(path);

    if (pixmap == 0)
        die(1, "out of memory for pixmap %s", path);
}

void loadIcon(char *path, char *base, const char *name, YIcon **icon) {
    strcpy(base, name);

    *icon = getIcon(path);

    if (*icon == 0)
        die(1, "out of memory for pixmap %s", path);
}

static void initIcons() {
    XIconSize *is;

    is = XAllocIconSize();
    assert(is != 0);
    is->min_width = 16;
    is->min_height = 16;
    is->max_width = 32; 
    is->max_height = 32; 
    is->width_inc = 16;
    is->height_inc = 16;
    XSetIconSizes(app->display(), app->root()->handle(), is, 1);
    XFree(is);
}

static void initPixmaps() {
    char path[1024], *p;

    strncpy(path, REDIR_ROOT(xpmPath), 1000); path[1000] = 0;
    if (strlen(path) > 0 && path[strlen(path) - 1] != '/')
        strcat(path, "/");
    strcat(path, "themes/");
    strcat(path, themeName);
    p = strrchr(path, '/');
    if (p)
        p[1] = 0;
#if 0
    switch (wmLook) {
#ifdef CONFIG_LOOK_WIN95
    case lookWin95: strcat(path, "/themes/win95/"); break;
#endif
#ifdef CONFIG_LOOK_NICE
    case lookNice:  strcat(path, "/themes/nice/");  break;
#endif
#ifdef CONFIG_LOOK_MOTIF
    case lookMotif: strcat(path, "/themes/motif/"); break;
#endif
#ifdef CONFIG_LOOK_WARP4
    case lookWarp4: strcat(path, "/themes/warp4/"); break;
#endif
#ifdef CONFIG_LOOK_WARP3
    case lookWarp3: strcat(path, "/themes/warp3/"); break;
#endif
    //TODO case lookWin31: strcat(path, "/win31/"); break;
    default: strcat(path, "/themes/"); break;
    }
#endif
    p = path + strlen(path);

    loadPixmap(path, p, "close.xpm", &closePixmap);
    loadPixmap(path, p, "maximize.xpm", &maximizePixmap);
    loadPixmap(path, p, "minimize.xpm", &minimizePixmap);
    loadPixmap(path, p, "restore.xpm", &restorePixmap);

    strncpy(path, xpmPath, 1000); path[1000] = 0;
    strcat(path, "/icons/");
    p = path + strlen(path);
    loadIcon(path, p, "app", &defaultAppIcon);

    strncpy(path, xpmPath, 1000); path[1000] = 0;
    strcat(path, "/ledclock/");
    p = path + strlen(path);
    
    loadPixmap(path, p, "n0.xpm", &PixNum[0]);
    loadPixmap(path, p, "n1.xpm", &PixNum[1]);
    loadPixmap(path, p, "n2.xpm", &PixNum[2]);
    loadPixmap(path, p, "n3.xpm", &PixNum[3]);
    loadPixmap(path, p, "n4.xpm", &PixNum[4]);
    loadPixmap(path, p, "n5.xpm", &PixNum[5]);
    loadPixmap(path, p, "n6.xpm", &PixNum[6]);
    loadPixmap(path, p, "n7.xpm", &PixNum[7]);
    loadPixmap(path, p, "n8.xpm", &PixNum[8]);
    loadPixmap(path, p, "n9.xpm", &PixNum[9]);
    loadPixmap(path, p, "nc.xpm", &PixNoColon);
    loadPixmap(path, p, "cl.xpm", &PixColon);

    strncpy(path, xpmPath, 1000); path[1000] = 0;
    strcat(path, "/taskbar/");
    p = path + strlen(path);

    loadPixmap(path, p, START_PIXMAP, &startPixmap);
    loadPixmap(path, p, "windows.xpm", &windowsPixmap);
    loadPixmap(path, p, "mail.xpm", &mailPixmap);
    loadPixmap(path, p, "unreadmail.xpm", &unreadMailPixmap);
    loadPixmap(path, p, "newmail.xpm", &newMailPixmap);
}

static void initCursors() {
    leftPointer = XCreateFontCursor(app->display(), XC_left_ptr);
    rightPointer = XCreateFontCursor(app->display(), XC_right_ptr);

    movePointer = XCreateFontCursor(app->display(), XC_fleur);

    sizeRightPointer = XCreateFontCursor(app->display(), XC_right_side);
    sizeTopRightPointer = XCreateFontCursor(app->display(), XC_top_right_corner);
    sizeTopPointer = XCreateFontCursor(app->display(), XC_top_side);
    sizeTopLeftPointer = XCreateFontCursor(app->display(), XC_top_left_corner);
    sizeLeftPointer = XCreateFontCursor(app->display(), XC_left_side);
    sizeBottomLeftPointer = XCreateFontCursor(app->display(), XC_bottom_left_corner);
    sizeBottomPointer = XCreateFontCursor(app->display(), XC_bottom_side);
    sizeBottomRightPointer = XCreateFontCursor(app->display(), XC_bottom_right_corner);
}

static void initMenus() {
    windowListMenu = new YWindowListMenu(app->root());
    
    windowMenu = new YMenu(app->root());
    assert(windowMenu != 0);

    occupyMenu = new YMenu(app->root());
    assert(occupyMenu != 0);

    occupyMenu->addItem("All", 0, "Alt+F2", cmdOccupyAllOrCurrent);
    //YMenu::YMenuItem *all = new YMenu::YMenuItem("All", 0, "Alt+F2",
    //                                             cmdOccupyAll,
    //                                             0, 0);
    //all->setCheck(1);
    //occupyMenu->add(all);
    occupyMenu->addSeparator();
    for (int w = 0; w < workspaceCount; w++) {
        char s[128];
        sprintf(s, "%d. %s", w + 1, workspaceNames[w]);
        occupyMenu->addItem(s, 0, 0, cmdOccupyWorkspace, (void *)w);
    }
    //occupyMenu->addSeparator();
    //occupyMenu->addItem("All/Previous", 0, "", cmdOccupyAll);

    windowMenu->addItem("Restore", 0, "Alt+F5", cmdRestore);
    windowMenu->addItem("Move", 0, "Alt+F7", cmdMove);
    windowMenu->addItem("Size", 0, "Alt+F8", cmdSize);
    windowMenu->addItem("Minimize", 2, "Alt+F9", cmdMinimize);
    windowMenu->addItem("Maximize", 2, "Alt+F10", cmdMaximize);
    windowMenu->addItem("Lower", 0, "Alt+F3", cmdLower);
    windowMenu->addItem("Hide", 0, "Alt+F11", cmdHide);
    windowMenu->addItem("Shade", 3, "Alt+F12", cmdShade);
    if (workspaceCount > 1) {
        windowMenu->addSeparator();
        //windowMenu->addSubmenu("Occupy Workspace", 7, occupyMenu);
        YMenu::YMenuItem *item = new YMenu::YMenuItem("Occupy Workspace", 7, 0,
                                                      cmdOccupyAllOrCurrent,
                                                      occupyMenu, 0);
        windowMenu->add(item);
    }
    windowMenu->addSeparator();
    windowMenu->addItem("Close", 0, "Alt+F4", cmdClose);
    windowMenu->addSeparator();
    windowMenu->addSubmenu("Window list", 0, windowListMenu);
    rootMenu = new YMenu(app->root());


    YMenu *closeSubmenu = new YMenu(app->root());
    assert(closeSubmenu != 0);

    closeSubmenu->addItem("Close", 0, "Delete", cmdClose);
    closeSubmenu->addSeparator();
    closeSubmenu->addItem("Kill Client", 0, "", cmdKill);
    closeSubmenu->addItem("Terminate Process", 0, "", cmdTermProcess)->disable();
    closeSubmenu->addItem("Kill Process", 0, "", cmdKillProcess)->disable();

    windowListPopup = new YMenu(app->root());
    windowListPopup->addItem("Show", 0, "", cmdActivate);
    windowListPopup->addItem("Hide", 0, "", cmdHide);
    windowListPopup->addSeparator();
    windowListPopup->addItem("Tile vertically", 0, "", cmdTileVertical)->disable();
    windowListPopup->addItem("Tile horizontally", 1, "", cmdTileHorizontal)->disable();
    windowListPopup->addItem("Cascade", 1, "", cmdCascade)->disable();
    windowListPopup->addSeparator();
    YMenu::YMenuItem *item = new YMenu::YMenuItem("Close", 0, 0,
                                cmdClose,
                                closeSubmenu, 0);
    windowListPopup->add(item);
    //windowListPopup->addSubmenu("Close", 0, "", closeSubmenu);;
}

int handler(Display *display, XErrorEvent *xev) {

    if (initializing && 
        xev->request_code == X_ChangeWindowAttributes && 
        xev->error_code == BadAccess) 
    {
        fprintf(stderr, "WM already running, exiting...\n");
        exit(1);
    }

    DBG {
        char msg[80], req[80], number[80];
        
        XGetErrorText(display,
                      xev->error_code,
                      msg, sizeof(msg));
        sprintf(number, "%d", xev->request_code);
        
        XGetErrorDatabaseText(display,
                              "XRequest",
                              number, "",
                              req, sizeof(req));
        if (!req[0])
            sprintf(req, "[request_code=%d]", xev->request_code);
        
        fprintf(stderr, "icewm: %s(0x%lX): %s\n", req, xev->resourceid, msg);
    }
    return 0;
}

#ifdef DEBUG
void dumpZorder(const char *oper, YFrameWindow *w, YFrameWindow *a) {
    YFrameWindow *p = app->top(w->layer());
    msg("---- %s ", oper);
    while (p) {
        if (p && p->client())
            msg(" %c %c 0x%lX: %s", (p == w) ? '*' : ' ',  (p == a) ? '#' : ' ', p->client()->clientWindow(), p->client()->windowTitle());
        else
            msg("?? 0x%lX: %s", p->handle());
        p = p->next();
    } 
}
#endif

YApplication::YApplication(const char *displayName) {
    app = this;
    fLoopLevel = 0;
    fExitApp = 0;
    fFocus = 0;
    lastEventTime = CurrentTime;
    fTop[0] = fBottom[0] = 0;
    fTop[1] = fBottom[1] = 0;
    fColormapWindow = 0;
    fGrabWindow = 0;
    fGrabMouse = 0;
    fPopup = 0;
    fActiveWorkspace = -1;

    if (displayName == 0)
        displayName = getenv("DISPLAY");

    if (!(fDisplay = XOpenDisplay(displayName)))
        die(1, "icewm: Can't open display: %s", displayName ? displayName : "<none>");

    DBG XSynchronize(display(), True);
    
    XSetErrorHandler(handler);

    initContexts();

    defaultColormap = DefaultColormap(display(), DefaultScreen(display()));

    initAtoms();
    initCursors();

    fRoot = new YRootWindow(0, RootWindow(display(),
                                          DefaultScreen(display())));
    assert(fRoot != 0);

    initGCs();
    initPixmaps();
    initIcons();

#ifdef SHAPE
    shapesSupported = XShapeQueryExtension(display(),
                                           &shapeEventBase, &shapeErrorBase);
#endif

    initMenus();

    statusMoveSize = new MoveSizeStatus(app->root());
    switchWindow = new SwitchWindow(app->root());
    taskBar = new TaskBar(app->root());
    if (showTaskBar)
        taskBar->show();
    windowList = new WindowList(app->root());
    //windowList->show();
    ctrlAltDelete = new CtrlAltDelete(app->root());

    XSetInputFocus(display(), PointerRoot, RevertToPointerRoot, CurrentTime);
    initializing = 0;
}

YApplication::~YApplication() {
    delete ctrlAltDelete; ctrlAltDelete = 0;
    delete taskBar; taskBar = 0;
    delete windowList; windowList = 0;
    delete switchWindow; switchWindow = 0;
    delete statusMoveSize; statusMoveSize = 0;
    
    
    delete rootMenu; rootMenu = 0;
    //delete windowMenu; windowMenu = 0;
    //delete windowListMenu; windowListMenu = 0;
    delete windowListPopup; windowListPopup = 0;    
    
    delete titleFont;
    delete menuFont;
    delete statusFont;
    delete normalTaskBarFont;
    delete activeTaskBarFont;
    delete windowListFont;

    delete closePixmap;
    delete minimizePixmap;
    delete maximizePixmap;
    delete restorePixmap;
    delete defaultAppIcon;
    delete startPixmap;
    delete windowsPixmap;
    delete mailPixmap;
    delete unreadMailPixmap;
    delete newMailPixmap;
    delete PixNoColon;
    delete PixColon;
    for (int n = 0; n < 10; n++) delete PixNum[n];
   
    XFreeGC(display(), outlineGC);
    XFreeGC(display(), clipPixmapGC);
    
    //XSync(app->display(), False);
    XCloseDisplay(display());
    app = 0;
}

int YApplication::mainLoop() {
    fLoopLevel++;
    fExitLoop = 0;

    struct timeval timeout;
    timeout.tv_sec = 0;
    timeout.tv_usec = 500 * 1000;

    while (!fExitApp && !fExitLoop) {
        if (XPending(display()) > 0) {
            XEvent xev;

            XNextEvent(display(), &xev);
            
            saveEventTime(xev);
            
#ifdef DEBUG
            DBG logEvent(xev);
#endif
            
            if (xev.type == KeymapNotify) {
                XRefreshKeyboardMapping(&xev.xmapping);
            } else {
                YWindow *window;
                int ge = (xev.type == ButtonPress ||
                          xev.type == ButtonRelease ||
                          xev.type == MotionNotify ||
                          xev.type == KeyPress ||
                          xev.type == KeyRelease ||
                          xev.type == EnterNotify ||
                          xev.type == LeaveNotify) ? 1 : 0;

                if (fPopup && ge) {
                    handleGrabEvent(fPopup, xev);
                } else if (fGrabWindow && ge) {
                    handleGrabEvent(fGrabWindow, xev);
                } else if (XFindContext(display(),
                                        xev.xany.window,
                                        windowContext,
                                        (XPointer *)&window) == 0)
                {
                    window->handleEvent(xev);
                } else {
                    MSG(("unknown window 0x%lX", xev.xany.window));
                }
            }
        } else {
            int rc;
            fd_set read_fds;

            FD_ZERO(&read_fds);
            FD_SET(ConnectionNumber(app->display()), &read_fds);

#if defined(HPUX9)
            rc = select(sizeof(fd_set), (int *)&read_fds, 0, 0, &timeout);
#else
            rc = select(sizeof(fd_set), &read_fds, 0, 0, &timeout);
#endif

            sigset_t mask;
            sigpending(&mask);
            if (sigismember(&mask, SIGINT) || sigismember(&mask, SIGTERM)) {
                unmanageClients();
                exit(0);
            }
            if (rc == 0) {
                if (taskBar->visible() && taskBar->clock()->visible())
                    taskBar->clock()->repaint();
                if (mailBoxStatus && mailBoxStatus->visible())
                    mailBoxStatus->checkMail();
                timeout.tv_usec = 500 * 1000;
            } else
                assert(rc != -1);
        }
    }
    fLoopLevel--;
    return fExitCode;
}

void YApplication::handleGrabEvent(YWindow *win, XEvent &xev) {
    Window child;

    if (xev.type == MotionNotify) {
    if (fGrabTree && xev.xbutton.subwindow != None) {
        XFindContext(display(),
                     xev.xbutton.subwindow,
                     windowContext,
                     (XPointer *)&win);
    }
        if (xev.xmotion.window != win->handle())
            if (XTranslateCoordinates(app->display(),
                                      xev.xany.window, win->handle(),
                                      xev.xmotion.x, xev.xmotion.y,
                                      &xev.xmotion.x, &xev.xmotion.y, &child) == True)
                xev.xmotion.window = win->handle();
            else
                return ;
    } else if (xev.type == ButtonPress || xev.type == ButtonRelease ||
              xev.type == EnterNotify || xev.type == LeaveNotify) {
    if (fGrabTree && xev.xbutton.subwindow != None) {
        XFindContext(display(),
                     xev.xbutton.subwindow,
                     windowContext,
                     (XPointer *)&win);
    }

        if (xev.xbutton.window != win->handle())
            if (XTranslateCoordinates(app->display(),
                                      xev.xany.window, win->handle(),
                                      xev.xbutton.x, xev.xbutton.y,
                                      &xev.xbutton.x, &xev.xbutton.y, &child) == True)
                xev.xbutton.window = win->handle();
            else
                return ;
    } else if (xev.type == KeyPress || xev.type == KeyRelease) {
    if (fGrabTree && xev.xbutton.subwindow != None) {
        XFindContext(display(),
                     xev.xbutton.subwindow,
                     windowContext,
                     (XPointer *)&win);
    }
        if (xev.xkey.window != win->handle())
            if (XTranslateCoordinates(app->display(),
                                      xev.xany.window, win->handle(),
                                      xev.xkey.x, xev.xkey.y,
                                      &xev.xkey.x, &xev.xkey.y, &child) == True)
                xev.xkey.window = win->handle();
            else
                return ;
    }
    win->handleEvent(xev);
}

int YApplication::grabEvents(YWindow *win, Cursor ptr, unsigned int eventMask, int grabMouse, int grabKeyboard, int grabTree) {
    int rc;
    
    if (fGrabWindow != 0)
        return 0;

    XSync(display(), 0);
    fGrabTree = grabTree;
    if (grabMouse) {
        fGrabMouse = 1;
        rc = XGrabPointer(display(), win->handle(),
                          False,
                          eventMask,
                          GrabModeAsync, GrabModeAsync,
                          None, ptr, CurrentTime);
        
        if (rc != Success) {
            MSG(("grab status = %d\x7", rc));
            return 0;
        }
    } else {
        fGrabMouse = 0;

        XChangeActivePointerGrab(display(),
                                 eventMask,
                                 ptr, CurrentTime);
    }

    if (grabKeyboard) {
        rc = XGrabKeyboard(display(), win->handle(),
                           False,
                           GrabModeAsync, GrabModeAsync, getEventTime());
        if (rc != Success) {
            MSG(("grab status = %d\x7", rc));
            XUngrabPointer(display(), CurrentTime);
            return 0;
        }
        
        XSetInputFocus(display(),
                       win->handle(),
                       RevertToPointerRoot,
                       CurrentTime);
    }

    installColormap(None);

    fGrabWindow = win;
    return 1;
}

int YApplication::releaseEvents() {
    if (fGrabWindow == 0)
        return 0;
    fGrabWindow = 0;
    if (fGrabMouse) {
        XUngrabPointer(display(), CurrentTime);
        fGrabMouse = 0;
    }
    XUngrabKeyboard(display(), CurrentTime);
    XSetInputFocus(display(),
                   (focus() && focus()->client()) ?
                    focus()->client()->clientWindow() : PointerRoot,
                   RevertToPointerRoot,
                   CurrentTime);
    if (colormapWindow() && colormapWindow()->client())
        installColormap(colormapWindow()->client()->colormap());
    return 1;
}

void YApplication::exitLoop(int exitCode) {
    fExitLoop = 1;
    fExitCode = exitCode;
}

void YApplication::exit(int exitCode) {
    fExitApp = 1;
    exitLoop(exitCode);
}

void YApplication::saveEventTime(XEvent &xev) {
    //lastEventTime = CurrentTime;
    //return ;
    switch (xev.type) {
    case ButtonPress:
    case ButtonRelease:
        lastEventTime = xev.xbutton.time;
        break;
        
    case MotionNotify:
        lastEventTime = xev.xmotion.time;
        break;
        
    case KeyPress:
    case KeyRelease:
        lastEventTime = xev.xkey.time;
        break;
        
    case EnterNotify:
    case LeaveNotify:
        lastEventTime = xev.xcrossing.time;
        break;
        
    case PropertyNotify:
        lastEventTime = xev.xproperty.time;
        break;
        
    case SelectionClear:
        lastEventTime = xev.xselectionclear.time;
        break;
        
    case SelectionRequest:
        lastEventTime = xev.xselectionrequest.time;
        break;
        
    case SelectionNotify:
        lastEventTime = xev.xselection.time;
        break;
    default:
        lastEventTime = CurrentTime;
        break;
    }
}

YFrameWindow *YApplication::getFrame(Window win) {
    YFrameWindow *frame;

    if (XFindContext(display(),
                     win,
                     frameContext,
                     (XPointer *)&frame) == 0)
        return frame;
    else
        return 0;
}

YFrameWindow *YApplication::createFrame(YFrameClient *client) {
    return new YFrameWindow(root(), client);
}

YFrameClient *YApplication::getClient(Window win) {
    YFrameClient *client;

    if (XFindContext(display(),
                     win,
                     clientContext,
                     (XPointer *)&client) == 0)
        return client;
    else
        return 0;
}

YFrameClient *YApplication::createClient(Window win) {
    return new YFrameClient(root(), 0, win);
}

void YApplication::setFocus(YFrameWindow *window) {
    assert((YWindow *)window != root());

    if (window == fFocus)
        return ;
    
    if (focus())
        focus()->loseFocus();
    fFocus = window;
    if (focus())
        focus()->setFocus();
    else
        XSetInputFocus(display(), PointerRoot, RevertToPointerRoot, getEventTime());
    if (!pointerColormap && colormapWindow() != focus())
        setColormapWindow(focus());
}

void YApplication::loseFocus(YFrameWindow *window) {
    assert(window != 0);
    if (!clickFocus)
        setFocus(0);
    else {
#ifdef DEBUG
        if (debug_z) dumpZorder("losing focus: ", window);
#endif
        YFrameWindow *w = window->findWindow(YFrameWindow::fwfNext | 
                                    YFrameWindow::fwfVisible | 
                                    YFrameWindow::fwfCycle |
                                    YFrameWindow::fwfFocusable |
                                    YFrameWindow::fwfWorkspace);
    
        assert (w != window);
        setFocus(w);
    }
}

void YApplication::activate(YFrameWindow *window) {
    if (window) {
        window->wmRaise();
        window->activate();
    }
//    else
//        XBell(display(), 100);
}

void YApplication::setTop(int layer, YFrameWindow *top) {
    if (fTop[layer]) {
        if (raiseOnClickClient)
            fTop[layer]->container()->grabButtons();
    }
    fTop[layer] = top;
    if (fTop[layer]) {
        if (raiseOnClickClient &&
           !(focusOnClickClient && !fTop[layer]->focused()))
            fTop[layer]->container()->releaseButtons();
    }
}

void YApplication::installColormap(Colormap cmap) {
    MSG(("installing colormap 0x%lX", cmap));
    if (fGrabWindow == 0) {
        if (cmap == None) {
            XInstallColormap(display(), defaultColormap);
        } else {
            XInstallColormap(display(), cmap);
        }
    }
}

void YApplication::setColormapWindow(YFrameWindow *frame) {
    if (fColormapWindow != frame) {
        fColormapWindow = frame;

        if (colormapWindow())
            installColormap(colormapWindow()->client()->colormap());
        else
            installColormap(None);
    }
}

void YApplication::manageClients() {
    unsigned int clientCount;
    Window winRoot, winParent, *winClients;
 
    XGrabServer(display());
    XQueryTree(display(), root()->handle(),
               &winRoot, &winParent, &winClients, &clientCount);

    if (winClients)
        for (unsigned int i = 0; i < clientCount; i++)
            if (findClient(winClients[i]) == 0)
                manageClient(winClients[i]);

    XUngrabServer(display());
    if (winClients)
        XFree(winClients);
}

void YApplication::unmanageClients() {
    Window w;

    setFocus(0);
    phase = phaseShutdown;
    XGrabServer(display());
    while (bottom(0)) {
        w = bottom(0)->client()->handle();
        unmanageWindow(w);
    }
    while (bottom(1)) {
        w = bottom(1)->client()->handle();
        unmanageWindow(w);
    }
    XUngrabServer(display());
}

YFrameClient *YApplication::manageClient(Window win) {
    YFrameClient *client = manageWindow(win);
    if (client && client->visible()) {
        mapWindow(win, 0);
    }

    return client;
}

YFrameClient *YApplication::findClient(Window win) {
    YFrameClient *client = getClient(win);
    if (client == 0) {
        YFrameWindow *frame = getFrame(win);
        if (frame)
            client = frame->client();
        else
            client = 0;
    }
    return client;
}

YFrameClient *YApplication::manageWindow(Window win) {
    YFrameClient *client = 0;

    MSG(("managing window 0x%lX", win));

    XEvent xev;
    XGrabServer(display());
    XSync(display(), 0);
    if (XCheckTypedWindowEvent(display(), win, DestroyNotify, &xev))
        goto end;

    client = getClient(win);

    if (client == 0) {
        XWindowAttributes attributes;
    
        if (!XGetWindowAttributes(display(),
                                  win,
                                  &attributes))
            goto end;

        if (attributes.override_redirect)
            goto end;

        client = createClient(win);

        client->setBorder(attributes.border_width);
        client->setGeometry(attributes.x + DO_BORDER(client->border()),
                            attributes.y + DO_BORDER(client->border()),
                            attributes.width,
                            attributes.height);

        client->setColormap(attributes.colormap);
    }
//    if (attributes.map_state == IsViewable)
//        client->show();
    //        mapWindow(win);
end:
    XUngrabServer(display());
    return client;
}

void YApplication::placeWindow(YFrameWindow *frame, int x, int y, int newClient) {
    YFrameClient *client = frame->client();
    int posX, posY;
    unsigned int posWidth, posHeight;

    posWidth = client->width() + 2 * frame->borderX();
    posHeight = client->height() + 2 * frame->borderY() + frame->titleY();

    if (newClient && client->sizeHints() &&
        !(client->sizeHints()->flags & (USPosition | PPosition)) &&
        !(client == windowList))
    {
        static int lastX = 0;
        static int lastY = 0;

        if (lastX < minX()) lastX = minX();
        if (lastY < minY()) lastY = minY();

        x = lastX;
        y = lastY;

        lastX += wsTitleBar;
        lastY += wsTitleBar;

        if (int(y + posHeight) >= maxY()) {
            y = minY();
            lastY = wsTitleBar;
        }
        if (int(x + posWidth) >= maxX()) {
            x = minX();
            lastX = wsTitleBar;
        }
        
        newClient = 0;
    }
    posX = x - frame->borderX() - DO_BORDER(frame->client()->border());
    posY = y - frame->borderY() - DO_BORDER(frame->client()->border());

    if (newClient) {
        int gx, gy;
        
        client->gravityOffsets(gx, gy);
        
        // is -1 correct here ?
        if (gx > 0)
            posX -= 0 - 1 - client->border();
        if (gy > 0)
            posY -= 0 + frame->titleY() - 1 - client->border();
    }

    // check size too ?
    //if (int(posX + posWidth) >= int(maxX())) posX = maxX() - posWidth;
    //if (int(posY + posHeight) >= int(maxY())) posY = maxY() - posHeight;
    //if (posX < minX()) posX = minX();
    //if (posY < minY()) posY = minY();
    
    frame->setGeometry(posX,
                       posY,
                       posWidth,
                       posHeight);
}

YFrameWindow *YApplication::mapWindow(Window win, int newClient) {
    YFrameWindow *frame = getFrame(win);
    
    MSG(("mapping window 0x%lX", win));

    if (frame == 0) {
        
        YFrameClient *client = getClient(win);
        if (client == 0) {
            client = manageWindow(win);
        }

        if (client == 0)
            return 0;

        int x = client->x() + minX();
        int y = client->y() + minY();

        frame = createFrame(client);

        placeWindow(frame, x, y, newClient);

#ifdef SHAPE
        frame->setShape();
#endif
    }

    MSG(("Map - Frame: %d", frame->visible()));
    MSG(("Map - Client: %d", frame->client()->visible()));

    FrameState st = frame->client()->getWMState();
    
    if (st == WithdrawnState) {
        XWMHints *h = frame->client()->hints();
        if (h && (h->flags & StateHint))
            st = h->initial_state;
    }
    if (!frame->visibleOn(activeWorkspace()))
        frame->workspaceHide();
    switch (st) {
    case IconicState:
        //frame->setFrameState(NormalState);
        frame->wmMinimize();
        break;

    case NormalState:
    case WithdrawnState:
        frame->changeState(NormalState);

        if (focusOnMap && (phase == phaseRunning || frame->visibleOn(activeWorkspace())))
            frame->activate();
        break;
    }
    return frame;
}

YFrameClient *YApplication::unmapWindow(Window win) {
    YFrameWindow *frame = getFrame(win);
    
    MSG(("unmapping window 0x%lX", win));

    if (frame) {
        YFrameClient *client = frame->client();

        client->hide();
        frame->hide();
        if (!frame->visibleOn(activeWorkspace()))
            frame->workspaceShow();
        frame->unmanage();
        app = this;
        delete frame;
        client->setPosition(client->x() - minX(),
                            client->y() - minY());
        return client;
    } else {
        YFrameClient *client = getClient(win);
        
        return client;
    }
}

void YApplication::unmanageWindow(Window win) {
    YFrameClient *client = unmapWindow(win);

    MSG(("unmanaging window 0x%lX", win));

    if (client) {
        client->show();
        delete client;
    }
}

void YApplication::destroyedWindow(Window win) {
    YFrameWindow *frame = getFrame(win);
    if (frame) 
        delete frame;
    else {
        YFrameClient *client = getClient(win);
        if (client) 
            delete client;
        else
            MSG(("destroyed? unknown window: 0x%lX", win));
    }
}

void YApplication::focusTopWindow() {
    if (top(0)) 
        setFocus(top(0)->findWindow(YFrameWindow::fwfVisible |
                                    YFrameWindow::fwfCycle |
                                    YFrameWindow::fwfFocusable |
                                    YFrameWindow::fwfWorkspace));
    if (focus() == 0 && top(1))
        setFocus(top(1)->findWindow(YFrameWindow::fwfVisible |
                                    YFrameWindow::fwfCycle |
                                    YFrameWindow::fwfFocusable |
                                    YFrameWindow::fwfWorkspace));
}

void YApplication::restackWindows(YFrameWindow *win, int oper) {
    int count = 0;
    YFrameWindow *f;

    if (oper == 0) {
        f = app->top(win->layer());
        for (; f; f = f->next()) count++;
    } else if (oper == 1) {
        f = win;
        for (; f; f = f->prev()) count++;
    } else if (oper == -1) {
        f = win;
        for (; f; f = f->next()) count++;
    }

    if (oper == 1) {

        if (win->layer() == 0) {
            f = app->bottom(1);
            for (; f; f = f->prev()) count++;
        }

        if (taskBar->visible())
            count++; // taskBar

        if (windowList && windowList->frame())
            count++;

        YPopupWindow *p = fPopup;
        while (p) {
            count++;
            p = p->prevPopup();
        }
        if (statusMoveSize->visible())
            count++;

        if (ctrlAltDelete->visible())
            count++;
    }

    Window *w = (Window *)malloc(sizeof(Window *) * count);
    if (w == 0)
        return ;
    
    int i = 0;

    if (oper == 1) {
        if (win->layer() == 0) {
            f = top(1);
            for (; f; f = f->next())
                w[i++] = f->handle();
        }
        
        if (statusMoveSize->visible())
            w[i++] = statusMoveSize->handle();

        YPopupWindow *p = fPopup;
        while (p) {
            w[i++] = p->handle();
            p = p->prevPopup();
        }

        if (taskBar->visible())
            w[i++] = taskBar->handle();

        
        if (windowList && windowList->frame())
            w[i++] = windowList->frame()->handle();

        if (ctrlAltDelete->visible())
            w[i++] = ctrlAltDelete->handle();
    }
    
    if (oper == -1) {
        f = win;
        for (; i < count; i++, f = f->next()) {
            w[i] = f->handle();
        }
    } else {
        f = app->top(win->layer());
        for (; i < count; i++, f = f->next()) {
            w[i] = f->handle();
        }
    }
    
    if (count > 0) {
        if (oper == 1) {
#if 1
            XRaiseWindow(app->display(), w[0]);
#else
            /* !!! must determine correct top window */
            if (win->next()) {
                XWindowChanges xwc;

                xwc.sibling = win->next()->handle();
                xwc.stack_mode = Above;
                XConfigureWindow(app->display(), w[0], CWSibling | CWStackMode, &xwc);
            }
#endif
        } else if (oper == -1) {
            //XLowerWindow(app->display(), w[0]);
            if (win->prev()) {
                XWindowChanges xwc;

                xwc.sibling = win->prev()->handle();
                xwc.stack_mode = Below;
                XConfigureWindow(app->display(), w[0], CWSibling | CWStackMode, &xwc);
            }
        }
        if (count > 1)
            XRestackWindows(app->display(), w, count);
    }
    free(w);
}

YMenu *YApplication::createWindowMenu(YMenu *menu, int workspace) {
    if (!menu)
        menu = new YMenu(root());
    
    int level, levelCount, windowLevel, layerCount;
    YFrameWindow *frame;

    for (int layer = 0 ; layer <= 1; layer++) {
        layerCount = 0;
        for (level = 0; level < 4; level++) {
            levelCount = 0;
            for (frame = app->top(layer); frame; frame = frame->next()) {
                if (frame->client() == windowList)
                    continue;
                if (!frame->visibleOn(workspace))
                    continue;
                if (frame->frameOptions() & YFrameWindow::foIgnoreWinList)
                    continue;
                if (workspace != app->activeWorkspace() &&
                    frame->visibleOn(app->activeWorkspace()))
                    continue;
                
                windowLevel = 0;
                if (frame->hidden())
                    windowLevel = 3;
                else if (frame->minimized())
                    windowLevel = 2;
                else if (frame->shaded())
                    windowLevel = 1;
                
                if (level != windowLevel)
                    continue;
                
                if ((levelCount == 0 && level > 0) || (layerCount == 0 && layer > 0))
                    menu->addSeparator();
                
                frame->addToMenu(menu);
                levelCount++;
                layerCount++;
            }
        }
    }
    return menu;
}

int YApplication::windowCount(int workspace) {
    int count = 0;
    
    for (int layer = 0 ; layer <= 1; layer++) {
        for (YFrameWindow *frame = app->top(layer); frame; frame = frame->next()) {
            if (!frame->visibleOn(workspace))
                continue;
            if (frame->frameOptions() & YFrameWindow::foIgnoreWinList)
                continue;
            if (workspace != app->activeWorkspace() &&
                frame->visibleOn(app->activeWorkspace()))
                continue;
            count++;
        }
    }
    return count;
}

YWindowListMenu::YWindowListMenu(YWindow *parent): YMenu(parent) {
}

void YWindowListMenu::updatePopup() {
    removeAll();

    assert(this == app->createWindowMenu(this, app->activeWorkspace()));

    int first = 1;
    
    for (int d = 0; d < app->workspaceCount(); d++) {
        if (d == app->activeWorkspace())
            continue;
        if (first) {
            addSeparator();
            first = 0;
        }
        char s[50];
        sprintf(s, "%d. %-.32s", d + 1, app->workspaceName(d));

        YMenu *sub = 0;
        if (app->windowCount(d) > 0)
            sub = app->createWindowMenu(0, d);
        
        YMenu::YMenuItem *item = new YMenuItem(s,
                                               (d < 10) ? 0 : -1, 0,
                                               cmdActivateWorkspace,
                                               sub, (void *)d);
        add(item);
                                               
        /*if (app->windowCount(d) == 0) {
            addItem(s, (d < 10) ? 1 : -1, 0,
                    cmdActivateWorkspace, (void *)d);
        } else {
            if (sub)
                addSubmenu(s, (d < 10) ? 1 : -1, sub);
        }*/
    }
}

void YApplication::popupWindowListMenu(int x, int y) {
    windowListMenu->popup(0, 0, x, y, -1, -1,
                          YPopupWindow::pfCanFlipVertical |
                          YPopupWindow::pfCanFlipHorizontal |
                          YPopupWindow::pfPopupMenu);
}

void runProgram(char *str, char **args) {
    if (fork() == 0) {
        resetSignals();
        if (args)
            execvp(str, args);
        else
            execlp(str, str, 0);
        exit(1);
    }
}

void YApplication::handleCommand(WMCommand command, void *context) {
    switch (command) {
    case cmdActivateWindow:
        {
            YFrameWindow *f = app->topLayer();

            while (f) {
                if ((void *)f == context) {
                    activate(f);
                    return ;
                }
                f = f->nextLayer();
            }
            
        }
        break;
//    case cmdTerminal: runProgram("xterm"); break;
//    case cmdRefresh:  runProgram("xrefresh"); break;
//    case cmdKill:     runProgram("xkill"); break;
//    case cmdLock:     runProgram("xlock"); break;
    case cmdExec:     
#ifdef CONFIG_GUIEVENTS
    app->signalGuiEvent(geLaunchApp);
#endif
        runProgram(progCmds[int(context)], progArgs[int(context)]);
        break;
    case cmdCloseSession:
        wmCloseSession();
        break;
    case cmdExit:
        unmanageClients();
        exit(0);
        break;
    case cmdRestart:
#ifdef CONFIG_GUIEVENTS
    app->signalGuiEvent(geRestart);
#endif
        unmanageClients();
        XSync(display(), 0);
        resetSignals();
        if (execlp("icewm", "icewm", 0) != 0) {
            XBell(display(), 100);
            fprintf(stderr, "icewm: Could not restart, not on $PATH?\n");
            manageClients();
        }
        break;
    case cmdActivateWorkspace:
        activateWorkspace(int(context));
        break;
    default:
        return ;
    }
}

void YFrameControl::setFrame(YFrameWindow *newFrame) {
    fFrame = newFrame;
}

void initWorkspaces() {
    if (workspaceCount == 0)
        addWorkspace(" 0 ");

    for (int w = 0; w < workspaceCount; w++)
        workspaceAtoms[w] = XInternAtom(app->display(), workspaceNames[w], False);
    workspaceAtomAll = XInternAtom(app->display(), "all", False);

    XChangeProperty(app->display(),
                    app->root()->handle(),
                    _XA_WM_WORKSPACES,
                    XA_ATOM,
                    32, PropModeReplace,
                    (unsigned char *)workspaceAtoms, workspaceCount);

    Atom r_type;
    int r_format;
    unsigned long count;
    unsigned long bytes_remain;
    unsigned char *prop;
    unsigned long ws = 0;

    if (XGetWindowProperty(app->display(),
                           app->root()->handle(),
                           _XA_WM_WORKSPACE,
                           0, 1, False, XA_ATOM,
                           &r_type, &r_format,
                           &count, &bytes_remain, &prop) == Success)
    {
        if (r_type == XA_ATOM && r_format == 32 && count == 1) {
            Atom *atom = (Atom *)prop;

            // could be optimized, if we are certain they are in order ???
            for (int w = 0; w < workspaceCount; w++)
                if (atom[0] == workspaceAtoms[w])
                    ws = w;
        }
        XFree(prop);
    }
    app->activateWorkspace(ws);
}

int YApplication::minX() {
    return 0;
}
int YApplication::minY() {
    if (taskBar->y() <= 0 && taskBar->visible())
        return taskBar->height() + taskBar->y();
    else
        return 0;
}

int YApplication::maxX() {
    return root()->width();
}

int YApplication::maxY() {
    if (taskBar->y() <= 0 || !taskBar->visible())
        return root()->height();
    else
        return taskBar->y(); //root()->height() - taskBar->height();
}

void YApplication::relocateWindows(int dx, int dy) {
    YFrameWindow *f = app->topLayer();

    while (f) {
        f->setPosition(f->x() + dx, f->y() + dy);
        f = f->nextLayer();
    }
}

void YApplication::activateWorkspace(int workspace) {
    if (workspace != fActiveWorkspace) {
        if (taskBarShowWorkspaces) {
            if (fActiveWorkspace != -1)
                taskBar->workspaceButton(fActiveWorkspace)->setPressed(0);
        }
        fActiveWorkspace = workspace;
        if (taskBarShowWorkspaces) {
            taskBar->workspaceButton(fActiveWorkspace)->setPressed(1);
        }
        
        XChangeProperty(app->display(),
                        app->root()->handle(),
                        _XA_WM_WORKSPACE,
                        XA_ATOM,
                        32, PropModeReplace,
                        (unsigned char *)(workspaceAtoms + fActiveWorkspace), 1);

        
        //msg("activating: %d", workspace);
        YFrameWindow *w;
        
        w = bottomLayer();
        while (w) {
            if (w->visibleNow() && !w->visibleOn(workspace))
                w->workspaceHide();
            w = w->prevLayer();
        }

        w = topLayer();
        while (w) {
            if (!w->visibleNow() && w->visibleOn(workspace))
                w->workspaceShow();
            w = w->nextLayer();
        }
        taskBar->relayout();
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWorkspaceChange);
#endif
    }
}

void YApplication::wmCloseSession() {
    YFrameWindow *f = app->topLayer();

    /* shutdown started */
    while (f) {
        f->wmClose();
        f = f->nextLayer();
    }
}

char *findProgFile(const char *name) {
    char p[1024];
    char *h;

    h = getenv("HOME");
    if (h) {
        strcpy(p, h);
        strcat(p, "/.icewm/");
        strcat(p, name);

        if (access(p, R_OK) == 0)
            return strdup(p);
    }

    strcpy(p, ETC_PATH);
    strcat(p, "/");
    strcat(p, name);

    if (access(p, R_OK) == 0)
        return strdup(p);

    strcpy(p, REDIR_ROOT(xpmPath));
    strcat(p, "/");
    strcat(p, name);

    if (access(p, R_OK) == 0)
        return strdup(p);

    return 0;
}

#ifdef CONFIG_GUIEVENTS
void YApplication::signalGuiEvent(GUIEvent ge) {
    static Atom GUIEventAtom = None;
    unsigned char num = (unsigned char)ge;
    
    if (GUIEventAtom == None)
        GUIEventAtom = XInternAtom(app->display(), XA_GUI_EVENT_NAME, False);
    XChangeProperty(app->display(), app->root()->handle(),
                    GUIEventAtom, GUIEventAtom, 8, PropModeReplace,
                    &num, 1);
}
#endif

int main(int argc, char *argv[]) {
    char *displayName = 0;
#ifndef NO_CONFIGURE
    char *configFile = 0;
#endif

    for (int i = 1; i < argc; i++) {
        if (argv[i][0] == '-') {
            if (strcmp(argv[i], "-display") == 0) {
                displayName = argv[++i];
#ifdef DEBUG
            } else if (strcmp(argv[i], "-debug") == 0) {
                debug = 1;
            } else if (strcmp(argv[i], "-debug_z") == 0) {
                debug_z = 1;
#endif
            }
#ifndef NO_CONFIGURE
            else if (strcmp(argv[i], "-c") == 0)
                configFile = argv[++i];
            else if (strcmp(argv[i], "-n") == 0)
                configurationLoaded = 1;
#endif
        }
    }
#ifndef NO_CONFIGURE
    if (!configurationLoaded) {
        if (configFile == 0)
            configFile = findProgFile("preferences");
        if (configFile)
            loadConfiguration(configFile);
        if (themeName != 0) {
            char theme[1024];
            
            char *themePath;

            strcpy(theme, "themes/");
            strcat(theme, themeName);

            themePath = findProgFile(theme);
            if (themePath) 
                loadConfiguration(themePath);
        }
    }
#endif

#ifndef NO_CONFIGURE_MENUS
    if (menuFile == 0)
        menuFile = findProgFile("menu");
#endif

#ifndef NO_WINDOW_OPTIONS
    if (winOptFile == 0)
        winOptFile = findProgFile("winoptions");
#endif

    char disp[256] = "DISPLAY=";
    
    if (displayName) {
        strcat(disp, displayName);
        putenv(disp);
    }


    setSignals();
    phase = phaseStartup;
    YApplication app(displayName);

    initWorkspaces();

#ifndef NO_WINDOW_OPTIONS
    if (winOptFile)
        loadWinOptions(winOptFile);
#endif
#ifndef NO_CONFIGURE_MENUS
    if (menuFile)
        loadMenus(menuFile);
#else
#if 0
    if (progCount == 0) {
        addProgram("XTerm", "xterm", 0);
        addProgram(0, 0, 0);
        addProgram("XLock", "xlock", 0);
        addProgram("XRefresh", "xrefresh", 0);
        addProgram("XKill", "xkill", 0);
    }
#endif
#endif

    rootMenu->addSeparator();
    YMenu *close = new YMenu(app.root());
    close->addItem("All windows", 0, "", cmdCloseSession);
    close->addSeparator();
    close->addItem("Exit Window Manager", 1, "", cmdExit);
    close->addItem("Restart Window Manager", 0, "", cmdRestart);
    rootMenu->addSubmenu("Windows", 0, windowListMenu);
    rootMenu->addSubmenu("Close", 0, close);

#ifdef CONFIG_GUIEVENTS
    app.signalGuiEvent(geStartup);
#endif
    app.manageClients();
    phase = phaseRunning;
    int rc = app.mainLoop();
#ifdef CONFIG_GUIEVENTS
    app.signalGuiEvent(geShutdown);
#endif
    app.unmanageClients();
    return rc;
}
