/*
 * status.c: handles the status line updating, etc for IRCII 
 *
 * Written By Michael Sandrof
 *
 * Copyright(c) 1990 
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */


#include "irc.h"

#include "ircterm.h"
#include "status.h"
#include "server.h"
#include "vars.h"
#include "hook.h"
#include "input.h"
#include "commands.h"
#include "window.h"
#include "screen.h"
#include "mail.h"
#include "output.h"
#include "names.h"
#include "ircaux.h"
#include "misc.h"
#include "hash2.h"
#include "cset.h"
#ifdef TRANSLATE
#include "translat.h"
#endif

#define MY_BUFFER 120

extern char *DCC_get_current_transfer _((void));
extern char *ltoa _((long));
extern	long	oper_kills;
extern	long	nick_collisions;

static	char	*convert_format _((Window *, char *, int));
static	char	*status_nickname _((Window *));
static	char	*status_query_nick _((Window *));
static	char	*status_right_justify _((Window *));
static	char	*status_chanop _((Window *));
static	char	*status_channel _((Window *));
static	char	*status_server _((Window *));
static	char	*status_mode _((Window *));
static	char	*status_umode _((Window *));
static	char	*status_insert_mode _((Window *));
static	char	*status_overwrite_mode _((Window *));
static	char	*status_away _((Window *));
static	char	*status_oper _((Window *));
static	char	*status_users _((Window *));
static	char	*status_user0 _((Window *));
static	char	*status_user1 _((Window *));
static	char	*status_user2 _((Window *));
static	char	*status_user3 _((Window *));
static	char	*status_user4 _((Window *));
static	char	*status_user5 _((Window *));
static	char	*status_user6 _((Window *));
static	char	*status_user7 _((Window *));
static	char	*status_user8 _((Window *));
static	char	*status_user9 _((Window *));
static	char	*status_user10 _((Window *));
static	char	*status_user11 _((Window *));
static	char	*status_user12 _((Window *));
static	char	*status_user13 _((Window *));
static	char	*status_user14 _((Window *));
static	char	*status_user15 _((Window *));
static	char	*status_user16 _((Window *));
static	char	*status_user17 _((Window *));
static	char	*status_user18 _((Window *));
static	char	*status_user19 _((Window *));
static	char	*status_lag   _((Window *));
static	char	*status_dcc _((Window *));
static	char	*status_oper_kills _((Window *));
static	char	*status_msgcount _((Window *));
static	char	*status_hold _((Window *));
static	char	*status_version _((Window *));
static	char	*status_clock _((Window *));
static	char	*status_hold_lines _((Window *));
static	char	*status_window _((Window *));
static	char	*status_mail _((Window *));
static	char	*status_refnum _((Window *));
static	char	*status_topic _((Window *));
static	char	*status_null_function _((Window *));
static	char	*status_notify_windows _((Window *));
static	char	*convert_sub_format _((char *, char));
static	char	*status_voice _((Window *));
static	char	*status_cpu_saver_mode _((Window *));
static	char	*status_dcccount _((Window *));
static	char	*status_cdcccount _((Window *));
static	char	*status_position _((Window *));
static	char	*status_lastjoin _((Window *));
static	char	*status_userlist _((Window *));
static	char	*status_shitlist _((Window *));
static	char	*status_nethack _((Window *));
static	char	*status_aop _((Window *));
static	char	*status_bitch _((Window *));
static	char	*status_newserver _((Window *));
static	char	*status_scrollback _((Window *));

#define cparse(format, str) convert_output_format(fget_string_var(format), "%s", str)

char	*time_format = NULL;	/* XXX Bogus XXX */
char	*strftime_24hour = "%R";
char	*strftime_12hour = "%I:%M%p";
char	time_str[61];

/* update_clock: figures out the current time and returns it in a nice format */
char	*update_clock(int flag)
{
	extern	time_t	start_time;
	static	int		min = -1,
				hour = -1;
	static	time_t		last_minute = -1;
	time_t			idlet;
	static struct tm	time_val;
	struct	timeval		tv;
		time_t		hideous;
		
	get_time(&tv);
	hideous = tv.tv_sec;
#ifndef NO_CHEATING
	if (hideous / 60 > last_minute)
	{
		last_minute = hideous / 60;
		time_val = *localtime(&hideous);
	}
#else
	time_val = localtime(&tv.tv_sec);
#endif
	if (flag == RESET_TIME || time_val.tm_min != min || time_val.tm_hour != hour)
	{
		int server = from_server;
		from_server = primary_server;
		
		if (time_format)	/* XXXX Bogus XXXX */
			strftime(time_str, 60, time_format, &time_val);
		else if (get_int_var(CLOCK_24HOUR_VAR))
			strftime(time_str, 60, strftime_24hour, &time_val);
		else
			strftime(time_str, 60, strftime_12hour, &time_val);

		lower(time_str);
		if ((time_val.tm_min != min) || (time_val.tm_hour != hour))
		{
			hour = time_val.tm_hour;
			min = time_val.tm_min;
			do_hook(TIMER_LIST, "%02d:%02d", hour, min);
			if (min == 0)
				do_hook(TIMER_HOUR_LIST, "%02d:%02d", hour, min);
		}
		idlet = (hideous - idle_time) / 60L;
		if ((do_hook(IDLE_LIST, "%ld", idlet)) && get_int_var(AUTO_AWAY_TIME_VAR))
		{
			if (idlet >= get_int_var(AUTO_AWAY_TIME_VAR)/60)
			{
				auto_away(idlet);
				if (!((hideous - server_list[from_server].awaytime)/60 % 60))
					logmsg(LOG_CRAP, NULL, NULL, 4);
			}
		}
		if ((hideous - start_time) > 20)
			check_serverlag(get_server_itsname(from_server));
		from_server = server;
		return (time_str);
	}
	if (flag == GET_TIME)
		return(time_str);
	else
		return (NULL);
}

void reset_clock(Window *win, char *unused, int unused1)
{
	update_clock(RESET_TIME);
	update_all_status(win, NULL, 0);
}

/*
 * convert_sub_format: This is used to convert the formats of the
 * sub-portions of the status line to a format statement specially designed
 * for that sub-portions.  convert_sub_format looks for a single occurence of
 * %c (where c is passed to the function). When found, it is replaced by "%s"
 * for use is a sprintf.  All other occurences of % followed by any other
 * character are left unchanged.  Only the first occurence of %c is
 * converted, all subsequence occurences are left unchanged.  This routine
 * mallocs the returned string. 
 */
static	char	* convert_sub_format(char *format, char c)
{
	char	buffer[BIG_BUFFER_SIZE + 1];
	static	char	bletch[] = "%% ";
	char	*ptr = NULL;
	int	dont_got_it = 1;
	
	if (format == NULL)
		return (NULL);
	*buffer = (char) 0;
	memset(buffer, 0, sizeof(buffer));
	while (format)
	{
		if ((ptr = (char *) strchr(format, '%')) != NULL)
		{
			*ptr = (char) 0;
			strmcat(buffer, format, BIG_BUFFER_SIZE);
			*(ptr++) = '%';
			if ((*ptr == c)/* && dont_got_it*/)
			{
				dont_got_it = 0;
				strmcat(buffer, "%s", BIG_BUFFER_SIZE);
			}
			else
			{
				bletch[2] = *ptr;
				strmcat(buffer, bletch, BIG_BUFFER_SIZE);
			}
			ptr++;
		}
		else
			strmcat(buffer, format, BIG_BUFFER_SIZE);
		format = ptr;
	}
	malloc_strcpy(&ptr, buffer);
	return (ptr);
}

static	char	*convert_format(Window *win, char *format, int k)
{
	char	buffer[BIG_BUFFER_SIZE + 1];
	char	*ptr;
	int	*cp;
	
	memset(buffer, 0, sizeof(buffer));
	while (format)
	{
		if ((ptr = (char *) strchr(format, '%')) != NULL)
		{
			int	extended = 0;
			char	*nextspot = NULL;

			*ptr = (char) 0;
			strmcat(buffer, format, BIG_BUFFER_SIZE);
			*(ptr++) = '%';

			cp = &win->func_cnt[k];

			if (*cp < MAX_FUNCTIONS)
			{
				if (*ptr == '!') {
					extended = 1;
					/* get over the '!' char */
					ptr++;
					nextspot = ptr + 1;
				}
				switch (*(ptr++))
				{
				case '&':
					new_free(&win->wset->dcccount_format);
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_dcccount;
					win->wset->dcccount_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_DCCCOUNT_WSET), '&');
					break;
				case '|':
					new_free(&win->wset->cdcccount_format);
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_cdcccount;
					win->wset->cdcccount_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_CDCCCOUNT_WSET), '|');
					break;
					
				case '%':
					strmcat(buffer, "%", BIG_BUFFER_SIZE);
					break;
				case '>':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_right_justify;
					break;

				case '^':
					new_free(&win->wset->msgcount_format);
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->wset->msgcount_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_MSGCOUNT_WSET), '^');
					win->status_func[k][(*cp)++] = status_msgcount;
					break;

				case '@':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_chanop;
					break;

				case '+':
					new_free(&win->wset->mode_format);
					win->wset->mode_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_MODE_WSET), '+');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_mode;
					break;
				case '-':
					new_free(&win->wset->topic_format);
					win->wset->topic_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_TOPIC_WSET), '-');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_topic;
					break;
				case '#':
					new_free(&win->wset->umode_format);
					win->wset->umode_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_UMODE_WSET), '#');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_umode;
					break;
				case '=':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_voice;
					break;
				case '*':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_oper;
					break;

				case 'A':
					new_free(&win->wset->away_format);
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->wset->away_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_AWAY_WSET), 'A');
					win->status_func[k][(*cp)++] = status_away;
					break;
				case 'B':
					new_free(&win->wset->hold_lines_format);
					win->wset->hold_lines_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_HOLD_LINES_WSET), 'B');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_hold_lines;
					break;
				case 'C':
					new_free(&win->wset->channel_format);
					win->wset->channel_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_CHANNEL_WSET), 'C');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_channel;
					break;
				case 'D':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_dcc;
					break;
				case 'E':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_scrollback;
					break;

				case 'f':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_shitlist;
					break;
				case 'h':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_nethack;
					break;
				case 'a':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_aop;
					break;
				case 'b':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_bitch;
					break;
				case 'l':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_lastjoin;
					break;
				case 's':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_newserver;
					break;
				
				case 'F':
					new_free(&win->wset->notify_format);
					win->wset->notify_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_NOTIFY_WSET), 'F');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_notify_windows;
					break;
				case 'G':
					break;
				case 'H':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_hold;
					break;
				case 'I':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_insert_mode;
					break;
				case 'J':
					new_free(&win->wset->cpu_saver_format);
					win->wset->cpu_saver_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_CPU_SAVER_WSET), 'J');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_cpu_saver_mode;
					break; 
				case 'K':
					new_free(&win->wset->status_oper_kills_format);
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->wset->status_oper_kills_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_OPER_KILLS_WSET), 'K');
					win->status_func[k][(*cp)++] = status_oper_kills;
					break; 
				case 'L':
					new_free(&win->wset->status_lag_format);
					win->wset->status_lag_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_LAG_WSET), 'L');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_lag;
					break; 
				case 'M':
					new_free(&win->wset->mail_format);
					win->wset->mail_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_MAIL_WSET), 'M');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_mail;
					break;
				case 'N':
					new_free(&win->wset->nick_format);
					win->wset->nick_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_NICKNAME_WSET), 'N');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_nickname;
					break;
				case 'O':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_overwrite_mode;
					break;
				case 'P':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_position;
					break;
				case 'Q':
					new_free(&win->wset->query_format);
					win->wset->query_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_QUERY_WSET), 'Q');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_query_nick;
					break;
				case 'R':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_refnum;
					break;
				case 'S':
					new_free(&win->wset->server_format);
					win->wset->server_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_SERVER_WSET), 'S');
					strmcat(buffer,"%s",BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_server;
					break;
				case 'T':
					new_free(&win->wset->clock_format);
					win->wset->clock_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_CLOCK_WSET), 'T');
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_clock;
					break;
				case 'U':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user0;
					break;
				case 'u':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_userlist;
					break;
				case 'V':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_version;
					break;
				case 'W':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_window;
					break;
				case 'X':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user1;
					break;
				case 'Y':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user2;
					break;
				case 'Z':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user3;
					break;
/* Replaced status format % chars with 1-9 instead of XYZ CDE */
				case '1':
					if (extended) 
					{
						switch (*ptr) 
						{
							case '0':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user10;
								break;
							case '1':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user11;
								break;
							case '2':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user12;
								break;
							case '3':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user13;
								break;
							case '4':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user14;
								break;
							case '5':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user15;
								break;
							case '6':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user16;
								break;
							case '7':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user17;
								break;
							case '8':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user18;
								break;
							case '9':
								strmcat(buffer, "%s", BIG_BUFFER_SIZE);
								win->status_func[k][(*cp)++] = status_user19;
								break;
							default:
								/*  empty  */
								break;
						}
						ptr++;
						extended = 0;
					}
					else 
					{
						strmcat(buffer, "%s", BIG_BUFFER_SIZE);
						win->status_func[k][(*cp)++] = status_user1;
					}
					break;
				case '2':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user2;
					break;
				case '3':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user3;
					break;
				case '4':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user4;
					break;
				case '5':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user5;
					break;
				case '6':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user6;
					break;
				case '7':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user7;
					break;
				case '8':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user8;
					break;
				case '9':
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
					win->status_func[k][(*cp)++] = status_user9;
					break;
				}
			}
			else
				ptr++;
			if (extended)
			{
				new_free(&win->wset->status_users_format);
				win->wset->status_users_format = convert_sub_format(get_wset_string_var(win->wset, STATUS_USERS_WSET), '!');
				strmcat(buffer, "%s", BIG_BUFFER_SIZE);
				win->status_func[k][(*cp)++] = status_users;
	
			}
		}
		else
			strmcat(buffer, format, BIG_BUFFER_SIZE);
		format = ptr;
	}
	return m_strdup(buffer);
}

char    *alias_special_char _((char **, char *, char *, char *, int *));

extern char *expand_alias(char *, char *, int *, char **);

void fix_status_buffer(Window *window, char *buffer, int ansi, int in_status)
{
	unsigned char	rhs_buffer[3*BIG_BUFFER_SIZE + 1];
	char	fillchar;
	int	in_rhs = 0,
		ch_lhs = 0, ch_rhs = 0,
		pr_lhs = 0, pr_rhs = 0,
		start_rhs = -1,
		i;
	int len = 0;

	if (get_int_var(STATUS_DOES_EXPANDOS_VAR))
	{
		int  af = 0;
		char *stuff;

		stuff = expand_alias(buffer, empty_string, &af, NULL);
		strmcpy(buffer, stuff, BIG_BUFFER_SIZE);
		new_free(&stuff);
	}
	
	if (ansi)
		len = count_ansi(buffer, -1);
	else
		strcpy(buffer, stripansicodes(buffer));
	/*
	 * Count out the characters.
	 * Walk the entire string, looking for nonprintable
	 * characters.  We count all the printable characters
	 * on both sides of the %> tag.
	 */
	for (i = 0; buffer[i]; i++)
	{
		/*
		 * The FIRST such %> tag is used.
		 * Using multiple %>s is bogus.
		 */
		if (buffer[i] == '\f' && start_rhs == -1)
		{
			start_rhs = i;
			in_rhs = 1;
		}
		/*
		 * Is it NOT a printable character?
		 */
		else if (buffer[i] == REV_TOG ||
			 buffer[i] == UND_TOG ||
			 buffer[i] == ALL_OFF ||
			 buffer[i] == BOLD_TOG ||
			 (ansi && vt100_decode(buffer[i])))
		{
			if (!in_rhs)
				ch_lhs++;
			else
				ch_rhs++;
		}
		/*
		 * So it is a printable character.
		 */
		else
		{
			if (!in_rhs)
				ch_lhs++, pr_lhs++;
			else
				ch_rhs++, pr_rhs++;
		}

		/*
		 * Dont allow more than CO printable characters
		 */
		if (pr_lhs + pr_rhs >= window->screen->co)
		{
			buffer[i] = 0;
			break;
		}
	}
	/* What will we be filling with? */
	if (get_int_var(STATUS_NO_REPEAT_VAR))
		fillchar = ' ';
	else
		fillchar = buffer[ch_lhs - 1];

	/*
	 * Now if we have a rhs, then we have to adjust it.
	 */
	if (start_rhs != -1)
	{
		strcpy(rhs_buffer, buffer + start_rhs + 1);
		if (get_int_var(STATUS_NO_REPEAT_VAR))
		{
			if (pr_lhs + pr_rhs < window->screen->co)
			{
				sprintf(buffer + start_rhs, "%*s%s",
					window->screen->co - 1 - pr_lhs - pr_rhs, 
					empty_string,
					rhs_buffer);
			} 
			else
				strcpy(buffer+start_rhs, rhs_buffer);
		}
		else
		{
			char c;
			if (pr_lhs + pr_rhs < window->screen->co)
			{
				if ((start_rhs - 1) >= 0)
					c = buffer[start_rhs - 1];
				else
					c = ' ';	
				/*
				 * now we put it back.
				 */
				strcpy(buffer + start_rhs, strfill(fillchar, window->screen->co - 1 - pr_rhs - pr_lhs)); 
				strcat(buffer, rhs_buffer);
			}
			else
				strcpy(buffer+start_rhs, rhs_buffer);
		}
	}

	/*
	 * No rhs?  If the user wants us to pad it out, do so.
	 */
	else if (get_int_var(FULL_STATUS_LINE_VAR) && in_status)
	{
		int chars = window->screen->co - pr_lhs - 1;
		int count;
		for (count = 0; count <= chars; count++)
			buffer[ch_lhs + count - 1] = fillchar;
		buffer[ch_lhs + chars] = 0;
	}

}

char	*stat_convert_format(Window *win, char *form)
{
	unsigned char	buffer[2*BIG_BUFFER_SIZE + 1];
	char	*ptr = form;
	int extended = 0;
	char *(*func_ptr) _((Window *));

	if (!form || !*form)
		return m_strdup(empty_string);	
	*buffer = 0;
	while (*ptr)
	{
		extended = 0;
		func_ptr = NULL;
		if (*ptr == '%')
		{
			char	*nextspot = NULL;

			ptr++;
			if (*ptr == '!') 
			{
				extended = 1;
				/* get over the '!' char */
				if (*(ptr+1) && isdigit(*(ptr+1)))
					ptr++;
				nextspot = ptr + 1;
			}
			switch (*(ptr++))
			{


				case 'u':
					func_ptr = status_userlist;
					break;
				case 'f':
					func_ptr = status_shitlist;
					break;
				case 'h':
					func_ptr = status_nethack;
					break;
				case 'a':
					func_ptr = status_aop;
					break;
				case 'b':
					func_ptr = status_bitch;
					break;
				case 'l':
					func_ptr = status_lastjoin;
					break;
				case 's':
					func_ptr = status_newserver;
					break;
				case '&':
					func_ptr = status_dcccount;
					break;
				case '|':
					func_ptr = status_cdcccount;
					break;
					
				case '%':
					strmcat(buffer, "%", BIG_BUFFER_SIZE);
					break;
				case '>':
					func_ptr = status_right_justify;
					break;

				case '^':
					func_ptr = status_msgcount;
					break;
				case '@':
					func_ptr = status_chanop;
					break;

				case '+':
					func_ptr = status_mode;
					break;
				case '-':
					func_ptr = status_topic;
					break;
				case '#':
					func_ptr = status_umode;
					break;
				case '=':
					func_ptr = status_voice;
					break;
				case '*':
					func_ptr = status_oper;
					break;

				case 'A':
					func_ptr = status_away;
					break;
				case 'B':
					func_ptr = status_hold_lines;
					break;
				case 'C':
					func_ptr = status_channel;
					break;
				case 'D':
					func_ptr = status_dcc;
					break;
				case 'E':
					func_ptr = status_scrollback;
					break;
				case 'F':
					func_ptr = status_notify_windows;
					break;
				case 'G':
					break;
				case 'H':
					func_ptr = status_hold;
					break;
				case 'I':
					func_ptr = status_insert_mode;
					break;
				case 'J':
					break; 
				case 'K':
					func_ptr = status_oper_kills;
					break; 
				case 'L':
					func_ptr = status_lag;
					break; 
				case 'M':
					func_ptr = status_mail;
					break;
				case 'N':
					func_ptr = status_nickname;
					break;
				case 'O':
					func_ptr = status_overwrite_mode;
					break;
				case 'P':
					func_ptr = status_position;
					break;
				case 'Q':
					func_ptr = status_query_nick;
					break;
				case 'R':
					func_ptr = status_refnum;
					break;
				case 'S':
					func_ptr = status_server;
					break;
				case 'T':
					func_ptr = status_clock;
					break;
				case 'U':
					func_ptr = status_user0;
					break;
				case 'V':
					func_ptr = status_version;
					break;
				case 'W':
					func_ptr = status_window;
					break;
				case 'X':
					func_ptr = status_user1;
					break;
				case 'Y':
					func_ptr = status_user2;
					break;
				case 'Z':
					func_ptr = status_user3;
					break;
/* Replaced status format % chars with 1-9 instead of XYZ CDE */
				case '1':
					if (extended) 
					{
						switch (*ptr) 
						{
							case '0':
								func_ptr = status_user10;
								break;
							case '1':
								func_ptr = status_user11;
								break;
							case '2':
								func_ptr = status_user12;
								break;
							case '3':
								func_ptr = status_user13;
								break;
							case '4':
								func_ptr = status_user14;
								break;
							case '5':
								func_ptr = status_user15;
								break;
							case '6':
								func_ptr = status_user16;
								break;
							case '7':
								func_ptr = status_user17;
								break;
							case '8':
								func_ptr = status_user18;
								break;
							case '9':
								func_ptr = status_user19;
								break;
							default:
								/*  empty  */
								break;
						}
						if (func_ptr)
							ptr++;
						else
							ptr--;
						extended = 0;
					}
					else 
						func_ptr = status_user1;
					break;
				case '2':
					func_ptr = status_user2;
					break;
				case '3':
					func_ptr = status_user3;
					break;
				case '4':
					func_ptr = status_user4;
					break;
				case '5':
					func_ptr = status_user5;
					break;
				case '6':
					func_ptr = status_user6;
					break;
				case '7':
					func_ptr = status_user7;
					break;
				case '8':
					func_ptr = status_user8;
					break;
				case '9':
					func_ptr = status_user9;
					break;
			}
			if (extended)
				func_ptr = status_users;
			if (func_ptr)
				strmcat(buffer, (func_ptr)(win), BIG_BUFFER_SIZE);
		}
		else
		{
			strmccat(buffer, *ptr, BIG_BUFFER_SIZE);
			ptr++;
		}
	}
	fix_status_buffer(win, buffer, get_int_var(DISPLAY_ANSI_VAR), 0);
	return m_strdup(buffer);
}

void build_status(Window *win, char *format, int unused)
{
	int	i, k;
	char	*f = NULL;
	if (!win)
		return;
	for (k = 0; k < 3; k++) 
	{
		if (!win->wset)
			continue;
		new_free(&(win->wset->status_format[k]));
		win->func_cnt[k] = 0;
		for (i = 0; i < MAX_FUNCTIONS; i++)
			win->status_func[k][i] = status_null_function;
		switch(k)
		{
			case 0:
				f = win->wset->format_status[1];
				break;
			case 1:
				f = win->wset->format_status[2];
				break;
			case 2:
				f = win->wset->format_status[3];
				break;
		}			

		if (f)
			win->wset->status_format[k] = convert_format(win, f, k);

		for (i = win->func_cnt[k]; i < MAX_FUNCTIONS; i++)
			win->status_func[k][i] = status_null_function;
	}
	update_all_status(win, NULL, 0);
}

void make_status(Window *win)
{
	unsigned char	buffer[3*BIG_BUFFER_SIZE + 1];
	char		*func_value[MAX_FUNCTIONS+10] = {NULL};
	unsigned char	**output;
	
	int len = 1,
	status_line,
	ansi = get_int_var(DISPLAY_ANSI_VAR);

	/* The second status line is only displayed in the bottom window
	 * and should always be displayed, no matter what SHOW_STATUS_ALL
	 * is set to - krys
	 */
	for (status_line = 0 ; status_line < 1+win->double_status + win->menu.lines; status_line++)
	{
		int	line = status_line,
			i;

		if (!win->wset || !win->wset->status_format[line])
			continue;
			
		for (i = 0; i < MAX_FUNCTIONS; i++)
			func_value[i] = (win->status_func[line][i]) (win);
		len = 1;
		
		if (get_int_var(REVERSE_STATUS_VAR))
			buffer[0] = get_int_var(REVERSE_STATUS_VAR) ? REV_TOG : ' ';
		else
			len = 0;
                                        
		snprintf((char *) buffer+len, BIG_BUFFER_SIZE* 3, 
			win->wset->status_format[line],
			func_value[0], func_value[1], func_value[2],
			func_value[3], func_value[4], func_value[5],
			func_value[6], func_value[7], func_value[8],
			func_value[9], func_value[10], func_value[11],
			func_value[12], func_value[13], func_value[14],
			func_value[15], func_value[16], func_value[17],
			func_value[18], func_value[19], func_value[20],
			func_value[21], func_value[22], func_value[23],
			func_value[24], func_value[25], func_value[26],
			func_value[27], func_value[28], func_value[29],
			func_value[30], func_value[31],func_value[32],
			func_value[33], func_value[34],func_value[35],
			func_value[36], func_value[37],func_value[38]);

			/*  Patched 26-Mar-93 by Aiken
			 *  make_window now right-justifies everything 
			 *  after a %>
			 *  it's also more efficient.
			 */

		fix_status_buffer(win, buffer, ansi, 1);
		output = split_up_line(buffer, win->screen->co+1);
		
		if (!win->wset->status_line[status_line] ||
			strcmp(output[0], win->wset->status_line[status_line]))

		{
			malloc_strcpy(&win->wset->status_line[status_line], output[0]);
			if (!dumb_mode)
			{
				unsigned char *st = NULL;
				output_screen = win->screen;
				if (ansi)
					st = cparse((line==3)?FORMAT_STATUS3_FSET:(line==2)?FORMAT_STATUS2_FSET:(line==1)?FORMAT_STATUS1_FSET:FORMAT_STATUS_FSET, output[0]);

				if (win->menu.lines && (line == win->double_status+win->menu.lines) && win->status_split)
					term_move_cursor(0,win->top);
				else if (win->menu.lines && !win->status_split)
					term_move_cursor(0,win->bottom+status_line-win->menu.lines);
				else
					term_move_cursor(0,win->bottom+status_line);

				output_line(st? st: output[0]);

				cursor_in_display();
				term_bold_off();
			} else
				do_hook(STATUS_UPDATE_LIST, "%d %d %s", win->refnum, status_line, win->wset->status_line[status_line]);
		}
	}
	cursor_to_input();
}

static	char	*status_nickname(Window *window)
{
static char my_buffer[MY_BUFFER+1];

	if ((connected_to_server) && !get_int_var(SHOW_STATUS_ALL_VAR)
	    && (!window->current_channel) &&
	    (window->screen->current_window != window))
		return empty_string;
	snprintf(my_buffer, MY_BUFFER, window->wset->nick_format, get_server_nickname(window->server));
	return my_buffer;
}

static	char	*status_server(Window *window)
{
	char	*name;
static	char	my_buffer[MY_BUFFER+1];
	if (connected_to_server)
	{
		if (window->server != -1)
		{
			if (window->wset->server_format)
			{
				if (!(name = get_server_itsname(window->server)))
					name = get_server_name(window->server);
				snprintf(my_buffer, MY_BUFFER, window->wset->server_format, name);
			}
			else
				return empty_string;
		}
		else
			strcpy(my_buffer, " No Server");
	}
	else
		return empty_string;
	return my_buffer;
}

static	char	*status_query_nick(Window *window)
{
static	char my_buffer[BIG_BUFFER_SIZE+1];

	if (window->query_nick && window->wset->query_format)
	{
		snprintf(my_buffer, BIG_BUFFER_SIZE, window->wset->query_format, window->query_nick);
		return my_buffer;
	}
	else
		return empty_string;
}

static	char	*status_right_justify(Window *window)
{
static	char	my_buffer[] = "\f";

	return my_buffer;
}

static	char	*status_notify_windows(Window *window)
{
	int	doneone = 0;
	char	buf2[MY_BUFFER+2];
static	char	my_buffer[MY_BUFFER+1];
	if (get_int_var(SHOW_STATUS_ALL_VAR) || window == window->screen->current_window)
	{
		*buf2='\0';
		window = NULL;
		while ((traverse_all_windows(&window)))
		{
			if (window->miscflags & WINDOW_NOTIFIED)
			{
				if (doneone++)
					strmcat(buf2, ",", MY_BUFFER);
				strmcat(buf2, ltoa(window->refnum), MY_BUFFER);
			}
		}
	}
	if (doneone && window->wset->notify_format)
	{
		snprintf(my_buffer, MY_BUFFER, window->wset->notify_format, buf2);
		return (my_buffer);
	}
	return empty_string;
}

static	char	*status_clock(Window *window)
{
static	char	my_buf[MY_BUFFER+1];

	if ((get_int_var(CLOCK_VAR) && window->wset->clock_format)  &&
	    (get_int_var(SHOW_STATUS_ALL_VAR) ||
	    (window == window->screen->current_window)))
		snprintf(my_buf, MY_BUFFER, window->wset->clock_format, update_clock(GET_TIME));
	else
		return empty_string;
	return my_buf;
}

static	char	*status_mode(Window *window)
{
char		*mode = NULL;
static  char	my_buffer[MY_BUFFER+1];
	if (window->current_channel)
	{
		mode = get_channel_mode(window->current_channel,window->server);
		if (mode && *mode && window->wset->mode_format)
			snprintf(my_buffer, MY_BUFFER, window->wset->mode_format, mode);
		else
			return empty_string;
	} else
		return empty_string;
	return my_buffer;
}


static	char	*status_umode(Window *window)
{
	char	localbuf[MY_BUFFER+1];
static char my_buffer[MY_BUFFER+1];

	if ((connected_to_server) && !get_int_var(SHOW_STATUS_ALL_VAR)
	    && (window->screen->current_window != window))
		*localbuf = 0;
	else {
		if (window->server > -1)
			strmcpy(localbuf, get_umode(window->server), MY_BUFFER);
		else
			*localbuf = 0;
	}
	
	if (*localbuf && window->wset->umode_format)
		snprintf(my_buffer, MY_BUFFER, window->wset->umode_format, localbuf);
	else
		return empty_string;
	return my_buffer;
}

static	char	*status_chanop(Window *window)
{
	char	*text;
	if (window->current_channel && get_channel_oper(window->current_channel, window->server) &&   
			(text = get_wset_string_var(window->wset, STATUS_CHANOP_WSET)))
		return (text);
	else
		return (empty_string);
}


static	char	*status_hold_lines(Window *window)
{
	int	num;
static  char	my_buffer[MY_BUFFER+1];
	
	if ((num = (window->lines_held /10) * 10))
	{
		snprintf(my_buffer, MY_BUFFER, window->wset->hold_lines_format, ltoa(num));
		return(my_buffer);
	}
	return (empty_string);
}

static	char	*status_msgcount(Window *window)
{
static  char	my_buffer[MY_BUFFER+1];

	if (get_int_var(MSGCOUNT_VAR) && window->wset->msgcount_format)
	{
		snprintf(my_buffer, MY_BUFFER, window->wset->msgcount_format, ltoa(get_int_var(MSGCOUNT_VAR)));
		return my_buffer;
	}
	return empty_string;
}

static	char	*status_channel(Window *window)
{
	char	channel[IRCD_BUFFER_SIZE + 1];
static	char	my_buffer[IRCD_BUFFER_SIZE + 1];

	if (window->current_channel/* && chan_is_connected(s, window->server)*/)
	{
extern const int MODE_PRIVATE, MODE_SECRET;
		int num;
		if (get_int_var(HIDE_PRIVATE_CHANNELS_VAR) &&
		    is_channel_mode(window->current_channel,
				MODE_PRIVATE | MODE_SECRET,
				window->server))
			strmcpy(channel, "*private*", IRCD_BUFFER_SIZE);
		else
			strmcpy(channel, window->current_channel, IRCD_BUFFER_SIZE);

		if ((num = get_int_var(CHANNEL_NAME_WIDTH_VAR)) &&
		    ((int) strlen(channel) > num))
			channel[num] = (char) 0;
		snprintf(my_buffer, IRCD_BUFFER_SIZE, window->wset->channel_format, channel);
		return my_buffer;
	}
	return empty_string;
}

static	char	*status_mail(Window *window)
{
	char	*number;
static	char	my_buffer[MY_BUFFER+1];

	if (window && (get_int_var(MAIL_VAR) && (number = check_mail()) && window->wset->mail_format) &&
	    (get_int_var(SHOW_STATUS_ALL_VAR) ||
	    (window == window->screen->current_window)))
	{
		snprintf(my_buffer, MY_BUFFER, window->wset->mail_format, number);
		return my_buffer;
	}
	return empty_string;
}

static	char	*status_insert_mode(Window *window)
{
char	*text;

	if (get_int_var(INSERT_MODE_VAR) && (get_int_var(SHOW_STATUS_ALL_VAR)
	    || (window->screen->current_window == window)))
		if ((text = get_string_var(STATUS_INSERT_VAR)))
			return text;
	return empty_string;
}

static	char	*status_overwrite_mode(Window *window)
{
char	*text;

	if (!get_int_var(INSERT_MODE_VAR) && (get_int_var(SHOW_STATUS_ALL_VAR)
	    || (window->screen->current_window == window)))
	{
	    if ((text = get_string_var(STATUS_OVERWRITE_VAR)))
		return text;
	}
	return empty_string;
}

static	char	*status_away(Window *window)
{
static char	my_buffer[MY_BUFFER+1];

	if (window && connected_to_server && !get_int_var(SHOW_STATUS_ALL_VAR)
	    && (window->screen->current_window != window))
		return empty_string;
	else if (window)
	{
		if (window->server != -1 && server_list[window->server].away)
		{
			snprintf(my_buffer, MY_BUFFER, window->wset->away_format, ltoa(get_int_var(MSGCOUNT_VAR)));
			return my_buffer;
		}
		else
			return empty_string;
	}
	return empty_string;
}

static	char	*status_user0(Window *window)
{
char	*text;

	if ((text = get_wset_string_var(window->wset, STATUS_USER_WSET)) &&
	    (get_int_var(SHOW_STATUS_ALL_VAR) ||
	    (window == window->screen->current_window)))
		return text;
	return empty_string;
}

static  char    *status_user1(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER1_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return text;
        return empty_string;
}

static  char    *status_user2(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER2_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return text;
	return empty_string;
}

static  char    *status_user3(Window *window)
{
char         *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER3_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return text;
       return empty_string;
}

static  char    *status_user4(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER4_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user5(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER5_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user6(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER6_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}


static  char    *status_user7(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER7_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}


static  char    *status_user8(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER8_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}


static  char    *status_user9(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER9_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user10(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER10_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user11(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER11_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user12(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER12_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user13(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER13_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user14(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER14_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user15(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER15_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user16(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER16_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user17(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER17_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user18(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER18_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}

static  char    *status_user19(Window *window)
{
char        *text;

        if ((text = get_wset_string_var(window->wset, STATUS_USER19_WSET)) &&
            (get_int_var(SHOW_STATUS_ALL_VAR) ||
            (window == window->screen->current_window)))
                return(text);
        return (empty_string);
}


static	char	*status_hold(Window *window)
{
char	*text;

	if (window->holding_something && (text = get_wset_string_var(window->wset, STATUS_HOLD_WSET)))
		return(text);
	return (empty_string);
}

static	char	*status_lag (Window *window)
{
static  char	my_buffer[MY_BUFFER+1];
	if ((window->server > -1) && window->wset->status_lag_format)
	{
		if (get_server_lag(window->server) > -1)
		{
			char p[40];
			sprintf(p, "%2d",get_server_lag(window->server)); 
			snprintf(my_buffer,MY_BUFFER, window->wset->status_lag_format, p);
		}
		else
			snprintf(my_buffer, MY_BUFFER, window->wset->status_lag_format, "??");
		return(my_buffer);
	}
	return empty_string;
}

static	char	*status_topic (Window *window)
{
static  char	my_buffer[MY_BUFFER+1];
char	tmp_buffer[MY_BUFFER+1];
ChannelList *chan;
	if (window && window->current_channel && window->wset->topic_format)
	{
		if ((chan = lookup_channel(window->current_channel, window->server, 0)))
		{
			strmcpy(tmp_buffer, chan->topic?chan->topic: "No Topic", MY_BUFFER-strlen(window->wset->topic_format));
			snprintf(my_buffer, MY_BUFFER, window->wset->topic_format, stripansicodes(tmp_buffer));
			return(my_buffer);
		}
	}
	return empty_string;
}

static	char	*status_oper(Window *window)
{
char	*text;

	if (get_server_operator(window->server) &&
			(text = get_string_var(STATUS_OPER_VAR)) &&
			(get_int_var(SHOW_STATUS_ALL_VAR) ||
			connected_to_server || 
			(window->screen->current_window == window)))
		return(text);
	return (empty_string);
}

static	char	*status_voice(Window *window)
{
char	*text;
	if (window->current_channel &&
	    get_channel_voice(window->current_channel, window->server) &&
	    !get_channel_oper(window->current_channel, window->server) &&
	    (text = get_wset_string_var(window->wset, STATUS_VOICE_WSET)))
		return text;
	return empty_string;
}

static	char	*status_window(Window *window)
{
char	*text;
	if ((number_of_windows_on_screen(window) > 1) && (window->screen->current_window == window) &&
	    (text = get_wset_string_var(window->wset, STATUS_WINDOW_WSET)))
		return(text);
	return (empty_string);
}

static	char	*status_refnum(Window *window)
{
static char my_buffer[40+1];
	strmcpy(my_buffer, window->name ? window->name : ltoa(window->refnum), 40);
	return (my_buffer);
}

static	char	*status_version(Window *window)
{
	if ((connected_to_server) && !get_int_var(SHOW_STATUS_ALL_VAR)
	    && (window->screen->current_window != window))
		return(empty_string);
	return ((char *)irc_version);
}

static	char	* status_oper_kills (Window *window)
{
static char my_buffer[MY_BUFFER+1];
	if (window->wset->status_oper_kills_format)
	{
		char tmp[30];
		snprintf(tmp, 29, "%d", nick_collisions); 
		snprintf(my_buffer, MY_BUFFER, window->wset->status_oper_kills_format, tmp, ltoa(oper_kills));
		return my_buffer;
	}
	return empty_string;	
}

static char *status_dcccount (Window *window)
{
extern int get_count_stat, send_count_stat;
static char my_buffer[MY_BUFFER+1];
	if (window->wset->dcccount_format)
	{
		char tmp[30];
		strcpy(tmp, ltoa(send_count_stat));
		snprintf(my_buffer, MY_BUFFER, window->wset->dcccount_format, ltoa(get_count_stat), tmp);
		return my_buffer;
	}
	return empty_string;
}

static char *status_cdcccount (Window *window)
{
#ifdef WANT_CDCC
extern int cdcc_numpacks, send_numpacks;
static char my_buffer[MY_BUFFER+1];
	if (window->wset->cdcccount_format)
	{
		char tmp[30];
		strcpy(tmp, ltoa(cdcc_numpacks));
		snprintf(my_buffer, MY_BUFFER, window->wset->cdcccount_format, ltoa(send_numpacks), tmp);
		return my_buffer;
	}
#endif
	return empty_string;
}

static char *status_cpu_saver_mode (Window *window)
{
	static char my_buffer[81];

	if (cpu_saver && window->wset->cpu_saver_format)
	{
		snprintf(my_buffer, 80, window->wset->cpu_saver_format, "cpu");
		return my_buffer;
	}

	return empty_string;
}

static	char	* status_users (Window *window)
{
static char my_buffer[200];
ChannelList *chan;
NickList *nick;
int serv = window->server;
	if (window->server != -1 && window->wset->status_users_format)
	{
		if ((chan = prepare_command(&serv, NULL, 3)))
		{
			int ops = 0, nonops = 0, voice = 0, ircop = 0, friends = 0;
			char buff[40], buff1[40], buff2[40], buff3[40], buff4[40];
			 
			for (nick = next_nicklist(chan, NULL); nick; nick = next_nicklist(chan, nick))
			{
				if (nick->chanop)
					ops++;
				else
					nonops++;
				if (nick->voice)
					voice++;
				if (nick->ircop)
					ircop++;
				if (nick->userlist)
					friends++;
			}
			strcpy(buff, ltoa(ops)); 
			strcpy(buff1, ltoa(nonops));
			strcpy(buff2,ltoa(ircop)); 
			strcpy(buff3, ltoa(voice));
			strcpy(buff4, ltoa(friends));
			snprintf(my_buffer, 199, window->wset->status_users_format, buff, buff1, buff2, buff3, buff4);
			return my_buffer;
		}
	}
	return empty_string;	
}

static	char	*status_null_function(Window *window)
{
	return (empty_string);
}


static char *status_dcc(Window *window)
{
	if (window->window_level & LOG_DCC || !window->window_level)
		return(DCC_get_current_transfer());
	return empty_string;
}

static char *status_position (Window *window)
{
	static char my_buffer[41];

	snprintf(my_buffer, 40, "(%d-%d)", window->lines_scrolled_back,
					window->distance_from_display);
	return my_buffer;
}
 
static	char	*status_userlist (Window *window)
{
ChannelList *chan;
int serv = window->server;
static char my_buffer[3];
	if (window->server != -1)
	{
		if ((chan = prepare_command(&serv, NULL, 3)))
		{
			my_buffer[0] = chan->csets->set_userlist ? 'U':'u';
			my_buffer[1] = 0;
			return my_buffer;
		}
	}
	return empty_string;
}

static	char	*status_shitlist (Window *window)
{
ChannelList *chan;
int serv = window->server;
static char my_buffer[3];
	if (window->server != -1)
	{
		if ((chan = prepare_command(&serv, NULL, 3)))
		{
			my_buffer[0] = chan->csets->set_shitlist ? 'S':'s';
			my_buffer[1] = 0;
			return my_buffer;
		}
	}
	return empty_string;
}

static	char	*status_nethack (Window *window)
{
ChannelList *chan;
int serv = window->server;
static char my_buffer[3];
	if (window->server != -1)
	{
		if ((chan = prepare_command(&serv, NULL, 3)))
		{
			my_buffer[0] = chan->csets->set_hacking ? 'H':'h';
			my_buffer[1] = 0;
			return my_buffer;
		}
	}
	return empty_string;
}

static	char	*status_aop (Window *window)
{
ChannelList *chan;
int serv = window->server;
static char my_buffer[3];
	if (window->server != -1)
	{
		if ((chan = prepare_command(&serv, NULL, 3)))
		{
			my_buffer[0] = chan->csets->set_aop ? 'A':'a';
			my_buffer[1] = 0;
			return my_buffer;
		}
	}
	return empty_string;
}

static	char	*status_bitch (Window *window)
{
ChannelList *chan;
int serv = window->server;
static char my_buffer[3];
	if (window->server != -1)
	{
		if ((chan = prepare_command(&serv, NULL, 3)))
		{
			my_buffer[0] = chan->csets->bitch_mode ? 'B':'b';
			my_buffer[1] = 0;
			return my_buffer;
		}
	}
	return empty_string;
}

static char *status_lastjoin (Window *window)
{
	if (window->server != -1)
	{
		if (joined_nick)
			return joined_nick;
	}
	return empty_string;
}

static char *status_newserver (Window *window)
{
extern char *ov_server(int);
	if (window->server != -1)	
		return ov_server(window->server);
	return empty_string;
}

static char *status_scrollback(Window *win)
{
char *stuff;
	if (win->scrollback_point &&
	    (stuff = get_wset_string_var(win->wset, STATUS_SCROLLBACK_WSET)))
		return stuff;
	else
		return empty_string;
}
