/* Copyright 1998 Yggdrasil Computing, Inc.
   Written by Adam J. Richter

   This file may be copied according to the terms and conditions of the
   GNU General Public License as published by the Free Software
   Foundation.
*/

#include <stdlib.h>
#include <stdio.h>

#include "resource.h"
#include "pnp.h"

#define ASSERT(x)	/* as nothing for now */

struct range_list {
    unsigned long start, end;	/* end = start+len */
    struct range_list *next;
};

static int
alloc_in_range_list (unsigned long start, unsigned long end,
		     struct range_list **list) {
    if (start == end) return 1;	/* adding empty range. */
    ASSERT(start < end);
    while (*list && start > (*list)->end) {
        list = &(*list)->next;
    }
    ASSERT(*list == NULL || start <= (*list)->end);
    if (!(*list) || end < (*list)->start) {
        /* There is a gap between us and next element, or no next element. */
        struct range_list *new = malloc(sizeof(struct range_list));
	if (!new) {
	    fprintf (stderr, "alloc_in_range_list: malloc failure.\n");
	    exit(1);
	}
	new->start = start;
	new->end = end;
	new->next = *list;
        *list = new;
	return 1;
    }
    ASSERT(start <= (*list)->end);
    if (end == (*list)->start) {
        /* We are contiguous with next element */
    	(*list)->start = start;
	return 1;
    }
    ASSERT(end > (*list)->start);
    ASSERT(start <= (*list)->end);
    if (start == (*list)->end) {
        /* Next element is contiguous with us. */
        if ((*list)->next && (*list)->next->start < end)
	    return 0; /* we overlap subsequent element */
        if ((*list)->next && (*list)->next->start == end) {
	    /* we exactly bridge to subsequent element. */
	    struct range_list *obselete = (*list)->next;
	    (*list)->end = obselete->end;
	    (*list)->next = obselete->next;
	    free(obselete);
	}
	else {
	    /* next element is contiguous with us and we do not reach
	       subsequent element. */
	    (*list)->end = end;
	}
	return 1;
    }
    ASSERT(end > (*list)->start);
    ASSERT(start < (*list)->end);
    return 0;			/* we overlap */
}

static void
dealloc_in_range_list (unsigned long start, unsigned long end,
		     struct range_list **list) {
    if (start == end) return;	/* deleting empty range. */
    ASSERT(start < end);
    while (*list && start >= (*list)->end) {
        list = &(*list)->next;
    }
    while (*list && end > (*list)->start) {
	/* we overlap with this element */
        ASSERT(start < (*list)->end);
	if((*list)->end > end) {
	    (*list)->start = end;
	    return;
	} else {
	    struct range_list *tmp = *list;
	    *list = (*list)->next;
	    free(tmp);
	}
    }
}

struct range_list *tag_range_lists[] = {NULL, NULL, NULL, NULL};

static int
tag_to_index (unsigned char tag) {
    switch (tag) {
        case IRQ_TAG: return 0;
        case DMA_TAG: return 1;
        case IOport_TAG:
        case FixedIO_TAG:
	    return 2;
        case MemRange_TAG:
        case Mem32Range_TAG:
        case FixedMem32Range_TAG:
	    return 3;
        default:		
	    return -1;		/* Unrecognized tag */
    }
}

int
tag_allocable (unsigned char tag) {
    return tag_to_index(tag) != -1;
}

int
allocate_resource(char tag, unsigned long start, unsigned long len) {
    return alloc_in_range_list(start, start + len,
			       &tag_range_lists[tag_to_index(tag)]);
}

void
deallocate_resource(char tag, unsigned long start, unsigned long len) {
    dealloc_in_range_list(start, start + len,
			  &tag_range_lists[tag_to_index(tag)]);
}

void
alloc_system_resources(void) {
    FILE *input;
    int interrupt, dma, io_start, io_end, mem_start;

    /* Avoid allocating DMA channels used by other devices in /proc. */
    if ((input = fopen("/proc/interrupts", "r")) != NULL) {
        fscanf(input, "%*[^\n]\n"); /* skip first line */
        while (fscanf (input, "%d:%*[^\n]\n", &interrupt) == 1) {
	    (void) allocate_resource(IRQ_TAG, interrupt, 1);
	}
        fclose(input);
    }
    if ((input = fopen("/proc/dma", "r")) != NULL) {
        while (fscanf (input, "%d:[^\n]\n", &dma) == 1) {
	    (void) allocate_resource(DMA_TAG, dma, 1);
	}
        fclose(input);
    }
    if ((input = fopen("/proc/ioports", "r")) != NULL) {
        while (fscanf (input, "%x-%x:[^\n]\n", &io_start, &io_end) == 2) {
	    (void) allocate_resource(IOport_TAG, io_start, io_end - io_start);
	}
        fclose(input);
    }

    if ((input = popen("lspci -n -v | egrep IRQ | awk '{print $NF}'", "r"))
	!= NULL) {
        while (fscanf (input, "%d\n", &interrupt) == 1) {
	    (void) allocate_resource(IRQ_TAG, interrupt, 1);
	}
        pclose(input);
    }
    if ((input = popen("lspci -n -v | egrep 'I/O ports at' | awk '{print $NF}'", "r"))
	!= NULL) {
        while (fscanf (input, "%x\n", &io_start) == 1) {
	    (void) allocate_resource(IOport_TAG, io_start, 1);
	}
        pclose(input);
    }
    if ((input = popen("lspci -n -v | egrep 'Memory at' | awk '{print $NF}'", "r"))
	!= NULL) {
        while (fscanf (input, "%x\n", &mem_start) == 1) {
	    (void) allocate_resource(MemRange_TAG, mem_start, 1);
	}
        pclose(input);
    }
}
