/*
 * $Id: zle_main.c,v 1.27 1995/06/21 05:32:24 coleman Exp coleman $
 *
 * zle_main.c - main routines for line editor
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1995 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */

/*
 * Note on output from zle (PWS 1995/05/23):
 *
 * All input and output from the editor should be to/from the file descriptor
 * `SHTTY' and FILE `shout'.  (Normally, the former is used for input
 * operations, reading one key at a time, and the latter for output
 * operations, flushing after each refresh()).  Thus fprintf(shout, ...),
 * putc(..., shout), etc., should be used for output within zle.
 *
 * However, the functions printbind() and printbinding() can be invoked from
 * the builtin bindkey as well as zle, in which case output should be to
 * stdout.  For this purpose, the static variable FILE *bindout exists, which
 * is set to stdout in bin_bindkey() and shout in zleread().
 */

#define ZLEGLOBALS
#define ZLE
#include "zsh.h"

static int emacs_cur_bindtab[256], eofchar, eofsent;
int viins_cur_bindtab[256], ungetok;	/* needed in zle_hist */

static Key cky;

/* set up terminal */

/**/
void
setterm(void)
{
    struct ttyinfo ti;

#if defined(CLOBBERS_TYPEAHEAD) && defined(FIONREAD)
    int val;

    ioctl(SHTTY, FIONREAD, (char *)&val);
    if (val)
	return;
#endif

/* sanitize the tty */
#ifdef HAS_TIO
    shttyinfo.tio.c_lflag |= ICANON | ECHO;
# ifdef FLUSHO
    shttyinfo.tio.c_lflag &= ~FLUSHO;
# endif
#else				/* not HAS_TIO */
    shttyinfo.sgttyb.sg_flags = (shttyinfo.sgttyb.sg_flags & ~CBREAK) | ECHO;
    shttyinfo.lmodes &= ~LFLUSHO;
#endif

    attachtty(mypgrp);
    ti = shttyinfo;
#ifdef HAS_TIO
    if (isset(NOFLOWCONTROL))
	ti.tio.c_iflag &= ~IXON;
    ti.tio.c_lflag &= ~(ICANON | ECHO
# ifdef FLUSHO
			| FLUSHO
# endif
	);
# ifdef TAB3
    ti.tio.c_oflag &= ~TAB3;
# else
#  ifdef OXTABS
    ti.tio.c_oflag &= ~OXTABS;
#  else
    ti.tio.c_oflag &= ~XTABS;
#  endif
# endif
    ti.tio.c_oflag |= ONLCR;
    ti.tio.c_cc[VQUIT] =
# ifdef VDISCARD
	ti.tio.c_cc[VDISCARD] =
# endif
# ifdef VSUSP
	ti.tio.c_cc[VSUSP] =
# endif
# ifdef VDSUSP
	ti.tio.c_cc[VDSUSP] =
# endif
# ifdef VSWTCH
	ti.tio.c_cc[VSWTCH] =
# endif
# ifdef VLNEXT
	ti.tio.c_cc[VLNEXT] =
# endif
	VDISABLEVAL;
# if defined(VSTART) && defined(VSTOP)
    if (isset(NOFLOWCONTROL))
	ti.tio.c_cc[VSTART] = ti.tio.c_cc[VSTOP] = VDISABLEVAL;
# endif
    eofchar = ti.tio.c_cc[VEOF];
    ti.tio.c_cc[VMIN] = 1;
    ti.tio.c_cc[VTIME] = 0;
    ti.tio.c_iflag |= (INLCR | ICRNL);
 /* this line exchanges \n and \r; it's changed back in getkey
	so that the net effect is no change at all inside the shell.
	This double swap is to allow typeahead in common cases, eg.

	% bindkey -s '^J' 'echo foo^M'
	% sleep 10
	echo foo<return>  <--- typed before sleep returns

	The shell sees \n instead of \r, since it was changed by the kernel
	while zsh wasn't looking. Then in getkey() \n is changed back to \r,
	and it sees "echo foo<accept line>", as expected. Without the double
	swap the shell would see "echo foo\n", which is translated to
	"echo fooecho foo<accept line>" because of the binding.
	Note that if you type <line-feed> during the sleep the shell just sees
	\n, which is translated to \r in getkey(), and you just get another
	prompt. For type-ahead to work in ALL cases you have to use
	stty inlcr.
	This workaround is due to Sven Wischnowsky <oberon@cs.tu-berlin.de>.

	Unfortunately it's IMPOSSIBLE to have a general solution if both
	<return> and <line-feed> are mapped to the same character. The shell
	could check if there is input and read it before setting it's own
	terminal modes but if we get a \n we don't know whether to keep it or
	change to \r :-(
	*/

#else				/* not HAS_TIO */
    ti.sgttyb.sg_flags = (ti.sgttyb.sg_flags | CBREAK) & ~ECHO & ~XTABS;
    ti.lmodes &= ~LFLUSHO;
    eofchar = ti.tchars.t_eofc;
    ti.tchars.t_quitc =
	ti.ltchars.t_suspc =
	ti.ltchars.t_flushc =
	ti.ltchars.t_dsuspc = ti.ltchars.t_lnextc = -1;
#endif

#if defined(TTY_NEEDS_DRAINING) && defined(TIOCOUTQ) && defined(HAVE_SELECT)
    if (baud) {			/**/
	int n = 0;

	while ((ioctl(SHTTY, TIOCOUTQ, (char *)&n) >= 0) && n) {
	    struct timeval tv;

	    tv.tv_sec = n / baud;
	    tv.tv_usec = ((n % baud) * 1000000) / baud;
	    select(0, NULL, NULL, NULL, &tv);
	}
    }
#endif

    settyinfo(&ti);
}

static char *kungetbuf;
static int kungetct, kungetsz;

/**/
void
ungetkey(int ch)
{
    if (kungetct == kungetsz)
	kungetbuf = realloc(kungetbuf, kungetsz *= 2);
    kungetbuf[kungetct++] = ch;
}

/**/
void
ungetkeys(char *s, int len)
{
    s += len;
    while (len--)
	ungetkey(*--s);
}

#if defined(pyr) && defined(HAVE_SELECT)
static int
breakread(int fd, char *buf, int n)
{
    fd_set f;

    FD_ZERO(&f);
    FD_SET(fd, &f);
    return (select(fd + 1, (SELECT_ARG_2_T) & f, NULL, NULL, NULL) == -1 ?
	    EOF : read(fd, buf, n));
}

# define read    breakread
#endif

/**/
int
getkey(int keytmout)
{
    char cc;
    unsigned int ret;
    long exp100ths;
    int die = 0, r, icnt = 0;
    int old_errno = errno;

#ifdef HAVE_SELECT
    fd_set foofd;

#else
# ifdef HAS_TIO
    struct ttyinfo ti;

# endif
#endif

    if (kungetct)
	ret = (int)(unsigned char)kungetbuf[--kungetct];
    else {
	if (keytmout) {
	    if (keytimeout > 500)
		exp100ths = 500;
	    else if (keytimeout > 0)
		exp100ths = keytimeout;
	    else
		exp100ths = 0;
#ifdef HAVE_SELECT
	    if (exp100ths) {
		struct timeval expire_tv;

		expire_tv.tv_sec = exp100ths / 100;
		expire_tv.tv_usec = (exp100ths % 100) * 10000L;
		FD_ZERO(&foofd);
		FD_SET(SHTTY, &foofd);
		if (select(SHTTY+1, (SELECT_ARG_2_T) & foofd,
			   NULL, NULL, &expire_tv) <= 0)
		    return EOF;
	    }
#else
# ifdef HAS_TIO
	    ti = shttyinfo;
	    ti.tio.c_lflag &= ~ICANON;
	    ti.tio.c_cc[VMIN] = 0;
	    ti.tio.c_cc[VTIME] = exp100ths / 10;
#  ifdef HAVE_TERMIOS_H
	    tcsetattr(SHTTY, TCSANOW, &ti.tio);
#  else
	    ioctl(SHTTY, TCSETA, &ti.tio);
#  endif
	    r = read(SHTTY, &cc, 1);
#  ifdef HAVE_TERMIOS_H
	    tcsetattr(SHTTY, TCSANOW, &ti.tio);
#  else
	    ioctl(SHTTY, TCSETA, &shttyinfo.tio);
#  endif
	    return (r <= 0) ? -1 : cc;
# endif
#endif
	}
	while ((r = read(SHTTY, &cc, 1)) != 1) {
	    if (r == 0) {
		/* The test for IGNOREEOF was added to make zsh ignore ^Ds
		   that were typed while commands are running.  Unfortuantely
		   this caused trouble under at least one system (SunOS 4.1).
		   Here shells that lost their xterm (e.g. if it was killed
		   with -9) didn't fail to read from the terminal but instead
		   happily continued to read EOFs, so that the above read
		   returned with 0, and, with IGNOREEOF set, this caused
		   an infinite loop.  The simple way around this was to add
		   the counter (icnt) so that this happens 20 times and than
		   the shell gives up (yes, this is a bit dirty...). */
		if (isset(IGNOREEOF) && icnt++ < 20)
		    continue;
		stopmsg = 1;
		zexit(1);
	    }
	    icnt = 0;
	    if (errno == EINTR) {
		die = 0;
		if (!errflag)
		    continue;
		errflag = 0;
		errno = old_errno;
		return EOF;
	    } else if (errno == EWOULDBLOCK) {
		fcntl(0, F_SETFL, 0);
	    } else if (errno == EIO && !die) {
		ret = jobbingv;
		jobbingv = OPT_SET;
		attachtty(mypgrp);
		refresh();	/* kludge! */
		jobbingv = ret;
		die = 1;
	    } else if (errno != 0) {
		zerr("error on TTY read: %e", NULL, errno);
		stopmsg = 1;
		zexit(1);
	    }
	}
	if (cc == '\r')		/* undo the exchange of \n and \r determined by */
	    cc = '\n';		/* setterm() */
	else if (cc == '\n')
	    cc = '\r';

	ret = (int)(unsigned char)cc;
    }
    if (vichgflag) {
	if (vichgbufptr == vichgbufsz)
	    vichgbuf = realloc(vichgbuf, vichgbufsz *= 2);
	vichgbuf[vichgbufptr++] = ret;
    }
    errno = old_errno;
    return ret;
}

/* Where to print out bindings:  either stdout, or the zle output shout */
static FILE *bindout;

/* read a line */

/**/
unsigned char *
zleread(char *lp, char *rp)
{
    int z;
    unsigned char *s;
    int old_errno = errno;

#ifdef HAVE_SELECT
    long costmult = (baud) ? 3840000L / baud : 0;
    struct timeval tv;
    fd_set foofd;

    tv.tv_sec = 0;
#endif

    fflush(shout);
    fflush(stderr);
    intr();
    insmode = unset(OVERSTRIKE);
    eofsent = 0;
    resetneeded = 0;
    lpmpt = lp;
    rpmpt = rp;
    permalloc();
    histline = curhist;
#ifdef HAVE_SELECT
    FD_ZERO(&foofd);
#endif
    undoing = 1;
    line = (unsigned char *)zalloc((linesz = 256) + 1);
    *line = '\0';
    virangeflag = vilinerange = lastcmd = done = cs = ll = mark = 0;
    curhistline = NULL;
    mult = 1;
    vibufspec = 0;
    gotmult = gotvibufspec = 0;
    bindtab = mainbindtab;
    addedsuffix = complexpect = vichgflag = 0;
    viinsbegin = 0;
    statusline = NULL;
    bindout = shout;		/* always print bindings on terminal */
    if ((s = (unsigned char *)getlinknode(bufstack))) {
	setline((char *)s);
	zsfree((char *)s);
	if (stackcs != -1) {
	    cs = stackcs;
	    stackcs = -1;
	    if (cs > ll)
		cs = ll;
	}
	if (stackhist != -1) {
	    histline = stackhist;
	    stackhist = -1;
	}
    }
    initundo();
    if (unset(NOPROMPTCR))
	putc('\r', shout);
    if (tmout)
	alarm(tmout);
    zleactive = 1;
    resetneeded = 1;
    refresh();
    errflag = retflag = 0;
    while (!done && !errflag) {
	struct zlecmd *zc;

	statusline = NULL;
	bindk = getkeycmd();
	if (!ll && c == eofchar) {
	    eofsent = 1;
	    break;
	}
	if (bindk != -1) {
	    int ce = complexpect;

	    zc = zlecmds + bindk;
	    if (!(lastcmd & ZLE_ARG)) {
		mult = 1;
		vibufspec = 0;
		gotmult = gotvibufspec = 0;
	    }
	    if ((lastcmd & ZLE_UNDO) != (zc->flags & ZLE_UNDO) && undoing)
		addundo();
	    if (bindk != z_sendstring) {
		if (!(zc->flags & ZLE_MENUCMP) && menucmp)
		    menucmp = 0;
		if (!(zc->flags & ZLE_MENUCMP) &&
		    addedsuffix && !(zc->flags & ZLE_DELETE) &&
		    !((zc->flags & ZLE_INSERT) && c != ' ')) {
		    backdel(addedsuffix);
		}
		if (!menucmp && !((zc->flags & ZLE_INSERT) &&
				  complexpect == 2 && c == '}'))
		    addedsuffix = 0;
	    }
	    if (zc->func)
		(*zc->func) ();
	    /* for vi mode, make sure the cursor isn't somewhere illegal */
	    if (bindtab == altbindtab && cs > findbol() &&
		(cs == ll || line[cs] == '\n'))
		cs--;
	    if (ce == complexpect && ce && !menucmp)
		complexpect = 0;
	    if (bindk != z_sendstring)
		lastcmd = zc->flags;
	    if (!(lastcmd & ZLE_UNDO) && undoing)
		addundo();
	} else {
	    errflag = 1;
	    break;
	}
#ifdef HAVE_SELECT
	if (baud && !(lastcmd & ZLE_MENUCMP)) {
	    FD_SET(SHTTY, &foofd);
	    if ((tv.tv_usec = cost * costmult) > 500000)
		tv.tv_usec = 500000;
	    if (!kungetct && select(SHTTY+1, (SELECT_ARG_2_T) & foofd,
				    NULL, NULL, &tv) <= 0)
		refresh();
	} else
#endif
	if (!kungetct)
	    refresh();
    }
    statusline = NULL;
    trashzle();
    zleactive = 0;
    alarm(0);
    z = strlen(UTOSCP(line));
    line[z] = '\n';
    line[z + 1] = 0;
    heapalloc();
    zsfree(curhistline);
    if (eofsent) {
	free(line);
	line = NULL;
    }
    free(lastline);		/* freeundo */
    errno = old_errno;
    return line;
}

/**/
int
getkeycmd(void)
{
    int ret;
    static int hops = 0;

    cky = NULL;

    if ((c = getkey(0)) < 0)
	return -1;
    if ((ret = bindtab[c]) == z_sequenceleadin) {
	int lastlen = 0, t0 = 1, buflen = 50;
	Key ky;
	char *buf;

	buf = (char *)zalloc(buflen);
	ungetok = 0;
	buf[0] = c, buf[1] = '\0';
	if ((cky = (Key) keybindtab->getnode(keybindtab, buf)) && cky->func == z_undefinedkey)
	    cky = NULL;
	else
	    lastlen = 1;
	if (!c)
	    buf[0] = (char)0x80;
	for (;;) {
	    if ((c = getkey(cky ? 1 : 0)) >= 0) {
		if (t0 == buflen - 1)
		    buf = (char *)realloc(buf, buflen *= 2);
		buf[t0++] = (c) ? c : 0x80;
		buf[t0] = '\0';
		ky = (Key) keybindtab->getnode(keybindtab, buf);
	    } else
		ky = NULL;
	    if (ky) {
		if (ky->func == z_undefinedkey)
		    continue;
		cky = ky;
		if (!ky->prefixct) {
		    ret = ky->func;
		    break;
		}
		lastlen = t0;
	    } else if (cky) {
		ungetkeys(&buf[lastlen], t0 - lastlen);
		ret = cky->func;
		c = buf[lastlen - 1];
		break;
	    } else
		return z_undefinedkey;
	}
	zfree(buf, buflen);
    } else
	ungetok = 1;		/* for doisearch() */
    if (ret == z_vidigitorbeginningofline)
	ret = (lastcmd & ZLE_DIGIT) ? z_digitargument : z_beginningofline;
    else if (ret == z_executenamedcmd && !statusline)
	ret = executenamedcommand("execute: ");
    else if (ret == z_executelastnamedcmd)
	ret = lastnamed;
    else if (ret == z_sendstring) {
#define MAXHOPS 20
	if (++hops == MAXHOPS) {
	    zerr("string inserting another one too many times", NULL, 0);
	    hops = 0;
	    return -1;
	}
    } else
	hops = 0;
    return ret;
}

/**/
void
sendstring(void)
{
    if (!cky) {
	char buf[2];

	buf[0] = c;
	buf[1] = '\0';
	cky = (Key) keybindtab->getnode(keybindtab, buf);
    }
    ungetkeys(cky->str, cky->len);
}

/**/
Key
makefunckey(int fun)
{
    Key ky = (Key) zcalloc(sizeof *ky);

    ky->func = fun;
    return ky;
}

/* initialize the bindings */

/**/
void
initxbindtab(void)
{
    int t0, vi = 0;
    char buf[3], *s;
    Key ky;

    lastnamed = z_undefinedkey;
    for (t0 = 0; t0 != 32; t0++)
	viins_cur_bindtab[t0] = viinsbind[t0];
    for (t0 = 32; t0 != 256; t0++)
	viins_cur_bindtab[t0] = z_selfinsert;
    viins_cur_bindtab[127] = z_backwarddeletechar;
    for (t0 = 0; t0 != 128; t0++)
	emacs_cur_bindtab[t0] = emacsbind[t0];
    for (t0 = 128; t0 != 256; t0++)
	emacs_cur_bindtab[t0] = z_selfinsert;
    newemkeybindtable();
    newvikeybindtable();
    if ((s = zgetenv("VISUAL"))) {
	if (strstr(s, "vi"))
	    vi = 1;
    } else if ((s = zgetenv("EDITOR")) && strstr(s, "vi"))
	vi = 1;
    if (vi) {
	mainbindtab = viins_cur_bindtab;
	keybindtab = vikeybindtab;
    } else {
	mainbindtab = emacs_cur_bindtab;
	keybindtab = emkeybindtab;
    }
    for (t0 = 0200; t0 != 0240; t0++)
	emacs_cur_bindtab[t0] = viins_cur_bindtab[t0] = z_undefinedkey;
    for (t0 = 0; t0 != 128; t0++)
	altbindtab[t0] = vicmdbind[t0];
    for (t0 = 128; t0 != 256; t0++)
	altbindtab[t0] = emacsbind[t0];
    bindtab = mainbindtab;
    if (!kungetbuf)
	kungetbuf = (char *)zalloc(kungetsz = 32);

    emkeybindtab->addnode(emkeybindtab, ztrdup("\33\133"), ky = makefunckey(z_undefinedkey));
    ky->prefixct = 4;
    emkeybindtab->addnode(emkeybindtab, ztrdup("\33\133C"), makefunckey(z_forwardchar));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\33\133D"), makefunckey(z_backwardchar));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\33\133A"), makefunckey(z_uplineorhistory));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\33\133B"), makefunckey(z_downlineorhistory));
    vikeybindtab->addnode(vikeybindtab, ztrdup("\33"), ky = makefunckey(z_vicmdmode));
    ky->prefixct = 4;
    vikeybindtab->addnode(vikeybindtab, ztrdup("\33\133"), ky = makefunckey(z_undefinedkey));
    ky->prefixct = 4;
    vikeybindtab->addnode(vikeybindtab, ztrdup("\33\133C"), makefunckey(z_forwardchar));
    vikeybindtab->addnode(vikeybindtab, ztrdup("\33\133D"), makefunckey(z_backwardchar));
    vikeybindtab->addnode(vikeybindtab, ztrdup("\33\133A"), makefunckey(z_uplineorhistory));
    vikeybindtab->addnode(vikeybindtab, ztrdup("\33\133B"), makefunckey(z_downlineorhistory));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30"), ky = makefunckey(z_undefinedkey));
    ky->prefixct = 15;
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30*"), makefunckey(z_expandword));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30g"), makefunckey(z_listexpand));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30G"), makefunckey(z_listexpand));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30\16"), makefunckey(z_infernexthistory));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30\13"), makefunckey(z_killbuffer));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30\6"), makefunckey(z_vifindnextchar));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30\17"), makefunckey(z_overwritemode));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30\25"), makefunckey(z_undo));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30\26"), makefunckey(z_vicmdmode));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30\12"), makefunckey(z_vijoin));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30\2"), makefunckey(z_vimatchbracket));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30s"), makefunckey(z_historyincrementalsearchforward));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30r"), makefunckey(z_historyincrementalsearchbackward));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30u"), makefunckey(z_undo));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\30\30"), makefunckey(z_exchangepointandmark));
    emkeybindtab->addnode(emkeybindtab, ztrdup("\33"), ky = makefunckey(z_undefinedkey));
    ky->prefixct = 4;

    strcpy(buf, "\33q");
    for (t0 = 128; t0 != 256; t0++)
	if (emacsbind[t0] != z_undefinedkey) {
	    buf[1] = t0 & 0x7f;
	    emkeybindtab->addnode(emkeybindtab, ztrdup(buf), makefunckey(emacsbind[t0]));
	    ky->prefixct++;
	}
    stackhist = stackcs = -1;
}

/**/
char *
getkeystring(char *s, int *len, int fromwhere, int *misc)
{
    char *buf = ((fromwhere == 2)
		 ? zalloc(strlen(s) + 1) : alloc(strlen(s) + 1));
    char *t = buf, *u = NULL;
    char svchar = '\0';
    int meta = 0, control = 0;

    for (; *s; s++) {
	if (*s == '\\' && s[1])
	    switch (*++s) {
	    case 'a':
#ifdef __STDC__
		*t++ = '\a';
#else
		*t++ = '\07';
#endif
		break;
	    case 'n':
		*t++ = '\n';
		break;
	    case 'b':
		*t++ = '\010';
		break;
	    case 't':
		*t++ = '\t';
		break;
	    case 'v':
		*t++ = '\v';
		break;
	    case 'f':
		*t++ = '\f';
		break;
	    case 'r':
		*t++ = '\r';
		break;
	    case 'E':
		if (!fromwhere) {
		    *t++ = '\\', s--;
		    continue;
		}
	    case 'e':
		*t++ = '\033';
		break;
	    case 'M':
		if (fromwhere) {
		    if (s[1] == '-')
			s++;
		    meta = 1 + control;	/* preserve the order of ^ and meta */
		} else
		    *t++ = '\\', s--;
		continue;
	    case 'C':
		if (fromwhere) {
		    if (s[1] == '-')
			s++;
		    control = 1;
		} else
		    *t++ = '\\', s--;
		continue;
	    case 'c':
		if (fromwhere < 2) {
		    *misc = 1;
		    break;
		}
	    default:
		if ((idigit(*s) && *s < '8') || *s == 'x') {
		    if (!fromwhere)
			if (*s == '0')
			    s++;
			else if (*s != 'x') {
			    *t++ = '\\', s--;
			    continue;
			}
		    if (s[1] && s[2] && s[3]) {
			svchar = s[3];
			s[3] = '\0';
			u = s;
		    }
		    *t++ = zstrtol(s + (*s == 'x'), &s,
				   (*s == 'x') ? 16 : 8);
		    if (svchar) {
			u[3] = svchar;
			svchar = '\0';
		    }
		    s--;
		} else {
		    if (!fromwhere && *s != '\\')
			*t++ = '\\';
		    *t++ = *s;
		}
		break;
	} else if (*s == '^' && fromwhere == 2) {
	    control = 1;
	    continue;
	} else
	    *t++ = *s;
	if (meta == 2) {
	    t[-1] |= 0x80;
	    meta = 0;
	}
	if (control) {
	    if (t[-1] == '?')
		t[-1] = 0x7f;
	    else
		t[-1] &= 0x9f;
	    control = 0;
	}
	if (meta) {
	    t[-1] |= 0x80;
	    meta = 0;
	}
    }
    *t = '\0';
    *len = t - buf;
    return buf;
}

/**/
void
printbind(char *s, int len)
{
    int ch;

    while (len--) {
	ch = (unsigned char)*s++;
	if (ch & 0x80) {
	    fprintf(bindout, "\\M-");
	    ch &= 0x7f;
	}
	if (icntrl(ch))
	    switch (ch) {
	    case 0x7f:
		fprintf(bindout, "^?");
		break;
	    default:
		fprintf(bindout, "^%c", (ch | 0x40));
		break;
	} else {
	    if (ch == '\\' || ch == '^')
		putc('\\', bindout);
	    putc(ch, bindout);
	}
    }
}

/**/
void
printbinding(char *str, Key k)
{
    int len;

    if (k->func == z_undefinedkey)
	return;
    putc('\"', bindout);
    printbind(str, (len = strlen(str)) ? len : 1);
    fprintf(bindout, "\"\t");
    if (k->func == z_sendstring) {
	putc('\"', bindout);
	printbind(k->str, k->len);
	fprintf(bindout, "\"\n");
    } else
	fprintf(bindout, "%s\n", zlecmds[k->func].name);
}

/**/
int
bin_bindkey(char *name, char **argv, char *ops, int junc)
{
    int i, *tab;

    if (ops['v'] && ops['e']) {
	zerrnam(name, "incompatible options", NULL, 0);
	return 1;
    }
    if (ops['v'] || ops['e'] || ops['d'] || ops['m']) {
	if (*argv) {
	    zerrnam(name, "too many arguments", NULL, 0);
	    return 1;
	}
	if (ops['d']) {
	    freehashtable(emkeybindtab);
	    freehashtable(vikeybindtab);
	    initxbindtab();
	}
	if (ops['e']) {
	    mainbindtab = emacs_cur_bindtab;
	    keybindtab = emkeybindtab;
	} else if (ops['v']) {
	    mainbindtab = viins_cur_bindtab;
	    keybindtab = vikeybindtab;
	}
	if (ops['m'])
	    for (i = 128; i != 256; i++)
		if (mainbindtab[i] == z_selfinsert)
		    mainbindtab[i] = emacsbind[i];
	return 0;
    }
    tab = (ops['a']) ? altbindtab : mainbindtab;
    bindout = stdout;	/* print bindings to stdout */
    if (!*argv) {
	char buf[2];

	buf[1] = '\0';
	for (i = 0; i != 256; i++) {
	    buf[0] = i;
	    putchar('\"');
	    printbind(buf, 1);
	    if (i < 254 && tab[i] == tab[i + 1] && tab[i] == tab[i + 2]) {
		printf("\" to \"");
		while (tab[i] == tab[i + 1])
		    i++;
		buf[0] = i;
		printbind(buf, 1);
	    }
	    printf("\"\t%s\n", zlecmds[tab[i]].name);
	}
	scanhashtable(keybindtab, (HFunc) printbinding);
	return 0;
    }
    while (*argv) {
	Key ky = NULL, cur = NULL;
	char *s;
	int func, len, firstzero = 0;

	s = getkeystring(*argv++, &len, 2, NULL);
	if (len > 1) {
	    if (s[0])
		firstzero = 0;
	    else
		firstzero = 1;
	    for (i = 0; i < len; i++)
		if (!s[i])
		    s[i] = (char)0x80;
	}
	if (!*argv || ops['r']) {
	    if (len == 1)
		func = tab[(unsigned char)*s];
	    else
		func = (ky = (Key) keybindtab->getnode(keybindtab, s)) ? ky->func
		    : z_undefinedkey;
	    if (func == z_undefinedkey) {
		zerrnam(name, "in-string is not bound", NULL, 0);
		zfree(s, len);
		return 1;
	    }
	    if (ops['r']) {
		if (len == 1 && func != z_sequenceleadin) {
		    tab[(unsigned char)*s] = z_undefinedkey;
		    if (func == z_sendstring)
			free(keybindtab->removenode(keybindtab, s));
		} else {
		    if (ky && ky->prefixct) {
			if (ky->func == z_sendstring)
			    zfree(ky->str, ky->len);
			ky->func = z_undefinedkey;
		    } else
			free(keybindtab->removenode(keybindtab, s));
		    if (len > 1) {
			s[--len] = '\0';
			while (len > 1) {
			    (ky = (Key) keybindtab->getnode(keybindtab, s))->prefixct--;
			    if (!ky->prefixct && ky->func == z_undefinedkey)
				free(keybindtab->removenode(keybindtab, s));
			    s[--len] = '\0';
			}
			(ky = (Key) keybindtab->getnode(keybindtab, s))->prefixct--;
			if (!ky->prefixct) {
			    tab[(unsigned char)*s] = ky->func;
			    if (ky->func != z_sendstring)
				free(keybindtab->removenode(keybindtab, s));
			}
		    }
		}
		zfree(s, len);
		continue;
	    }
	    if (func == z_sendstring) {
		if (len == 1)
		    ky = (Key) keybindtab->getnode(keybindtab, s);
		printbind(ky->str, ky->len);
		putchar('\n');
	    } else
		printf("%s\n", zlecmds[func].name);
	    zfree(s, len);
	    return 0;
	}
	if (!ops['s']) {
	    for (i = 0; i != ZLECMDCOUNT; i++)
		if (!strcmp(*argv, zlecmds[i].name))
		    break;
	    if (i == ZLECMDCOUNT) {
		zerr("undefined function: %s", *argv, 0);
		zfree(s, len);
		return 1;
	    }
	    func = i;
	} else
	    func = z_sendstring;

	if (len == 1 && tab[(unsigned char)*s] != z_sequenceleadin) {
	    if (ops['s']) {
		keybindtab->addnode(keybindtab, ztrdup(s), cur = makefunckey(z_sendstring));
	    } else if (tab[(unsigned char)*s] == z_sendstring)
		free(keybindtab->removenode(keybindtab, s));
	    tab[(unsigned char)*s] = func;
	} else {
	    if (!(cur = (Key) keybindtab->getnode(keybindtab, s))
		|| (cur->func == z_undefinedkey))
		for (i = len - 1; i > 0; i--) {
		    char sav;

		    sav = s[i];
		    s[i] = '\0';
		    if (i == 1 && firstzero)
			*s = '\0';
		    if (!(ky = (Key) keybindtab->getnode(keybindtab, s)))
			keybindtab->addnode(keybindtab, ztrdup(s), ky = makefunckey(z_undefinedkey));
		    ky->prefixct++;
		    s[i] = sav;
		    if (i == 1 && firstzero)
			*s = (char)0x80;
		}
	    if (cur) {
		cur->func = func;
		zfree(cur->str, cur->len);
	    } else
		keybindtab->addnode(keybindtab, ztrdup(s), cur = makefunckey(func));
	    if (firstzero)
		*s = 0;
	    if (tab[(unsigned char)*s] != z_sequenceleadin) {
		cur->func = tab[(unsigned char)*s];
		tab[(unsigned char)*s] = z_sequenceleadin;
	    }
	}
	if (ops['s']) {
	    cur->str = getkeystring(*argv, &cur->len, 2, NULL);
	    cur->str = (char *)realloc(cur->str, cur->len);
	}
	argv++;
	zfree(s, len);
    }
    return 0;
}

/**/
void
freekey(void *x)
{
    Key k = (Key) x;

    if (k->str)
	zsfree(k->str);
    zfree(k, sizeof(struct key));
}

extern int clearflag;

/**/
void
describekeybriefly(void)
{
    int cmd;
    int len;

    if (statusline)
	return;
    statusline = "Describe key briefly: _";
    refresh();
    cmd = getkeycmd();
    statusline = NULL;
    if (cmd < 0)
	return;
    trashzle();
    clearflag = (isset(USEZLE) && termok &&
		 (isset(ALWAYSLASTPROMPT) && mult == 1)) ||
	(unset(ALWAYSLASTPROMPT) && mult != 1);
    if (!cky) {
	char ch = c;
	printbind(&ch, 1);
    }
    else
	printbind(cky->nam, (len = strlen(cky->nam)) ? len : 1);
    fprintf(shout, " is ");
    if (cmd == z_sendstring) {
	if (!cky) {
	    char buf[2];

	    buf[0] = c;
	    buf[1] = '\0';
	    cky = (Key) keybindtab->getnode(keybindtab, buf);
	}
	fputc('"', shout);
	printbind(cky->str, cky->len);
	putc('"', shout);
    }
    else
	fprintf(shout, "%s", zlecmds[cmd].name);
    if (clearflag)
	putc('\r', shout), tcmultout(TCUP, TCMULTUP, nlnct);
    else
	putc('\n', shout);
}

static int func, funcfound;
#define MAXFOUND 4

static void
printfuncbind(char *str, Key k)
{
    int len = strlen(str);

    if (k->func != func || funcfound >= MAXFOUND || len <= 1)
	return;
    if (!funcfound++)
	fprintf(shout, " on");
    putc(' ', shout);
    printbind(str, len);
}

/**/
void
whereis(void)
{
    int i;

    if ((func = executenamedcommand("Where is: ")) == z_undefinedkey)
	return;
    funcfound = 0;
    trashzle();
    clearflag = (isset(USEZLE) && termok &&
		 (isset(ALWAYSLASTPROMPT) && mult == 1)) ||
	(unset(ALWAYSLASTPROMPT) && mult != 1);
    if (func == z_selfinsert)
	fprintf(shout, "%s is on many keys", zlecmds[func].name);
    else {
	fprintf(shout, "%s is", zlecmds[func].name);
	for (i = 0; funcfound < MAXFOUND && i < 256; i++)
	    if (mainbindtab[i] == func) {
		char ch = i;
		if (!funcfound++)
		    fprintf(shout, " on");
		putc(' ', shout);
		printbind(&ch, 1);
	    }
	if (funcfound < MAXFOUND)
	    scanhashtable(keybindtab, (HFunc) printfuncbind);
	if (!funcfound)
	    fprintf(shout, " not bound to any key");
    }
    if (clearflag)
	putc('\r', shout), tcmultout(TCUP, TCMULTUP, nlnct);
    else
	putc('\n', shout);
}

/**/
void
trashzle(void)
{
    if (zleactive) {
	refresh();
	moveto(nlnct, 0);
	if (clearflag && tccan(TCCLEAREOD)) {
	    tcout(TCCLEAREOD);
	    clearflag = 0;
	}
	if (postedit)
	    fprintf(shout, "%s", postedit);
	fflush(shout);
	resetneeded = 1;
	settyinfo(&shttyinfo);
    }
    if (errflag)
	kungetct = 0;
}
