/*
 * Copyright (c) 1990 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <limits.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include "internal.h"

/*
 * Convert a string to an unsigned long integer.
 *
 * Ignores `locale' stuff.  Assumes that the upper and lower case
 * alphabets and digits are each contiguous.
 */

unsigned long strtoul (const char *pszInput, char **ppszEnd, int nBase)
    {
    const char   *pszCurrent;
    unsigned long ulAccum;
    int           nChar;
    unsigned long ulCutoff;
    int           fNeg;
    int           fAny;
    int           nCutlim;
/*
 *  Ignore leading whitespace in the string.
 */
    fNeg       = 0;
    pszCurrent = pszInput;

    do
        {
	nChar = (int) *pszCurrent++;
        }
    while (isspace(nChar));
/*
 *  Set the negative sign should one be present.
 */
    if (nChar == '-')
        {
	fNeg  = 1;
	nChar = (int) *pszCurrent++;
        }
/*
 *  Ignore a leading plus sign.
 */
    else
        {
	if (nChar == '+')
	    {
	    nChar = (int) *pszCurrent++;
	    }
        }
/*
 *  If the base is undefined then look at the first character. If the
 *  character is not a leading zero then the base is decimal.
 */
    if (nBase == 0)
        {
	if (nChar != '0')
	    {
	    nBase = 10;
	    }
/*
 *  Look the next character. If the character is "x" then the item is
 *  hexadecimal. Otherwise, it is octal.
 */
	else
	    {
	    nChar = (int) *pszCurrent++;
	    if (nChar == 'x' || nChar == 'X')
	        {
		nChar = (int) *pszCurrent++;
		nBase  = 16;
	        }
	    else
	        {
		nBase = 8;
	        }
	    }
        }
/*
 *  A base was specified. Ignore a leading zero if it is present.
 */
    else
        {
	if (nChar == '0')
	    {
	    nChar = (int) *pszCurrent++;
/*
 *  If the base is hexadecimal then ignore the "x" character which may
 *  follow the leading zero.
 */
	    if (nBase == 16)
	        {
		if (nChar == 'x' || nChar == 'X') 
		    {
		    nChar = (int) *pszCurrent++;
		    }
	        }
	    }
        }
/*
 * Compute the cutoff value between legal numbers and illegal
 * numbers.  That is the largest legal value, divided by the
 * base.  An input number that is greater than this value, if
 * followed by a legal input character, is too big.  One that
 * is equal to this value may be valid or not; the limit
 * between valid and invalid numbers is then based on the last
 * digit.  For instance, if the range for longs is
 * [-2147483648..2147483647] and the input base is 10,
 * cutoff will be set to 214748364 and cutlim to either
 * 7 (neg==0) or 8 (neg==1), meaning that if we have accumulated
 * a value > 214748364, or equal but the next digit is > 7 (or 8),
 * the number is too big, and we will return a range error.
 *
 * Set any if any `digits' consumed; make it negative to indicate
 * overflow.
 */
    ulCutoff = (unsigned long) (ULONG_MAX / (unsigned long) nBase);
    nCutlim  = (int)           (ULONG_MAX % (unsigned long) nBase);
/*
 *  Convert the next digit to a binary entry
 */
    ulAccum = 0;
    fAny    = 0;
    nChar   = (int) *pszCurrent++;
/*
 *  Process a decimal digit
 */
    while (nChar != '\0')
        {
	if (isdigit (nChar))
	    {
	    nChar -= '0';
	    }
/*
 *  Process a lower case hexadecimal sequence.
 */
	else
	    {
	    if (islower (nChar))
	        {
		nChar -= 'a' - 10;
	        }
/*
 *  Process a upper case hexadecimal sequence.
 */
	    else
	        {
		if (isupper (nChar))
		    {
		    nChar -= 'A' - 10;
		    }
/*
 *  All others are invalid.
 */
		else
		    {
		    break;
		    }
	        }
	    }
/*
 *  If the character is out of range then terminate the scan.
 */
	if (nChar >= nBase)
	    {
	    break;
	    }
/*
 *  If the value has overflowed then do not allow it to continue with
 *  the processing.
 */
	if (fAny >= 0)
	    {
	    if (ulAccum > ulCutoff ||
	       (ulAccum == ulCutoff && nChar > nCutlim))
	        {
		fAny = -1;
	        }
/*
 *  Accumulate the next digit.
 */
	    else
	        {
		fAny     = 1;
		ulAccum *= (unsigned long) nBase;
		ulAccum += (unsigned long) nChar;
	        }
	    }
/*
 *  Fetch the next character from the input.
 */
	nChar = *pszCurrent++;
        }
/*
 *  Generate the domain error if it is indicated
 */
    if (fAny < 0)
        {
	ulAccum = ULONG_MAX;
	errno   = ERANGE;
        }
/*
 *  Negate the value should it be required.
 */
    else
        {
	if (fNeg)
	    {
	    ulAccum = (unsigned long) -((long) ulAccum);
	    }
	pszInput = &pszCurrent[-1];  /* Update resulting pointer */
        }
/*
 *  Update the termination pointer if one was given.
 */
    if (ppszEnd != NULL)
        {
	*ppszEnd = (char *) pszInput;
        }
/*
 *  Return the accumulated value.
 */
    return (ulAccum);
    }
