/*
 * pcsel.c
 *
 * /dev/pcsp implementation
 *
 * Copyright (C) 1993  Michael Beck
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <getopt.h>
#include <sys/ioctl.h>
#include <sys/pcsp.h>
 
#define AUDIO		"/dev/pcsp"

int audio, omode = O_WRONLY;

/* trying to define independent ioctl for sounddrivers version 1 and 2+ */
#ifdef SOUND_VERSION
#define IOCTL(a,b,c)            ioctl(a,b,&c)
#else
#define IOCTL(a,b,c)            (c = ioctl(a,b,c) )
#endif

struct _devices {
	char		*name;
	char		*nick;
	unsigned	code;
} devs[] = {
	{ "Stereo-on-One", "Sto1", SNDCARD_STO1 },
	{ "Mono DAC", "DACm",      SNDCARD_DACM },
	{ "Stereo DAC", "DACs",    SNDCARD_DACS },
	{ "PC-Speaker", "PCSP",    SNDCARD_PCSP }
};

#define LP_NO		3
#define CARD_NO		4

void actual_conf(unsigned long dev, unsigned long port, unsigned long portS)
{
	int i, volume;

	printf("Actual PCSP output device: ");
	for (i = 0; i < CARD_NO; ++i)
		if (devs[i].code == dev) {
			printf("%s ", devs[i].name);
			if (dev == SNDCARD_PCSP) {
				IOCTL(audio, PCSP_GET_VOL, volume);
				printf("Volume : %d %%\n", volume * 100 / 256);
			}
			else {
				if (dev != SNDCARD_DACS)
					printf("at lp%d\n", port);
				else
					printf("at lp%d & lp%d\n", port, portS);
			}
			break;
		}
	if (i == CARD_NO) {
		printf("unknown device %d !!!\n", dev);
		exit (EXIT_FAILURE);
	}	
}

unsigned long set_dev(char *devname)
{
	int i;
	int dev = 0;

	for (i = 0; i < CARD_NO; ++i)
		if (! strcasecmp(devname, devs[i].nick)) {
			dev = devs[i].code;
			if (IOCTL(audio, PCSP_SET_DEV, dev) < 0) {
				perror(AUDIO);
				exit(EXIT_FAILURE);
			}
		}
	return dev;
}

int main(int argc, char *argv[])
{
  int i;
  int dev, port, portS, arg, stereo, samplerate, volume = 0;
  char c, flag, acc, opt;

  audio = open(AUDIO, omode, 0);
  if (audio < 0) {
    perror(AUDIO);
    exit(EXIT_FAILURE);
  }

  IOCTL(audio, PCSP_GET_DEV, dev);
  if (dev != SNDCARD_PCSP) {
    IOCTL(audio, PCSP_GET_PORTS, port);
    portS = port & 0xF0 >> 8;
    port &= 0x0F;
  }
  else {
    port = 1; portS = 2;
  }
  stereo = 2; samplerate = 0;

  acc = 0; opt = 1;
  while ((c = getopt (argc, argv, "d:p:r:l:s:v:MSh")) != EOF) {
    flag = 1; opt = 0;
    switch (c) {
      case 'd': 
        dev = set_dev(optarg); 
        if (! dev) {
          fprintf(stderr, "%s: unknown device '%s'\n", argv[0], optarg);
          exit(-1);
        }
        break;
      case 'p':
      case 'r': flag = 0;
      case 'l':
        arg = atoi(optarg);
        if (arg >= LP_NO)
          fprintf(stderr, "%s: wrong port\n", argv[0]);
        else {
          (flag) ? portS : port = arg;
          acc = 1;
        }
        break;
      case 'M':
        stereo = 0; break;
      case 'S':
        stereo = 1; break;
      case 's':
        samplerate = atoi(optarg); break;
      case 'v':
	volume = atoi(optarg); break;
      case 'h':
      default:
        fprintf (stderr,
         "Usage: %s [-d device] [-p port] [-r rport -l lport] [-s Hz] [-v volume] [-SM]\n", argv[0]);
        fprintf (stderr, "  supported devices :\n");
        for (i = 0; i < CARD_NO; ++i)
          fprintf (stderr, "  '%s' : %s\n", devs[i].nick, devs[i].name);
        exit (EXIT_FAILURE);
    }
  }
  if (opt) {
    actual_conf(dev, port, portS);
    exit(EXIT_SUCCESS);
  }

  if (acc) {
    if (dev == SNDCARD_DACS && port == portS) {
      fprintf(stderr, "%s: a Stereo DAC need two different ports\n", argv[0]);
      exit(EXIT_FAILURE);
    }
    port |= portS << 8;
    if (IOCTL(audio, PCSP_SET_PORTS, port) < 0) {
      perror(AUDIO);
      exit(EXIT_FAILURE);
    }
  }
  if (stereo == 2)
    stereo = (dev == SNDCARD_STO1) || (dev == SNDCARD_DACS) ? 1 : 0;
  if (IOCTL(audio, SNDCTL_DSP_STEREO, stereo) < 0) 
    fprintf(stderr, "%s: can't set stereo\n", argv[0]);
  if (samplerate)
    if (IOCTL(audio, SNDCTL_DSP_SPEED, samplerate) < 0)
      fprintf(stderr, "%s: can't set samplerate to %d Hz\n", argv[0], samplerate);
  if (volume > 0) {
    volume = (volume * 256) / 100;
    if (IOCTL(audio, PCSP_SET_VOL, volume) < 0)
      fprintf(stderr, "%s: can't set volume for PC-Speaker\n", argv[0]);
  }

  return (EXIT_SUCCESS);
}
