/*
 * APM BIOS driver for Linux
 * by Stephen Rothwell
 *    Stephen.Rothwell@pd.necisa.oz.au
 *
 * $Id: apm_bios.c,v 0.8 1995/01/10 12:16:24 sfr Exp $
 */

#include <asm/system.h>
#include <asm/segment.h>

#include <linux/types.h>
#include <linux/stddef.h>
#include <linux/timer.h>
#include <linux/fcntl.h>
#include <linux/malloc.h>
#include <linux/apm_bios.h>

extern unsigned long get_cmos_time(void);

/*
 * define to always call the APM BIOS busy routine
 * even if the clock was not slowed by the idle routine
 */
#define ALWAYS_CALL_BUSY
/*
 * define to have debug messages
 */
#define APM_DEBUG

/*
 * Need to poll the APM BIOS every second
 */
#define APM_CHECK_TIMEOUT	(HZ)

/*
 * Maximum number of events stored
 */
#define MAX_EVENTS		20

/*
 * These are the actual BIOS calls in assembler
 */
#define APM_BIOS_CALL(error_reg) \
	__asm__ __volatile__("lcall _apm_bios_entry\n\t" \
		"setc %%" # error_reg

#define APM_GET_EVENT(event, error)	\
	APM_BIOS_CALL(al) : "=a" (error), "=b" (event) : "0" (0x530b))
#define APM_SET_POWER_STATE(state, error) \
	APM_BIOS_CALL(al) : "=a" (error) \
		: "0" (0x5307), "b" (0x0001), "c" (state))
#define APM_DRIVER_VERSION(ver, ax, error) \
	APM_BIOS_CALL(bl) : "=a" (ax), "=b" (error) \
		: "0" (0x530e), "1" (0), "c" (ver))
#define APM_GET_POWER_STATUS(bx, cx, dx, error) \
	APM_BIOS_CALL(al) : "=a" (error), "=b" (bx), "=c" (cx), "=d" (dx) \
		: "0" (0x530a), "1" (1))
#define APM_SET_CPU_IDLE(error) \
	APM_BIOS_CALL(al) : "=a" (error) : "0" (0x5305))
#define APM_SET_CPU_BUSY(error) \
	APM_BIOS_CALL(al) : "=a" (error) : "0" (0x5306))

/*
 * Forward declarations
 */
static void	suspend(void);
static void	standby(void);
static void	set_time(void);

static void	check_events(unsigned long);

static int	do_open(struct inode *, struct file *);
static void	do_release(struct inode *, struct file *);
static int	do_read(struct inode *, struct file *, char *, int);
static int	do_select(struct inode *, struct file *, int,
			  select_table *);
static int	do_ioctl(struct inode *, struct file *, u_int, u_long);

/*
 * Local variables
 */
static asmlinkage struct {
	unsigned long	offset;
	unsigned short	segment;
}				apm_bios_entry;
static int			apm_enabled = 0;
static int			apm_bios_opened = 0;
static int			clock_slowed = 0;
static int			apm_major;

static long			clock_cmos_diff;
static int			got_clock_diff = 0;

static struct wait_queue *	process_list = NULL;
static apm_event_t		events[MAX_EVENTS];
static int			event_head;
static int			event_tail;

static struct timer_list	apm_timer =
	{ NULL, NULL, APM_CHECK_TIMEOUT, 0, &check_events };

static char			driver_version[] = "0.4";

static int			(*old_idle_hook)(void);
static void			(*old_busy_hook)(void);

#ifdef APM_DEBUG
static char *	apm_event_name[] = {
	"system standby",
	"system suspend",
	"normal resume",
	"critical resume",
	"low battery",
	"power status change",
	"update time",
	"critical suspend",
	"user standby",
	"user suspend",
	"system standby resume"
};
#define NR_APM_EVENT_NAME	\
		(sizeof(apm_event_name) / sizeof(apm_event_name[0]))
#endif

static struct file_operations apm_bios_fops = {
	NULL,		/* lseek */
	do_read,
	NULL,		/* write */
	NULL,		/* readdir */
	do_select,
	do_ioctl,
	NULL,		/* mmap */
	do_open,
	do_release,
	NULL,		/* fsync */
	NULL		/* fasync */
};

typedef struct callback_list_t {
	int (*				callback)(apm_event_t);
	struct callback_list_t *	next;
} callback_list_t;

static callback_list_t *	callback_list = NULL;

typedef struct lookup_t {
	int	key;
	char *	msg;
} lookup_t;

static const lookup_t error_table[] = {
	{ APM_SUCCESS,		"Operation succeeded" },
	{ APM_DISABLED,		"Power management disabled" },
	{ APM_NOT_CONNECT,	"Interface not connected" },
	{ APM_BAD_DEVICE,	"Unrecognized device ID" },
	{ APM_BAD_PARAM,	"Parameter out of range" },
	{ APM_NOT_PRESENT,	"Interface not engaged" }
};
#define ERROR_COUNT	(sizeof(error_table)/sizeof(lookup_t))

static int apm_driver_version(u_short *val)
{
	u_short	error;

	APM_DRIVER_VERSION(*val, *val, error);
	if (error & 0xff)
		return (*val >> 8);
	return APM_SUCCESS;
}

static int apm_get_event(apm_event_t *event)
{
	u_short	error;

	APM_GET_EVENT(*event, error);
	if (error & 0xff)
		return (error >> 8);
	return APM_SUCCESS;
}

static int apm_set_power_state(u_short state)
{
	u_short	error;

	APM_SET_POWER_STATE(state, error);
	if (error & 0xff)
		return (error >> 8);
	return APM_SUCCESS;
}

static int apm_get_power_status(u_short *status, u_short *bat, u_short *life)
{
	u_short	error;

	APM_GET_POWER_STATUS(*status, *bat, *life, error);
	if (error & 0xff)
		return (error >> 8);
	return APM_SUCCESS;
}

static void apm_error(char *str, int err)
{
	int	i;

	for (i = 0; i < ERROR_COUNT; i++)
		if (error_table[i].key == err) break;
	if (i < ERROR_COUNT)
		printk("apm_bios: %s: %s\n", str, error_table[i].msg);
	else
		printk("apm_bios: %s: unknown error code %#2.2x\n", str, err);
}

int apm_register_callback(int (*callback)(apm_event_t))
{
	callback_list_t *	new;

	new = kmalloc(sizeof(callback_list_t), GFP_KERNEL);
	if (new == NULL)
		return -ENOMEM;
	new->callback = callback;
	new->next = callback_list;
	callback_list = new;
	return 0;
}

void apm_unregister_callback(int (*callback)(apm_event_t))
{
	callback_list_t **	ptr;
	callback_list_t *	old;

	ptr = &callback_list;
	for (ptr = &callback_list; *ptr != NULL; ptr = &(*ptr)->next)
		if ((*ptr)->callback == callback)
			break;
	old = *ptr;
	*ptr = old->next;
	kfree_s(old, sizeof(callback_list_t));
}
	
static int queue_empty(void)
{
	return event_head == event_tail;
}

static apm_event_t get_queued_event(void)
{
	apm_event_t	event;

	event_tail = (event_tail + 1) % MAX_EVENTS;
	event = events[event_tail];
	return event;
}

static void queue_event(apm_event_t event)
{
	if (!apm_bios_opened)
		return;
	event_head = (event_head + 1) % MAX_EVENTS;
	if (event_head == event_tail)
		event_tail = (event_tail + 1) % MAX_EVENTS;
	events[event_head] = event;
	wake_up_interruptible(&process_list);
}

static void set_time(void)
{
	unsigned long	flags;

	if (!got_clock_diff)
		return;

	save_flags(flags);
	cli();
	CURRENT_TIME = get_cmos_time() + clock_cmos_diff;
	restore_flags(flags);
}

static void suspend(void)
{
	unsigned long	flags;
	int		err;

	save_flags(flags);
	cli();
	clock_cmos_diff = CURRENT_TIME - get_cmos_time();
	got_clock_diff = 1;
	restore_flags(flags);
	err = apm_set_power_state(APM_STATE_SUSPEND);
	if (err)
		apm_error("suspend", err);
	set_time();
}

static void standby(void)
{
	int	err;

	err = apm_set_power_state(APM_STATE_STANDBY);
	if (err)
		apm_error("standby", err);
}

static apm_event_t get_event(void)
{
	int		error;
	apm_event_t	event;

	static int notified = 0;

	error = apm_get_event(&event);
	if (error == APM_SUCCESS)
		return event;

	if ((error != APM_NO_EVENTS) && (notified++ == 0))
		apm_error("get_event", error);

	return 0;
}

static int send_event(apm_event_t event, apm_event_t undo)
{
	callback_list_t *	call;
	callback_list_t *	fix;
    
	for (call = callback_list; call != NULL; call = call->next) {
		if (call->callback(event) && undo) {
			for (fix = callback_list; fix != call; fix = fix->next)
				fix->callback(undo);
			if (apm_bios_info.version > 0x100)
				apm_set_power_state(APM_STATE_REJECT);
			return 1;
		}
	}

	queue_event(event);
	/* defer events that might be rejected */
	return (apm_bios_opened && undo);
}

static void check_events(unsigned long unused)
{
	apm_event_t	event;

	while ((event = get_event()) != 0) {
		switch (event) {
		case APM_SYS_STANDBY:
		case APM_USER_STANDBY:
			if (send_event(event, APM_STANDBY_RESUME) == 0)
				standby();
			break;

		case APM_SYS_SUSPEND:
		case APM_USER_SUSPEND:
			if (send_event(event, APM_NORMAL_RESUME) == 0)
				suspend();
			break;

		case APM_NORMAL_RESUME:
		case APM_CRITICAL_RESUME:
		case APM_STANDBY_RESUME:
			set_time();
			send_event(event, 0);
			break;

		case APM_LOW_BATTERY:
		case APM_POWER_STATUS_CHANGE:
			send_event(event, 0);
			break;

		case APM_UPDATE_TIME:
			set_time();
			break;

		case APM_CRITICAL_SUSPEND:
			suspend();
			break;
		}
#ifdef APM_DEBUG
		if (event <= NR_APM_EVENT_NAME)
			printk("APM BIOS received %s notify\n",
			       apm_event_name[event - 1]);
		else
			printk("APM BIOS received unknown event 0x%x\n",
			       event);
#endif
	}

	init_timer(&apm_timer);
	apm_timer.expires = APM_CHECK_TIMEOUT;
	add_timer(&apm_timer);
}

static int do_idle(void)
{
	unsigned short	error;

	if (!apm_enabled)
		return 0;

	APM_SET_CPU_IDLE(error);
	if (error & 0xff)
		return 0;

	clock_slowed = (apm_bios_info.flags & APM_IDLE_SLOWS_CLOCK) != 0;

	return 1;
}

static void do_busy(void)
{
	unsigned short	error;

	if (old_busy_hook != NULL)
		(*old_busy_hook)();

#ifndef ALWAYS_CALL_BUSY
	if (!clock_slowed)
		return;
#endif

	APM_SET_CPU_BUSY(error);

	clock_slowed = 0;
}

static int do_read(struct inode *inode, struct file *fp, char *buf, int count)
{
	struct wait_queue	wait =
		{ current,	NULL };
	int			i;
	apm_event_t		event;

	if (count < sizeof(apm_event_t))
		return -EINVAL;
	if (queue_empty()) {
		if (fp->f_flags & O_NONBLOCK)
			return -EAGAIN;
		add_wait_queue(&process_list, &wait);
repeat:
		current->state = TASK_INTERRUPTIBLE;
		if (queue_empty() && !(current->signal & ~current->blocked)) {
			schedule();
			goto repeat;
		}
		current->state = TASK_RUNNING;
		remove_wait_queue(&process_list, &wait);
	}
	i = count;
	while ((i >= sizeof(event)) && !queue_empty()) {
		event = get_queued_event();
		memcpy_tofs(buf, &event, sizeof(event));
		buf += sizeof(event);
		i -= sizeof(event);
	}
	if (i < count)
		return count - i;
	if (current->signal & ~current->blocked)
		return -ERESTARTSYS;
	return 0;
}

static int do_select(struct inode *inode, struct file *fp, int sel_type,
		     select_table * wait)
{
	if (sel_type != SEL_IN)
		return 0;
	if (!queue_empty())
		return 1;
	select_wait(&process_list, wait);
	return 0;
}

static int do_ioctl(struct inode * inode, struct file *filp,
		    u_int cmd, u_long arg)
{
	int	ret = 0;

	switch (cmd) {
	case APM_IOC_STANDBY:
		standby();
		break;

	case APM_IOC_SUSPEND:
		suspend();
		break;

	default:
		ret = -EINVAL;
	}
	return ret;
}

static void do_release(struct inode * inode, struct file * filp)
{
	apm_bios_opened = 0;
}

static int do_open(struct inode * inode, struct file * filp)
{
	if (!suser())
		return -EPERM;
	if (apm_bios_opened)
		return -EBUSY;
	apm_bios_opened = 1;
	event_tail = event_head = 0;
	return 0;
}

static int apm_setup(void)
{
	unsigned short	bx;
	unsigned short	cx;
	unsigned short	dx;
	unsigned short	error;
	char *		power_stat;
	char *		bat_stat;

	if (apm_bios_info.version == 0) {
		printk("APM BIOS not found.\n");
		return -1;
	}
    
	printk("APM BIOS ver %c.%c flags 0x%x (driver version %s)\n",
	       ((apm_bios_info.version >> 8) & 0xff) + '0',
	       (apm_bios_info.version & 0xff) + '0',
	       apm_bios_info.flags,
	       driver_version);

	if ((apm_bios_info.flags & APM_32_BIT_SUPPORT) == 0) {
		printk("    No 32 bit BIOS support\n");
		return -1;
	}

	/*
	 * Fix for the Compaq Contura 3/25c which reports BIOS version 0.1
	 * but is reportedly a 1.0 BIOS.
	 */
	if (apm_bios_info.version == 0x001)
		apm_bios_info.version = 0x100;

	printk("    entry %x:%lx cseg16 %x dseg %x",
	       apm_bios_info.cseg, apm_bios_info.offset,
	       apm_bios_info.cseg_16, apm_bios_info.dseg);
	if (apm_bios_info.version > 0x100)
		printk(" cseg len %x, dseg len %x",
		       apm_bios_info.cseg_len, apm_bios_info.dseg_len);
	printk("\n");

	apm_bios_entry.offset = apm_bios_info.offset;
	apm_bios_entry.segment = APM_CS;

	set_base(gdt[APM_CS >> 3],
		 0xc0000000 + ((unsigned long)apm_bios_info.cseg << 4));
	set_base(gdt[APM_CS_16 >> 3],
		 0xc0000000 + ((unsigned long)apm_bios_info.cseg_16 << 4));
	set_base(gdt[APM_DS >> 3],
		 0xc0000000 + ((unsigned long)apm_bios_info.dseg << 4));
	if (apm_bios_info.version == 0x100) {
		set_limit(gdt[APM_CS >> 3], 64 * 1024);
		set_limit(gdt[APM_CS_16 >> 3], 64 * 1024);
		set_limit(gdt[APM_DS >> 3], 64 * 1024);
	} else {
		set_limit(gdt[APM_CS >> 3], apm_bios_info.cseg_len);
		set_limit(gdt[APM_CS_16 >> 3], apm_bios_info.cseg_len);
		set_limit(gdt[APM_DS >> 3], apm_bios_info.dseg_len);
		apm_bios_info.version = 0x0101;
		error = apm_driver_version(&apm_bios_info.version);
		if (error != 0)
			apm_bios_info.version = 0x100;
	}
    
	error = apm_get_power_status(&bx, &cx, &dx);
	if (error)
		printk("    Power status not available\n");
	else {
		switch ((bx >> 8) & 0xff) {
		case 0: power_stat = "off line"; break;
		case 1: power_stat = "on line"; break;
		case 2: power_stat = "on backup power"; break;
		default: power_stat = "unknown"; break;
		}
	
		switch (bx & 0xff) {
		case 0: bat_stat = "high"; break;
		case 1: bat_stat = "low"; break;
		case 2: bat_stat = "critical"; break;
		case 3: bat_stat = "charging"; break;
		default: bat_stat = "unknown"; break;
		}
	
		printk("    AC %s, battery %s, battery life ",
		       power_stat, bat_stat);
		if ((cx & 0xff) == 0xff)
			printk("unknown\n");
		else
			printk("%d%%\n", cx & 0xff);
	
		if (apm_bios_info.version > 0x100) {
			printk("    battery flag = 0x%x, battery life ",
			       (cx >> 8) & 0xff);
			if (dx == 0xffff)
				printk("unknown\n");
			else
				printk("%d %s\n", dx & 0x7fff,
				       ((dx & 0x8000) == 0)
				       ? "seconds" : "minutes");
		}
	}

	if ((apm_major = register_chrdev(0, "apm_bios", &apm_bios_fops)) < 0) {
		printk("APM BIOS: Cannot allocate major device number\n");
		return -1;
	}

	init_timer(&apm_timer);
	add_timer(&apm_timer);

	old_idle_hook = idle_hook;
	idle_hook = do_idle;
	old_busy_hook = busy_hook;
	busy_hook = do_busy;

	apm_enabled = 1;

	return 0;
}

long apm_bios_init(long kmem_start)
{
	apm_setup();
	return kmem_start;
}
