/* encaps.c */
/* Copyright (C) 1994 Yggdrasil Computing, Incorporated
   4880 Stevens Creek Blvd. Suite 205
   San Jose, CA 95129-1034
   USA
   Tel (408) 261-6630
   Fax (408) 261-6631

   Encaps is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by Ross Biro Nov 94 */ 

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <malloc.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <unistd.h>
#include <getopt.h>

#include "ansidecl.h"
#include <bfd.h>

#define obstack_chunk_alloc malloc
#define obstack_chunk_free free

#define DEF_SEC_FLAGS SEC_ALLOC|SEC_LOAD|SEC_RELOC|SEC_HAS_CONTENTS

/* This is the routine that does most of the work. */
/* file is the name of the output file 
   target is the target type "a.out-i386-linux" for example
   secflags is ored with DEFAULT_SEC_FLAGS and then applied.
            It is used to say if the section is data or code. (Normally 
	    SEC_DATA.
   format  is the file format to write (Normally bfd_object)
   symname is the name to give the symbol.
   secname is the name of the section (.data is good)
   fd      is the file descriptor for the data to be put in the file. 
*/

int
encapsulate (const char *file, const char *target, unsigned long secflags,
	     bfd_format format, const char *symname,  char *lengthname,
	     const char *secname,
	     int fd) {
  bfd *abfd;
  asection *sec;
  size_t length;
  unsigned char *ptr;
  struct stat st;
  asymbol *syms[3];
  asymbol *new;

  bfd_init();
  bfd_check_init();

  abfd = bfd_openw(file, target);
  
  if (abfd == NULL) {
    fprintf (stderr, "Error Openning %s file %s: %s\n",target, file, 
	     bfd_errmsg (bfd_get_error()));
    return (bfd_get_error()<<16| errno);
  }

#ifdef __i386__  
  /* This wonderful poorly document call is absolutely necessary. */
  /* The linker deals with this in a much better way.  Perhaps the
     code should be borowed. */
  bfd_default_set_arch_mach (abfd, bfd_arch_i386, 0);
#endif

  if (!bfd_set_format (abfd, format)) {
    fprintf (stderr, "Error settting format: %s\n",
	     bfd_errmsg (bfd_get_error()));
    return (bfd_get_error()<<16| errno);
  }

  sec = bfd_make_section (abfd, secname);

  if (sec == NULL) {
    fprintf (stderr, "Error Making Section: %s\n",
	     bfd_errmsg (bfd_get_error()));
    return (bfd_get_error()<<16| errno);
  }

  if (!bfd_set_section_flags(abfd, sec, DEF_SEC_FLAGS | secflags)) {
    fprintf (stderr, "Error settting section flags %s\n",
	     bfd_errmsg (bfd_get_error()));
    return (bfd_get_error()<<16| errno);
  }
  
  /* We are ready for the data. */
  if (fstat (fd, &st)) {
    fprintf (stderr, "Error Stating Data File: %s\n", 
	     strerror (errno));
    return (errno);
  }
  
  length = st.st_size;

  /* create a new symbol */

  new = bfd_make_empty_symbol (abfd);
  if ((unsigned long)new < 4096) {
    fprintf (stderr, "Error allocating symbol: %s\n",
	     bfd_errmsg (bfd_get_error()));
    return (bfd_get_error()<<16| errno);
  }

  new->name = symname;
  new->section = sec;
  new->flags = BSF_EXPORT;
  new->value =sizeof (char *);
  syms[0]=new;


  new = bfd_make_empty_symbol (abfd);
  if ((unsigned long)new < 4096) {
    fprintf (stderr, "Error allocating symbol: %s\n",
	     bfd_errmsg (bfd_get_error()));
    return (bfd_get_error()<<16| errno);
  }

  new->name = lengthname;
  new->section = sec;
  new->flags = BSF_EXPORT;
  new->value =0;
  syms[1]=new;

  syms[2]=NULL;

  if (!bfd_set_symtab (abfd, syms, 2)) {
    fprintf (stderr, "Error setting symbol table: %s\n",
	     bfd_errmsg (bfd_get_error()));
    return (bfd_get_error()<<16| errno);
  }

  /* mmap would be good here, but if I do it, I can't easily
     add the length info.  Oh well
  ptr = mmap (0, length, PROT_READ, MAP_FILE|MAP_SHARED,  fd, 0); 

  if (ptr == NULL) {
    fprintf (stderr, "Error mmapping Data File: %s\n", 
	     strerror (errno));
    return (errno);
  }
*/
  ptr = malloc (length + sizeof (char *));

  if (ptr == NULL) {
    fprintf (stderr, "Error allocating memory for data file: %s\n", 
	     strerror (errno));
    return (errno);
  }

  /* this is technically a bug.  For example a ctrl-z bg may break this. */
  if (read (fd, ptr+sizeof (char *), length) != length) {
    fprintf (stderr, "Error reading file: %s\n", 
	     strerror (errno));
    return (errno);
  }
  *(unsigned long*)ptr = length;

  if (! bfd_set_section_size (abfd, sec, length+sizeof(char *))) {
    fprintf (stderr, "Error settting section size %s:\n",
	     bfd_errmsg (bfd_get_error()));
    return (bfd_get_error()<<16| errno);
  }

  if (! bfd_set_section_contents (abfd, sec, ptr, 0, length+sizeof (char *))) {
    fprintf (stderr, "Error settting section contents: %s\n",
	     bfd_errmsg (bfd_get_error()));
    return (bfd_get_error()<<16| errno);
  }

  /* write out everything. */
  bfd_close (abfd);
  return (0);
}

void 
usage (char *name) {
  fprintf (stderr, "usage %s: [-f format] [-n secname] [-t target] [-s secflags] output-file input-file symbol-name length-name\n", name);
  fprintf (stderr,"    --format format        file format to write.\n");
  fprintf (stderr,"    --section_name         name of the section (normally .data)\n");
  fprintf (stderr,"    --target target        target name (a.out-linux-i386 elf32-i386)\n");
  fprintf (stderr,"    --sectionflags flags   flags to put in section header\n");
  exit (1);
	   
}


int main (int argc, char *argv[]) {
  int c;
  char *outfile=NULL;
  char *infile=NULL;
  char *target="a.out-i386-linux";
  int secflags = SEC_DATA;
  bfd_format format = bfd_object;
  char *symname = NULL;
  char *lengthname = NULL;
  char *secname = ".data";
  int fd = -1;

  int option_index;
  static struct option long_options[] = {
    {"help", 0, 0, 'h'},
    {"format", 1, 0, 'f'},
    {"section-name", 1, 0, 'n'},
    {"target", 1, 0, 't'},
    {"sectionflags", 1, 0, 's'},
  };
  
  while ((c = getopt_long (argc, argv, "hf:t:n:s:",
			   long_options, &option_index)) > 0) {
    switch (c) {
    case 'h':
    case '?':
    default:
      usage(argv[0]);

    case 'f':
      format = atoi(optarg);
      break;

    case 'n':
      secname = optarg;
      break;

    case 't':
      target = optarg;
      break;

    case 's':
      secflags=atoi (optarg);
      break;

    }
  }
  if (argc - optind != 4) usage(argv[0]);

  outfile = argv[optind];
  infile= argv[optind+1];
  symname = argv[optind+2];
  lengthname = argv[optind+3];
  fd = open (infile, O_RDONLY);
  if (fd < 0) {
    fprintf (stderr, "error opening infile %s: %s\n", infile,
	     strerror(errno));
    exit (1);
  }

  if (encapsulate (outfile, target, secflags, format, symname, lengthname, 
		   secname, fd))
    return (1);

  return (0);
}
