/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 * portions of the NFS exclusive-lock emulation code are taken from the
 * Procmail & formail mail processing package,
 * Copyright (c) 1990-1994, S.R. van den Berg, The Netherlands.
 ***************************************************************************
 * MODULE: lockfile.c
 * lock file manipulation procedures.
 ***************************************************************************
 * File Locking Routines:
 * FILE * Lockcf( char *filename )
 *     locks the indicated file if it exists
 *     Returns: a FILE * for RW if successful, NULL otherwise;
 * FILE * Readlockfile(char *filename,pid_t *pid,char *str,int len,struct stat *s)
 *     reads the information in the lock file.
 * int Checklockfile(char *filename,pid_t *pid,char *str,int len,struct stat *s)
 *     Checks for the presence of a server by using kill(0,pid).
 *     Returns 1 if server present, 0 if not.
 *     Side Effect: if the server is present, returns PID and string from file
 *       in pid and str, stats the file as well
 * FILE * Getlockfile(char *filename,pid_t *pid,char *str,int len,struct stat *s)
 *     locks the indicated file;
 *     Returns a FILE * if successful, NULL otherwise;
 *     Side Effect: if the file is locked, returns PID and string from file
 *       in pid and str, stats the file as well
 * Setlockfile( FILE *fp; pid_t pid; char *str);
 *     write PID and str into the file
 * Closelockfile( FILE *fp );
 *     truncates the lock file and closes it.
 * int Exlockcf( char *filename )
 *    creates and locks the file.
 *    Returns: FILE * for writing, NULL if file exists and is locked
 ***************************************************************************
 * Lock File Manipulation:
 * Each active server has a lock file, which it uses to record its
 * activity.  The lock file is created and then locked;
 * the deamon will place its PID and an activity in the lock file.
 * Programs wanting to know the server status will read the file.
 * Note:  only active server, not status programs, will lock the file.
 * This prevents a status program from locking out a server.
 * Note that the information in the status file may be stale,  as the
 * may update the file without the knowledge of the checker.  This
 * appears to be a small price to pay;  the only place where it may
 * have an effect is when a job is being removed.
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"

extern off_t lseek ();
static int do_lock (), do_unlock ();

FILE *Lockcf (char *filename) {
    int fd;			/* fd for file descriptor */
    FILE *fp;			/* fp for FILE * */
    int lock, retries;

    /*
     * Open the lock file for RW
     */
    fp = NULL;
    assert (filename != NULL);
    user_to_daemon ();
    if ((fd = open (filename, O_RDWR, 0644)) < 0) {
	logerr (XLOG_DEBUG, "Lockcf: cannot open file %s", filename);
    }
    assert (fd >= 3);

    lock = do_lock (fd, filename);
    if (lock < 0) {
	logerr_die (XLOG_CRIT, "Lockcf: lock '%s' failed", filename);

    } else if (lock == 0) {
	log (XLOG_DEBUG, "Lockcf: lock failed: '%s' is already locked",
		filename);
	(void) close (fd);
	fd = -1;

    } else if ((fp = fdopen (fd, "r+")) == NULL) {
	logerr_die (XLOG_CRIT, "Lockcf: fdopen '%s' failed", filename);
    }
    daemon_to_user ();
    if (Debug > 5)
	log (XLOG_DEBUG, "Lockcf: file '%s', fd %d", filename, fd);
    return (fp);
}

FILE *Unlockcf (char *filename) {
    int fd;			/* fd for file descriptor */
    FILE *fp;			/* fp for FILE * */
    int lock;			/* lock status */

    /*
     * Open the lock file for RW
     */
    fp = NULL;
    assert (filename != NULL);
    user_to_daemon ();
    if ((fd = open (filename, O_RDWR, 0644)) < 0) {
	logerr (XLOG_DEBUG, "Unlockcf: cannot open file %s", filename);
    }
    assert (fd >= 3);

    if ((lock = do_unlock (fd, filename)) < 0) {
	logerr_die (XLOG_CRIT, "Unlockcf: lock '%s' failed", filename);

    } else if (lock == 0) {
	if (Debug > 3)
	    logerr (XLOG_DEBUG, "Unlockcf: lock failed '%s' active",
		    filename);
	(void) close (fd);
	fd = -1;

    } else if ((fp = fdopen (fd, "r+")) == NULL) {
	logerr_die (XLOG_CRIT, "Unlockcf: fdopen '%s' failed", filename);
    }
    if (Debug > 5)
	log (XLOG_DEBUG, "Unlockcf: file '%s', fd %d", filename, fd);
    daemon_to_user ();
    return (fp);
}

/***************************************************************************
 * File *Readlockfile( .... )
 * Utility function used by Getlockfile and Checklockfile
 * Implements the lock manipulation and reading of the lock file.
 ***************************************************************************/
FILE *Readlockfile (char *filename, pid_t *pid, char *str,
	int len, struct stat *sb)
{
    int fd;			/* file descriptor */
    FILE *fp = NULL;		/* stream */
    pid_t p;			/* ACME Integer and Buggy Works, Inc. */
    char buffer[BUFSIZ];	/* holds the pid read from file */
    char *cp;			/* ACME Pointer, Inc. */
    struct stat s;		/* try stating first */
    time_t before_open;

    if (Debug > 5)
	log (XLOG_DEBUG, "Readlockfile: lockfile '%s'", filename);

    before_open = time (NULL);	/* so we can check if we created it */

    /*
     * Open the lock file, creating if necessary
     */

    if ((fd = open (filename, O_RDWR | O_CREAT, 0644)) < 0) {
	fix_SD ();
	if ((fd = open (filename, O_RDWR | O_CREAT, 0644)) < 0) {
	    logerr_die (XLOG_DEBUG,
		"Readlockfile: cannot open/create lock file '%s'", filename);
	}
    }
    /* check to see that you are not clobbering stdin, stdout, stderr */
    assert (fd >= 3);

    /* need a FILE * for stdio */
    if ((fp = fdopen (fd, "r+")) == NULL) {
	logerr_die (XLOG_CRIT, "Readlockfile: fdopen '%s' failed", filename);
    }

    /* stat the file for use later */
    if (fstat (fd, &s) < 0) {
	logerr_die (XLOG_INFO, "Readlockfile: fstat '%s' failed", filename);
    }

    if (s.st_ctime >= before_open) {
	/* the lockfile was created _after_ we opened it -- we probably
	 * created it ourselves. Check the ownership, in case we're running
	 * as root; if we are, it may be owned by root, which will cause
	 * problems later, so fix it here. (note: we don't need to use
	 * fchown(); it's either owned by root or daemon, never user).
	 *
	 * Doing it this way is faster than using open_daemon() above.
	 */
	if (s.st_uid != DaemonUID) {
	    chown (filename, DaemonUID, DaemonGID);
	}
    }

    if (sb) {
	*sb = s;
    }
    if (Debug > 6)
	log (XLOG_DEBUG, "Readlockfile: %s, perms 0%o",
	     filename, s.st_mode);
    /* read the process number */
    if (fseek (fp, 0L, 0) < 0) {
	logerr_die (XLOG_INFO, "Readlockfile: fseek '%s' failed", filename);
    }
    p = 0;
    if (fgets (buffer, sizeof (buffer), fp)) {
	p = (pid_t) atoi (buffer);
    }
    if (Debug > 5)
	log (XLOG_DEBUG, "Readlockfile: '%s' pid %ld len %d",
	     filename, (long) p, s.st_size);
    if (fgets (buffer, sizeof (buffer), fp)) {
	if ((cp = (char *) strchr (buffer, '\n'))) {
	    *cp = 0;
	}
	if (Debug > 5)
	    log (XLOG_DEBUG, "Readlockfile: info '%s'", buffer);
	if (str) {
	    (void) strcpy (str, buffer);
	}
    }
    if (pid) {
	*pid = p;
    }
    if (fseek (fp, 0L, 0) < 0) {
	logerr_die (XLOG_INFO, "Readlockfile: fseek '%s' failed", filename);
    }
    return (fp);
}


/***************************************************************************
 * pid_t Checklockfile( ... )
 * Calls Readlockfile() to read the interesting information, passes
 * things onwards.
 ***************************************************************************/

pid_t Checklockfile (char *filename, pid_t *pid, char *str,
	int len, struct stat *sb)
{
    FILE *fp;			/* stream */
    pid_t p;			/* not another useless comment... */

    user_to_root ();		/* need to be root to use kill() */

    /*
     * Read the information
     */
    if ((fp = Readlockfile (filename, &p, str, len, sb)) != NULL) {
	(void) fclose (fp);
    }

    /* check to see if daemon present by using kill() */
    if (p != 0 && kill (p, 0) < 0) {
	if (Debug > 4) {
	    log (XLOG_DEBUG, "Checklockfile: server %d not present", p);
	}
	p = 0;
    }

    if (pid) {
	*pid = p;
    }

    root_to_user ();

    if (Debug > 4)
	log (XLOG_DEBUG, "Checklockfile: %s server %d", filename, p);
    return (p);
}


/***************************************************************************
 * FILE * Getlockfile( ... )
 * Calls Readlockfile() to read the interesting information, passes
 * things onwards.  Will try to lock the file; if it fails, assume status
 * correct.  Slight race condition here.
 ***************************************************************************/

FILE *
Getlockfile (char *filename, pid_t *pid, char *str,
	int len, struct stat *statb)
{
    FILE *fp;			/* lockfile */
    pid_t p;
    int lock;

    user_to_daemon ();          /* get the lock file (start) */

    fp = Readlockfile (filename, &p, str, len, statb);
    if (fp == NULL) {
	/* we can't even read the lock file, let alone lock it. fail. */
	if (Debug > 2)
	    log (XLOG_DEBUG, "Getlockfile: Readlockfile failed for '%s'", filename);
	daemon_to_user ();	/* get the lock file (broken) */
	return (fp);
    }
    if (pid != NULL) {
	*pid = p;
    }

    if ((lock = do_lock (fileno (fp), filename)) < 0) {
	logerr_die (XLOG_CRIT, "Getlockfile: lock '%s' failed", filename);

    } else if (lock > 0) {
	daemon_to_user ();          /* get the lock file (end) */
	return (fp);

    } else {
	if (Debug > 3) {
	    log (XLOG_DEBUG, "Getlockfile: '%s' is locked", filename);
	}
	(void) fclose (fp);
	fp = NULL;
    }

    daemon_to_user ();          /* get the lock file (end) */

    if (Debug > 2)
	log (XLOG_DEBUG, "Getlockfile: %s %s",
	     filename, fp != NULL ? "success" : "fail");
    return (fp);
}

/***************************************************************************
 * Setlockfile( FILE *lockfile; pid_t pid; char *str )
 * will write things into the lock file
 * This has the format:
 * line 1: pid\n
 * line 2: string\n
 ***************************************************************************/

void
Setlockfile (char *name, FILE *fp, pid_t pid, char *str) {
    user_to_daemon ();		/* clear the lock file (start) */
    assert (fp != NULL);

    if ((fseek (fp, 0L, 0) < 0) ||
		(ftruncate (fileno (fp), (off_t) 0) < 0))
    {
	logerr_die (XLOG_WARNING, "fseek/ftruncate %s", name);
    }

    if ((fprintf (fp, "%ld\n%s\n", (long) pid, str ? str : "") == EOF)
		|| (fflush (fp) == EOF))
    {
	/* make this friendlier: "no space left on device"
	 * may cause either to fail.
	 */
	logerr_die (XLOG_WARNING, "couldn't write to %s", name);
    }

    daemon_to_user ();		/* clear the lock file (end) */
    if (Debug > 4)
	log (XLOG_DEBUG, "Setlockfile: %s (%d), %s", name, pid,
	     str ? str : "(nil)");
}

/***************************************************************************
 * Closelockfile( FILE *lockfile)
 * Truncate the file.
 ***************************************************************************/

void
Closelockfile (char *name, FILE *fp) {
    if (Debug > 4)
	log (XLOG_DEBUG, "Closelockfile: closing %s", name);
    if (fp == NULL) {
	return;
    }
    user_to_daemon ();		/* clear the lock file (start) */
    (void) fflush (fp);
    if (ftruncate (fileno (fp), (off_t) 0) < 0) {
	logerr (XLOG_CRIT, "Closelockfile: cannot truncate %s", name);
	return;
    }
    if (fclose (fp) == EOF) {
	logerr (XLOG_CRIT, "Closelockfile: fclose failed %s", name);
	return;
    }
    daemon_to_user ();		/* clear the lock file (end) */
}

/***************************************************************************
 * int Exlockcf( char *userfile )
 * Create a user file in the directory and lock it. Prevents things
 * from getting confused.
 * jmason: made Exlockcf _really_ exclusive, and added NonExlockcf for
 * not-really-exclusive locks ;) (we need REALLY exclusive locks to avoid
 * getting burnt by the [8lgm] lpr security hole).
 * The NFS exclusive locks are slower than the real thing, but we need
 * them, otherwise we've got a crackable race condition. Bummer.
 ***************************************************************************/


#ifdef HAVE_SYS_TIMES_H
/* for times() */
#include <sys/times.h>
#endif

/* get a unique name for the temporary file used in NFS O_EXCL emulation.
 * This should be very difficult to guess, otherwise we end up with
 * our own version of the SunOS fsirand bug (bugid 1063470):
 *
 * >Non-random file handles can be guessed, leading to security hole.
 */
static char *get_unique_name (char *filename) {
    static unsigned long hash_value;
    unsigned long secs, usecs;
#ifdef HAVE_GETRUSAGE
    struct rusage rusageb;
#else
#ifdef HAVE_TIMES
    struct tms timesb;
#endif
#endif
    char *i, *unique_word, *dir;

    dir = strdup (filename);
    if ((i = strrchr (dir, '/'))) {
	*(i + 1) = '\0';
    } else {
	*dir = '\0';
    }

    hash_value ^= getpid ();		/* both longs */
    secs = usecs = (unsigned long) 0;	/* shut up purify */
    hires_time (&secs, &usecs);
    hash_value ^= secs ^ usecs;

    /* these should make it a bit less guessable. */
#ifdef HAVE_GETRUSAGE
    /* the preferred method -- more crud involved. ;) */
    getrusage (RUSAGE_SELF, &rusageb);
    hash_value ^= rusageb.ru_utime.tv_sec ^ rusageb.ru_utime.tv_usec ^
		  rusageb.ru_stime.tv_sec ^ rusageb.ru_stime.tv_usec ^
		  rusageb.ru_maxrss ^ rusageb.ru_msgrcv ^
		  rusageb.ru_nvcsw ^ rusageb.ru_nivcsw ^
    		  ((unsigned long) rusageb.ru_idrss << 010) ^
    		  ((unsigned long) rusageb.ru_minflt << 020) ^
    		  ((unsigned long) rusageb.ru_majflt << 030) ^
    		  (   ((((unsigned long) rusageb.ru_nswap)) ^
		      (((unsigned long) rusageb.ru_inblock) << 010) ^
		      (((unsigned long) rusageb.ru_oublock) << 020) ^
		      (((unsigned long) rusageb.ru_msgsnd) << 030)) <<
#if (SIZEOF_UNSIGNED_LONG < 5)	/* this is only here to avoid warnings */
			000		/* non-64-bit */
#else
			040		/* 64-bit */
#endif
		  );
#else
#ifdef HAVE_TIMES
    times (&timesb);
    hash_value ^= ((unsigned long) timesb.tms_utime) ^
		  ((unsigned long) timesb.tms_stime << 010) ^
		  ((unsigned long) timesb.tms_cutime << 020) ^
		  ((unsigned long) timesb.tms_cstime << 030);
#endif
#endif

    /* finally, throw in a few muls to stir it all up. */
    for (i = Host; *i; i++) { hash_value = hash_value * *i; }

    /* now guess that! ;) */

    unique_word = (char *) malloc (MAXPATHLEN);
    (void) sprintf (unique_word, "%s.nfs.%lx", dir, hash_value);
    free (dir);
    return (unique_word);
}

/* portions of this NFS exclusive-lock emulation code are taken from the
 * Procmail & formail mail processing package,
 * Copyright (c) 1990-1994, S.R. van den Berg, The Netherlands.
 *
 * deobfuscated & hacked on by jmason so it suits PLP's needs better.
 */
static int nfs_exopen (char *filename, int flags) {
    int fd;
    char *unique_name;
    struct stat stbuf, fdbuf;

    assert (flags & O_EXCL); /* sanity check */

    unique_name = NULL;
    /* get unique names until we can open it "exclusively". */
    while (1) {
	if (unique_name)
	    free (unique_name);
        unique_name = get_unique_name (filename);

	if (Debug > 4) log (XLOG_DEBUG, "file: '%s' tmp lock file: '%s'",
	    filename, unique_name);

	if (lstat (unique_name, &stbuf) == 0) {
	    /* this really shouldn't happen */
	    log (XLOG_INFO, "%s exists -- possible crack attempt?!", unique_name);
	    continue;
	}
	/* there's a race condition between the above line and here. */
	if ((fd = open (unique_name, flags, FILMOD)) <= 0) {
	    if (errno != EEXIST) {
		logerr (XLOG_INFO, "cannot open '%s'", unique_name);
		unlink (unique_name); free (unique_name);
		return -1;
	    }
	} else {
	    break;
	}
    }

    lstat (unique_name, &stbuf);
    fstat (fd, &fdbuf);

/* check the file descriptor we (and hopefully only we) have open,
 * against the filesystem's version of it.
 */
    if ((stbuf.st_nlink != 1) || (stbuf.st_size) ||
        (stbuf.st_dev != fdbuf.st_dev) || (stbuf.st_ino != fdbuf.st_ino) ||
        (stbuf.st_uid != fdbuf.st_uid) || (stbuf.st_gid != fdbuf.st_gid) ||
	(S_ISLNK(stbuf.st_mode)))
    {
	if (Debug) log (XLOG_DEBUG, "nfs-exopen: stat mismatch, bombing");
        close (fd);
	unlink (unique_name); free (unique_name);
	return -1;
    }
    if (link (unique_name, filename) < 0) {
	if (errno == EEXIST) {
	    if (Debug) log (XLOG_DEBUG, "'%s' already exists", filename);
	    close (fd);
	    unlink (unique_name); free (unique_name);
	    return -1;
	} else {
	    logerr_die (XLOG_INFO, "nfs exopen: link failed");
	}
    }
    if ((stat (filename, &stbuf) < 0) || (stbuf.st_nlink != 2) ||
	    (S_ISLNK(stbuf.st_mode)))
    {
	if (Debug) log (XLOG_DEBUG, "link() fucked up silently");
	close (fd);
	unlink (unique_name); free (unique_name);
	return -1;
    }

    flags ^= O_EXCL;	/* switch off exclusive flag here. */
    flags |= O_TRUNC;	/* turn on truncate flag */

    close (fd);
    if ((fd = open (filename, flags, FILMOD)) < 0) {
	unlink (unique_name); free (unique_name);
	logerr_die (XLOG_INFO, "nfs exopen: couldn't open '%s'", filename);
    }
    unlink (unique_name);
    free (unique_name);

    if (Debug > 4) log (XLOG_DEBUG, "exclusively opened '%s' (NFS-safe)",
	filename);
    return (fd);
}

#ifdef HAVE_SYS_MOUNT_H
#include <sys/mount.h>
#endif
#ifdef HAVE_SYS_STATVFS_H
#include <sys/statvfs.h>
#else
#ifdef HAVE_SYS_STATFS_H
#include <sys/statfs.h>
#endif
#endif
#ifdef HAVE_SYS_VFS_H
#include <sys/vfs.h>
#endif

/* Function to tell if a dir is on an NFS-mounted filesystem.
 */
int nfs_filesystem (char *dir) {
    {
	/* BSD systems: NFS filesystems have device ids of < 0.
	 * If only everything was BSD...
	 */
	struct stat buf;
	if (stat (dir, &buf) < 0) {
	    if (Debug > 4) {
		log (XLOG_DEBUG, "nfs_filesystem: cannot stat '%s'", dir);
	    }
	    return (0);
	}
	/* System V: st_dev is now unsigned long, but that can be
	 * cast anyway, so we don't need a #ifdef BSD or anything.
	 */
	if (buf.st_dev < 0) {
	    return (1);
	}
    }

#ifdef USE_STATVFS
    {
	/* If the statvfs f_basetype field contains the identifying
	 * string "nfs", it's an NFS filesystem (kludge! ;).
	 * (purify finds UMRs in the statvfs() and strcmp() calls here;
	 * This is a feature of the statvfs() implementation, not a PLP bug).
	 */
	struct statvfs buf;
	if (statvfs (dir, &buf) < 0) {
	    if (Debug > 4) {
		log (XLOG_DEBUG, "nfs_filesystem(USE_STATVFS): cannot statvfs '%s'", dir);
	    }
	    return (0);
	}
	if (buf.f_basetype && strsame (buf.f_basetype, "nfs")) {
	    return (1);
	}
    }
#endif

#ifdef USE_STATFS
    {
	/* this is a REAL cheat. If the active inode count is <= 0,
	 * we should use guaranteed-atomic locking (a la NFS); on
	 * some systems, this is always true if fs is nfs.
	 */
	struct statfs buf;
	if (statfs (dir, &buf) < 0) {
	    if (Debug > 4) {
		log (XLOG_DEBUG, "nfs_filesystem(USE_STATFS): cannot statfs '%s'", dir);
	    }
	    return (0);
	}
	if (buf.f_files <= 0) {
	    return (1);
	}
    }
#endif

    /* default: we can't really tell. If the :nw: printcap flag is
     * set, use NFS locking, otherwise don't bother.
     */
    if (NW)
	return (1);
    else
	return (0);
}

static int
_exlockcf (char *filename, int flags)	/* (this is the backend) */
{
    int fd;			/* fd for file descriptor */
    int lock;

    user_to_daemon ();		/* exlock the file (start) */

    /*
     * NFS does not handle O_EXCL correctly -- introduces race condition.
     * Use a special O_EXCL emulation function here.
     *
     * Julian Anderson <jules@comp.vuw.ac.nz>, 29 Sep 1994
     * This fragment was supposed to create the lock file, but the first thing
     * it does is stats the filename to attempt to determine if the spool
     * directory is NFS mounted.  Rather, we should stat the _directory_.
     */
    if (flags & O_EXCL) {
	char *dname;
	char *dend;

	dname = (char *) strdup (filename);
	if ((dend = strrchr(dname,'/'))) {
	  /* if there's a / in there, we can chop it at this point. */
	  *dend = '\0';
	} else {
	  /* just the filename. Replace it with ".". */
	  dname[0] = '.'; dname[1] = '\0';
	}

        if (nfs_filesystem(dname)) {
	  fd = nfs_exopen (filename, flags);
	} else {
	  fd = open (filename, flags, FILMOD);
	}
	free (dname);
    } else {
        fd = open (filename, flags, FILMOD);
    }

    if (fd < 0) {
	logerr_die (XLOG_CRIT, "lockcf: cannot create file '%s'", filename);
    }

    lock = do_lock (fd, filename);

    if (lock < 0) {
	logerr_die (XLOG_CRIT, "lockcf: lock '%s' failed", filename);

    } else if (lock == 0) {
	logerr_die (XLOG_CRIT, "lockcf: '%s' is already locked", filename);

    } else if (ftruncate (fd, (off_t) 0) < 0) {
	/* truncate the file to zero bytes -- security */
	logerr_die (XLOG_CRIT, "lockcf: cannot truncate %s", filename);
    }

    daemon_to_user ();		/* exlock the file (end) */
    return (fd);
}

inline int NonExlockcf (char *filename) {
    return (_exlockcf (filename, O_RDWR | O_CREAT));
}
inline int Exlockcf (char *filename) {
    return (_exlockcf (filename, O_RDWR | O_CREAT | O_EXCL));
}

/***************************************************************************
 * do_lock( fd , char *filename)
 * does a lock on a file;
 * Returns: 1 if successful; 0 if locked; -1 if lock fn failed
 ***************************************************************************/
static int
do_lock (int fd, char *filename) {
    int code;
#ifdef HAVE_FCNTL_LOCK
    struct flock file_lock;
#endif

    code = -1;
    if (Debug > 5) {
	log (XLOG_DEBUG, "do_lock: file %s, fd %d", filename, fd);
    }

#ifdef HAVE_FCNTL_LOCK
    file_lock.l_type = F_WRLCK;
    file_lock.l_whence = SEEK_SET;
    file_lock.l_start = 0;
    file_lock.l_len = 0;
    if (fcntl (fd, F_SETLK, &file_lock) < 0) {
	if (errno == EAGAIN || errno == EACCES) {
	    code = 0;
	} else {
	    if (Debug > 5)
		log (XLOG_DEBUG, "fcntl failed: %s", Errormsg (errno));
	    code = -1;
	}
    } else {
	code = 1;
    }

#else
#ifdef HAVE_LOCKF
    /*
     * want to try F_TLOCK
     */
    if (lockf (fd, F_TLOCK, 0L) < 0) {
	if (errno == EAGAIN) {
	    code = 0;
	} else {
	    if (Debug > 5)
		log (XLOG_DEBUG, "lockf failed: %s", Errormsg (errno));
	    code = -1;
	}
    } else {
	code = 1;
    }

#else
    /* last resort -- doesn't work over NFS */
    if (flock (fd, LOCK_EX | LOCK_NB) < 0) {
	if (errno == EWOULDBLOCK) {
	    code = 0;
	} else {
	    if (Debug > 5)
		log (XLOG_DEBUG, "flock failed: %s", Errormsg (errno));
	    code = -1;
	}
    } else {
	code = 1;
    }

#endif				/* HAVE_LOCKF */
#endif				/* HAVE_FCNTL_LOCK */

    if (Debug > 5)
	log (XLOG_DEBUG, "do_lock: status %d", code);
    return (code);
}

/***************************************************************************
 * do_unlock( fd , char *filename)
 * unlocks a lock on a file;
 * Returns: 1 if successful; 0 if locked; -1 otherwise
 * jmason: I haven't included the Edinburgh hack. if you don't know what
 * this is, you don't _want_ to know. ;)
 ***************************************************************************/
static int
do_unlock (int fd, char *filename) {
    int code;
#ifdef HAVE_FCNTL_LOCK
    struct flock file_lock;
#endif

    code = -1;
    if (Debug > 5)
	log (XLOG_DEBUG, "do_unlock: file %s, fd %d", filename, fd);

#ifdef HAVE_FCNTL_LOCK
    file_lock.l_type = F_UNLCK;

    if (Debug > 5)
	log (XLOG_DEBUG, "do_unlock: using fcntl with SEEK_SET");
    file_lock.l_whence = SEEK_SET;

    file_lock.l_start = 0;
    file_lock.l_len = 0;
    if (fcntl (fd, F_UNLCK, &file_lock) < 0) {
	if (Debug > 5)
	    log (XLOG_DEBUG, "fcntl failed: %s", Errormsg (errno));
	code = 0;
    } else {
	code = 1;
    }

#else
#ifdef HAVE_LOCKF
    /*
     * want to try F_ULOCK
     */
    if (lockf (fd, F_ULOCK, 0L) < 0) {
	code = 0;
    } else {
	code = 1;
    }

#else
    /* last resort */
    if (flock (fd, LOCK_UN) < 0) {
	if (errno == EWOULDBLOCK) {
	    code = 0;
	}
    } else {
	code = 1;
    }

#endif				/* HAVE_LOCKF */
#endif				/* HAVE_FCNTL_LOCK */

    if (Debug > 5)
	log (XLOG_DEBUG, "do_lock: status %d", code);
    return (code);
}


#ifdef HAVE_FTIME
#include <sys/timeb.h>
#endif

/*
 * get high-resolution timing information, using whatever's available.
 *
 */
void
hires_time (unsigned long *lo, unsigned long *hi) {
#ifdef HAVE_FTIME
    struct timeb tv;
    ftime (&tv);
    *lo = (unsigned long) tv.time;
    *hi = (unsigned long) tv.millitm;
#else

# ifdef HAVE_GETTIMEOFDAY
    struct timeval tv;
    (void) gettimeofday (&tv, NULL);
    *lo = (unsigned long) tv.tv_sec;
    *hi = (unsigned long) tv.tv_usec;
# else

#  ifdef HAVE_GETHRTIME
    *hi = (unsigned long) gethrtime ();
#  else

    *hi = 0;		/* no method of getting hi-res timing */
#  endif

    *lo = (unsigned long) time ((time_t *) 0);
# endif
#endif
}
