/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: Link_support.c
 ***************************************************************************
 * Support for the inter-machine communications
 * Link_open(int retry)
 *  opens a link to the remote host
 * Link_close()
 *  closes the link to the remote Printer
 * Link_send( char *l)
 *  sends a line to the remote host
 * Link_line( int retry, char *l)
 *  opens the link (with or without retry)
 *  sends a line to the remote host
 * Link_confirm()
 *  gets a single character back, confirming the action
 * Link_ack( c )
 *  sends a single character back, confirming the action
 * Link_copy( FILE *fp, count)
 *  copies a file to the remote host;
 *  if count bytes not transfered, an error is generated
 * Link_get()
 *  gets and prints all information  on stdout
 * Link_port_num()
 *	gets remote port number for connection
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"

static int Link_fd;		/* fd for the socket */
static int reserveport ();

/* these are used for non-RFC-compliant lpd's (like WSLPD).
 * Set up alarms so PLP doesn't hang forever during transfers.
 */
#define ALARM_TIMEOUT	30
static plp_signal_t (*old_alarm_fun)(int);
static char alarm_timed_out;

extern plp_signal_t got_alarm (int);

/***************************************************************************
 * Link_open(retry)
 * 1. Set up an inet socket;  a socket has a local host/local port and
 *    remote host/remote port address part.  The LPR software runs SUID
 *    root,  and will attempt to open a privileged port (number less than
 *    PRIV);  at the remote end this is checked to make sure that the
 *    remote machine is running SUID root.  Primitive,  but it is adequate
 *    in a trusting environment.
 *
 * (See Advanced 4.3 BSD IPC Tutorial for more information) The server on
 * the remote machine will be listening on the port given by the entry in
 * the /etc/service for the particular service desired.  If we are using
 * the "real" version of lpd this will be the "Printer" service, the test
 * version will use a "test" service.  The LPD daemon will open a socket
 * and bind to the appropriate "port" number.  The
 * getservbyname("printer","tcp"), is used to get the port information
 * (sp->s_port), which is used in a bind call.
 *
 * When we want to communicate with the lpd daemon on the remote machine,
 * listening on that particular port, we call getseverbyname(...) to get
 * the port number and protocol.  The remote host expects the local port
 * to be in a range that is available only to a root UID process, i.e.-
 * less than IPPORT_RESERVED.  When we open the local port, we get a local
 * port in this range.  At the remote end, the port number is checked to
 * ensure that it is in a valid range.  Since the reserved ports can only
 * be accessed by UID 0 (root) processes, this would appear to prevent
 * ordinary users from directly contacting the remote daemon.
 *
 ***************************************************************************/

/*
 * getport() gets a port,  in the correct range,  to the remote machine This code comes
 * from a description of the RLOGIN and LPD materials
 */
int
getport (char *hostname, int port_num) {
    struct hostent *host;	/* host entry pointer */
    int sock;			/* socket */
    int i;			/* generic integers */
    unsigned long inet_a, inet_addr ();	/* translate to INET address */

#if (SOCK_FAMILY==AF_UNIX)
    struct sockaddr_un sin;	/* unix socket address */
#else
    struct sockaddr_in sin;	/* inet socket address */
#endif

    /*
     * Zero out the sockaddr_in struct
     */
    bzero ((char *) &sin, sizeof (sin));
    /*
     * Get the host address and port number to connect to.
     */
    if (DbgRem > 4)
	log (XLOG_DEBUG, "getport: host %s", hostname);
#if (SOCK_FAMILY==AF_INET)
    if ((host = (gethostbyname (hostname)))) {
	/*
	 * set up the address information
	 */
	bcopy (host->h_addr, (caddr_t) & sin.sin_addr, host->h_length);
	sin.sin_family = host->h_addrtype;
    } else {
	if (DbgRem > 3) {
	    log (XLOG_DEBUG,
		 "getport: unknown host '%s', trying inet address", hostname);
	}
	inet_a = inet_addr (hostname);
	if (inet_a == (unsigned long) (-1)) {
	    fatal (XLOG_INFO, "getport: unknown host '%s'", hostname);
	}
	sin.sin_addr.s_addr = inet_a;
	sin.sin_family = AF_INET;
	if (DbgRem > 4)
	    log (XLOG_DEBUG, "getport: inet address %s", hostname);
    }
#endif
    /*
     * get the server name and protocol information from /etc/services
     */
    if (port_num == 0) {
	port_num = Link_port_num ();
    }
    /*
     * set up the address information
     */
#if (SOCK_FAMILY==AF_UNIX)
    sin.sun_family = SOCK_FAMILY;
    sprintf (sin.sun_path, "/tmp/lpdsock%d", port_num);
#else
    sin.sin_port = port_num;
#endif

    /*
     * Try connecting to the server.
     */
    if (DbgRem > 3)
	log (XLOG_DEBUG, "trying connection to %s, port %d",
	     hostname, port_num);

    /*
     * After this call, WE ARE ROOT! Remember to change back.
     */
    sock = reserveport (Maxportno, Minportno);
    if (sock < 0) {
	root_to_user ();
	return (-1);
    }
    old_alarm_fun = plp_signal (SIGALRM, got_alarm);
    alarm_timed_out = 0; alarm (ALARM_TIMEOUT);

#if (SOCK_FAMILY==AF_UNIX)
    i = connect (sock, (struct sockaddr *) & sin,
	sizeof (sin.sun_family) + strlen (sin.sun_path));
    if (i < 0) {
	if (DbgRem > 2) {
	    /* ?? solaris returns an ECHILD here! */
	    if (errno == EINTR || errno == ECHILD) {
		log (XLOG_INFO, "timed out waiting for connect to unix sock");
	    } else {
		logerr (XLOG_DEBUG, "connect failed to unix sock");
	    }
	}
	(void) close (sock);
	(void) plp_signal (SIGALRM, old_alarm_fun);
	alarm (0);
	root_to_user ();
	return (-1);
    }
#else
    i = connect (sock, (struct sockaddr *) & sin, sizeof (sin));
    if (i < 0) {
	if (DbgRem > 2) {
	    /* ?? solaris returns an ECHILD here! */
	    if (errno == EINTR || errno == ECHILD) {
		log (XLOG_INFO, "timed out waiting for connect to %s", RM);
	    } else {
		logerr (XLOG_DEBUG, "connect failed to %s", RM);
	    }
	}
	(void) close (sock);
	(void) plp_signal (SIGALRM, old_alarm_fun);
	alarm (0);
	root_to_user ();
	return (-1);
    }
#endif
    (void) plp_signal (SIGALRM, old_alarm_fun);
    alarm (0);
    root_to_user ();
    return (sock);
}

plp_signal_t 
got_alarm (int signal) {
    alarm_timed_out = 1;
}

/*
 * reserveport(int port_no, min_port_no) Reserve a port, starting at port_no, down to
 * min_port_no. while port_no > min_port_no try to grab the port; if you can, then return
 * socket else return -1 Returns: socket if successful, -1 if not
 */
#if (SOCK_FAMILY==AF_UNIX)
static int
reserveport (int port_no, int min_port_no) {
    int sock;

    user_to_root ();
    sock = socket (SOCK_FAMILY, SOCK_STREAM, 0);
    if (sock < 0) {
	root_to_user ();
	logerr_die (XLOG_INFO, "reserveport socket call failed");
    } else if (sock == 0) {
	root_to_user ();
	logerr_die (XLOG_INFO, "reserveport: socket returns 0");
    }
    return (sock);
}

#else
#ifdef HAVE_RRESVPORT

static int
reserveport (int port_no, int min_port_no) {
    int sock;

    user_to_root ();		/* WE STAY ROOT AFTER THIS! Call root_to_user() yourself */

    if ((sock = rresvport (&port_no)) == -1) {
	logerr_die (XLOG_INFO, "rresvport");
    }
    if (DbgRem > 4)
	log (XLOG_DEBUG, "reserveport got socket %d", port_no);
    return (sock);
}

#else				/* no rresvport() */

/*
 * This often doesn't work. YMMV; SunOS and Solaris don't like it, at least. Use
 * rresvport() if you can.
 */
static int
reserveport (int port_no, int min_port_no) {
    struct sockaddr_in sin;
    int sock;

    sock = socket (AF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
	logerr_die (XLOG_INFO, "reserveport socket call failed");
    } else if (sock == 0) {
	logerr_die (XLOG_INFO, "reserveport: socket returns 0");
    }
    user_to_root ();		/* grep: reserveport, no rresvport() */
#ifdef TEST_PRIV
    printf ("reserveport: invoked by: uid=%d, euid=%d\n", getuid (), geteuid ());
#endif
    while (port_no >= min_port_no) {
	bzero ((char *) &sin, sizeof (sin));

	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = 0;
	sin.sin_port = htons ((u_short) port_no);

	if (bind (sock, (struct sockaddr *) & sin, sizeof (sin)) != -1) {
	    if (DbgRem > 4)
		log (XLOG_DEBUG, "reserveport got socket %d", port_no);
	    /* WE STAY ROOT AFTER THIS! Call root_to_user() yourself */
	    return (sock);
	}
	if (errno != EADDRINUSE && errno != EADDRNOTAVAIL) {
	    /* WE STAY ROOT AFTER THIS! Call root_to_user() yourself */
	    logerr_die (XLOG_INFO, "reserveport: bind failed");
	}
	--port_no;
    }
    /* WE STAY ROOT AFTER THIS! Call root_to_user() yourself */
    (void) close (sock);
    return (-1);
}

#endif				/* no rresvport() */
#endif				/* AF_INET sockets */

int
Link_open (int retry) {
    unsigned int i = 1;
    int l;

    if (Link_fd) {
	return (JSUCC);
    }
    while (Link_fd == 0) {
	if (DbgRem > 3)
	    log (XLOG_DEBUG, "open link: attempt %d (max retries %d)", i, retry);
	l = getport (RM, 0);
	if (l < 0) {
	    if (i++ > retry) {
		if (DbgRem > 3)
		    log (XLOG_DEBUG, "retries expired; failed to open link");
		return (JFAIL);
	    }
	    sleep ((unsigned) 15);

	} else if (l == 0) {
	    fatal (XLOG_INFO, "Argh! cannot happen - fd = 0");
	} else {
	    Link_fd = l;
	}
    }
    if (DbgRem > 3)
	log (XLOG_DEBUG, "opened connection to %s", RM);
    return (JSUCC);
}

/***************************************************************************
 * 1. close the link
 ***************************************************************************/
void
Link_close (void) {
    if (Link_fd) {
	if (DbgRem > 4)
	    log (XLOG_DEBUG, "closing connection");
	(void) close (Link_fd);
    }
    Link_fd = 0;
}

/***************************************************************************
 * Link_line( int retry; char *line )
 * send a line to the remote end
 * if retry != 0, blocks until connection made
 ***************************************************************************/

int
Link_line (int retry, char *str) {
    if (Link_open (retry) != JSUCC) {
	return (JFAIL);
    }
    return (Link_send (str));
}

int
Link_send (char *str) {
    int i, l;			/* ACME Integers, Inc. */

    l = strlen (str);
    if (Link_fd == 0) {
	log (XLOG_INFO, "Link_send: link to %s not open", RM);
	return (JFAIL);
    }
    if (l != 0 && (i = write (Link_fd, str, l)) != l) {
	if (i < 0) {
	    logerr (XLOG_INFO, "write error to remote site %s", RM);
	}
	Link_close ();
	return (JFAIL);
    }
    if (DbgRem > 4)
	log (XLOG_DEBUG, "sent to %s: '%d'%s", RM, *str, str + 1);
    return (JSUCC);
}

/***************************************************************************
 * Link_confirm()
 *  gets a single character back, confirming the action
 ***************************************************************************/

int
Link_confirm (void) {
    char buf[256];		/* buffer, one line (ish) */
    int n;

    if (Link_fd == 0) {
	log (XLOG_INFO, "link to %s not open", RM);
	return (JFAIL);
    }
    old_alarm_fun = plp_signal (SIGALRM, got_alarm);
    alarm_timed_out = 0; alarm (ALARM_TIMEOUT);

    n = read (Link_fd, buf, 1);
    (void) plp_signal (SIGALRM, old_alarm_fun);
    alarm (0);

    if (n != 1) {
	/* ?? solaris returns an ECHILD here! */
	if (errno == EINTR || errno == ECHILD) {
	    log (XLOG_INFO, "timed out waiting for confirm from %s", RM);
	} else {
	    if (n == 0) {
		logerr (XLOG_INFO, "remote host %s: failed to ack transfer", RM);
	    } else {
		logerr (XLOG_INFO, "remote host %s: illegal ack code '%s'", RM, buf);
	    }
	}
	Link_close ();
	return (JNOCONF);		/* the job may still have printed */
    }

    if (buf[0] == 0) {
	if (DbgRem > 4)
	    log (XLOG_DEBUG, "successful confirm from %s", RM);
	return (JSUCC);
    } else if (buf[0] == '\002') {
	if (DbgRem > 4)
	    log (XLOG_DEBUG, "spool area full from %s", RM);
	return (JFULL);
    } else {
	/* read more, in case it's a line (shouldn't be, though!) */
	n = read (Link_fd, buf + sizeof(char), sizeof (buf) - 2);
	if (n >= 0) { buf[n] = '\0'; }	/* just in case */
	if (DbgRem > 4)
	    log (XLOG_DEBUG, "bad confirm from %s: \"%s\"", RM, buf);
	Link_close ();
	return (JFAIL);
    }
}

/***************************************************************************
 * Link_copy( FILE *fp; long count; char *name)
 *  copies a file to the remote nost;
 *  if count bytes not transfered, an error is generated
 ***************************************************************************/

int
Link_copy (FILE *fp, long count, char *name) {
    char buf[BUFSIZ];		/* buffer */
    int i, l;			/* ACME Integer, Inc. */
    long total;			/* total number of bytes */

    if (DbgRem > 4)
	log (XLOG_DEBUG, "sending %s (%d bytes) to %s", name, count, RM);
    if (Link_fd == 0) {
	log (XLOG_INFO, "link is not open");
	goto error;
    }
    total = 0;
    while ((i = fread (buf, 1, sizeof (buf), fp)) > 0) {
	total = total + i;
	if (total > count) {
	    log (XLOG_DEBUG,
		 "file '%s': length is %d instead of %d bytes",
		 name, total, count);
	    goto error;
	}
	if ((l = write (Link_fd, buf, i)) != i) {
	    if (l < 0) {
		logerr (XLOG_INFO, "write error while sending '%s' to %s",
			name, RM);
	    } else {
		logerr (XLOG_INFO, "partial write while sending '%s' to %s",
			name, RM);
	    }
	    goto error;
	}
    }
    if (i < 0) {
	logerr (XLOG_INFO, "file '%s' read error", name);
	goto error;
    } else if (total != count) {
	log (XLOG_DEBUG,
	     "file '%s', copied %d instead of %d bytes to %s",
	     name, total, count, RM);
	goto error;
    }
    if (DbgRem > 4) {
	log (XLOG_DEBUG, "sent %s (%d bytes) to %s", name, count, RM);
    }
    return (JSUCC);
error:
    Link_close ();
    return (JFAIL);
}

int
Link_ack (int c) {
    char buf[1];
    int succ;

    buf[0] = c;
    succ = JFAIL;

    if (write (Link_fd, buf, 1) != 1) {
	if (DbgRem > 4)
	    logerr (XLOG_DEBUG, "ack '%d' write error to %s", c, RM);
    } else {
	succ = JSUCC;
	if (DbgRem > 4)
	    log (XLOG_DEBUG, "ack '%d' sent to site %s", c, RM);
    }
    if (succ != JSUCC) {
	Link_close ();
    }
    return (succ);
}

/***************************************************************************
 * reads all information from the link, and prints on stdout
 ***************************************************************************/
void
Link_get (void) {
    int i;			/* ACME Integers, Inc. */
    char buf[BUFSIZ];		/* buffer */

    if (Link_fd == 0) {
	fatal (XLOG_INFO, "link is not open");
    }
    while ((i = read (Link_fd, buf, sizeof (buf))) > 0) {
	(void) write (1, buf, i);
    }
}

/***************************************************************************
 * - look up the service in the service directory using getservent
 * - if the port number has been set, don't do it a second time.
 ***************************************************************************/

int
Link_port_num (void) {
    struct servent *sp;
    unsigned short int port;

    if (Lpr_port_num == 0) {
	if (!Printer_port || !*Printer_port) {
	    logerr_die (XLOG_INFO, "printer-port is not set!");
	}
	if ((*Printer_port >= '0') && (*Printer_port <= '9')) {
	    /*
	     * starts with a number -> must be a port number. let sscanf do our range
	     * checking: %hu -> unsigned short int
	     */
	    if (sscanf (Printer_port, "%hu", &port) != 1) {
		logerr_die (XLOG_INFO,
			    "%s is not in the valid port range", Printer_port);
	    }
	    Lpr_port_num = port;

	} else {
	    if ((sp = getservbyname (Printer_port, "tcp")) == 0) {
		logerr_die (XLOG_INFO,
			    "getservbyname(\"%s\",tcp) failed", Printer_port);
	    }
	    Lpr_port_num = sp->s_port;
	}
    }
    return (Lpr_port_num);
}
