/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: Errormsg.c
 * error messages and logging
 * log:
 *  --message on stderr and on stdout if "Echo_on_stdout" is set
 *  --if the severity is high enough, will also log using syslog().
 *  --saves the last "non-Debug" log message
 * logerr:
 *  --formats the error message corresponding to "errno" and calls log();
 * fatal:
 *  -- log() and exit() with Errorcode
 * logerr_die:
 *  -- logerr() and exit() with Errorcode
 **************************************************************************/

#include "lp.h"
#include "library/errormsg.h"

static void use_syslog ();

#ifdef HAVE_STRERROR
extern char *strerror ();
#else
extern char *sys_errlist[];
#define strerror(err)	(sys_errlist[err])
#endif

#ifdef HAVE_SYS_NERR
/* unused; all systems tested so far have NSIG
 * to be honest, I can't remember where I got sys_nerr from ;)
 */
#define num_signals()	(sys_nerr)
#else
#ifdef NSIG
#define num_signals()	(NSIG)
#else
/* need other values here! */
#define num_signals()	(99)
#endif
#endif

#define LOG_MESSAGE_LEVEL	/* matter of preference */

/*
 * Errormsg(err) returns a printable form of the errormessage corresponding to errno.
 */

char *
Errormsg (err)
    int err;
{
    char *cp;
    static char msgbuf[16];	/* holds "errno=%d". */

    if (err >= 0 && err <= num_signals()) {
	cp = strerror (err);
    } else {
	(void) sprintf (msgbuf, "errno=%d", err);
	cp = msgbuf;
    }
    return (cp);
}

#ifdef LOG_MESSAGE_LEVEL
struct msgkind {
    int *var;
    char *str;
};
static struct msgkind msg_name[] = {
    {&XLOG_CRIT, " (CRIT)"},
    {&XLOG_ERR, " (ERR)"},
    {&XLOG_WARNING, " (WARN)"},
    {&XLOG_NOTICE, " (NOTICE)"},
    {&XLOG_INFO, " (INFO)"},
    {&XLOG_DEBUG, ""},
    {0}
};

static
char *
logmsg (kind)
    int kind;
{
    int i;
    static char b[35];

    for (i = 0; msg_name[i].var; ++i) {
	if (*msg_name[i].var == kind) {
	    return (msg_name[i].str);
	}
    }
    (void) sprintf (b, "bad message type %d", kind);
    return (b);
}

#endif

#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#else
static FILE *f;
#endif
static int init = 0;

static void
use_syslog (int kind, char *msg) {
    if (Debug > 7) {
	(void) fprintf (stderr, "syslog: %s\n", msg);
    }

#ifndef HAVE_SYSLOG_H
    if (f != NULL || (f = fopen ("/dev/console", "w")) != NULL) {
	(void) fprintf (f, "syslog: LPD %s\n", msg);
	(void) fflush (f);
    }
#else				/* HAVE_SYSLOG_H */
#if defined(LOG_PID) && defined(LOG_NOWAIT)
    /* assume we can use openlog(). */
    if (init == 0) {
	if (Name && *Name) {
	    openlog (Name, LOG_PID | LOG_NOWAIT, SYSLOG_FACILITY);
	} else {
	    /* guess lpd. */
	    openlog ("lpd", LOG_PID | LOG_NOWAIT, SYSLOG_FACILITY);
	}
	init = 1;
    }
    (void) syslog (kind, msg);
#else
    (void) syslog (SYSLOG_FACILITY | kind, msg);
#endif
#endif				/* HAVE_SYSLOG_H */
}

static char log_buf[1024];

static void
log_backend (int kind) {
    static char stamp_buf[99];

    plp_block_signals ();
    /*
     * syslogd does most of the things we do after this point, so there's no point in
     * filling up the logs with useless stuff; log the message here.
     */
    if (kind <= XLOG_INFO)
	use_syslog (kind, log_buf);

    *stamp_buf = '\0';
    (void) sprintf (stamp_buf, "%s", Time_str ());
    if (ShortHost && *ShortHost) {
	(void) sprintf (stamp_buf + strlen (stamp_buf), " %s", ShortHost);
    }
    if (Name && *Name) {
	(void) sprintf (stamp_buf + strlen (stamp_buf), " %s", Name);
    }
    if (Debug) {
        (void) sprintf (stamp_buf + strlen (stamp_buf), "[%ld]", (long) getpid ());
#ifdef LOG_MESSAGE_LEVEL
	(void) sprintf (stamp_buf + strlen (stamp_buf), "%s", logmsg (kind));
#endif
    }
    if (Echo_on_stdout) {
	(void) fprintf (stdout, "%s %s\n", stamp_buf, log_buf);
	(void) fflush (stdout);
    }
    (void) fprintf (stderr, "%s %s\n", stamp_buf, log_buf);

    if (kind != XLOG_DEBUG) {
	(void) sprintf (Last_errormsg, "%s %s\n", stamp_buf, log_buf);
    }
    plp_unblock_signals ();
}

/* VARARGS2 */
void
#ifdef HAVE_STDARGS
log (int kind, char *msg,...)
#else
log (va_alist) va_dcl
#endif
{
#ifndef HAVE_STDARGS
    int kind;
    char *msg;
#endif
    VA_LOCAL_DECL

    *log_buf = '\0';
    if (Printer && *Printer) { (void) sprintf (log_buf, "%s: ", Printer); }
    VA_START (msg);
    VA_SHIFT (kind, int);
    VA_SHIFT (msg, char *);
    (void) vsprintf (log_buf + strlen (log_buf), msg, ap);
    VA_END;
    log_backend (kind);
}

/* VARARGS2 */
void
#ifdef HAVE_STDARGS
fatal (int kind, char *msg,...)
#else
fatal (va_alist) va_dcl
#endif
{
#ifndef HAVE_STDARGS
    int kind;
    char *msg;
#endif
    VA_LOCAL_DECL

    *log_buf = '\0';
    if (Printer && *Printer) { (void) sprintf (log_buf, "%s: ", Printer); }

    VA_START (msg);
    VA_SHIFT (kind, int);
    VA_SHIFT (msg, char *);
    (void) vsprintf (log_buf + strlen (log_buf), msg, ap);
    VA_END;

    log_backend (kind);
    cleanup (0);
    exit (Errorcode);
}

/* VARARGS2 */
void
#ifdef HAVE_STDARGS
logerr (int kind, char *msg,...)
#else
logerr (va_alist) va_dcl
#endif
{
#ifndef HAVE_STDARGS
    int kind;
    char *msg;
#endif
    VA_LOCAL_DECL
    int err = errno;

    *log_buf = '\0';
    if (Printer && *Printer) { (void) sprintf (log_buf, "%s: ", Printer); }
    VA_START (msg);
    VA_SHIFT (kind, int);
    VA_SHIFT (msg, char *);
    (void) vsprintf (log_buf + strlen (log_buf), msg, ap);
    VA_END;
    (void) sprintf (log_buf + strlen (log_buf), " - %s", Errormsg (err));
    log_backend (kind);
    errno = err;
}

/* VARARGS2 */
void
#ifdef HAVE_STDARGS
logerr_die (int kind, char *msg,...)
#else
logerr_die (va_alist) va_dcl
#endif
{
#ifndef HAVE_STDARGS
    int kind;
    char *msg;
#endif
    VA_LOCAL_DECL
    int err = errno;

    *log_buf = '\0';
    if (Printer && *Printer) { (void) sprintf (log_buf, "%s: ", Printer); }

    VA_START (msg);
    VA_SHIFT (kind, int);
    VA_SHIFT (msg, char *);
    (void) vsprintf (log_buf + strlen (log_buf), msg, ap);
    VA_END;

    (void) sprintf (log_buf + strlen (log_buf), " - %s", Errormsg (err));
    log_backend (kind);
    cleanup (0);
    exit (Errorcode);
}

/***************************************************************************
 * Diemsg( char *m1, *m2, ...)
 * print error message to stderr, and die
 ***************************************************************************/

/* VARARGS1 */
void
#ifdef HAVE_STDARGS
Diemsg (char *msg,...)
#else
Diemsg (va_alist) va_dcl
#endif
{
#ifndef HAVE_STDARGS
    char *msg;
#endif
    VA_LOCAL_DECL

    (void) fprintf (stderr, "%s: Fatal error - ", Name);

    VA_START (msg);
    VA_SHIFT (msg, char *);
    (void) vfprintf (stderr, msg, ap);
    VA_END;

    (void) fprintf (stderr, "\n");
    cleanup (0);
    exit (1);
}

/***************************************************************************
 * Warnmsg( char *m1, *m2, ...)
 * print warning message to stderr
 ***************************************************************************/

/* VARARGS1 */
void
#ifdef HAVE_STDARGS
Warnmsg (char *msg,...)
#else
Warnmsg (va_alist) va_dcl
#endif
{
#ifndef HAVE_STDARGS
    char *msg;
#endif
    VA_LOCAL_DECL

    (void) fprintf (stderr, "%s: Warning - ", Name);

    VA_START (msg);
    VA_SHIFT (msg, char *);
    (void) vfprintf (stderr, msg, ap);
    VA_END;

    (void) fprintf (stderr, "\n");
}

/*
 * this shouldn't be called before Setuplog() or Std_environ(), as they'll close the
 * syslog's fd (which this keeps open).
 */

/* VARARGS1 */
void
#ifdef HAVE_STDARGS
log_request (char *fmt,...)
#else
log_request (va_alist) va_dcl
#endif
{
#ifndef HAVE_STDARGS
    char *fmt;
#endif
    VA_LOCAL_DECL
    static int log_is_open = 0;
    char buf[BUFSIZ];

    if (!log_is_open) {
        openlog ("lpd", LOG_PID, SYSLOG_FACILITY);
        log_is_open++;
    }

    VA_START (fmt);
    VA_SHIFT (fmt, char *);
    (void) vsprintf (buf, fmt, ap);
    VA_END;

    (void) syslog (LOG_INFO, buf);
    if (Debug) {
	(void) strcpy (log_buf, buf);
	log_backend (XLOG_DEBUG);
    }

    /* exiting will close the logfile */
}
