/***************************************************************************
* Module: hold.c
* Purpose: holds the job in printer queue
****************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "library/utils.h"
#include "common/hold.h"

/***************************************************************************
 *  a simple application of the hold decision table
 *
 *  user   local  job   from  perms inlist    hold?
 *  ===============================================================
 *  root   yes    *     *     *     *         yes
 *  root   yes    *     *     *     yes       yes
 *  root   no     *     mach  *     *         yes
 *  root   no     *     mach  *     yes       yes
 *  user   -      user  mach  R     yes       yes
 *  user   -      *     *     C     yes       yes
 *
 * Returns: 1 if hold is indicated, 0 otherwise
 ***************************************************************************/
int
shouldhold (struct plp_queue *q, int control_perms) {
    int i, same_host;

    if (Debug > 5)
	log (XLOG_DEBUG, "shouldhold: Person = %s, From = %s, job %s@%s",
	     Person, FQDN, q->q_user, &q->q_from);
    same_host = (hostcmp (FQDN, &q->q_from) == 0);
    i = Match_entry (q);
    if (Debug > 5)
	log (XLOG_DEBUG, "shouldhold: Match_entry = %d", i);
    if (Is_root && Is_local && i)
	return (1);
    if (Is_root && !Is_local && same_host && i)
	return (1);
    if (strsame (Person, q->q_user) && same_host && i)
	return (1);
    if (i && control_perms)
	return (1);
    return (0);
}

/***************************************************************************
 *  a simple application of the release decision table
 *
 *  user   local  job   from  perms inlist    release?
 *  ===============================================================
 *  root   yes    *     *     *     *         yes
 *  root   yes    *     *     *     yes       yes
 *  root   no     *     mach  *     *         yes
 *  root   no     *     mach  *     yes       yes
 *  user   -      user  mach  R     yes       yes
 *  user   -      *     *     C     yes       yes
 *
 * Returns: 1 if release is indicated, 0 otherwise
 ***************************************************************************/
static int
shouldrelease (struct plp_queue *q, int control_perms) {
    int i, same_host;

    assert(q!=(struct plp_queue*)0);
    if (Debug > 5)
	log (XLOG_DEBUG, "shouldrelease: Person = %s, From = %s, job %s@%s",
	     Person, FQDN, q->q_user, &q->q_from);
    same_host = (hostcmp (FQDN, &q->q_from) == 0);
    i = Match_entry (q);
    if (Debug > 5)
	log (XLOG_DEBUG, "shouldrelease: Match_entry = %d", i);
    if (Is_root && Is_local && i)
	return (1);
    if (Is_root && !Is_local && same_host && i)
	return (1);
    if (strsame (Person, q->q_user) && same_host && i)
	return (1);
    if (i && control_perms)
	return (1);
    return (0);
}

/***************************************************************************
 * hold_job(FILE *fp; struct plp_queue *q)
 * Hold a job
 * ACTIONS:
 * 1. Change first character of control file name from 'c' to 'h'
 ***************************************************************************/

static int
hold_job (FILE *fp, struct plp_queue *q) {
    char newname[CFNAMELEN + 1];
    char oldname[CFNAMELEN + 1];

    (void) strcpy (newname, q->q_name);
    (void) strcpy (oldname, q->q_name);
    *newname = 'h';
    if (Debug > 4) {
	log (XLOG_DEBUG, "hold_job: renaming %s to %s ", oldname, newname);
	    }
    Setuid_debug("hold_job");
    user_to_daemon ();
    if (rename (q->q_name, newname) < 0) {
	/* Problem renaming file */
	logerr (XLOG_INFO, "move: rename of %s to %s failed", oldname, newname);
	daemon_to_user (); /* We are still daemon in this case, so switch back */
	Setuid_debug("hold_job");
	return (1);
    }
    daemon_to_user ();
    Setuid_debug("hold_job");
    return (0);
}

/***************************************************************************
 * dohold(struct plp_queue *q)
 * hold the job
 * 1. Lock the control file.
 * 2. If unsuccessful, find the server PID and kill it off.
 * 3. hold the job.
 ***************************************************************************/
static void
dohold (struct plp_queue *q) {
    FILE *cfp;

    if ((cfp = Lockcf (q->q_name)) == NULL) {
	/* hmmm... looks like an active server */
	if ((cfp = fopen_daemon (q->q_name, "rw")) == NULL) {
	    /* nope, the file has really gone */
	    logerr (XLOG_INFO, "control file %s not readwritable", q->q_name);
	    return;
	}
	/* well, we will just have to kill off the server */
	if (q->q_daemon == 0) {
	    /*
	     * Hmmm... we have this fellow running the file, and it is locked.  That
	     * means that it just started running this guy. Better check again.
	     */
	    (void) Checkactive ();
	}
	if (q->q_daemon) {
	    (void) fprintf (stdout, "Killing off %s server %d.",
			    q->q_server, q->q_daemon);
	    if (killpg (q->q_daemon, SIGINT) < 0) {
		if (Debug > 2)
		    log (XLOG_DEBUG,
			 "server %s (%d) was not alive",
			 q->q_server, q->q_daemon);
	    }
	}
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "dohold: holding files for job %s", q->q_name);
    hold_job (cfp, q);
    (void) fclose (cfp);
}

/***************************************************************************
 * only issues a message...
 ***************************************************************************/

static void
Remote_hold (void) {
    fprintf (stderr, "Remote hold asked for.\n");
}

/***************************************************************************
 * dorelease(struct plp_queue *q)
 * release the job
 * 1. Lock the control file.
 * 2. If unsuccessful, find the server PID and kill it off.
 * 3. release the job.
 ***************************************************************************/
static void
dorelease (struct plp_queue *q) {
    FILE *cfp;

    assert(q!=(struct plp_queue*)0);
    if ((cfp = Lockcf (q->q_name)) == NULL) {
	/* hmmm... looks like an active server */
	if ((cfp = fopen_daemon (q->q_name, "rw")) == NULL) {
	    /* nope, the file has really gone */
	    logerr (XLOG_INFO, "control file %s not readwritable", q->q_name);
	    return;
	}
	/* well, we will just have to kill off the server */
	if (q->q_daemon == 0) {
	    /*
	     * Hmmm... we have this fellow running the file, and it is locked.  That
	     * means that it just started running this guy. Better check again.
	     */
	    (void) Checkactive ();
	}
	if (q->q_daemon) {
	    (void) fprintf (stdout, "Killing off %s server %d.",
			    q->q_server, q->q_daemon);
	    if (killpg (q->q_daemon, SIGINT) < 0) {
		if (Debug > 2)
		    log (XLOG_DEBUG,
			 "server %s (%d) was not alive",
			 q->q_server, q->q_daemon);
	    }
	}
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "release files for job %s", q->q_name);
    release_job (cfp, q);
    (void) fclose (cfp);
}

/***************************************************************************
 * only issues a message...
 ***************************************************************************/
static void
Remote_release (void) {
    fprintf (stderr, "Remote release asked for.\n");
}

/***************************************************************************
 * release_job(FILE *fp; struct plp_queue *q)
 * Release a job
 * ACTIONS:
 * 1. Change first character of control file name from 'h' to 'c'
 ***************************************************************************/
int 
release_job (FILE *fp, struct plp_queue *q) {
    char newname[CFNAMELEN + 1];
    char oldname[CFNAMELEN + 1];

    (void) strcpy (newname, q->q_name);
    *newname = 'c';
    if (Debug > 4) {
	log (XLOG_DEBUG, "release_job: renaming %s to %s ", oldname, newname);
	    }
    Setuid_debug("release_job");
    user_to_daemon ();
    if (rename (q->q_name, newname) < 0) {
	/* Problem renaming file */
	logerr (XLOG_INFO, "move: rename of %s to %s failed", oldname, newname);
	daemon_to_user (); /* We are still daemon in this case, so switch back */	
	Setuid_debug("release_job");
	return (1);
    }
    daemon_to_user ();
    Setuid_debug("release_job");
    return (0);
}

/***************************************************************************
* hold ()
* Purpose: holds the job in printer queue
****************************************************************************/
void
hold () {
    int i;
    struct plp_queue *q;	/* job entry */
    int perms;			/* hold perms values */
    int control_perms;		/* has control perms */

    /*
     * If explicitly asked for debug then echo messages to stdout otherwise (db flag in
     * printcap) just log messages in file.
     */
    Echo_on_stdout = Debug;
    /*
     * get the printcap entry
     */
    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	(void) fprintf (stdout, "Printer or Queue '%s' does not exist!\n", Printer);
	(void) fflush (stdout);
	return;
    }
    if (SD == 0 || *SD == 0) {
	if (Debug > 2)
	    log (XLOG_DEBUG, "not a Printer");
	return;
    }
    chdir_SD ();
    /*
     * set the flags needed
     */
    Is_local = hostcmp (FQDN, Host) == 0;
    Is_root = strsame (Person, "root");
    /*
     * check to see that the user has RMJOB privs on this machine
     */
    perms = 'R';		/* must be able to at least use the Printer */
    if (!(Is_root || Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) > 0)) {
	(void) fprintf (stdout, "Sorry user %s, but you do not have hold permission on '%s'!", Person, First_name);
	return;
    }
    perms = 'C';		/* check for control perms */
    control_perms = 1;
    if (!(Is_root || Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) > 0)) {
	control_perms = 0;
    }
    if (Debug > 4)
	log (XLOG_DEBUG,
	     "hold: Is_root %d, Is_local %d, control_perms %d",
	     Is_root, Is_local, control_perms);
    /*
     * check for remote machine and networked file system
     */
    if (RM && NW) {
	Remote_hold ();
	return;
    }
    /*
     * get the job queue
     */
    Jobcount = Getq ();
    (void) Checkactive ();
    /*
     * run down list
     */
    (void) fprintf (stdout, "Printer '%s' (%s):\n", Printer, Host);
    (void) fflush (stdout);
    if (Debug > 5)
	log (XLOG_DEBUG, "hold: Jobcount = %d", Jobcount);
    for (i = 0; i < Jobcount; ++i) {
	q = &Queue[i];
	if (Debug > 5)
	    log (XLOG_DEBUG, "hold: checking %s, %d, %s",
		 q->q_name, q->q_num, q->q_user);
	if (shouldhold (q, control_perms)) {
	    (void) fprintf (stdout, "Holding %s, job %d owner %s on printer '%s'.\n",
			    q->q_name, q->q_num, q->q_user, Printer);
	    (void) fflush (stdout);
	    dohold (q);
	}
    }
    /*
     * check for remote machine
     */
    if (RM) {
	Remote_hold ();
    }
    /*
     * give the server a kick
     */
    (void) Startserver ();
}

/***************************************************************************
 * void release ()
 * Releases the held job
 ***************************************************************************/
void
release () {
    int i;
    struct plp_queue *q;	/* job entry */
    int perms;			/* hold perms values */
    int control_perms;		/* has control perms */

    /*
     * If explicitly asked for debug then echo messages to stdout otherwise (db flag in
     * printcap) just log messages in file.
     */
    Echo_on_stdout = Debug;
    /*
     * get the printcap entry
     */
    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	(void) fprintf (stdout, "Printer or Queue '%s' does not exist!\n", Printer);
	(void) fflush (stdout);
	return;
    }
    if (SD == 0 || *SD == 0) {
	if (Debug > 2)
	    log (XLOG_DEBUG, "not a Printer");
	return;
    }
    chdir_SD ();
    /*
     * set the flags needed
     */
    Is_local = hostcmp (FQDN, Host) == 0;
    Is_root = strsame (Person, "root");
    /*
     * check to see that the user has RMJOB privs on this machine
     */
    perms = 'R';		/* must be able to at least use the Printer */
    if (!(Is_root || Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) > 0)) {
	(void) fprintf (stdout, "Sorry user %s, but you do not have release permission on '%s'!", 
			Person, First_name);
	return;
    }
    perms = 'C';		/* check for control perms */
    control_perms = 1;
    if (!(Is_root || Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) > 0)) {
	control_perms = 0;
    }
    if (Debug > 4)
	log (XLOG_DEBUG,
	     "release: Is_root %d, Is_local %d, control_perms %d",
	     Is_root, Is_local, control_perms);
    /*
     * check for remote machine and networked file system
     */
    if (RM && NW) {
	Remote_release ();
	return;
    }
    /*
     * get the job queue
     */
    Jobcount = Getq ();
    (void) Checkactive ();
    /*
     * run down list
     */
    (void) fprintf (stdout, "Printer '%s' (%s):\n", Printer, Host);
    (void) fflush (stdout);
    if (Debug > 5)
	log (XLOG_DEBUG, "release: Jobcount = %d", Jobcount);
    for (i = 0; i < Jobcount; ++i) {
	q = &Queue[i];
	if (Debug > 5)
	    log (XLOG_DEBUG, "release: checking %s, %d, %s",
		 q->q_name, q->q_num, q->q_user);
	if (shouldrelease (q, control_perms)) {
	    (void) fprintf (stdout, "Releasing %s, job %d owner %s from printer '%s'.\n",
			    q->q_name, q->q_num, q->q_user, Printer);
	    (void) fflush (stdout);
	    dorelease (q);
	}
    }
    /*
     * check for remote machine
     */
    if (RM) {
	Remote_release ();
    }
    /*
     * give the server a kick
     */
    (void) Startserver ();
}
