/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: lpr_temp.c
 * Routines to manage the Temp_file array
 ***************************************************************************/

#include <assert.h>
#include "library/errormsg.h"
#include "lpr.h"
#include "lpr_global.h"
#include "lpr_temp.h"

/***************************************************************************
 * Get_cf()
 *   return the name of the control file.
 * Add_temp( char *s )
 *   Add the name of a file to the temporary file list
 * Remove_temp()
 *   Unlink all the files in the Temp_file[] list
 ***************************************************************************/

/***************************************************************************
 * Generate the name of a data file
 * The name has the form: df<seq><job><host>
 ***************************************************************************/
char *Get_tmp_data(void)
{
    static int seq;
    static char buf[MAXPATHLEN];	/* Big Buffers, Inc. */
    int c;
    static char st[]=
    	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";

    /*
     * convert sequence number to A-Z, a-z, 0-9
     */
    if (seq >= strlen (st)) {
	Diemsg ("too many input files, split job up");
    }
    c = st[seq];
    ++seq;
    /*
     * generate name
     *
     * Julian Anderson <jules@comp.vuw.ac.nz>, 27 Sep 1994
     * ...to ensure that the H record and control file name are both
     * either FQDN or shortnames
     */
#ifdef SHORTHOSTNAME
    assert(9+strlen(ShortHost)<sizeof(buf));
    (void) sprintf (buf, "df%c%03d%s", c, (int) Job_number, ShortHost);
#else
    assert(9+strlen(Host)<sizeof(buf));
    (void) sprintf (buf, "df%c%03d%s", c, (int) Job_number, Host);
#endif 

    if (Debug > 3)
	log (XLOG_DEBUG, "Get_tmp_data: %s", buf);

    /*
     * Add to list of temporary files to be removed in case of error
     */
    return (Add_temp (buf));
}

/***************************************************************************
 * return the name of the control file.  This has the form
 * tf<Priority><Sequence><Host>
 ***************************************************************************/
char *Get_cf(void)
{
    static char buf[MAXPATHLEN];
    char *sp;			/* ACME Pointers and Buggy Whips */

    /* Julian Anderson <jules@comp.vuw.ac.nz>, 27 Sep 1994
     * ...to ensure that the H record and control file name are both
     * either FQDN or shortnames
     */
#ifdef SHORTHOSTNAME
    (void) sprintf (buf, "tf%c%03d%s", Priority, (int) Job_number, ShortHost);
#else
    (void) sprintf (buf, "tf%c%03d%s", Priority, (int) Job_number, Host);
#endif    
    /*
     * check to see that the job file name fits in the limits
     */
    if (strlen (buf) >= CFNAMELEN) {
	fatal (XLOG_INFO, "Get_cf: control file name too long '%s'",
	       buf);
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "Get_cf: control file %s", buf);
    sp = Add_temp (buf);
    return (sp);
}

/***************************************************************************
 * Add the name of a file to the temporary file list
 * Note: these will be deleted on abnormal exit
 ***************************************************************************/
static char *Temp_space;
static int Temp_size;

char *Add_temp(char *s)
{
    int i;
    char *sp;			/* ACME Pointers and Integers */

    assert(s!=(char*)0);
    if (Temp_max == 0) {
	Temp_max = 10;
	Temp_file = (char **) malloc ((unsigned) sizeof (char *) * Temp_max);
	if (Temp_file == 0) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
    }
    if (Temp_max <= Temp_count) {
	Temp_max += 10;
	Temp_file = (char **) realloc ((char *) Temp_file,
				       (unsigned) (sizeof (char *) * Temp_max));
	if (Temp_file == 0) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
    }
    i = strlen (s) + 1;
    if (i > Temp_size) {
	Temp_size = BUFSIZ;
	Temp_space = (char *) malloc (BUFSIZ);
	if (Temp_space == 0) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
    }
    sp = Temp_space;
    (void) strcpy (Temp_space, s);
    Temp_file[Temp_count] = sp;
    ++Temp_count;
    Temp_space += i;
    Temp_size -= i;
    if (Debug > 5) {
	for (i = 0; i < Temp_count; ++i)
	    log (XLOG_DEBUG, "temp %d %s", i, Temp_file[i]);
    }
    return (sp);
}

/***************************************************************************
 * Remove the named file from the Temp_file[] list
 ***************************************************************************/

void Remove_tempfile(char *s)
{
    int i;

    if (!s) return;

    for (i = 0; i < Temp_count; ++i) {
	if (Temp_file[i] && strsame (s, Temp_file[i])) {
            Temp_file[i] = (char*)NULL;
            if (Debug > 3) {
                log (XLOG_DEBUG, "Remove_tempfile: removing file %s from list", s);
            }
        }    
    }
}

/***************************************************************************
 * Unlink all the files in the Temp_file[] list
 ***************************************************************************/

void Remove_temp(void)
{
    char buf[BUFSIZ];		/* hold the full pathname */
    int i;			/* ACME Integers, Inc. */
    char *sd_end;		/* end of the SD name in buf */

    if (Temp_count <= 0 || SD == 0 || *SD == 0
	|| (strlen (SD) + CFNAMELEN) > sizeof (buf)) {
	return;
    }
    (void) strcpy (buf, SD);
    (void) strcat (buf, "/");
    sd_end = &buf[strlen (buf)];
    user_to_daemon ();		/* to remove temp files */
    for (i = 0; i < Temp_count; ++i) {
        if (! Temp_file[i]) continue; /* might be deleted by Remove_tempfile() */
	(void) strcpy (sd_end, Temp_file[i]);
	if (Debug > 3)
	    log (XLOG_DEBUG, "Remove_temp: removing temp file %s", buf);
	if (unlink (buf) < 0) {
	    if (*(Temp_file[i]) == 't') {	/* temp control file */
		*(Temp_file[i]) = 'c';
		unlink (buf);
	    } else {
		if (Debug > 3)
		    log (XLOG_DEBUG, "Remove_temp: cannot remove temp file %s", buf);
	    }
	}
    }
    daemon_to_user ();		/* to remove temp files */
}
