/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: lpr_parms.c
 * get the parameters for lpr program
 ***************************************************************************/

#include <assert.h>
#include "common/mailaddr.h"
#include "library/errormsg.h"
#include "lpr.h"
#include "lpr_global.h"
#include "lpr_parms.h"

#ifdef NIS
#include <rpc/rpc.h>
#include <rpcsvc/ypclnt.h>
#include <rpcsvc/yp_prot.h>
#endif

int num_euent = 0;
struct eu_entry EU_entry[MAXEUENTRY];
static int setup_mailaddress, no_banner;

static void Dienoarg(int option)
{
    Diemsg ("option '%c' missing argument", option);
}

/***************************************************************************
 * Check_int_dup (int option, int *value, char *arg)
 * 1.  check to see if value has been set
 * 2.  if not, then get integer value from arg
 ***************************************************************************/

static void Check_int_dup (int option, int *value, char *arg)
{
    assert(value!=(int*)0);
    if (*value) {
	Diemsg ("duplicate option %c", option);
    }
    if (arg == 0) {
	Dienoarg (option);
    }
    if (sscanf (arg, "%d", value) != 1 || *value <= 0) {
	Diemsg ("option %c parameter (%s) is not positive integer",
		option, arg);
    }
}

/***************************************************************************
 * 1.  check to see if value has been set
 * 2.  if not, then set it
 ***************************************************************************/

static void Check_str_dup(int option, char *value, char *arg)
{
    assert(value!=(char*)0);
    if (*value) {
	Diemsg ("duplicate option %c", option);
    }
    if (arg == 0) {
	Dienoarg (option);
    }
    if ((int) strlen (arg) > MAXPARMLEN) {
	Diemsg ("option %c arguement too long (%s)", option, arg);
    }
    (void) strcpy (value, arg);
}

/***************************************************************************
 * 1.  check to see if value has been set
 * 2.  if not, then set it
 ***************************************************************************/

static void Check_dup(int option, int *value)
{
    if (*value) {
	Diemsg ("duplicate option %c", option);
    }
    *value = 1;
}

/***************************************************************************
 * 1.  check to see if root
 ***************************************************************************/

static void Root_only(int option)
{
    Setuid_debug("Root_only");
    
    if (UserUID) {
        Diemsg ("option %c can be used only by root", option);
    }
}

#ifdef EUCS_ZOPTIONS
/*****************************************************************************
 * Set up default EUCS options
 ****************************************************************************/
static void Setup_euentry(void)
{
    int i;			/* ACME integers */
    struct passwd *pw_ent;	/* user entry in /etc/passwd */
    struct group *grent;
    char buf[MAXPARMLEN];
    char *jobname = "(stdin)";

    EU_entry[num_euent].name = "V";
    EU_entry[num_euent].value = "1.0";
    num_euent++;

    EU_entry[num_euent].name = "E";
    EU_entry[num_euent].value = ercc_reg ();
    num_euent++;

    EU_entry[num_euent].name = "SF";

    for (i = 0; i < Parmcount; ++i) {
	if (Parms[i].str) {
	    jobname = Parms[i].str;
	    break;
	}
    }
    EU_entry[num_euent].value = strdup (jobname);
    num_euent++;

    EU_entry[num_euent].name = "N";
    EU_entry[num_euent].value = LOGNAME;
    num_euent++;

    EU_entry[num_euent].name = "U";
    sprintf (buf, "%d", UserUID);
    EU_entry[num_euent].value = strdup (buf);
    num_euent++;

    if ((pw_ent = getpwnam (LOGNAME)) && (grent = getgrgid (pw_ent->pw_gid))) {
	EU_entry[num_euent].name = "GN";
	EU_entry[num_euent].value = strdup (grent->gr_name);
	num_euent++;
    }
}

/******************************************************************************
 * Read potential -Z option patterns from config file.
 * Config file contains three fields; the last one is optional.
 * EU format, -Z option pattern, default value.
 *****************************************************************************/
static void Read_zoptions(void)
{
    FILE *fp;			/* file pointer */
    int found, i;
    char *zoption;
    char buf[BUFSIZ];		/* for reading lines */
    char format[MAXPARMLEN + 1];/* format code */
    char pattern[MAXPARMLEN + 1];	/* pattern */
    char value[MAXPARMLEN + 1];	/* default */

    fp = fopen_path (Zoptions_path, "r");
    if (fp == NULL) {
	if (Debug > 4)
	    logerr (XLOG_DEBUG,
		    "Read_zoptions: cannot find any zopt files on path '%s'",
		    Zoptions_path);
	return;
    }
    while (fgets (buf, sizeof (buf), fp) != NULL) {
	if (buf[0] == '#' || buf[0] == '\n')
	    continue;
	i = sscanf (buf, "%s %s %[^\n]", format, pattern, value);
        /* at least die before a corrupted stack comes into effect */
        assert(i<1 || strlen(format)<sizeof(format));
        assert(i<2 || strlen(pattern)<sizeof(pattern));
        assert(i<3 || strlen(value)<sizeof(value));
	if (i < 2) {
	    log (XLOG_INFO, "Read_zoptions: bad line ignored; %s", buf);
	    continue;
	} else if (i == 2) {
	    *value = '\0';
	}
	if (Debug > 4)
	    log (XLOG_DEBUG, "Read_zoptions: read line %s %s %s",
		 format, pattern, value);

	if ((zoption = Parse_zoption (pattern, value))) {
	    /* search for option to override */
	    for (i = 0; i < num_euent; i++) {
		if (strsame (EU_entry[i].name, format)) {
		    EU_entry[i].value = zoption;
		    break;
		}
	    }
	    if (i == num_euent) {	/* add new option */
		if (num_euent < MAXEUENTRY) {
		    EU_entry[num_euent].name = strdup (format);
		    EU_entry[i].value = zoption;
		    num_euent++;
		} else {
		    log (XLOG_INFO,
			 "Read_zoptions: too many Z-options, discarding %s=%s", format, zoption);
		}
	    }
	}
    }
    fclose (fp);
}

/******************************************************************************
 * Parse the ZOPTS as a comma-separated list of options looking for pattern,
 * return default if not found.
 * Pattern must contain one input specifier only; (or be - to substitute the
 * default value) this can be:
 *      %s      String
 *      %f      Forms (one of FL options)
 *****************************************************************************/

static char *Parse_zoption(char *patt, char *value)
{
    char *ret = NULL;		/* return value */
    char *zopt = ZOPTS;		/* ZOPTS pointer */
    char *nextz;		/* next ZOPTS pointer */
    int before, after;		/* length before & after specifier */
    char *tail;			/* pointer after specifier in pattern */
    char spec;			/* specifier character */

    assert(patt!=(char*)0);
    assert(value!=(char*)0);
    if (strsame (patt, "-")) {		/* use default value */
	if (*value) {
	    ret = strdup (value);
	}
	return (ret);
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "Parse_zoption: zopt is '%s'", zopt);

    /* search for %specifier character */
    if ((tail = strchr (patt, '%')) && (spec = tail[1])) {
	before = tail - patt;	/* length before specifier */
	tail += 2;		/* portion after specifier */
	after = strlen (tail);
	if (strchr (tail, '%')) {
	    log (XLOG_INFO,
		 "Parse_zoption: too many specifiers in %s, ignored", patt);
	    return NULL;
	}
    } else {
	log (XLOG_INFO,
	     "Parse_zoption: no specifier in %s, ignored", patt);
	return NULL;
    }

    if (Debug > 4)
	log (XLOG_DEBUG, "Parse_zoption: before=%d after=%d spec=%c patt='%s' tail='%s'", before, after, spec, patt, tail);

    while (zopt && *zopt) {
	int zlen;
	if ((nextz = strchr (zopt, ',')))	/* temporarily truncate zopt */
	    *nextz = '\0';
	zlen = strlen (zopt);

	if (Debug > 5)
	    log (XLOG_DEBUG, "Parse_zoption: part '%s' length %d", zopt, zlen);

	/* check if portion zopt[before..zlen-after] matches */
	if (zlen >= before + after &&
	    strncmp (zopt, patt, before) == 0 &&
	    strncmp (zopt + zlen - after, tail, after) == 0) {
	    int ok = 0;
	    char *flist, *fnext;
	    int flen;

	    zlen -= before + after;	/* length of matched portion */
	    if (Debug > 5)
		log (XLOG_DEBUG, "Parse_zoption: spec matches '%s' length %d",
		     zopt + before, zlen);

	    switch (spec) {
	    case 'f':		/* check if in FL list */
		for (flist = FL; flist && *flist; flist = fnext) {
		    if ((fnext = strchr (flist, ','))) {
			flen = fnext - flist;
			fnext++;
		    } else
			flen = strlen (flist);
		    if (zlen == flen &&
			!strncmp (zopt + before, flist, zlen)) {
			ok = 1;
			break;
		    }
		}
		if (!ok && (before || after))
		    logerr_die (XLOG_INFO,
				"Parse_zoption: forms value '%s' not recognised (%s valid)", zopt, FL ? FL : "none");
		break;
	    default:		/* unknown format */
		log (XLOG_INFO,
		     "Parse_zoption: specification %%%c not understood, treating as %%s", spec);
	    case 's':		/* arbitrary string */
		ok = 1;
		break;
	    }
	    if (ok) {
		malloc_or_die (ret, zlen + 1);
		strncpy (ret, zopt + before, zlen);
		ret[zlen] = '\0';
		if (nextz) {	/* shift parameters */
		    while ((*zopt++ = *++nextz));
		} else {	/* remove final parameter */
		    if (zopt > ZOPTS)
			zopt--;
		    *zopt = '\0';
		}
		return (ret);
	    }
	}
	if (nextz)		/* restore comma */
	    *nextz++ = ',';
	zopt = nextz;
    }

    /* fall back on default value */
    if (*value) {
	ret = strdup (value);
    }
    return (ret);
}

#ifdef LOCAL
static char *ercc_reg_file(char *name)
{
    FILE *fp;			/* file pointer */
    int found;
    char buf[BUFSIZ];		/* for reading lines */
    char key[MAXPARMLEN + 1];	/* user/group name */
    static char code[MAXPARMLEN + 1];	/* cost code */
    struct stat statb;		/* for statting files */

    assert(name!=(char*)0);

    if (!Costcode_path || !*Costcode_path) {
	return (NULL);
    }
    fp = fopen_path (Costcode_path, "r");
    if (fp == NULL) {
	if (Debug > 4)
	    logerr (XLOG_DEBUG,
		    "ercc_reg_file: cannot find any cost code files on path '%s'",
		    Costcode_path);
	return (NULL);
    }
    while (fgets (buf, sizeof (buf), fp) != NULL) {
	if (buf[0] == '#' || buf[0] == '\n')
	    continue;
	if (sscanf (buf, "%s%s", key, code) == 2 && strsame (name, key)) {
	    fclose (fp);
	    return (code);
	}
    }
    fclose (fp);
    return (NULL);
}

#endif

#ifdef NIS
static char *ercc_reg_NIS(char *name)
{
    char *outval;
    int outvallen;
    char *stddomain;
    char *cp;

    assert(name!=(char*)0);
    if (yp_get_default_domain (&stddomain))
	return NULL;

    if (!NIS_costcode_byname || !*NIS_costcode_byname) {
	logerr_die ("nis-costcode not set!");
    }
    if (yp_match (stddomain, NIS_costcode_byname, name, strlen (name),
		  &outval, &outvallen) == 0) {
	if ((cp = strchr (outval, '\n'))) {
	    *cp = '\0';
	}
	free (stddomain);
	return outval;
    }
    free (stddomain);
    return NULL;
}

#endif

#ifdef HESIOD
static char *ercc_reg_hes(char *name)
{
    char **hp;

    assert(name!=(char*)0);
    if (!Hesiod_costcode_key || !*Hesiod_costcode_key) {
	logerr_die ("hesiod-costcode-key not set!");
    }
    if ((hp = hesiod_match (name, Hesiod_costcode_key, NULL))) {
	return *hp;
    }
    return NULL;
}

#endif

/* ercc_reg() looks up the ERCC registration for user LOGNAME */
static char *ercc_reg(void)
{
    int i, j;			/* ACME integers */
    char *res = NULL;
    int nlookup = 0;		/* number of lookups */
    char *lookup[10];		/* lookup keys in order */
    struct passwd *pwent;
    struct group *grent;

    lookup[nlookup++] = LOGNAME;/* try user first */
    if ((pwent = getpwnam (LOGNAME)) &&
	(grent = getgrgid (pwent->pw_gid))) {
	char *group;
	
	malloc_or_die (group, strlen (grent->gr_name) + 2);
	*group = '+';
	strcpy (group + 1, grent->gr_name);
	lookup[nlookup++] = group;	/* then try group */
    }
    lookup[nlookup++] = "default_code";	/* then default */

    for (j = 0; j < nlookup; j++)
	for (i = 0; i < num_confs && res == NULL; i++)
	    if (conf[i].domain == NULL)	/* local domain only */
		switch (conf[i].type) {
#ifdef LOCAL
		case LOCAL_TYPE:
		    res = ercc_reg_file (lookup[j]);
		    break;
#endif
#ifdef NIS
		case NIS_TYPE:
		    res = ercc_reg_NIS (lookup[j]);
		    break;
#endif
#ifdef HESIOD
		case HESIOD_TYPE:
		    res = ercc_reg_hes (lookup[j]);
		    break;
#endif
		default:
		    fprintf (stderr, "Cannot lookup cost code of type %d.\n",
			     conf[i].type);
		}

    return (res);
}

#endif				/* EUCS_ZOPTIONS */

/***************************************************************************
 * Check for permissions and priority
 * 2. printcap: check for restrictions
 * 3. XU: check for restrictions
 ***************************************************************************/

static void Can_use(void)
{
    int prior;			/* find the max priority */
    int op;			/* 'R' for lpr */

    /*
     * check for priority
     */
    Priority = CLASSNAME[0];
    if (Priority == 0) {
	Priority = DEFPRIORITY;
    }
    if (!isascii (Priority) || !isupper (Priority)) {
	Priority = DEFPRIORITY;

#ifdef EUCS_ZOPTIONS
	/*
	 * don't print error message - we use the -J field to pass in extra info so the
	 * first character won't be a valid priority Paul Haldane, EUCS, 20/11/90
	 */
	Warnmsg ("Priority set to %c", Priority);
#endif
    }

    if (Debug>4)
	log (XLOG_DEBUG, "Checking perms for %s@%s on printer '%s'",
	    Person, Host, Printer);
    /*
     * check to see if we have restricted access
     */
    prior = Priority;
    op = 'R';			/* able to use lpr */
    if (Checkperm (Host, Person, First_name, &op, (int *) 0, 0) <= 0) {
	Diemsg ("Sorry %s@%s, you don't have use permission for '%s'",
		Person, ShortHost, Printer);
    }
    if (Checkperm (Host, Person, First_name, &op, &prior, 1) <= 0) {
	Diemsg ("Sorry %s@%s, no more pages allowed on '%s'",
		Person, ShortHost, Printer);
    }
    /*
     * now check to see if you have not exceeded your page limits
     */
    if (prior > Priority) {
	Warnmsg ("maximum Priority allowed is %c", prior);
	Priority = prior;
    }
    if (Debug > 2)
	log (XLOG_DEBUG, "Can_use: %s can use %s, priority %c",
	     Person, Printer, Priority);
}

/***************************************************************************
 * -- check to see if person is a member of one of the groups
 * Returns: 1 if  person is a member, 0 if not
 ***************************************************************************/

#define GRPLEN 32

static int Checkgrp(char *name, char *list)
{
    char buf[BUFSIZ];
    char *cp, *bp;
    struct group *gr;
    char **grplist;
    struct passwd *pwdp;
    char login_group[GRPLEN + 1];

    assert(name!=(char*)0);
    assert(list!=(char*)0);
    if (Debug > 4)
	log (XLOG_DEBUG, "Checkgrp: checking for %s in %s", name, list);
    if ((pwdp = getpwnam (name)) == NULL) {
	Diemsg ("No passwd entry for %s\n", name);
	return (0);
    }
    if ((gr = getgrgid (pwdp->pw_gid)) == NULL) {
	Diemsg ("No group entry for group %d\n", pwdp->pw_gid);
	return (0);

    }
    (void) strncpy (login_group, gr->gr_name, GRPLEN);
    login_group[GRPLEN] = '\0';

    cp = list;
    while (cp && *cp) {
	for (bp = buf; *cp && (*bp = *cp) && (*bp != ','); ++bp, ++cp);
	*bp = 0;
	if (*cp) {
	    ++cp;
	}
	if (Debug > 5)
	    log (XLOG_DEBUG, "Checkgrp: checking group %s", buf);
	if (strsame (login_group, buf)) {
	    if (Debug > 4)
		log (XLOG_DEBUG, "Checkgrp: %s in (login group) %s",
		     name, login_group);
	    return (1);
	}
	if ((gr = getgrnam (buf))) {
	    for (grplist = gr->gr_mem; *grplist; ++grplist) {
		if (strsame (name, *grplist)) {
		    if (Debug > 4)
			log (XLOG_DEBUG, "Checkgrp: %s in %s",
			     name, gr->gr_name);
		    return (1);
		}
	    }
	}
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "Checkgrp: %s not in any group", name);
    return (0);
}

/***************************************************************************
 *	Check for parameter consistency
 *  1. Check to see if the format is allowed
 *  2. Check on the multiple copies
 ***************************************************************************/

static void Check_parms(void)
{
    char *msg;
    /* check format and options */
    if (FX && *FX && strchr (FX, Format) == 0) {
	msg = NULL;
	switch (Format) {
	case 'f':
	    msg = "Normal or plain files";
	    break;
	case 't':
	    msg = "Troff files, use -Fn for Ditroff";
	    break;
	case 'n':
	    msg = "Ditroff files, use -Ft for (old) troff files";
	    break;
	case 'v':
	    msg = "Varian raster images";
	    break;
	case 'd':
	    msg = "TeX intermediate files (DVI)";
	    break;
	case 'g':
	    msg = "Plot intermediate files";
	    break;
	case 'c':
	    msg = "Cifplot intermediate files";
	    break;
	default:
	    msg = "format 'X' files";
	    msg[8] = Format;
	}
	Diemsg ("Printer %s does not know how to interpret %s\n", Printer, msg);
    }
    if (SC && Copies > 1) {
	Warnmsg ("multiple copies are not allowed");
	Copies = 0;
    }
    if (MC > 0 && Copies > MC) {
	Warnmsg ("only %d copies are allowed", MC);
	Copies = MC;
    }
    if ((Remove && !Is_root) && (Restrict_symlinks ?
				 (LN == 0 || *LN == 0 || !Checkgrp (Person, LN)) :
				 (LN && *LN && !Checkgrp (Person, LN)))) {
	Warnmsg ("-r (remove) not allowed");
	Remove = 0;
    }

#ifdef NO_SYMLINKS
    if (Use_links) {
	Warnmsg ("-s (symbolic links) disabled");
	Use_links = 0;
    }
#else				/* NOSYMLINK */
    if (Use_links) {
	/*
	 * check to see if we are a member of the right group
	 */
	if (!Is_root && (Restrict_symlinks ?
			 (LN == 0 || *LN == 0 || !Checkgrp (Person, LN)) :
			 (LN && *LN && !Checkgrp (Person, LN)))) {
	    Warnmsg ("-s (symbolic links) not allowed by user %s", Person);
	    Use_links = 0;
	}
	if (RM && NW) {
	    Warnmsg ("-s (symbolic links) not available for this queue");
	    Use_links = 0;
	}
    }
#endif				/* NOSYMLINK */
}

/***************************************************************************
 * Get the job sequence number
 * 1. Check to see if queuing is enabled
 * 2. Get the job number from the .job file
 ***************************************************************************/

static void Get_sequence(void)
{
    char buf[MAXPATHLEN];	/* holds the pathname */
    FILE *fp;			/* for sequence number */

    /*
     * check to see if printing is enabled
     */
    if (LO == (char*)0 || *LO == '\0' || SD == (char*)0 || *SD == '\0') {
	fatal (XLOG_CRIT, "Get_sequence: bad printcap entry");
    }
    fix_SD ();	/* check to see if the perms need fixing */

    assert((strlen(LO)+2+strlen(SD))<=sizeof(buf));
    if (LO[0] == '/') {
	(void) strcpy (buf, LO);
    } else {
	(void) sprintf (buf, "%s/%s", SD, LO);
    }
    (void) Checklockfile (buf, (pid_t *) 0, (char *) 0, 0, &LO_statb);
    if (!Is_root && (LO_statb.st_mode & DISABLE_QUEUE)) {
	Diemsg ("printer %s- queueing disabled", Printer);
    }
    /*
     * get the sequence file name
     */
    assert((strlen(SD)+7+strlen(Host))<=sizeof(buf));
    (void) sprintf (buf, "%s/.seq.%s", SD, Host);
    if (Debug > 3)
	log (XLOG_DEBUG, "sequence file name '%s'", buf);
    
#define MAX_RETRIES	10
    /*
     * lock the sequence file and get a new number (retrying quietly!)
     */
    {
	int retry;
	for (retry = 0; retry < MAX_RETRIES; retry++) {
	    if ((fp = Getlockfile (buf, &Job_number, (char *) 0, 0, &LO_statb))
			!= NULL)
	    {
		break;
	    }
	}
    }

    if (fp == NULL) {
	Diemsg ("cannot lock sequence file %s, try later (%s)", buf,
		Errormsg (errno));
    }
    /*
     * set the sequence number to the new value mod 1000;
     */
    Job_number = (Job_number + 1) % 1000;

    /* update the lockfile immediately (avoid race condition
     * causing "cannot lock" message)
     */
    Setlockfile (buf, fp, Job_number, Time_str ());
    (void) fclose (fp);

    if (Debug > 4)
	log (XLOG_DEBUG, "Get_sequence: number %d", Job_number);
}

/***************************************************************************
 * Display the values of the parameters that were read.
 * 1. Print the CFparm array
 * 2. Print the other parameters
 ***************************************************************************/

static void Show_parms(void)
{
    int i;
    char *s;

    /*
     * CFparm first:
     */
    log (XLOG_DEBUG, "CFparm:");
    for (i = 'A'; i <= 'Z'; ++i) {
	s = CFparm[i - 'A'];
	if (s[0]) {
	    log (XLOG_DEBUG, " %c  '%s'", i, s);
	}
    }
    log (XLOG_DEBUG, " Format: %c", Format);
    log (XLOG_DEBUG, " Copies: %d", Copies);
    log (XLOG_DEBUG, " Binary: %d", Binary);
    log (XLOG_DEBUG, " Use_links: %d", Use_links);
    log (XLOG_DEBUG, " Exper: %d", Exper);
    log (XLOG_DEBUG, " Priority: %d", Priority);
    log (XLOG_DEBUG, " Job_number: %d", Job_number);
    if (Read_stdin) {
	log (XLOG_DEBUG, " Read_stdin: %s", Read_stdin);
    }
    if (Filter_stdin) {
	log (XLOG_DEBUG, " Filter_stdin: %s", Filter_stdin);
    }
    log (XLOG_DEBUG, " Temp_count: %d", Temp_count);
    log (XLOG_DEBUG, " Temp_max: %d", Temp_max);
    for (i = 0; i < Temp_count; ++i) {
	s = Temp_file[i];
	log (XLOG_DEBUG, " %d  '%s'", i, s);
    }
}

/***************************************************************************
 * 1. get the parameters;
 * 2. set up the get the printcap entry
 * 3. check to see if any parameters violate the printcap restrictions
 * 4. get the sequence number
 ***************************************************************************/

void Setup_parms(void)
{
    if (setup_mailaddress) {
	Make_mail_address (MAILNAME, Mail_addressing, Person, Host);
    }

    if (no_banner) {
	/* make bannername a null string (inhibits banner output) */
	BNRNAME[0] = '\0';
    }

    /*
     * print the parameters passed
     */
    if (Debug > 4)
	Show_parms ();
    /*
     * get the printcap entry
     */
    Get_Printer (1);

    if (*(FROMHOST) == '\0') {
	if (BK) {
	    /* if we're in backwards-contemptibility mode,
	     * use the short hostname in the config file.
	     */
	    (void) strcpy (FROMHOST, ShortHost);
	} else {
	    (void) strcpy (FROMHOST, Host);
	}
    }
    /*
     * check authorizations
     */
    Can_use ();
    /*
     * check consistency of parameters
     */
    Check_parms ();

#ifdef EUCS_ZOPTIONS
    /*
     * set up EUCS format -Z options
     */
    if (EU) {
	Setup_euentry ();
	Read_zoptions ();
    }
#endif
    /*
     * get the sequence number
     */
    Get_sequence ();
    if (Debug > 4)
	Show_parms ();
}

/***************************************************************************
  * 1. Scan the argument list and get the flags
 ***************************************************************************/

static char *optstr = "#:C:D:F:J:P:R:T:U:Z:bcdfghi:lm?nprstvw:";

void Get_parms(int argc, char *argv[])
{
    int option;
    int i;

    setup_mailaddress = 0;
    no_banner = 0;
    while ((option = Getopt (argc, argv, optstr)) != EOF) {
        switch (option) {
        case '#':
            Check_int_dup (option, &Copies, Optarg);
            break;
        case 'C':
            Check_str_dup (option, CLASSNAME, Optarg);
            break;
        case 'D':
            Parse_debug (Optarg);
            break;
        case 'F':
            if (!Optarg) {
                Dienoarg (option);
            }
            if (strlen (Optarg) != 1) {
                Diemsg ("bad -F format string '%s'\n", Optarg);
            }
            if (Format) {
                Diemsg ("duplicate format specification -F%s\n", Optarg);
            } else {
                Format = *Optarg;
            }
            break;
        case 'J':
            Check_str_dup (option, JOBNAME, Optarg);
            break;
        case 'P':
            if (Printer) {
                Check_str_dup (option, Printer, Optarg);
            }
            if (!Optarg || !*Optarg) {
                Diemsg ("missing printer name in -P option\n");
            }
            Printer = Optarg;
            break;
        case 'R':
            Check_str_dup (option, ACCNTNAME, Optarg);
            break;
        case 'T':
            Check_str_dup (option, PRTITLE, Optarg);
            break;
        case 'U':
            Root_only (option);
            Check_str_dup (option, BNRNAME, Optarg);
            u_option = 1; /* will be used later in lpr_job.c */
            (void) strcpy (LOGNAME, Optarg);
            break;
        case 'Z':
            Check_str_dup (option, ZOPTS, Optarg);
            break;
        case 'b':
            Check_dup (option, &Binary);
            break;
        case 'h':
            Check_dup (option, &Noheader);
	    no_banner = 1;
            break;
        case 'i':
            Check_str_dup (option, INDENT, Optarg);
            break;
        case 'm':
            /*
             * -m[Mailname]
             */
            if (Optarg == NULL) {
                setup_mailaddress = 1;

                /* use a dummy arg...
                 * it's still valid if Optarg is NULL.
                 */
                Check_str_dup (option, MAILNAME, "dummy");
            } else {
                Check_str_dup (option, MAILNAME, Optarg);
            }
            break;
        case 'r':
            Check_dup (option, &Remove);
            break;
        case 's':
            Check_dup (option, &Use_links);
            break;
        case 'c':
        case 'd':
        case 'f':
        case 'g':
        case 'l':
        case 'n':
        case 'p':
        case 't':
        case 'v':
            if (Format) {
                Diemsg ("duplicate format specification -%c\n", option);
                exit (1);
            } else {
                Format = option;
            }
            break;
        case 'w':
            Check_str_dup (option, PWIDTH, Optarg);
            i = atoi (PWIDTH);
            if (i <= 0) {
                Diemsg ("-w <width>, width value '%s' bad", PWIDTH);
                exit (1);
            }
            (void) sprintf (PWIDTH, "%d", i);
            break;
        case '?':
            break;
        default:
            fatal (XLOG_INFO, "Get_parms: badparm %c", option);
        }
    }

    /*
     * set up the Parms[] array
     */
    allocParmsIfNeeded ();
    for (; Optind < argc; ++Optind) {
	if (Debug > 4) log (XLOG_DEBUG, "adding '%s'", argv[Optind]);
	Parms[Parmcount].str = argv[Optind];
	*Parms[Parmcount].filename = '\0';
	++Parmcount;
	growParmsIfNeeded ();
    }

    /*
     * set default format
     */
    if (Format == 0) {
	Format = 'f';
    }
}
