/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: lpr_job.c
 * Submit job in the spool directory
 ***************************************************************************/

#include <assert.h>
#include "library/errormsg.h"
#include "lpr.h"
#include "lpr_canprint.h"
#include "lpr_filters.h"
#include "lpr_global.h"
#include "lpr_job.h"
#include "lpr_parms.h"
#include "lpr_temp.h"

#ifdef NIS
#include <rpc/rpc.h>
#include <rpcsvc/ypclnt.h>
#include <rpcsvc/yp_prot.h>
#endif

/***************************************************************************
 * Get_fullpath(): get the current directory
 * this fails if the user cannot read the cwd, so don't call it unless
 * it's necessary!!
 ***************************************************************************/

static char *Get_fullpath (void)
{
    static char cwd_dir[MAXPATHLEN] = "";
    /*
     * get current working directory
     */
    if (*cwd_dir) {
	return (cwd_dir);
    }
    if (getwd (cwd_dir) == 0) {
	logerr_die (XLOG_INFO, "Get_fullpath: getwd failed");
    }
    return (cwd_dir);
}

/***************************************************************************
 * return the full path name
 ***************************************************************************/

static char *Full_path (char *file)
{
    static char full[MAXPATHLEN];
    static char no_automount[MAXPATHLEN];
    struct stat stbuf1, stbuf2;

    assert(file!=(char*)0);

    if (file[0] == '/') {
        assert((strlen(file) + 1) <= sizeof(full));
	(void) strcpy (full, file);

    } else {
	char *home;
	int st_result;

	home = Get_fullpath ();
        assert((strlen(home) + 2 + strlen(file)) <= sizeof(full));	    
	(void) sprintf (full, "%s/%s", home, file);
	/*
	 * if this fails, there's a problem with the code; we
	 * should NOT be calling Get_fullpath() after
	 * changing the directory. Don't put the stat call
	 * in the assert, otherwise it won't happen if we're
	 * compiled with -DNDEBUG.
	 */
	st_result = stat (file, &stbuf2);
	assert (st_result >= 0);
    }

    /* try and be a little automounter-conscious -- this will
     * help in case the filesystem is unmounted by the automounter
     * between spool-time and print-time.
     */
    if (strncmp (full, AUTOMOUNT_STR, AUTOMOUNT_LEN) == 0) {
	(void) strcpy (no_automount, full + AUTOMOUNT_LEN);

	if (stat (no_automount, &stbuf1) != NULL) {
	    /* file exists on this path -> check inode */
	    if ((stbuf1.st_ino == stbuf2.st_ino)
		    && (stbuf1.st_dev == stbuf2.st_dev))
	    {
		/* it's valid, return the stripped string */
		return (no_automount);
	    }
	}
    }
    return (full);
}

/***************************************************************************
 * makes a link to the target, and puts out entry
 ***************************************************************************/

static char *Make_link(char *target, struct stat *statb)
{
#ifdef NO_SYMLINKS
    fatal (XLOG_INFO, "no symbolic links allowed for security reasons");

#else				/* NO_SYMLINKS */
    char *lf;			/* link file */
    char buf[MAXPATHLEN];
    int i;

    assert(target!=(char*)0);
    assert(statb!=(struct stat*)0);
    assert(SD!=(char*)0);
    /*
     * get temp file name
     */
    lf = Get_tmp_data ();
    (void) sprintf (buf, "%s/%s", SD, lf);
    /*
     * generate absolute path name
     */
    target = Full_path (target);
    if (Debug > 3)
	log (XLOG_DEBUG, "Make_link: from %s to %s", buf, target);

    user_to_daemon ();
    i = symlink (target, buf);
    daemon_to_user ();
    if (i < 0) {
	logerr_die (XLOG_INFO, "Make_link: symbolic link from %s to %s failed",
		    buf, target);
    }
    if (stat (target, statb) < 0) {
	logerr_die (XLOG_INFO, "Make_link: stat failed '%s'", target);
    }
    return (lf);
#endif				/* NO_SYMLINKS */
}

/***************************************************************************
 * stat the file in the specified directory
 ***************************************************************************/

static void Get_stat(char *dir, char *file, struct stat *statb)
{
    static char buf[BUFSIZ];	/* for holding the pathname */

    assert(file!=(char*)0);
    assert(statb!=(struct stat*)0);
    if (dir && *dir) {
        assert((strlen(dir)+2+strlen(file))<=sizeof(buf));
	(void) sprintf (buf, "%s/%s", dir, file);
    } else {
        assert((strlen(file)+1)<sizeof(buf));
	(void) strcpy (buf, file);
    }
    if (stat (buf, statb) < 0) {
	logerr_die (XLOG_INFO, "Get_stat: cannot stat %s", buf);
    }
    if (Debug > 5)
	log (XLOG_DEBUG, "Get_stat: %s is %ld", buf, statb->st_size);
}

/***************************************************************************
 * Prints a line out in the control file
 ***************************************************************************/

static void Entry (char c, char *cp, FILE *fp)
{
    int d;

    assert(fp!=(FILE*)0);
    assert(cp!=(char*)0);
    d = 0;

    /* If the U option is set, the job may not appear under the bannername "root",
     * but under this name on which the U option has been set 
     */
    if (u_option) {
	if (c == 'P') {
	    cp = BNRNAME;
	} else if (c == 'L') {
	    cp = "U-OPTION";
	}
    }

    if (Debug > 4)
	log (XLOG_DEBUG, "Entry: %c '%s'", c, cp);

    if ((int) strlen (cp) > MAXPARMLEN) {
	d = cp[MAXPARMLEN];
	cp[MAXPARMLEN] = 0;
    }
    if (fprintf (fp, "%c%s\n", c, cp) == EOF) {
	logerr_die (XLOG_INFO, "error while writing control file");
    }
    if (d) {
	cp[MAXPARMLEN] = d;
    }
}

void Make_symlink_entry (dev_t dev, ino_t ino, FILE *cfp) {
    char s[32];

    (void) sprintf (s, "%ld %ld", dev, ino);
    Entry ('S', s, cfp);
}

/***************************************************************************
 * 1. Get a control file (actually a temp file)
 * 2. Write the header information into the file
 * 3. If you have a filter output, queue that
 * 4. otherwise if you have a simple file, queue that
 * 5. otherwise copy all of the files
 ***************************************************************************/

void Make_job(void) {
    char *cf;			/* control file name */
    FILE *cfp;			/* control file fp */
    int fd;			/* data file fd */
    int i, c;			/* ACME Integers, Inc. */
    long j, jobsize;
    char *s;			/* ACME Pointers, Inc. */
    struct stat f_stat, c_stat;	/* source and copy files status */
    unsigned long secs, usecs;
    ino_t ino;			/* used to record data for file symlink points to */
    dev_t dev;

    jobsize = 0;
    ino = dev = 0;
    /*
     * get the control file
     */
    cf = Get_cf ();
    i = Open_SD (cf);
    if ((cfp = fdopen (i, "w")) == NULL) {
	logerr_die (XLOG_INFO, "Make_job: IMPOSSIBLE- fdopen failed %s", cf);
    }

    /*
     * place in the current time as accurately as possible.
     */
    hires_time (&secs, &usecs);
    if (usecs) {
	(void) sprintf (WHENSP, "%lu.%lu", secs, usecs);
    } else {
	(void) sprintf (WHENSP, "%lu", (unsigned long) time ((time_t *) 0));
    }

    /* set the class name to the priority if not specified */
    if (CLASSNAME[0] == 0) {
	CLASSNAME[0] = Priority;
	CLASSNAME[1] = 0;
    }

    /* set the job name to the first file if not specified */
    if (Parmcount == 0) {
	if (JOBNAME[0] == 0) {
	    strncpy (JOBNAME, "(stdin)", MAXPARMLEN);
	    JOBNAME[MAXPARMLEN] = '\0';
	}
    } else {
	for (i = 0; i < Parmcount; ++i) {
	    if (Parms[i].str) {
		if (JOBNAME[0] == 0) {
		    strncpy (JOBNAME, Parms[i].str, MAXPARMLEN);
		    JOBNAME[MAXPARMLEN] = '\0';
		}
	    }
	}
    }

    /*
     * Insert EU entry into JOBNAME (J) entry. This bit is unnecessarily paranoid about
     * commas getting in, they can't get in from the user options.
     */
    if (EU) {
	int jlen = 0;
	JOBNAME[0] = '\0';
	for (i = 0; i < num_euent; i++) {
	    char *value = EU_entry[i].value;
	    char *name = EU_entry[i].name;
	    if (value && *value) {
		if ((int) (jlen + strlen (name) + strlen (value) + 2) < MAXPARMLEN)
		{
		    if (jlen)
			JOBNAME[jlen++] = ',';
		    /* no commas allowed in name */
		    for (; *name; name++) {
			if (*name == ',')
			    JOBNAME[jlen++] = ' ';
			else
			    JOBNAME[jlen++] = *name;
		    }
		    JOBNAME[jlen++] = '=';

		    /* no commas allowed in value */
		    for (; *value; value++) {
			if (*value == ',')
			    JOBNAME[jlen++] = ' ';
			else
			    JOBNAME[jlen++] = *value;
		    }
		} else {
		    log (XLOG_INFO,
			"Make_job: EU control file entry too long: ignoring %s=%s",
			name, value);
		}
	    }
	}
	JOBNAME[jlen] = '\0';
    }
    /*
     * Put entries in for each character in the Parms list
     *
     * BSD lpd uses the L record as the trigger for doing the banner.
     * If the L record comes before the J then the banner will show
     * the job string for the previous job.  To work around this we
     * don't put out the L record here but do so just before the
     * data file records. -- Paul Haldane (EUCS)
     */
    for (i = 'A'; i <= 'Z'; ++i) {
	s = CFparm[i - 'A'];
        if (strchr (s, '\n')) {		/* check for sneaky newlines */
            fatal (XLOG_CRIT,
                "Newline in job specs: possible crack attempt by %s!", LOGNAME);
        }
	if ((i != 'L') && s[0]) {
	    /* backward compatibility flag. Stick to the RFC as much as
	     * possible, including truncating possibly overlong fields.
	     */
	    if (BK) {
		int j;
		switch (j = i) {
		case 'D': case 'E': case 'Q': case 'R': case 'S': case 'Z':
		    break;

		case 'J':
		    s[99] = '\0'; goto bk_makeentry;

		case 'N':
		    s[131] = '\0'; goto bk_makeentry;

		case 'T':
		    s[79] = '\0'; goto bk_makeentry;

		case 'C': case 'H': case 'M': case 'P':
		    s[31] = '\0';
bk_makeentry:
		default:
		    Entry (j, s, cfp);
		    break;
		}
	    } else {
#ifdef SHORTHOSTNAME
		/*
		 * this is for machines that will always 
		 * have their fqdn attached in the H field
		 * of the control file; shorthostname is ok.
		 */
		if (i == 'H') {
		    s = ShortHost;
		}
#endif
	        Entry (i, s, cfp);
	    }
	}
    }

    /* For explanation of why we do this here see note at top of function.
     * -- Paul Haldane (EUCS)
     */
    s = CFparm['L'-'A'];
    if (s[0]) {
	if (BK) {
	    s[31] = '\0';
	}
        Entry('L', s, cfp);
    }

    /*
     * put out the created files or copy the ones passed; first find if there are any.
     */

    if (Read_stdin || Filter_stdin || Pr_stdin) {
	if (Filter_stdin) {
	    s = Filter_stdin;
	} else if (Pr_stdin) {
	    s = Pr_stdin;
	} else {
	    s = Read_stdin;
	}
	Get_stat (SD, s, &f_stat);
	j = (f_stat.st_size + 1023) / 1024;
	for (c = 0; c == 0 || c < Copies; ++c) {
	    Entry (Format, s, cfp);
	    jobsize += j;
	    if (MX > 0 && jobsize > MX) {
		Diemsg ("job too large. split up");
	    }
	}
	Entry ('U', s, cfp);
	Entry ('N', "(stdin)", cfp);

    } else {
	/*
	 * we create an entry for each file in the list
	 */
	for (i = 0; i < Parmcount; ++i) {
	    if (!Parms[i].str) {	/* skip if non-printable */
		continue;
	    }
	    /*
	     * if the filename field is set, a prefilter has
	     * been run and has already put the files in the
	     * spool directory, so we just have to stat 'em.
	     */
	    if (*Parms[i].filename) {
		char fullpath[MAXPATHLEN];

		if (Debug > 3)
		    log (XLOG_DEBUG, "prefilter output in spool dir: %s",
			Parms[i].filename);

                assert((strlen(SD)+2+strlen(Parms[i].filename))<=sizeof(fullpath));
		(void) sprintf (fullpath, "%s/%s", SD, Parms[i].filename);

		if (stat (fullpath, &f_stat) < 0) {
		    fatal (XLOG_CRIT, "Make_job: cannot stat %s", fullpath);
		}

	    } else {
		if (Debug > 3)
		    log (XLOG_DEBUG, "Make_job: copying/linking to spool dir: %s",
			Parms[i].str);
		/*
		 * repeat the check for printability
		 */
		if ((fd = Is_printable (Parms[i].str, &f_stat)) < 0) {
		    fatal (XLOG_CRIT, "Make_job: file %s become bad, %s@%s",
			   Parms[i].str, Person, Host);
		}
		if (Use_links) {
		    /*
		     * try to make a symbolic link to the file
		     */
		    s = Make_link (Parms[i].str, &c_stat);
		    dev = f_stat.st_dev;
		    ino = f_stat.st_ino;	/* setting these also acts as a flag */
		} else {
		    /*
		     * make a copy in the spool directory
		     */
		    s = Copy_file (fd, Parms[i].str, &c_stat);
		}
		(void) strcpy (Parms[i].filename, s);
		(void) close (fd);
		if (f_stat.st_size != c_stat.st_size) {
		    fatal (XLOG_CRIT, "Make_job: file %s changed, %s@%s",
			   Parms[i].str, Person, Host);
		}
	    }

	    Parms[i].size = (f_stat.st_size + 1023) / 1024;
	    if (Debug > 4)
		log (XLOG_DEBUG, "Make_job: %s -> %s, %d",
		     Parms[i].str, Parms[i].filename, Parms[i].size);
	}

	for (i = 0; i < Parmcount; ++i) {
	    if (!Parms[i].str) {	/* skip if non-printable */
		continue;
	    }
	    for (c = 0; c == 0 || c < Copies; ++c) {
		if (dev && ino)
		    Make_symlink_entry (dev, ino, cfp);
		Entry (Format, Parms[i].filename, cfp);
		jobsize += Parms[i].size;
		if (MX > 0 && jobsize > MX) {
		    Diemsg ("job too large (over %ldk). split up.", jobsize);
		}
	    }
	    Entry ('U', Parms[i].filename, cfp);
	    Entry ('N', Parms[i].str, cfp);

	    if (Remove && (Use_links == 0)) {
		if (unlink (Parms[i].str) < 0) {
		    Warnmsg ("could not remove %s: %s",
			     Parms[i].str, Errormsg (errno));
		}
	    }
	}
    }
    /*
     * flush the control file and rename it
     */
    if (fflush (cfp) == EOF) {
	logerr_die (XLOG_INFO, "cannot flush %s", cf);
    }
    if (fclose (cfp) == EOF) {
	logerr_die (XLOG_INFO, "cannot close %s", cf);
    }
    if (Ctrlfile_logdir && *Ctrlfile_logdir) {
	char buff[128];
	(void) sprintf (buff, ".lpr-out.%s", ShortHost);
	Log_cf (cf, buff, fileno (cfp));
    }

    Rename_cf (cf);
}
