/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: recvfiles.c
 * Receive files from a remote site
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "library/filestack.h"
#include "LPD/setproctitle.h"
#include "LPD/statfs.h"

/**************************************************************************/

#define ACK_SUCCESS	0	/* succeeded; delete remote copy */
#define ACK_STOP_Q	1	/* failed; stop the remote queue */
#define ACK_RETRY	2	/* failed; retry later */

static int send_ack (int), client_cmd (int *, long *, char *),
    add_to_tmpfiles (char *), get_file (long, int, char *),
    Validate_cf (int, FILE *, char *), check_free_space (int);

int minfree;			/* minimum number of blocks to leave free */
static void get_minfree (void);
void rm_tmpfiles (void);

/***************************************************************************
 * This module implements the file transfer protocol at the receiving
 * site. For each job:
 * 1.  The data files are transferred.
 * 2.  The control files are transferred.
 *     If the transfer is unsuccessful,  all job files are removed.
 *
 * Individual files are transferred using the following protocol.
 *
 * 1.  The sending site sends a line with the size and name of the file,
 * 	and a command code indicating if it is a data or control file.
 * 2.  The receiving site will acknowledge reception.
 * 3.  The remote site will send the file.
 * 4.  The receiving site will acknowledge reception.
 *
 * The transfer protocol is implemented by a simple FSM:
 * INIT:  no files transferred
 * DATA:  data file transferred
 * CONTROL: control file transferred
 *
 * A list of all files associated with the job is maintained,
 * and if any errors result,  the list of files will be deleted.
 *
 ***************************************************************************/

static long jobsize;

static int
get_file_from_remote (long size, char *fname) {
    int i, succ, fd;

    i = add_to_tmpfiles (fname);		/* add fname to list of tmp files */

    /*
     * check to see that file size does not exceed the total
     */
    Parms[i].num = 1;
    Parms[i].size = ((size + 1023) / 1024);
    jobsize = jobsize + Parms[i].size;

    if (MX > 0 && jobsize > MX) {
	log (XLOG_INFO, "incoming job exceeds file size limit (mx=%d)",
		fname, MX);
	send_ack (ACK_RETRY);	/* non-crit; won't stop the remote queue. */
	Link_close (); return -1;
    }

    if (!check_free_space (size)) {
	log (XLOG_INFO, "disk space in \"%s\" is below minfree (mi)", SD);
	send_ack (ACK_RETRY);	/* what Sun's lpd sends in this situation. */
	Link_close (); return -1;
    }

    fd = Exlockcf (fname);              /* we lock the file (die otherwise) */

    if (((succ = get_file (size, fd, fname)) != JSUCC) ||	/* get file */
    		(send_ack (ACK_SUCCESS) != JSUCC))		/* say thanks */
    {
	log (XLOG_INFO, "recvfiles: failed to get file");
	(void) close (fd); (void) unlink (fname);
	return -1;
    }
    return fd;
}

static void
protocol_error (void) {
    setproctitle ("%s: %s: incoming: protocol violation", Printer, From);
    log (XLOG_INFO, "recvfiles: protocol violation");
    rm_tmpfiles ();
    (void) send_ack (ACK_STOP_Q);
    Link_close (); return;
}

static void
remote_error (void) {
    setproctitle ("%s: %s: incoming: remote error", Printer, From);
    if (DbgRem > 0)
	log (XLOG_DEBUG, "recvfiles: remote error");
    rm_tmpfiles ();
    (void) send_ack (ACK_STOP_Q);
    Link_close (); return;
}

/***************************************************************************
 * recvfiles()
 *
 * Gets a job from the remote end.
 *
 * The old Berkeley protocol was to send data files and then the control
 * file.  The PLP protocol will send the control file first, then the data
 * files, followed by the name of the control file.
 *
 * If there is an error during transfer, all the jobs associated with the
 * file are removed.
 *
 * In order to be compatible with the Berkeley LPD, old Berkeley format is
 * also supported.
 ***************************************************************************/

#define BSD_PROTO	0	/* BSD (std) version of LPD protocol */
#define PLP_PROTO	1	/* PLP (extended) */

void
recvfiles (void)
{
    char fname[CFNAMELEN + 1];	/* file name */
    long size;			/* file size */
    int fd;			/* file descriptor */
    char cfname[CFNAMELEN + 1];	/* controlfile name */
    int ctrl_fd;		/* controlfile fd */
    FILE *cfp;			/* controlfile FILE* */
    int i, succ, cmd_code;
    int Protocol_variant;

    /*
     * get the printcap entry
     */
    if (Log_LPRs) {
	log_request ("job received from %s for printer '%s'", From, Printer);
    }
    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	log (XLOG_INFO, "recvfiles: non-existent printer: %s");
    }
    /*
     * set Debug Level from Printcap
     */
    if (DB && DbgRem < DB) {
	DbgRem = DB;
    }
    if ((SD == NULL) || !*SD) {
	/*
	 * no spooling directory, not a Printer
	 */
	if (DbgRem > 0)
	    log (XLOG_DEBUG, "recvfiles: not a printer (no sd)");
	return;
    }
    chdir_SD ();
    (void) Checklockfile (LO, (pid_t *) 0, (char *) 0, 0, &LO_statb);
    if (LO_statb.st_mode & DISABLE_QUEUE) {
	if (DbgRem > 3) {
	    log (XLOG_DEBUG, "recvfiles: spooling disabled");
	}
	send_ack (ACK_STOP_Q);
	return;
    }
    get_minfree ();

    Setuplog (LF, 1);

    setproctitle ("%s: receiving job from %s", Printer, From);

    if (DbgRem > 3)
	log (XLOG_DEBUG, "recvfiles: starting transfer");
    if (send_ack (ACK_SUCCESS) != JSUCC) {
	if (DbgRem > 3)
	    log (XLOG_DEBUG, "recvfiles: confirm failed");
	return;
    }

    while (1) {

	/*
	 * set up the file transfers
	 */
	Protocol_variant = BSD_PROTO;
	jobsize = 0;
	fd = ctrl_fd = -1;
	cfp = (FILE *) NULL;
	Rec_cnt = 0;
	for (i = 0; i < 26; ++i) {
	    CFparm[i][0] = 0;
	}
	while (((succ = client_cmd (&cmd_code, &size, fname)) == JSUCC)
		    && (cmd_code != 0))
	{
	    if (cmd_code == CFILE || cmd_code == CNAME) {
		if (cmd_code == CNAME) {
		    /* CNAME indicates PLP. */
		    Protocol_variant = PLP_PROTO;
	 	}
		if (ctrl_fd != -1) {
		    log (XLOG_INFO, "recvfiles: duplicate ctrl! (%s, %s)", cfname, fname);
		    protocol_error (); return;
		}
		(void) strcpy (cfname, fname); fname[0] = 't';
		if (DbgRem > 3)
		    log (XLOG_DEBUG, "recvfiles: getting cfile %s as %s", cfname, fname);

		ctrl_fd = get_file_from_remote (size, fname);
		if (ctrl_fd < 0) { remote_error (); return; }
		if ((Rec_cnt > 1) && (Protocol_variant != PLP_PROTO)) {
		    break;
		}

	    } else if (cmd_code == DFILE) {
		if (DbgRem > 3)
		    log (XLOG_DEBUG, "recvfiles: getting datafile %s", fname);
		fd = get_file_from_remote (size, fname);
		if (fd < 0) { protocol_error (); return; }
		(void) close (fd);

		if ((ctrl_fd != -1) && (Protocol_variant != PLP_PROTO)) {
		    break;
		}
	    } else if (cmd_code == CEND) {
		if (DbgRem > 3)
		    log (XLOG_DEBUG, "recvfiles: got PLP end-of-job marker");
		break;
	    }
	}

	if (cmd_code == 0) {
	    /* that's all the jobs we're going to get */
	    break;
	}
	/* otherwise, check the files for consistency and reenter the loop */

	if (succ != JSUCC) {
	    if (DbgRem > 3)
		log (XLOG_DEBUG, "recvfiles: error, cmd_code=%d", cmd_code);
	    remote_error (); return;
	}

	/* now that we've got all the files, let's check them out.
	 * ctrl_fd -> open fd into the control file
	 * cfname -> control file name
	 * Parms -> all files (including the control file!)
	 */

	if (ctrl_fd == -1) {
	    log (XLOG_INFO, "no control file in job!");
	    protocol_error (); return;
	}

	if (Rec_cnt == 1) {		/* only the control file */
	    log (XLOG_INFO, "no data files in job!");
	    protocol_error (); return;
	}

	for (i = 0; i < Rec_cnt; ++i) {
	    if (Job_match (cfname, Parms[i].filename) == 0) {
		log (XLOG_INFO, "recvfiles: file with bad format %s", fname);
		protocol_error (); return;
	    }
	}

	/* scan the controlfile for entries, and check to see that all
	 * the datafiles are present.
	 */
	if ((cfp = fdopen (ctrl_fd, "r")) == NULL) {
	    logerr_die (XLOG_CRIT, "recvfiles: fdopen failed");
	}

	/* if ctrlfile-logdir is set, copy the control file to it.
	 */
	if (Ctrlfile_logdir && *Ctrlfile_logdir) {
	    char buff[128];
	    (void) sprintf (buff, ".lpd-in.%s", ShortHost);
	    Log_cf (cfname, buff, fileno (cfp));
	}

	if ((Validate_cf (0, cfp, cfname)) == 0) {
	    (void) fclose (cfp);
	    send_ack (ACK_RETRY); Link_close (); return;
	}

	/* the controlfile is valid -- rename it and reset Rec_cnt.
	 * This way, even if the current lpd is killed, (1) the job
	 * will be printed when another lpd is started, and (2) the
	 * current copies of the files won't get deleted by rm_tmpfiles().
	 */
     
	(void) strcpy (fname, cfname);
	*fname = 't';
	if (rename (fname, cfname) < 0) {
	    logerr (XLOG_WARNING, "recvfiles: cannot rename %s to %s", fname, cfname);
	    rm_tmpfiles ();
	    send_ack (ACK_RETRY); Link_close (); return;
	}
	Rec_cnt = 0;

	/*
	 * We can confirm transfer to the other end
	 */
	if (send_ack (ACK_SUCCESS) != JSUCC) {
	    remote_error (); return;
	}
    }

    setproctitle ("%s: %s: starting printer", Printer, From);
    if (DbgRem > 0)
	log (XLOG_DEBUG, "recvfiles: starting printer");

    Link_close (); Startprinter (); return;
}

/***************************************************************************
 * add_to_tmpfiles(char * Name)
 * add a file Name to the received files
 ***************************************************************************/

static int
add_to_tmpfiles (char *filename)
{
    int i;

    if (DbgRem > 6)
	log (XLOG_DEBUG, "add_to_tmpfiles: temp file \"%s\"", filename);
    /*
     * add a control file name to the parameter list entries if there is nothing in the
     * list, clobber the entry.
     */
    if (Rec_cnt == 0) {
	Parmcount = 0;
    }
    i = Add_name (filename);
    Rec_cnt = Parmcount;
    return (i);
}

/***************************************************************************
 * rm_tmpfiles()
 * remove the files that are in the receive list; do this as daemon
 ***************************************************************************/

void
rm_tmpfiles (void)
{
    int i;			/* ACME Integer, Inc. */

    if (DbgRem > 6)
	log (XLOG_DEBUG, "rm_tmpfiles: removing temporary files");

    for (i = 0; i < Rec_cnt; ++i) {
	(void) unlink (Parms[i].filename);
    }
    Rec_cnt = 0;
}

/***************************************************************************
 * client_cmd(int *cmd_code; long *size; char *name)
 * 1. read a line from the far end
 * 2. the line has the format <cmd_code><size> <name>\n
 *      "%c%d %s\n" format
 * 3. extract the information from the line
 * 4. acknowledge the information
 ***************************************************************************/
static int
client_cmd (int *cmd_code, long *size, char *name)
{
    char buf[BUFSIZ];		/* buffer for reading */
    int i;			/* ACME Integers, Inc. */
    char *cp;			/* ACME Pointers, Inc. */

    /*
     * read a line from the remote end use Bomb-proof Read (bpread), which reads up to
     * the first \n
     */
    if (DbgRem > 3)
	log (XLOG_DEBUG, "client_cmd: waiting for command", buf);
    if ((i = bpread (1, buf, sizeof (buf))) < 0) {
	logerr (XLOG_INFO, "client_cmd: read error from remote");
	*cmd_code = 0;
	remote_error (); return (JFAIL);
    } else if (i == 0) {
	if (DbgRem > 3)
	    log (XLOG_DEBUG, "client_cmd: end of input");
	*cmd_code = 0;
	return (JSUCC);
    }

    i = buf[0];
    if (i != DFILE && i != CFILE && i != CNAME && i != CEND) {
	log (XLOG_INFO, "client_cmd: bad first char (%d)", i);
	protocol_error (); return JFAIL;
    }
    *cmd_code = i;

    /*
     * now pull off the size information
     */
    *size = 0;
    for (cp = &buf[1]; isdigit (*cp); ++cp) {
	*size = 10 * (*size) + ((*cp) - '0');
    }

    if (*cp != ' ') {
	log (XLOG_INFO, "client_cmd: no space separator (%d)", *cp);
	protocol_error (); return JFAIL;
    }
    while (*cp == ' ')		/* skip whitespace */
	++cp;

    if (strlen (cp) >= CFNAMELEN) {
	log (XLOG_INFO, "client_cmd: job name too long '%s'", cp);
	protocol_error (); return JFAIL;
    }

    if (Job_match (cp, cp) == 0) {
	log (XLOG_INFO, "client_cmd: bad job name '%s'", cp);
	protocol_error (); return JFAIL;
    }

    /*
     * must be a data or control file only
     */
    (void) strcpy (name, cp);

    /*
     * send a confirmation
     */
    if (send_ack (0) != JSUCC) {
	remote_error (); return JFAIL;
    }
    if (DbgRem > 3) {
	log (XLOG_DEBUG, "client_cmd: command '%03o', size %d, name \"%s\"",
	     *cmd_code, *size, name);
    }
    return JSUCC;
}

/***************************************************************************
 * get_file (long size; int fd; char *name)
 * 1. copy size bytes from fd 1 to fd
 * 2. the next byte read should be 0; if not, then you have error
 ***************************************************************************/

static int
get_file (long size, int fd, char *name) {
    char buf[BUFSIZ];		/* buffer for reading */
    long cnt;			/* number of bytes */
    int i;			/* ACME Integers, Inc. */

    /*
     * we simply copy the file from the remote end
     */
    if (DbgRem > 3)
	log (XLOG_DEBUG, "get_file: get '%s' (%d bytes) from %s", name, size, From);
    cnt = size;
    while (cnt > 0) {
	if (cnt > sizeof (buf)) {
	    i = sizeof (buf);
	} else {
	    i = cnt;
	}
	i = read (1, buf, i);
	if (i < 0) {
	    logerr (XLOG_INFO, "get_file (from %s): read error", From);
	    remote_error (); return JFAIL;
	}
	if (i == 0) {
	    log (XLOG_INFO, "get_file (from %s): nothing received", From);
	    remote_error (); return JFAIL;
	}
	if (write (fd, buf, i) != i) {
	    logerr (XLOG_INFO, "get_file (from %s): write error", From);
	    remote_error (); return JFAIL;
	}
	cnt = cnt - i;
    }
    i = read (1, buf, 1);
    if (i < 0) {
	logerr (XLOG_INFO, "get_file (from %s): post-file ack: error", From);
	remote_error (); return JFAIL;
    }
    if (i == 0) {
	log (XLOG_INFO, "get_file (from %s): post-file ack not recieved", From);
	remote_error (); return JFAIL;
    }
    if (buf[0] != 0) {
	log (XLOG_INFO, "get_file (from %s): post-file ack: bad (%d)", From, buf[0]);
	remote_error (); return JFAIL;
    }
    return (JSUCC);
}

/***************************************************************************
 * send_ack (int c)
 * acknowledge by sending back single char c
 ***************************************************************************/

static int
send_ack (int c) {
    char buf[1];

    buf[0] = (char) c;
    if (write (1, buf, 1) != 1) {
	if (errno != EBADF) {	/* the fd has already been closed */
	    logerr (XLOG_INFO, "send_ack: '%03o' to %s failed", c, From);
	}
	return JFAIL;
    }
    if (c == 0) {
	if (DbgRem > 5)
	    log (XLOG_DEBUG, "sent acknowledge byte");
    } else {
	if (DbgRem > 5)
	    log (XLOG_DEBUG, "sent bad result byte");
    }
    return JSUCC;
}

/***************************************************************************
 * Validate_cf (scan, fp, fname)
 * Check to ensure that the files contained in this control file
 * were actually sent as part of the control file.
 * 1. fseek to the start of the file
 * 2. read the file, looking for data file entries
 * 3. Check that the names of the data files are consistent with the
 *    name of the control file
 * 4. Check that the file was sent
 * returns: 0 if error, 1 if successful, -1 if use of printer is not permitted
 ***************************************************************************/

static int
Validate_cf (int scan, FILE *fp, char *fname) {
    char buf[BUFSIZ];		/* Buffer, Inc. */
    int c, l;			/* ACME Aluminum Siding and Integers, Inc. */
    char *bp;			/* Buffer */

    if (DbgRem > 6) {
	log (XLOG_DEBUG, "Validate_cf: %d files to check...", Parmcount);
	for (c = 0; c < Parmcount; ++c) {
	    log (XLOG_DEBUG, ", %d '%s'(%d)", c,
			    Parms[c].filename, Parms[c].num);
	}
    }
    
    if (fseek (fp, 0L, 0) < 0) {
	logerr_die (XLOG_CRIT, "Validate_cf %s: fseek failed", fname);
    }

    while (fgets (buf, sizeof (buf), fp)) {
	l = strlen (buf);
	/* make sure that they can't overrun our buffers */
	if (l > MAXPARMLEN) {
	    log (XLOG_INFO, "Validate_cf %s: line too long: '%s'", fname, buf);
	    return (0);
	}

	if (buf[l - 1] != '\n') {
	    log (XLOG_INFO, "Validate_cf %s: no newlines!", fname);
	    return (0);
	}

	buf[l - 1] = 0;
	c = *buf;
	bp = buf + 1;
	if (!isascii (c) || !isalnum (c)) {
	    log (XLOG_INFO, "Validate_cf %s: bad line: '%s'", fname, bp);
	    return (0);
	}

	/*
	 * Check to see that data file information is OK
	 */
	if (c == 'X')
	    c = 'T';		/* old PLP compatibility -- stick with the rfc */

	if (isupper (c) && c != 'L' && c != 'U') {
	    /* put it into CFparm for later use. */
	    (void) strcpy (CFparm[c - 'A'], bp);
	}

	if (islower (c) || c == 'U') {
	    /*
	     * check to see that the file is in the list
	     */
	    if (Job_match (fname, bp) == 0) {
		log (XLOG_INFO, "Validate_cf: %s@%s, file %s, bad data file '%s'",
		     LOGNAME, FROMHOST, fname, bp);
		return (0);
	    }
	    
	    if ((l = Find_name (bp)) < 0) {
		log (XLOG_INFO,
		     "Validate_cf: %s@%s, file %s, data file '%s' not in list",
		     fname, bp);
		return (0);

	    } else if (Parms[l].num == 0) {
		log (XLOG_INFO,
		     "Validate_cf: %s@%s, file %s, data file '%s' not transferred",
		     fname, bp);
		return (0);
	    }
	    /* we've already checked the size of the incoming file-set,
	     * so don't do it again.
	     */
	}
    }

    /*
     * check to see if the remote submitter has valid authorizations
     */
    if (LOGNAME[0] == 0) {
	log (XLOG_INFO, "Validate_cf: job %s from host %s: missing username", fname,
	     FROMHOST);
	return (0);
    }

    if (FD) {
	/*
	 * check to see if you will accept forwarded files; if FD is set, then no
	 * forwarded files are allowed
	 */
	if (hostcmp (FROMHOST, FQDN)) {
	    log (XLOG_INFO, "Validate_cf: %s@%s, forwarded job '%s' refused",
		 LOGNAME, FROMHOST, fname);
	    return (0);
	}
    }

    /* jmason: important note!
     * 
     * if we check the printer_perms file and refuse a job here, the user will not be
     * notified and the remote queue will be stopped. This is not good, so just pass the
     * job on to the unspooling stage; it'll be stopped there.
     */
    return (1);
}

static int
check_free_space (int size) {
    /*
     * returns 1 if enough space left in spool partition, 0 if out of space
     */
    plp_struct_statfs fsb;
    int space;

    if (plp_statfs (SD, &fsb) == -1) {
	logerr (XLOG_WARNING, "failed to statfs for %s", SD);
	return (1);
    }
    size = (size + 1023) / 1024;
    space = (plp_fs_free_bytes (fsb) + 1023) / 1024;
    if (minfree + size > space)
	return (0);

    return (1);
}

static void
get_minfree (void) {
    FILE *fp;

    minfree = 0;

    if (!MI || !*MI) {			/* minfree is not being used */
	return;
    }

    if (*MI >= '0' && *MI <= '9') {
        minfree = atol (MI);		/* value is numeric */
	return;
    }

    /* we don't bother caching the value, the lookup
     * only happens once per invocation.
     */
    if ((fp = fopen (MI, "r")) != NULL) {
	int retval;
	retval = fscanf (fp, "%d", &minfree);
	if (retval != 1) {
	    log (XLOG_INFO, "couldn't read value from minfree file '%s'", MI);
	    minfree = 0;
	}
	fclose (fp);
    }

    if (DbgRem > 2) {
	log (XLOG_DEBUG, "minfree value is set to %d", minfree);
    }
}
