/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: lpd.c
 * lpd main program
 ***************************************************************************
 * lpd -- line Printer daemon.
 * 1. Get options
 * 2. Check for existence of another daemon and exit if it exists.
 *    Note: this is done by locking a file
 * 3. Initialize error logging
 * 4. Start up all the queue handlers
 * 5. Set up communication sockets
 * 6. Loop forever doing:
 *     Listen for a connection and perform the requested operation.
 *     Operations are:
 *	    \1Printer\n
 *	    	check the queue for jobs and print any found.
 *	    \2Printer\n
 *	    	receive a job from another machine and queue it.
 *	    \3Printer [users ...] [jobs ...]\n
 *	    	return the current state of the queue (short form).
 *	    \4Printer [users ...] [jobs ...]\n
 *	    	return the current state of the queue (long form).
 *	    \5Printer Person [users ...] [jobs ...]\n
 *	    	remove jobs from the queue.
 *	    \6Printer Person operation
 *	    	enable/disable queueing, etc.
 *    Note: a process is forked for each operation
 *
 * Strategy to maintain protected spooling area:
 *	1. Spooling area is writable only by daemon and daemon group
 *	2. lpr runs setuid ROOT; it uses the user UID to access any
 *         file it wants and sets ownership of files in the spooling
 *         directory to be owner daemon, group daemon.
 *	3. Critical files in spooling area are owned by daemon, group daemon
 *	   with mode 660.
 *	4. lpd, lpq and lprm run setuid ROOT, and change group to daemon
 *	   Users can't get to anything w/o help of lpq and lprm programs.
 *	5. LPD forks a server process to service each device
 *	   in the printcap entry.  These processes run setuid root
 *	   and setgrp to their PID.  They open neccessary files, and
 *	   fork "filter" processes to process the output.  Filter
 *	   processes run UID daemon.
 *	6. The server process group is set to the main server PID.  This is used
 *	   by the LPRM program, which signals them with killpg(2)
 *
 * Error reporting:
 *	lpd will open a standard logging file. If not already present,
 *      no messages will be logged.  Critical messages will also be logged
 *      using syslog(8). see errormsg.c.
 *	The individual daemon processes will open log files also.  If
 *	they cannot be opened, then no logging will be done for the process.
 ***************************************************************************/

#include "lp.h"
#include "patchlevel.h"
#include "library/errormsg.h"
#include "library/utils.h"
#include "library/printcap_file.h"
#include "library/checkperm.h"
#include "library/readconf.h"
#include "LPD/setproctitle.h"

plp_signal_t cleanup (int);	/* file system scullery maid, cleans up files */
plp_signal_t rereadconfs (int);	/* re-read the config files */

static pid_t lpdpid;		/* lpd process id */
static int finet;		/* socket for incoming requests */

int Inetd_mode;			/* invoked by inetd? */
int Queue_mode;			/* just handle queue management? */
int Single_printer;		/* debug the queue management? */
int Foreground;			/* standalone: no queue mgmt or forking */
char NeedReread;		/* we need to reload the databases */

static void Startup (), Getoptions ();
static int From_host ();

#if (SOCK_FAMILY==AF_UNIX)	/* networking addresses */
struct sockaddr_un sa_us, sa_them;
#else
struct sockaddr_in sa_us, sa_them;
#endif

#ifdef HAVE_SELECT_H
#include <select.h>
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

void 
standalone_main (void) {
    pid_t f;			/* No ACME Integer this time, dude... */
    int nfds;			/* accept return value */
    int req;			/* request on this socket */
    int port_num;		/* used to get the inet server information */
    int fromlen;
#ifdef SO_REUSEADDR
    int one = 1;
#endif
    int defreadfds, readfds;	/* select()ed connections */

    /*
     * check to see that the is not a spooler present.  The LO is used when we exit to
     * clean out the lock file information. Note that we check first, as the initial
     * process, and then we fork another process.  This allows the output message to be
     * printed on the stdout device in a synchronous manner.
     */
    LO = Masterlock;
    if ((Lfd = Getlockfile (LO, &f, (char *) 0, 0, (struct stat *) 0)) == NULL) {
	if (Inetd_mode) {
	    logerr_die (XLOG_INFO, "PLP lpd: active LPD in standalone mode, pid %d", f);
	    exit (1);
	} else {
	    (void) fprintf (stdout, "Active PLP lpd, pid %ld\n", (long) f);
	    exit (0);
	}
    }
    (void) fclose (Lfd);

    if (!Foreground) {
	/* We can fork and let the parent die */
	if ((f = fork ()) < 0) {
	    logerr_die (XLOG_INFO, "lpd: fork failed");
	} else if (f) {
	    exit (0);
	}
	Setuplog (Lpdlogf, 0);	/* We can now start logging... */
    }
    if ((Lfd = Getlockfile (LO, &f, (char *) 0, 0, (struct stat *) 0)) == NULL) {
	printf ("Active PLP lpd, pid %ld\n", (long) f);
	exit (1);
    }				/* ...and locking */
    /*
     * save the PID of the LPD process;  it will be useful later
     */
    lpdpid = getpid ();

    if (!Foreground) {
	/*
	 * set pgrp to pid, so we can kill our kids later; also, drop the control tty.
	 * This prevents the user which started LPD from inadvertenly generation a signal
	 * and clobbering LPD.
	 */
#ifdef TIOCNOTTY
	if ((f = open ("/dev/tty", O_RDWR, 0)) >= 0) {
	    (void) ioctl (f, TIOCNOTTY, (struct sgttyb *) 0);
	    (void) close (f);
	}
#endif
	if (plp_setsid () == -1)
	    logerr_die (XLOG_INFO, "setsid failed");
    }
    /*
     * set up signals; note that SIGPIPE may not be needed, but it never hurts to make
     * sure.
     */
    (void) plp_signal (SIGCHLD, Reapchild);
    (void) plp_signal (SIGPIPE, (plp_sigfunc_t)SIG_IGN);
    (void) plp_signal (SIGHUP, rereadconfs);
    (void) plp_signal (SIGINT, cleanup);
    (void) plp_signal (SIGQUIT, cleanup);
    (void) plp_signal (SIGTERM, cleanup);

    /*
     * Set up the IPC.  This is very ugly,  and is stolen directly from the original LPD
     * code and the 4.3 BSD Interprocess Communication Tutorial, which seems to have
     * borrowed it from RLOGIN, etc.
     * 
     * Set up INET socket
     */
    if ((finet = socket (SOCK_FAMILY, SOCK_STREAM, 0)) < 0) {
	logerr_die (XLOG_CRIT, "lpd: cannot create socket");
    }
    port_num = Link_port_num ();

#ifdef SO_REUSEADDR
    /* set SO_REUSEADDR if it's supported */
    if ((setsockopt (finet, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof (one))) == -1) {
	logerr_die (XLOG_CRIT, "lpd: could not set socket option");
    }
#endif

    /* zero the sockaddr structure before using it */
    bzero ((char *) &sa_us, sizeof (sa_us));

#if (SOCK_FAMILY==AF_UNIX)
    sa_us.sun_family = SOCK_FAMILY;
    sprintf (sa_us.sun_path, "/tmp/lpdsock%d", port_num);

    f = bind (finet, (struct sockaddr *) & sa_us,
	      sizeof (sa_us.sun_family) + strlen (sa_us.sun_path));

    if (f < 0) {
	logerr_die (XLOG_CRIT, "lpd: bind failed domain: %d %s",
		    sa_us.sun_family, sa_us.sun_path);
    }
#else
#if (SOCK_FAMILY==AF_INET)
    sa_us.sin_family = SOCK_FAMILY;
    /* the INADDR_ANY will allow different nets to be used */
    sa_us.sin_addr.s_addr = INADDR_ANY;
    sa_us.sin_port = (unsigned short) port_num;

    f = bind (finet, (struct sockaddr *) & sa_us, sizeof (sa_us));

    if (f < 0) {
	logerr_die (XLOG_CRIT, "lpd: bind to internet socket failed");
    }
#else
    sa_us.sin_family = SOCK_FAMILY;
    sa_us.sin_addr.s_addr = INADDR_ANY;
    sa_us.sin_port = port_num;

    f = bind (finet, (struct sockaddr *) & sa_us, sizeof (sa_us));

    if (f < 0) {
	logerr_die (XLOG_CRIT, "lpd: bind failed in unknown domain");
    }
#endif
#endif

    /* put PID and time started in the lockfile */
    Setlockfile (LO, Lfd, lpdpid, Time_str ());

    if (listen (finet, 5) < 0) {
	logerr_die (XLOG_CRIT, "lpd: listen failed internet domain");
    }
    if (Debug > 2)
	log (XLOG_DEBUG, "lpd: socket %d", finet);

    /* Set up the sockets on which we will do an accept */
    defreadfds = (1 << finet);

    Startup ();			/* catch up on any old jobs. */

    /* Main loop: accept, do a request, continue. */
    for (;;) {
	/*
	 * Defensive Coding a la Brain Damaged Unix Implementations: a particular UNIX
	 * implementation will not cause SIGCHLD signals in a particularly reliable way.
	 * Apparently if there is a SIGCHLD while it is masked off (error printing does
	 * this) IT IS IGNORED!!!  So we put a Reapchild() here; at least we will gather
	 * them up each time we start a server.
	 */
	(plp_signal_t) Reapchild (0);

	if (NeedReread) {
	    NeedReread = 0;
	    log (XLOG_INFO, "rereading config files.");
	    Readconf ();
	}

	if (Debug > 2)
	    log (XLOG_DEBUG, "lpd: starting select");
	setproctitle ("accepting connections");		/* a la sendmail! */

	readfds = defreadfds;
	nfds = select (20, (fd_set *) &readfds, (fd_set *) 0, (fd_set *) 0,
		       (struct timeval *) 0);		/* wait patiently */
	if (nfds <= 0) {
	    if (nfds < 0 && errno != EINTR) {
		logerr_die (XLOG_CRIT, "lpd: select error");
	    }
	    continue;
	}
	/*
	 * incredible, but true: could have NO requests
	 */
	if (readfds) {
	    if (Debug > 2)
		log (XLOG_DEBUG, "lpd: doing select 0x%x", readfds);
	    /* get the socket causing the select */
	    fromlen = sizeof (sa_them);
	    req = accept (finet, (struct sockaddr *) & sa_them, &fromlen);
	    if (req < 0) {
		logerr (XLOG_CRIT, "lpd: finet accept");
		continue;
	    }
	    if (!From_host ()) {
		logerr (XLOG_INFO, "lpd: From_host failed");
		close (req);
		continue;
	    }
	    /*
	     * Fork process to handle activity requested
	     */
	    (void) fflush (stdout);
	    (void) fflush (stderr);
	    if ((f = fork ()) == 0) {	/* daughter */
		(void) fclose (Lfd);
		(void) close (finet);
		Lfd = NULL;
		if (dup2 (req, 1) < 0) {
		    logerr_die (XLOG_CRIT, "lpd: dup2 for server failed");
		}
		(void) close (req);
		servicereq ();
		(void) close (1);
		exit (0);
	    } else if (f < 0) {
		logerr (XLOG_CRIT, "lpd: fork failed");
	    }
	    (void) close (req);
	}
    }
}

/*
 * inetd_main: handle one request and exit.
 */

#if (SOCK_FAMILY==AF_INET)
void 
inetd_main (void) {
    int len;			/* length of the sockaddr structures */

    Setuplog (Lpdlogf, 1);	/* start logging, but don't close stdin */

    /* save the PID of the LPD process;  it will be useful later */
    lpdpid = getpid ();

    /*
     * set up signals; note that SIGPIPE may not be needed, but it never hurts to make
     * sure.
     */
    (void) plp_signal (SIGCHLD, Reapchild);
    (void) plp_signal (SIGPIPE, (plp_sigfunc_t)SIG_IGN);
    (void) plp_signal (SIGHUP, rereadconfs);
    (void) plp_signal (SIGINT, cleanup);
    (void) plp_signal (SIGQUIT, cleanup);
    (void) plp_signal (SIGTERM, cleanup);

    len = sizeof (sa_them);
    if (getpeername (0, (struct sockaddr *) & sa_them, &len) == -1) {
	logerr_die (XLOG_INFO, "lpd: getpeername failed");
    }
    if (!From_host ()) {	/* failure due to insecure remote port */
	logerr (XLOG_INFO, "lpd: From_host failed");
	exit (0);
    }
    len = sizeof (sa_us);
    if (getsockname (0, (struct sockaddr *) & sa_us, &len) == -1) {
	logerr_die (XLOG_INFO, "lpd: getsockname failed");
    }
    if (dup2 (0, 1) < 0) {
	logerr_die (XLOG_CRIT, "lpd: dup2 for server failed");
    }
    servicereq ();		/* this'll read from fd 1 */
    close (0);
    close (1);
}

#endif

/*
 * queue-managing mode: just fork off the queue lpds.
 */
void 
queue_main (void) {
    pid_t f;			/* No Acme Integer, Inc., again !! */

    if (Single_printer) {
	/* don't lock the main lockfile, otherwise the other printer queues
	 * won't be runnable!
	 */
	Startprinter ();
    } else {
	/*
	 * check to see that the is not a spooler present.  The LO is used when we exit to
	 * clean out the lock file information.
	 */
	LO = Masterlock;
	if ((Lfd = Getlockfile (LO, &f, (char *) 0, 0, (struct stat *) 0)) == NULL) {
	    (void) fprintf (stdout, "Active PLP lpd, pid %ld\n", (long) f);
	    exit (1);
	}
	Startup ();
	Closelockfile (LO, Lfd);
	Lfd = 0;
    }
    exit (0);
}

int
main (int argc, char **argv, char **envp) {

    (void) umask (0);		/* explicitly set umask */
    set_user_uid (1);
    root_to_daemon ();		/* adopt safe perms (main) */

    Getoptions (argc, argv);	/* process command line options */

    if (Inetd_mode) {
	close (2);	
	/* otherwise, log output may cause 'lpr' requests to fail
	 * (resulting in "bad confirm" messages)
	 */
    }
    Std_environ (argc, argv, envp);
    Readconf ();

    if (Inetd_mode) {
	inetd_main ();
    } else if (Queue_mode) {
	queue_main ();
    } else {
	standalone_main ();	/* never exits */
    }
    return 0;
}

/***************************************************************************
 * Getoptions(argv, argc)
 *    char **argv; int argc;
 * Purpose:
 * extracts options and arguments from command line using Getopt(2)
 * Side Effects:
 *	-X : calls setup_test() to modify defaults
 *  -D nn : sets Debug level
 *  -L file : sets log file
 ***************************************************************************/

static char usage_msg[] =
"usage: lpd [-iqv] [-D dbg] [-L log] [-P printer] [-F]\n(\"lpd -v\" for more help)\n";

static void 
show_version (void) {
    printf ("PLP version %s\n\n", PLP_PATCHLEVEL);

    printf ("by Patrick Powell, U. Minnesota\n");
    printf ("Justin Mason, Iona Technologies <jmason@iona.ie>\n");
    printf ("Angus Duggan, Harlequin Ltd. <angus@harlequin.co.uk>\n");
    printf ("Julian Turnbull, Edinburgh University <jst@dcs.edinburgh.ac.uk>\n");

    printf ("\n%s\narguments:\n", usage_msg);
#if (SOCK_FAMILY==AF_INET)
    printf ("        -i inetd mode\n");
#endif
    printf ("        -q queue-processing mode\n");
    printf ("        -v version info and extra usage (this message)\n");
    printf ("        -D diagnostics level\n");
    printf ("        -L alternate log file\n");
    printf ("        -P printer (handle only one queue)\n");
    printf ("        -F operate in foreground\n\n");

    printf ("This version of PLP uses the following definitions:\n\n");

    printf ("        locking:     %s\n",
#ifdef HAVE_FCNTL_LOCK
			"fcntl (preferred)"
#else
#ifdef HAVE_LOCKF
			"lockf"
#else
#ifdef HAVE_FLOCK
			"flock (does NOT work over NFS)"
#else
			"none"
#endif
#endif
#endif
    );

    printf ("        stty:        %s\n",
#ifdef USE_SGTTY
			"sgtty"
#else
#ifdef USE_TERMIO
			"termio"
#else
#ifdef USE_TERMIOS
			"termios"
#else
			"none"
#endif
#endif
#endif
    );
    printf ("        arch string: %s\n", ARCHITECTURE);
    exit (0);
}

static void
Getoptions (int argc, char **argv) {
    int c;
    Inetd_mode = 0;
    Queue_mode = 0;
    Single_printer = 0;
    Foreground = 0;
    while ((c = Getopt (argc, argv, "iqvD:L:P:F")) != EOF) {
	switch (c) {
	case 'D':		/* turn on Debugging */
	    if (Optarg == NULL) {
		(void) fprintf (stderr, usage_msg);
		exit (1);
	    }
	    Parse_debug (Optarg);
	    break;
	case 'L':
	    if (Optarg == NULL) {
		(void) fprintf (stderr, usage_msg);
		exit (1);
	    }
	    allocandcpy (Lpdlogf, Optarg);
	    break;
#if (SOCK_FAMILY==AF_INET)
	case 'i':		/* inetd mode */
	    Inetd_mode = 1;
	    break;
#endif
	case 'q':		/* queue management mode */
	    Queue_mode = 1;
	    break;
	case 'P':		/* queue debugging mode */
	    Single_printer = 1;
	    Queue_mode = 1;
	    if (Optarg == NULL) {
		(void) fprintf (stderr, usage_msg);
		exit (1);
	    }
	    allocandcpy (Printer, Optarg);
	    break;
	case 'F':
	    Foreground = 1;
	    break;
	case 'v':
	    show_version ();
	    break;
	default:
	    (void) fprintf (stderr, usage_msg);
	    exit (1);
	}
    }
    if (Optind < argc) {
	(void) fprintf (stderr, "%s: extra argument %s\n", Name, argv[Optind]);
	exit (1);
    }
}

/***************************************************************************
 * Setuplog( char *logfile, int saveme )
 * Purpose: to set up a standard error logging environment
 * saveme will prevent stdin from being clobbered
 *   1.  open /dev/null on fd 0 unless saveme is set
 *   2.  dup 0 to fd 1 unless saveme is set
 *   3.  If logfile is "-" or NULL, output file is alread opened
 *   4.  Open logfile; if unable to, then open /dev/null for output
 ***************************************************************************/
void
Setuplog (char *logfile, int saveme) {
    int fd;

    /*
     * we want 0 (stdin) to be /dev/null
     */
    (void) fflush (stdout);
    (void) fflush (stderr);
    if (saveme != 1) {
	(void) close (0);
	if ((fd = open ("/dev/null", O_RDWR, 0)) != 0) {
	    logerr_die (XLOG_CRIT, "Setuplog: /dev/null opened as %d", fd);
	}
	/*
	 * do the dup
	 */
	(void) close (1);
	if (dup2 (0, 1) < 0) {
	    logerr_die (XLOG_CRIT, "Setuplog: dup2 failed");
	}
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "Setuplog: opening log file %s", logfile);
    /*
     * open logfile;  if it is "-", use stderr
     */
    if (logfile && *logfile && strcmp (logfile, "-")) {
	if ((fd = open (logfile, O_WRONLY | O_APPEND, 0)) < 0) {
	    if (Debug > 0)
		logerr (XLOG_DEBUG, "cannot open logfile %s", logfile);
	    /* write to stderr if debugging, /dev/null otherwise */
	    if (Debug > 0) {
		fd = 2;
	    } else {
		fd = 0;
	    }
	}
	if (fd != 2 && dup2 (fd, 2) < 0) {
	    logerr_die (XLOG_CRIT, "Setuplog: dup2 failed");
	}
	if (fd > 2) {
	    (void) close (fd);
	}
    }
}

/*
 * When a child dies, it will generate a SIGCHLD, and Reapchild() will lay the body to
 * rest with a wait3(). This is masked off after Startprinter() so filters won't
 * get accidentally caught by it.
 */
plp_signal_t 
Reapchild (int signal) {
    int pid;
    plp_status_t status;

    while ((pid = plp_wait_kid (&status, WNOHANG)) > 0) {
	if (Debug > 3)
	    log (XLOG_DEBUG, "Reapchild: process %d, status %s", pid,
		 Decode_status (&status));
    }
#ifdef SVR4
    (void) plp_signal (SIGCHLD, Reapchild);
#endif
}

/*
 * Clean up; kill off children and remove sockets and files
 */
plp_signal_t
cleanup (int signal) {
    int pid;

    plp_block_signals ();

    if (Lfd) {
	Closelockfile (LO, Lfd);
	Lfd = 0;
    }
    pid = getpid ();
    if (Debug > 0)
	log (XLOG_DEBUG, "cleanup");
    rm_tmpfiles ();		/* clean up temp files (if there are any) */

    if (pid == lpdpid) {	/* we are session leader */
	log (XLOG_INFO, "cleanup: lpd process %d terminating", pid);

	(void) killpg (lpdpid, SIGKILL);
	shutdown (finet, 2);	/* shutdown the socket, and */
	close (finet);		/* close it */
    }
    /*
     * Wait for children until the bodies rot
     */
    (plp_signal_t) Reapchild (0);
    exit (Errorcode);
}

/***********************************************************************
 * From_host()
 *   This routine will extract the remote Host Name and remote Host
 *   port number information.  The remote process must connect on
 *   a port in a "reserved" range;  these are reserved for use
 *   by priviledged processes (setuid/uid root).
 * Side Effects: sets the "from" string to the Name of the remote Host.
 * Returns: 1 if privileged and successful, 0 if not.
 * NOTE: this code was derived from the orignal 4.2BSD LPR source,
 *   the 4.3BSD Interprocess Communications Tutorials, and other places.
 ***********************************************************************/

#if (SOCK_FAMILY==AF_UNIX)
static int
From_host (void) {
    if (From != NULL) {
	if (Debug > 4)
	    log (XLOG_INFO, "From_host: freeing old From");
	free (From);
    }
    From = (char *) strdup (Host);
    return (1);
}

#else
static int
From_host (void) {
    struct hostent *hp;		/* Host entry */
    int port;			/* from port number */
    char *dot_ptr;

    port = ntohs (sa_them.sin_port);
    if (sa_them.sin_family != SOCK_FAMILY) {
	logerr (XLOG_CRIT,
		"From_host: malformed from address, family %d, not right %d",
		sa_them.sin_family, SOCK_FAMILY);
	return (0);
    }
    /* FQDN may be overwritten later, btw -- it's supposed to come from
     * the control file.
     */
    if (FQDN) free (FQDN);
    if (From) free (From);
    hp = gethostbyaddr ((char *) (&sa_them.sin_addr),
			sizeof (struct in_addr), sa_them.sin_family);
    if (hp == 0) {
	char *s;		/* temp string */
	s = inet_ntoa (sa_them.sin_addr);
	log (XLOG_INFO, "hostname for address '%s' unknown", s);
	FQDN = (char *) strdup (s);
	From = (char *) strdup (s);

    } else {
	/* the perms lookups may rely on this being fully-qualified,
	 * therefore we must try our best to make it so.
	 */
	char *fqdn = get_fqdn (hp->h_name);
	FQDN = (char *) strdup (fqdn);
	From = (char *) strdup (fqdn);
	if ((dot_ptr = (char *) strchr (From, '.')) != NULL) {
	    *dot_ptr = '\0';
	}
    }

    /* if Maxportno > IPPORT_RESERVED, we're in non-setuid mode. */
    if ((port > IPPORT_RESERVED) && (Maxportno <= IPPORT_RESERVED)) {
	if (!AllowNonPrivPorts) {
	    log (XLOG_CRIT, "refused insecure request from host %s", From);
	    return (0);
	} else if (LogNonPrivPorts) {
	    log (XLOG_INFO, "processing insecure request from host %s", From);
	}
    }
    return (1);
}

#endif

static int
Printer_needs_start (char *printer) {
    Printer = printer;
    if (Get_pc_entry (Printer, All_pc_vars, All_pc_len) == 0) {
	log (XLOG_INFO, "trying to start non-existent printer");
	return (0);
    }
    if (SD == 0 || *SD == 0) {
	/*
	 * no spooling directory, not a Printer
	 */
	if (Debug > 0)
	    log (XLOG_DEBUG, "%s is not a printer (no spool directory)!", Printer);
	exit (0);
    }
    /* Startprinter will do the other sanity checks */
    return (Getq ());
}

/************************************************************************
 * start queue servers
 ************************************************************************/
static void
Startup (void) {
    int pid;
    char **pr;
    plp_status_t status;

    /*
     * Restart the server. Start scanning the printcap for entries
     */
    if ((pid = fork ()) < 0) {
	logerr (XLOG_CRIT, "queue server: cannot fork");
	sleep (60);
	Startup ();		/* retry */
	return;

    } else if (pid == 0) {

	/* stop Reapchild so we can handle our own kids from now on. */
	(void) plp_signal (SIGCHLD, (plp_sigfunc_t)SIG_IGN);

	if (Debug > 4)
	    log (XLOG_DEBUG, "starting servers");
	for (pr = All_printers (); *pr; ++pr) {
	    if (!Printer_needs_start (*pr)) {
		if (Debug > 2)
		    log (XLOG_DEBUG, "empty queue, no need to start server", *pr);
		continue;
	    }
	    {
		int retries = 0;

		while (((pid = fork ()) < 0) && (retries++ < 12)) {
		    logerr (XLOG_CRIT, "server for %s: cannot fork", Printer);
		    sleep (10 * retries);
		}
	    }
	    if (pid == 0) {
		if (Debug > 2)
		    log (XLOG_DEBUG, "started server for %s (pid %d)",
			 Printer, getpid ());
		(void) fclose (Lfd);
		Lfd = 0;
		Startprinter ();
		exit (0);
	    }
	    sleep ((unsigned) 1);
	}
	while ((pid = wait (&status)) > 0) {
	    if (Debug > 3)
		log (XLOG_DEBUG, "Startup: process %d, status %s", pid,
		     Decode_status (&status));
	}
	if (Debug > 4)
	    log (XLOG_DEBUG, "all servers done");
	exit (0);
    }
}

plp_signal_t 
rereadconfs (int signal) {
#ifdef SVR4
    (void) plp_signal (signal, rereadconfs);
#endif
    NeedReread = 1;

    /* we don't want to send a SIGHUP to our process group,
     * as that would kill most filters designed to work with
     * BSD lpds. oh well. :(
     */
}
