/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: localprinter.c
 * local Printer queue job handler
 ***************************************************************************/

#include "lp.h"
#include "LPD/localprinter.h"
#include "library/errormsg.h"
#include "library/utils.h"

static int prjobs;		/* numbers of jobs done */
static int old_PLP_nobanner_flag;	/* obsolete backwards-compat S option */

long jobsize;			/* global so that longbanner() can access it */

static int queuejob (FILE *cfp, struct plp_queue *q);
static int Banner_print (char *prog);
static int printfile (int format, char *file);

/**********************************************************************
 * 1. set up the state to indicated 0 completed jobs
 * 2. get the printer ready
 **********************************************************************/

int
Printinit (void) {
    int n;

    if (DbgLocal > 3)
	log (XLOG_DEBUG, "Printinit: called with prjobs %d", prjobs);
    n = Print_ready ();
    return (n);
}

/**********************************************************************
 * called when there is an error in the job handling
 * 1. set up the state to indicated 0 completed jobs
 * 2. close the printer
 **********************************************************************/
void
Printerror (void) {
    if (DbgLocal > 3)
	log (XLOG_DEBUG, "Printerror: called with prjobs %d", prjobs);

    prjobs = 0;
    Print_close ();
}

/**********************************************************************
 * called when there is an error in the job handling
 * 1. if there are completed jobs, print the trailer
 * 2. close the printer
 **********************************************************************/
void
Printfinal (void) {
    if (DbgLocal > 3)
	log (XLOG_DEBUG, "Printfinal: called with prjobs %d", prjobs);
    /*
     * print out trailer on close with a job out
     */
    if (prjobs) {
	/*
	 * FQ set means FF needed on open
	 */
	if (FQ && FF && *FF) {
	    if (DbgLocal > 3)
		log (XLOG_DEBUG, "Printfinal: FF on final");
	    (void) Print_string (FF);
	}
	if (TR && *TR) {
	    /*
	     * TR is sent out
	     */
	    if (DbgLocal > 2)
		log (XLOG_DEBUG, "Printfinal: TR '%s'", TR);
	    (void) Print_string (TR);
	}
    }
    prjobs = 0;
    Print_close ();
}

/* RFC-1179 has provision for checking a symlink in the spool dir versus
 * what's pointed to. If they fail, the user has done something like this:
 * 
 *   ln -s /etc/motd x; lpr -s x;rm -f x; ln -s /etc/shadow x
 *
 * which is pretty obviously a security hole. Here's what a news posting
 * from Keith Bostic (bostic@vangogh.CS.Berkeley.EDU) has to say on
 * a similar matter (local mail delivery):
 *
 * >A better defense (credit to Eric Allman) is that the local delivery
 * >agent does the lstat, the open and the fstat on the returned file
 * >descriptor, and then *compares* the values of the stat calls, most
 * >notably the dev/ino numbers.  This is what the current 4BSD local mail
 * >delivery agent does.  This should be absolutely safe on most systems,
 * >and statistically safe on systems with multiple NFS mounted filesystems.
 *
 * That's what we're doing here.
 */

int is_race_symlink (char *filename, struct stat *stbuf) {
    struct stat lstbuf;
    unsigned long Sdev;
    unsigned long Sino;
    char *spc;

    if (lstat (filename, &lstbuf) < 0) {
	logerr (XLOG_INFO, "is_race_symlink: cannot lstat file %s", filename);
	return (1);
    }
    if (!S_ISLNK(lstbuf.st_mode) ||	/* not a symlink so we needn't worry */
    	(SLINKDATA[0] == '\0')) 	/* no valid info, we have no choice */
    {
	return (0);
    }

    if ((spc = strchr (SLINKDATA, ' ')) == NULL) {
	return (0);			/* not a valid format */
    }

    *spc = '\0';
    /* need to use strtoul(); dev_t is unsigned long on SVR4, and if we
     * overflow, we'll produce an incorrect error.
     */
    Sdev = strtoul (SLINKDATA, (char **) NULL, 10);
    Sino = strtoul (spc + 1, (char **) NULL, 10);
    *spc = ' ';

    if ((stbuf->st_dev != Sdev) || (stbuf->st_ino != Sino)) {
	return (1);		/* it's a different file; shout about it */
    }
    return (0);			/* it's the same file. */
}

/**********************************************************************
 *   1.  First scan extracts information which controls printing
 *   2.  Set any default values not passed
 *   3.  Second scan does the printing.
 * Returns:  JBUSY, JFAIL, JSUCC, JABORT
 * Side effects:  sets information vector  CFparm[].
 **********************************************************************/
int
Printjob (FILE *cfp, struct plp_queue *q) {
    int i;			/* ACME Integer, Inc. */
    char parm[BUFSIZ];		/* holds a line read in */
    char *arg;			/* control line argument */
    char opt;			/* control line option */
    int jstatus;		/* job status */
    int perms = 'R';		/* file perms */

    if (fseek (cfp, 0L, 0) < 0) {
	logerr_die (XLOG_INFO, "Printjob: start- fseek failed");
    }
    /*
     * set job status
     */
    jstatus = JABORT;		/* default */

    /*
     * initialize the CFparm array, which holds value read from file
     */
    for (i = 0; i < 26; ++i) {
	CFparm[i][0] = 0;
    }
 
    /*
     * read the control file and extract user information
     */
    jobsize = 0;
    Parmcount = 0;
    while (fgets (parm, sizeof (parm), cfp)) {
	if ((arg = strchr (parm, '\n')) == NULL) {
	    log (XLOG_INFO, "bad controlfile (%s), no end-of-line",
		 q->q_name);
	    return (-1);
	}
	*arg = 0;
	opt = parm[0];
	arg = parm + 1;
	if (!isascii (opt) || !isalnum (opt)) {
	    log (XLOG_INFO, "bad controlfile (%s), line='0x%x %s'",
		 q->q_name, opt, arg);
	    return (-1);
	}
	old_PLP_nobanner_flag = 0;
	/*
	 * copy it into the appropriate place if needed
	 */
	if (isupper (opt)) {
	    switch (opt) {
	    case 'U':
	    case 'N':
		break;
	    case 'H':	
		if (strchr (arg, '.') == NULL) {
		    char *fqdn;
		    /* See if this host is in our domain; if it is,
		     * rewrite this into FQDN to save time/hassle
		     * later.
		     */
		    if ((DomainNameUnset == 0) &&
		   	(fqdn = host_in_domain (arg, Host_domain)))
		    {
			(void) strcpy (arg, fqdn);
		    }
		}
		goto add_to_cfparm;

	    case 'S':		/* old PLP compat; support it (urgh) */
		if (arg[0] == 'X') {
		    old_PLP_nobanner_flag = 1;
		    break;
		}
		goto add_to_cfparm;

	    case 'X':
		opt = 'T';	/* old PLP compat; stick with the rfc */
add_to_cfparm:
	    default:
		if (CFparm[opt - 'A'][0]) {
		    log (XLOG_INFO,
			 "duplicate in controlfile: '%c %s' (previous: '%s')",
			 opt, arg, CFparm[opt - 'A']);
		    return (-1);
		}
		if ((int) strlen (arg) >=  MAXPARMLEN) {
		    log (XLOG_INFO,
			 "controlfile %s: line too long: '%s'",
			 q->q_name, arg);
		    return (-1);
		}
		(void) strcpy (CFparm[opt - 'A'], arg);
		break;
	    }

	} else if (islower (opt)) {
	    if (Job_match (q->q_name, arg) == 0) {
		logerr (XLOG_INFO, "Printjob: bad control file '%s' entry '%s'",
			q->q_name, parm);
		return (-1);
	    }
	    if (stat (arg, &LO_statb) < 0) {
		if (SS && *SS) {
		    /* this is a multiple-server queue; chances are that
		     * some other server has nabbed the job before we
		     * got to it.
		     */
		    if (Debug > 4) {
			log (XLOG_DEBUG,
			    "cannot stat file (already printed by other server)");
		    }
		    return (JSUCC);
		} else {
		    logerr (XLOG_INFO, "Printjob: cannot stat file %s", arg);
		    return (-1);
		}
	    }
	    if (is_race_symlink (arg, &LO_statb)) {
		log (XLOG_CRIT,
		    "POSSIBLE CRACK ATTEMPT: 'lpr -s' symlink has been rerouted: %s@%s",
		    LOGNAME, FROMHOST);

		/* return a "successfully printed" status so the queue
		 * won't be stopped (denial-of-service attack). */
		return (JSUCC);
	    }
	    if ((i = Add_name (arg)) < 0) {
		logerr (XLOG_INFO, "Printjob: too many files %s", q->q_name);
		return (-1);
	    }
	    jobsize = jobsize + (LO_statb.st_size + 1023) / 1024;
	    switch (opt) {
	    case 'f':		/* uses the IF filter */
	    case 'l':
	    case 'p':
		break;
	    default:
		if (Filter_name[opt - 'a'] == 0) {
		    log (XLOG_INFO, "no filter for format '%c'!", opt);
		    return (-1);
		}
		break;
	    }
	}
    }

    if (MX > 0 && jobsize > MX) {
	logerr (XLOG_INFO, "job %s: larger than mx: %dk.", q->q_name, jobsize);
	goto error;
    }

    /*
     * Set up any parameters that were not provided to default values
     */
    if (PWIDTH[0] == 0) {
	(void) sprintf (PWIDTH, "%d", PW);
    }
    /*
     * check permissions
     */
    if (hostcmp (&q->q_from, FROMHOST)) {
	log (XLOG_INFO,
	     "Warning: control file origin '%s' and H entry '%s' do not match (user=%s)",
	     &q->q_from, FROMHOST, LOGNAME);
    }
    if (Debug > 0) {
	log (XLOG_DEBUG, "(controlfile host: %s, getpeername host: %s)",
	    FROMHOST, FQDN ? FQDN : "(null)");
    }

    /* TODO: deal with forwarded jobs using host%host in printer_perms file. */

    if ((Checkperm (FROMHOST, LOGNAME, First_name, &perms, (int *) 0, 0) <= 0) ||
    	    (Checkperm (FQDN, LOGNAME, First_name, &perms, (int *) 0, 0) <= 0))
    {
	log (XLOG_INFO,
	     "Sorry %s@%s, but you don't have permission to use '%s'!",
	     LOGNAME, FROMHOST, First_name);
	goto error;

    } else if ((Checkperm (FROMHOST, LOGNAME, First_name, &perms, (int *) 0, 1) <= 0) ||
    	    (Checkperm (FQDN, LOGNAME, First_name, &perms, (int *) 0, 1) <= 0))
    {
	log (XLOG_INFO, "Sorry %s@%s, no more pages allowed on '%s'",
	     LOGNAME, FROMHOST, First_name);
	goto error;
    }
    /*
     * See if there are any files to print
     */
    if (Parmcount == 0) {
	/* no files */
	if (DbgLocal > 4)
	    log (XLOG_DEBUG, "Printjob: no files");
	jstatus = JSUCC;
	goto error;
    }
    /*
     * Pass 2: Do the printing now
     */
    if (fseek (cfp, 0L, 0) < 0) {
	/* this is impossible */
	logerr (XLOG_NOTICE, "Printjob: fseek %s failed", q->q_name);
	goto error;
    }

    if (Log_LPRs) {
	if (Log_no_filenames) {
	    log_request ("%s: printing job %d for %s@%s, %d bytes", Printer,
			 q->q_data, q->q_num, LOGNAME, FROMHOST, q->q_size);
	} else {
	    log_request ("%s: printing job %s(%d) for %s@%s, %d bytes", Printer,
			 q->q_data, q->q_num, LOGNAME, FROMHOST, q->q_size);
	}
    }
    if (QH) {
	/*
	 * queuejob returns one of the Job Status settings
	 */
	jstatus = queuejob (cfp, q);
	goto error;
    }

    /*
     * CL: close printer between jobs. This implies that FFs should
     * not be sent either on open or between jobs.
     */
    if (CL) {
	FO = 0;		/* FF on open = no */
	FQ = 0;		/* FF at end = no */
	SF = 1;		/* suppress trailing FF = yes */
    }

    /*
     * Put out FF on open and between jobs
     */
    if (prjobs == 0) {
	/*
	 * FO set means FF needed on open
	 */
	if (LD && *LD) {
	    if (DbgLocal > 3)
		log (XLOG_DEBUG, "Printjob: putting out LD on open");
	    jstatus = Print_string (LD);
	    if (jstatus != JSUCC) {
		log (XLOG_INFO, "Printjob: LD on open printing failed");
		goto error;
	    }
	}
	if (FO && FF && *FF) {
	    if (DbgLocal > 3)
		log (XLOG_DEBUG, "Printjob: putting out FF on open");
	    jstatus = Print_string (FF);
	    if (jstatus != JSUCC) {
		log (XLOG_INFO, "Printjob: FF on open printing failed");
		goto error;
	    }
	}
    }
    /*
     * print banner if it has not been disabled by the SH flag or the lpr -h flag was
     * used and the AB (always banner) is clear
     */
    if (!(SH || ((old_PLP_nobanner_flag || !BNRNAME[0]) && !AB))) {
	jstatus = Print_banner ();
	if (jstatus != JSUCC) {
	    log (XLOG_INFO, "Printjob: banner printing failed");
	    goto error;
	}
	if (PL == 0) {		/* separator if not at page start */
  	    prjobs = 1;
  	}
    }
    /*
     * Banner Printing Program
     */
    if (BP && *BP) {
	jstatus = Banner_print (BP);
	if (jstatus != JSUCC) {
	    log (XLOG_INFO, "Printjob: BP program %s failed", BP);
	    goto error;
 	}
	prjobs = 1;
    }
    /*
     * print individual jobs
     */
    while (fgets (parm, sizeof (parm), cfp)) {
	if ((arg = strchr (parm, '\n')) == NULL) {
	    log (XLOG_INFO, "bad controlfile (%s), no end-of-line",
		 q->q_name);
	    goto error;
	}
	*arg = 0;
	opt = parm[0];
	arg = parm + 1;
	if ((opt == 0) || (!isalnum (opt))) {
	    log (XLOG_INFO, "bad controlfile (%s), line='0x%x %s'",
		 q->q_name, opt, arg);
	    goto error;
	}
	if (islower (opt)) {
	    if (prjobs && SF == 0 && FF && *FF) {
		if (DbgLocal > 3)
		    log (XLOG_DEBUG, "putting out FF between jobs");
		jstatus = Print_string (FF);
		if (jstatus != JSUCC) {
		    log (XLOG_INFO, "Printjob: FF printing failed");
		    goto error;
		}
	    }
	    /*
	     * print the file
	     */
	    prjobs = 1;
	    jstatus = printfile (opt, arg);
	    if (jstatus != JSUCC) {
		goto error;
	    }
	}
    }
    /*
     * End Printing Program
     */
    if (EP && *EP) {
	jstatus = Banner_print (EP);
	if (jstatus != JSUCC) {
	    log (XLOG_INFO, "Printjob: EP program %s failed", EP);
	    goto error;
	}
	prjobs = 1;
    }
    /*
     * error and status reporting
     */
error:
    /* don't log a serious (but vague) msg, it just clutters
     * the logs and confuses the users when they get mail.
     * A _useful_ message has already been logged.
     */
    if (DbgLocal > 3)
	log (XLOG_DEBUG, "Printjob: '%s' finished with status %d", q->q_name, jstatus);

    return (jstatus);
}

/***********************************************************************
 *   1. determine the format and corresponding filter command
 *   2. if 'p' format, set up pr process to format output
 *   3. output to Printer
 * Returns: JFAIL if retry, JSUCC if success, JABORT if ugly
 ***********************************************************************/

static int
printfile (int format, char *file) {
    int dfd;			/* data file fd */
    int status;			/* return status */
    char *filter = NULL;	/* program Name */
    int nofilter = 0;		/* no filter, just copy */
    char buf[BUFSIZ];		/* hold the Printer command */
    int p[2];			/* pipe file descriptors */
    int prchild;		/* Printer process */

    if (DbgLocal > 3)
	log (XLOG_DEBUG, "printfile: format %c, file %s", format, file);

    /*
     * Open input file
     */
    if ((dfd = open (file, O_RDONLY, 0)) < 0) {
	logerr (XLOG_NOTICE, "printfile: open failed '%s'", file);
	return (JABORT);
    }
    /*
     * find filter chain to be generated
     */
    switch (format) {
	/*
	 * use the filter Name tagged by 'Xf', where X is format
	 */
    default:
	filter = Setup_filter (format, Filter_name[format - 'a']);
	break;
	/*
	 * 'f' and 'l' formats use the IF filter, and OF if not present
	 */
    case 'f':			/* default */
    case 'l':			/* ignore control */
    case 'p':			/* print file using 'pr' */
	if (IF == 0 || *IF == 0) {
	    /*
	     * you have to use the OF filter; we don't set up a filter
	     */
	    nofilter = 1;
	} else if (format == 'l') {
	    filter = Setup_filter ('l', IF);
	} else {
	    filter = Setup_filter ('f', IF);
	}
	if (format != 'p') {
	    break;
	}
	/*
	 * create a process to invoke 'pr' first, set up the pr command
	 */
	(void) sprintf (buf, "%s -w%d -l%d %s %s",
			PR, PW, PL, PRTITLE[0] ? "-h" : "", PRTITLE);
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "printfile: pr command '%s'", buf);
	/*
	 * create PR process
	 */
	if (pipe (p) < 0) {
	    logerr_die (XLOG_NOTICE, "printfile: cannot make pipe");
	}
	/*
	 * make a pipe, fork PR process Run this process as daemon
	 */
	if ((prchild = fork ()) == 0) {	/* child */
	    if (dup2 (dfd, 0) < 0	/* file is stdin */
		|| dup2 (p[1], 1) < 0) {	/* pipe stdout */
		logerr_die (XLOG_NOTICE, "printfile: dup2 failed, PR process");
	    }
	    setup_close_on_exec ();	/* this'll close the unused pipe fds */
	    full_daemon_perms ();
	    secure_exec (buf);	/* we are "daemon" */

	} else if (prchild < 0) {
	    logerr_die (XLOG_NOTICE, "printfile: fork for pr failed");
	}
	(void) close (p[1]);	/* close output side */
	(void) close (dfd);
	dfd = p[0];		/* use pipe for input */
	break;
    }

    if (LA && AF && *AF) {      /* remote logging */
        time_t t;
        FILE *f;

        t = time ((time_t *) NULL);
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "writing acct entry");

        if ((f = fopen (AF, "a")) != NULL) {
            fprintf (f, "%s\t%s\t%s\t%7d\t%c\t%s",
                     (LOGNAME && *LOGNAME) ? LOGNAME : "NULL",
                     (FROMHOST && *FROMHOST) ? FROMHOST : "NULL",
                     (Printer && *Printer) ? Printer : "NULL",
                     1 /* npages */ , format, ctime (&t));
            (void) fclose (f);

        } else {
            logerr (XLOG_INFO,
                    "Can't write acct entry for printer %s", Printer);
        }
    }
    /*
     * start filter
     */
    if (nofilter) {
	status = Print_copy (dfd);

    } else {
	if (DbgLocal > 0)
	    log (XLOG_DEBUG,
		 "printfile: format %c, file %s, filter %s", format, file, filter);
	status = Print_filter (dfd, filter, 0);
    }
    (void) close (dfd);
    return (status);
}

/********************************************************************
 * Print the banner using the user specified program
 * 1. setup the filter
 * 2. start program
 * 3. wait for completion.
 ********************************************************************/
static int
Banner_print (char *prog) {
    char *filter;
    int status;

    filter = Setup_filter ('f', prog);

    if (DbgLocal > 3)
	log (XLOG_DEBUG, "Banner_print: filter %s", filter);
    status = Print_filter (0, filter, 0);
    return (status);
}

/********************************************************************
 * use the user specified Queue Handler.
 * exec'd
 *  progname arguments \   <- from qh options
 *      -Ccontrol_file \
 *      -PPrinter -wwidth -llength -xwidth -ylength [-c] [-iindent] \
 *              [-Zoptions] [-Cclass] [-Jjob] [-Raccntname] -nlogin -hHost
 *      -FX [affile]
 *
 *  exit codes are used to indicate the different success:
 *   0 - successful; 1 retry; anything else- abandon
 * Returns: JSUCC, etc.
 * NOTE: the queue handler is invoked with the control file as
 *  stdin and the Printer as stdout.
 ********************************************************************/

static int
queuejob (FILE *cfp, struct plp_queue *q) {
    plp_status_t status;
    int ret;
    pid_t id, pid;
    char buf[BUFSIZ];
    char *filter;

    /*
     * set up arguments
     */
    (void) sprintf (buf, "%s -X%s", QH, q->q_name);
    if (DbgLocal > 0)
	log (XLOG_DEBUG, "queuejob: %s", buf);
    filter = Setup_filter ('X', buf);
    /*
     * Open the Printer
     */
    Print_open ();
    (void) plp_signal (SIGCHLD, (plp_sigfunc_t)SIG_IGN);   /* don't reap this one */

    if ((pid = fork ()) == 0) {	/* child */
	if (dup2 (fileno (cfp), 0) < 0) {	/* stdin is cf */
	    logerr_die (XLOG_NOTICE, "queuejob: dup2 failed");
	}
	if (dup2 (Print_fd, 1) < 0) {		/* stdout is printer */
	    logerr_die (XLOG_NOTICE, "queuejob: dup2 failed");
	}
	setup_close_on_exec (); /* this'll close the unused pipe fds */
	full_daemon_perms ();
	secure_exec (filter);	/* we are "daemon" */

    } else if (pid < 0) {
	logerr_die (XLOG_INFO, "queuejob: fork failed");
    }
    /*
     * wait for process
     */
    if (DbgLocal > 0)
	log (XLOG_DEBUG, "waiting for queuejob %d", pid);

    id = plp_waitpid (pid, &status, WUNTRACED);
    if (DbgLocal > 0)
	log (XLOG_DEBUG, "queuejob: pid %d status %s", id,
	     Decode_status (&status));
    if (id < 0) {
	logerr (XLOG_INFO, "waitpid");
    }

    if (PLP_WIFSTOPPED (status) || (unsigned) PLP_WEXITSTATUS (status) > 1)
    {
	log (XLOG_INFO, "queuejob process %d failed, status (%s)",
		id, Decode_status (&status));

	ret = JABORT;
    } else if (PLP_WEXITSTATUS (status) == 1) {
	ret = JFAIL;
    } else {
	ret = JSUCC;
    }
    return (ret);
}
