/*
 * clk.c -- timer routines for lightweight processes.
 * Copyright (C) 1991-3 Stephen Crane.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * author: Stephen Crane, (jsc@doc.ic.ac.uk), Department of Computing,
 * Imperial College of Science, Technology and Medicine, 180 Queen's
 * Gate, London SW7 2BZ, England.
 */

#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include "lwp.h"

/* delta queue for the timer */
static struct lpq delayq;

/*
 * tvsub calculate to=a-b; for timevals.  assumes a>b
 */
static void tvsub (struct timeval *to, struct timeval *a, struct timeval *b)
{
	if (a->tv_usec < b->tv_usec) {
		a->tv_usec += 1000000;
		a->tv_sec--;
	}
	to->tv_usec = a->tv_usec - b->tv_usec;
	to->tv_sec = a->tv_sec - b->tv_sec;
}

static struct timeval tstart;

/*
 * delay -- block the invoker for >t microseconds.  If t==0, reschedule.
 */
void delayp (long ut)
{
	extern struct pcb *currp;
	struct pcb *p=delayq.head, *q=0;
	struct itimerval itime;
	struct timeval t, *left = &itime.it_value;

	if (!ut) {
		yieldp ();
		return;
	}
	t.tv_sec = ut/1000000;
	t.tv_usec = ut%1000000;
	if (p) {
		getitimer (ITIMER_REAL, &itime);
		if (timercmp(left, &t, >)) {
			/* new delay less than old */
			delayq.head = currp;
			tvsub (&p->dt, left, &t);
		} else {
			/* insert delay in body of queue */
			p->dt = *left;
			do {
				tvsub (&t, &t, &p->dt);
				q=p; p=p->next;
			} while (p && !timercmp(&t, &p->dt, <));
			q->next = currp;
			if (p)	/* adjust ensuing delta */
				tvsub (&p->dt, &p->dt, &t);
		}
	} else	/* the queue is empty */
		toq (&delayq, currp);

	/* time to wait has changed */
	currp->next = p;
	currp->dt = t;
	*left = delayq.head->dt;
	timerclear(&itime.it_interval);	/* one-shot */
	if (!timerisset(left)) left->tv_usec++;
	setitimer (ITIMER_REAL, &itime, (struct itimerval *)0);
	gettimeofday (&tstart, (struct timezone *)0);
	reschedp ();
}

/*
 * onalarm -- process alarm signals
 * this routine contains an adjustment for any current timer over-run
 */
void onalarm ()
{
	struct pcb *p;
	struct itimerval itime;
	struct timeval tnow, dt;

	gettimeofday (&tnow, (struct timezone *)0);
	tvsub (&dt, &tnow, &tstart);
	tstart = tnow;
	while (p = delayq.head)
		if (timercmp(&dt, &p->dt, >)) {	
			/* reduce the delta and ready the process */
			tvsub (&dt, &dt, &p->dt);
			p->dt.tv_sec = p->dt.tv_usec = 0;
			readyp (hoq (&delayq));
		} else {
			/* reduce time-to-wait */
			tvsub (&p->dt, &p->dt, &dt);
			break;
		}
	if (p) {
		/* start a new one-shot timer */
		timerclear(&itime.it_interval);
		itime.it_value = p->dt;
		setitimer (ITIMER_REAL, &itime, (struct itimerval *)0);
	}
}
