// -*- C++ -*-
// Management for the filesystem's list of inodes
//
// This program is distributed under the terms of the
// Free Software Foundation General Public Licence.
// Copyright Jeremy Fitzhardinge <jeremy@sw.oz.au> 1993

#ifndef __INODE_H_SEEN__
#define __INODE_H_SEEN__

#include <linux/userfs_fs.h>
#include "userfs_types.h"
#include <assert.h>

#pragma interface

typedef unsigned long Handle;
class Filesystem;

class InodeList;
class DirInode;		// XXX

class Inode
{
	friend InodeList;
	friend DirInode;	// XXX
	
private:
	Inode		*next, *prev;
	short		alive, onlist;
	
protected:
	// Common inode parts
	uid_t	uid;
	gid_t	gid;
	umode_t	mode;
	nlink_t	nlink;
	off_t	size;
	time_t	atime;
	time_t	mtime;
	time_t	ctime;
	dev_t	rdev;
	
	// Userfs protocol handle
	Handle	handle;

	// Filesystem we belong to
	Filesystem	&filesys;

	Inode	*findino(Handle h);

	// Misc hook functions for derived classes
//	virtual void iread(up_inode &);
	virtual void update() {}
	// Called at umount time before anything is deleted.
	virtual void cleanup() {}
public:
	Inode(Filesystem &, Handle);
	virtual ~Inode();

	Handle gethandle()		{ return handle; }
	Filesystem &getfilesys()	{ return filesys; }

	virtual void incref()	{ nlink++; }
	virtual void decref()	{ nlink--; }

	virtual void beforeop(enum up_ops)	{ }
	virtual void afterop(enum up_ops)	{ }
	
	virtual int link(const char *, int, Inode *)	{ return ENOSYS; }
	virtual int unlink(const char *, int)		{ return ENOSYS; }
	
#define DO(op)	 virtual int do_##op(const up_preamble &, upp_repl &)
#define DOa(op)	 virtual int do_##op(const up_preamble &, upp_repl &, \
				     const upp_##op##_s &)
#define DOr(op)	 virtual int do_##op(const up_preamble &, upp_repl &, \
				     upp_##op##_r &)
#define DOar(op) virtual int do_##op(const up_preamble &, upp_repl &, \
				     const upp_##op##_s &, upp_##op##_r &)
#define opINODE
#define opFILE
#include "operations.h"
};

#include <sys/types.h>

// Manage a list of Inodes
class InodeList
{
private:
	Inode		*list;

public:
	InodeList();
	~InodeList();
	Inode *find(Handle);
	void add(Inode *);
	void del(Inode *);
};

#endif // __INODE_H_SEEN__
