# include	<stdio.h>
# include	<ctype.h>
# include	<unistd.h>
# include	<fcntl.h>
# include	<string.h>
# include	<malloc.h>
# include	<errno.h>
# include	<time.h>
# include	<sys/stat.h>
# include	"oskfs.h"

Convert	convert = Binary;

lsn0		root;
int		dfd = -1;	/* the file descriptor for the mounted
				   disk or file				*/
int		secsize = 256;	/* size of a sector			*/
unchar		*sector = NULL;	/* a global sector buffer		*/
int		fsegsize = 48;	/* # of entries in file segment list	*/
openlog		*olog = NULL;	/* logging all open/close calls		*/
ulong		osiz = 0;	/* allocated space for olog		*/

static Status
do_init (int ac, char **av)
{
	char	*ptr;
	Bool	usecache;

	if (ac < 1) {
		fprintf (stderr, "Need file/device to mount\n");
		return (Fail);
	}
	usecache = False;
	if (ac > 1)
		for (ptr = av[1]; *ptr; ++ptr)
			switch (tolower (*ptr)) {
			case 'b':
				convert = Binary;
				break;
			case 't':
			case 'a':
				convert = Text;
				break;
			case 'g':
				convert = Guess;
				break;
			case '0':
				secsize = 256;
				break;
			case '1':
				secsize = 512;
				break;
			case '2':
				secsize = 1024;
				break;
			case '3':
				secsize = 2048;
				break;
			case '4':
				secsize = 4096;
				break;
			case '5':
				secsize = 8192;
				break;
			case '6':
				secsize = 16384;
				break;
			case '7':
				secsize = 32768;
				break;
			case 'c':
				usecache = True;
				break;
			}
	if (init_cache (usecache ? 64 : 0) == Fail) {
		fprintf (stderr, "Unable to initialize cache\n");
		return (Fail);
	}
	if (! (sector = malloc (secsize + 16))) {
		fprintf (stderr, "Can't alloc sector\n");
		return (Fail);
	}
	fsegsize = (secsize - 16) / 5;
	if ((dfd = open (av[0], O_RDONLY)) < 0) {
		fprintf (stderr, "Can't open %s\n", av[0]);
		return (Fail);
	}
	if (read (dfd, sector, 256) != 256) {
		fprintf (stderr, "Can't read LSN 0\n");
		close (dfd);
		return (Fail);
	}
	root = *osk2lsn0 (sector);
	if (! (root.almap = (unchar *) malloc ((root.amap + 2) * sizeof (unchar)))) {
		fprintf (stderr, "Can't alloc for allocation map\n");
		close (dfd);
		return (Fail);
	}
	if ((lseek (dfd, secsize, SEEK_SET) < 0) || (read (dfd, root.almap, root.amap) != root.amap)) {
		fprintf (stderr, "Can't read allocation map\n");
		close (dfd);
		return (Fail);
	}
	set_fsdata (secsize, 28);
	return (Ok);
}

static ulong
do_lookup (ulong dir, char *fname)
{
	TBuffer		*tb;
	fdes		*fd;
	directory	*dr;
	int		off;
	int		siz;
	int		cnt;
	unchar		*ptr;

	if (! (fd = read_fdes (dir))) {
		errno = ENOTDIR;
		return (0);
	}
	off = 0;
	siz = 32 * 128;
	while (tb = read_part (fd, off, siz)) {
		cnt = tb -> size;
		ptr = tb -> ptr;
		while (cnt >= 32) {
			dr = osk2directory (ptr);
			if (dr -> name[0] && (! strcasecmp (dr -> name, fname)))
				return (dr -> lsn);
			cnt -= 32;
			ptr += 32;
		}
	}
	errno = ENOTDIR;
	return (0);
}

static Status
do_close (ulong handle, ulong ctok)
{
	--ctok;
	if ((ctok >= osiz) || (olog[ctok].handle != handle) || (! olog[ctok].ocnt)) {
		errno = EBADF;
		return (Fail);
	}
	olog[ctok].ocnt--;
	if (! olog[ctok].ocnt) {
		olog[ctok].handle = 0;
		olog[ctok].cvt = convert;
	}
	return (Ok);
}

static TBuffer *
do_read (ulong handle, off_t off, off_t size, ulong ctok)
{
	fdes		*fd;
	TBuffer		*tb;
	TBuffer		ltb;
	register unchar	*ptr;
	register int	n;
	openlog		*ol;
	int		doconv;

	if (! (fd = read_fdes (handle))) {
		errno = ENOENT;
		return (NULL);
	}
	if (off == fd -> size) {
		ltb.ptr = NULL;
		ltb.size = 0;
		return (& ltb);
	}
	if (! (tb = read_part (fd, off, size)))
		if (! errno)
			errno = ENOENT;
	if (tb) {
		--ctok;
		if (ctok < osiz)
			ol = & olog[ctok];
		else
			ol = NULL;
		if (ol && (ol -> cvt == Guess)) {
			if (fd -> mode & (S_IXOTH | S_IXGRP | S_IXUSR))
				ol -> cvt = Binary;
			else if ((tb -> size > 1) && (tb -> ptr[0] == 0x4a) && (tb -> ptr[1] == 0xfc))
				ol -> cvt = Binary;
			else
				ol -> cvt = Text;
		}
		doconv = ((ol ? ol -> cvt : Binary) == Text);
		if (doconv) {
			ptr = tb -> ptr;
			n = tb -> size;
			while (n-- > 0)
				if (*ptr == '\015')
					*ptr++ = '\012';
				else if (*ptr == '\012')
					*ptr++ = '\015';
				else
					++ptr;
		}
	}
	return (tb);
}

static Directory *
do_readdir (ulong dir, off_t off, ulong ctok)
{
	static Directory	d;
	TBuffer			*tb;
	fdes			*fd;
	directory		*dr;
	int			siz;
	int			moff;
	int			cnt;
	unchar			*ptr;

	if (! (fd = read_fdes (dir))) {
		errno = ENOTDIR;
		return (0);
	}
	moff = 0;
	siz = 32 * 128;
	while (tb = read_part (fd, moff, siz)) {
		cnt = tb -> size;
		ptr = tb -> ptr;
		while (cnt >= 32) {
			dr = osk2directory (ptr);
			if (dr -> name[0] && (off-- <= 0)) {
				d.handle = dr -> lsn;
				d.off = 1;
				d.fname = dr -> name;
				return (& d);
			}
			cnt -= 32;
			ptr += 32;
		}
		moff += siz;
	}
	d.handle = 0;
	d.off = 0;
	d.fname = NULL;
	return (& d);
}

static ulong
do_mount (void)
{
	return (root.rootdir);
}

static Inode *
do_iread (ulong handle)
{
	static Inode	ino;
	fdes		*fd;

	if (! (fd = read_fdes (handle))) {
		errno = ENOENT;
		return (NULL);
	}
	ino.mode = fd -> mode;
	ino.nlink = fd -> nlink;
	ino.uid = fd -> uid;
	ino.gid = fd -> gid;
	ino.size = fd -> size;
	ino.atime =
	ino.mtime = fd -> mtime;
	ino.ctime = fd -> ctime;
	ino.rdev = 0;
	ino.blksize = secsize;
	ino.blocks = (ino.size + secsize - 1) / secsize;
	return (& ino);
}

static FS_Status *
do_statfs (void)
{
	FS_Status	fss;
	register int	n, fre;

	fre = 0;
	if ((root.sects >> 3) > root.amap) {
		fprintf (stderr, "Warning: Allocation map smaller than # of available sectors!\n");
		return (NULL);
	}
	for (n = 0; n < root.sects; n += root.csize)
		if (! (root.almap[n >> 3] & (1 << ((7 - (n & 0x7))))))
			fre += root.csize;
	fss.bsize = secsize;
	fss.blocks = root.sects;
	fss.bfree = fre;
	fss.bavail = fre;
	fss.files = 0;
	fss.ffree = 0;
	fss.fsid.val[0] = fss.fsid.val[1] = 0;
	fss.namelen = 28;
	return (& fss);
}

static ulong
do_open (ulong handle, Permission *p)
{
	ulong	ctok;
	ulong	run;

	run = 0;
	for (ctok = 0; ctok < osiz; ++ctok)
		if (! olog[ctok].ocnt) {
			if (! run)
				run = ctok + 1;
		} else if (olog[ctok].handle == handle) {
			olog[ctok].ocnt++;
			return (ctok + 1);
		}
	if (run)
		ctok = run - 1;
	if (ctok == osiz) {
		osiz += 32;
		if (! (olog = (openlog *) realloc (olog, (osiz + 2) * sizeof (openlog)))) {
			osiz = 0;
			return (0);
		}
		for (run = ctok; run < osiz; ++run)
			olog[run].ocnt = 0;
	}
	olog[ctok].ocnt = 1;
	olog[ctok].cvt = convert;
	return (ctok + 1);
}

static Directory *
do_multireaddir (ulong dir, off_t off, ulong ctok, ulong *count)
{
	static Directory	d[66];
	TBuffer			*tb;
	fdes			*fd;
	directory		*dr;
	int			siz;
	int			moff;
	int			cnt;
	int			n;
	unchar			*ptr;

	if (! d[0].fname) {
		for (n = 0; n < 65; ++n)
			if (! (d[n].fname = malloc (32))) {
				d[0].fname = NULL;
				errno = ENOMEM;
				return (NULL);
			}
	}
	*count = 0;
	if (! (fd = read_fdes (dir))) {
		errno = ENOTDIR;
		return (NULL);
	}
	errno = 0;
	moff = 0;
	siz = 32 * 128;
	while (tb = read_part (fd, moff, siz)) {
		cnt = tb -> size;
		ptr = tb -> ptr;
		while (cnt >= 32) {
			dr = osk2directory (ptr);
			if (dr -> name[0]) {
				if (off > 0)
					--off;
				else {
					d[*count].handle = dr -> lsn;
					d[*count].off = 1;
					strcpy (d[*count].fname, dr -> name);
					(*count)++;
					if (*count >= 64)
						break;
				}
			}
			cnt -= 32;
			ptr += 32;
		}
		moff += siz;
	}
	d[*count].handle = 0;
	d[*count].off = 0;
	strcpy (d[*count].fname, "");
	return (d);
}

static fscalls	calls = {
	def_errorlog,		/* errorlog */
	do_init,		/* init */
	NULL,			/* deinit */
	NULL,			/* create */
	do_lookup,		/* lookup */
	do_close,		/* close */
	do_read,		/* read */
	NULL,			/* write */
	NULL,			/* truncate */
	NULL,			/* fsync */
	do_readdir,		/* readdir */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* readlink */
	NULL,			/* followlink */
	do_mount,		/* mount */
	NULL,			/* umount */
	do_iread,		/* iread */
	NULL,			/* iwrite */
	do_statfs,		/* statfs */
	NULL,			/* iput */
	do_open,		/* open */
	NULL,			/* permission */
	NULL,			/* rename */
	do_multireaddir,	/* multireadddir */
	NULL			/* notify_change */
};

void
inform_user (UserInfo ui, void *ptr)
{
}

int
main (int argc, char **argv)
{
# if	0
	char	*av[3] = { "osk.disk", "g0c", NULL };

	do_init (2, av);
	do_lookup (3, "startup");
	do_read (266, 37404 - 1000, 1024, 0);
	exit (0);
# endif
	return (userfs (argc, argv, & calls));
}
