# include	<stdio.h>
# include	<string.h>
# include	<malloc.h>
# include	<errno.h>
# include	<time.h>
# include	<sys/stat.h>
# include	"fslib.h"

# define	OFFSET		2
# define	DEF_BLKSIZE	1024
# define	DEF_NAMELEN	256

static finode	*ilst = NULL;
static int	icnt = 0;
static int	isiz = 0;

static ulong	blksize = DEF_BLKSIZE;
static ulong	namelen = DEF_NAMELEN;

void
set_fsdata (ulong blk, ulong nam)
{
	blksize = blk;
	namelen = nam;
}

filenam *
alloc_filenam (char *name, ulong use, Bool isdir, void *priv)
{
	filenam	*f;
	finode	*i;

	if (! (f = (filenam *) malloc (sizeof (filenam))))
		return (NULL);
	if (! (f -> fname = strdup (name))) {
		free (f);
		return (NULL);
	}
	f -> next = NULL;
	if (! use) {
		if (icnt >= isiz) {
			isiz += 256;
			if (! (ilst = (finode *) realloc (ilst, sizeof (finode) * (isiz + 2)))) {
				isiz = 0;
				icnt = 0;
				free (f -> fname);
				free (f);
				return (NULL);
			}
		}
		i = & ilst[icnt];
		i -> nr = icnt + OFFSET;
		i -> linked = 0;
		i -> ino.rdev = 0;
		++icnt;
		i -> priv = priv;
		i -> slink = NULL;
		if (isdir) {
			i -> ino.mode = S_IFDIR | 0555;
			i -> ino.size = namelen * 2;
			i -> dir = f;
		} else {
			i -> ino.mode = S_IFREG | 0444;
			i -> ino.size = 0;
			i -> dir = NULL;
		}
		i -> ino.nlink = 1;
		i -> ino.uid = 0;
		i -> ino.gid = 0;
		i -> ino.mtime = i -> ino.ctime = i -> ino.atime = 0;
		i -> ino.blksize = blksize;
		i -> ino.blocks = 0;
		f -> i = i -> nr;
	} else {
		f -> i = use;
		if (i = get_finode (use)) {
			i -> ino.nlink++;
			if ((name[0] == '.') && (name[1] == '\0'))
				i -> dir = f;
		}
	}
	return (f);
}

void
set_filesize (register ulong ino, register ulong size)
{
	register finode	*i;

	if (i = get_finode (ino)) {
		i -> ino.size = size;
		i -> ino.blocks = (i -> ino.size + (i -> ino.blksize - 1)) / i -> ino.blksize;
	}
}

void
free_filenam (register filenam *f)
{
	if (f -> fname)
		free (f -> fname);
	free (f);
}

finode *
get_finode (register int inr)
{
	if ((inr >= OFFSET) && (inr < icnt + OFFSET))
		return (& ilst[inr - OFFSET]);
	errno = EBADF;
	return (NULL);
}

filenam *
make_root (void)
{
	register filenam	*root;
	register finode		*i;

	if (! (root = alloc_filenam (".", 0, True, NULL)))
		return (NULL);
	if (! (root -> next = alloc_filenam ("..", root -> i, True, NULL))) {
		free_filenam (root);
		return (NULL);
	}
	if (i = get_finode (root -> i))
		i -> ino.mtime = time (NULL);
	return (root);
}

filenam *
make_directory (char *fname, ulong up, void *priv)
{
	filenam	*f, *down;

	if (f = alloc_filenam (fname, 0, True, priv)) {
		if (down = alloc_filenam (".", f -> i, True, NULL)) {
			if (down -> next = alloc_filenam ("..", up, True, NULL))
				return (f);
			free_filenam (down);
		}
		free_filenam (f);
	}
	return (NULL);
}

filenam *
add_fullpath (register filenam *root, register char *path, register Bool isdir)
{
	char		*tmp;
	filenam		*up, *cur, *prev;
	register char	*ptr, *sav;
	finode		*i;

	if (! (tmp = strdup (path)))
		return (NULL);
	sav = tmp;
	while (*sav == '/')
		++sav;
	up = root;
	while (ptr = strchr (sav, '/')) {
		*ptr++ = '\0';
		while (*ptr == '/')
			++ptr;
		prev = NULL;
		for (cur = root; cur; cur = cur -> next)
			if (! strcmp (cur -> fname, sav))
				break;
			else
				prev = cur;
		if (! cur) {
			if (! prev)
				return (NULL);
			if (! (cur = make_directory (sav, root -> i, NULL)))
				return (NULL);
			prev -> next = cur;
		}
		if (! cur)
			return (NULL);
		if ((i = get_finode (cur -> i)) && i -> dir)
			cur = i -> dir;
		else
			cur = NULL;
		if (! cur)
			return (NULL);
		up = root;
		root = cur;
		sav = ptr;
	}
	if (! root)
		return (NULL);
	for (prev = root; prev -> next; prev = prev -> next)
		;
	if (isdir)
		prev -> next = make_directory (sav, root -> i, NULL);
	else
		prev -> next = alloc_filenam (sav, 0, False, NULL);
	if ((cur = prev -> next) && (i = get_finode (root -> i)))
		i -> ino.size += namelen;
	free (tmp);
	return (cur);
}

static void
do_dump (char *pre, filenam *root)
{
	char	*npre;
	finode	*i;

	if (! (npre = malloc (strlen (pre) + 4)))
		return;
	sprintf (npre, "  %s", pre);
	while (root) {
		printf ("%s%6lu: %s\n", pre, root -> i, root -> fname);
		if (strcmp (root -> fname, ".") && strcmp (root -> fname, ".."))
			if (i = get_finode (root -> i))
				if (S_ISDIR (i -> ino.mode) && i -> dir)
					do_dump (npre, i -> dir);
		root = root -> next;
	}
	free (npre);
}

void
dump_filesystem (filenam *root)
{
	do_dump ("", root);
}

void
def_errorlog (register char *str)
{
	fprintf (stderr, ": %s\n", str);
}

ulong
def_create (register ulong dir, register long mode, register long rdev,
	register Permission *p, register char *fname)
{
	register finode		*i;
	register filenam	*f, *run, *down;
	register databuf	*db;

	if ((! (i = get_finode (dir))) || (! (run = i -> dir)))
		return (0);
	if (! (f = alloc_filenam (fname, 0, S_ISDIR (mode) ? True : False, NULL))) {
		errno = ENOMEM;
		return (0);
	}
	if (S_ISDIR (mode)) {
		if (! (down = alloc_filenam (".", f -> i, True, NULL))) {
			free_filenam (f);
			errno = ENOMEM;
			return (0);
		}
		if (! (down -> next = alloc_filenam ("..", run -> i, True, NULL))) {
			free_filenam (down);
			free_filenam (f);
			errno = ENOMEM;
			return (0);
		}
	}
	i -> ino.size += namelen;
	while (run -> next)
		run = run -> next;
	run -> next = f;
	if (i = get_finode (f -> i)) {
		i -> ino.mode = (umode_t) mode;
		i -> ino.rdev = (dev_t) rdev;
		i -> ino.ctime =
		i -> ino.mtime =
		i -> ino.atime = time (NULL);
		if (db = (databuf *) malloc (sizeof (databuf))) {
			db -> magic = DATABUF_MAGIC;
			db -> ptr = NULL;
			db -> cnt = 0;
			db -> siz = 0;
			i -> priv = db;
		}
	}
	inform_user (UI_Create, f);
	return (f -> i);
}

ulong
def_lookup (register ulong dir, register char *fname)
{
	register finode		*i;
	register filenam	*f;

	if (! (i = get_finode (dir)))
		return (0);
	if (! i -> dir) {
		errno = ENOTDIR;
		return (0);
	}
	f = i -> dir;
	while (f) {
		if (! strcmp (fname, f -> fname)) {
			inform_user (UI_Lookup, f);
			return (f -> i);
		}
		f = f -> next;
	}
	errno = ENOENT;
	return (0);
}

Status
def_close (register ulong handle, register ulong ctok)
{
	register finode	*i;

	if ((i = get_finode (handle)) && i -> linked) {
		i -> linked--;
		inform_user (UI_Close, i);
		return (Ok);
	}
	return (Fail);
}

TBuffer *
def_read (register ulong handle, register off_t off, register off_t size,
	register ulong ctok)
{
	static TBuffer		tb;
	register finode		*i;
	register databuf	*db;
	ui_io			uio;

	if ((! (i = get_finode (handle))) || (! (db = i -> priv)) || (db -> magic != DATABUF_MAGIC))
		return (NULL);
	if (db -> cnt < off)
		return (NULL);
	if (off + size > db -> cnt)
		size = db -> cnt - off;
	uio.off = off;
	uio.size = size;
	uio.i = i;
	inform_user (UI_Read, & uio);
	tb.size = (ulong) size;
	tb.ptr = db -> ptr + off;
	return (& tb);
}

off_t
def_write (register ulong handle, register off_t off, register TBuffer *tb,
	register ulong ctok)
{
	register finode		*i;
	register databuf	*db;
	ui_io			uio;

	if ((! (i = get_finode (handle))) || (! (db = i -> priv)) || (db -> magic != DATABUF_MAGIC))
		return (0);
	if (off + tb -> size >= db -> siz) {
		db -> siz = off + tb -> size + 8192;
		if (! (db -> ptr = (unchar *) realloc (db -> ptr, (db -> siz + 16) * sizeof (unchar)))) {
			db -> siz = 0;
			db -> cnt = 0;
			errno = ENOMEM;
			return (0);
		}
	}
	uio.off = off;
	uio.size = tb -> size;
	uio.i = i;
	inform_user (UI_Write, & uio);
	memcpy (db -> ptr + off, tb -> ptr, tb -> size);
	if (off + tb -> size > db -> cnt) {
		db -> cnt = off + tb -> size;
		set_filesize (i -> nr, db -> cnt);
	}
	return (tb -> size);
}

Status
def_truncate (register ulong handle, register off_t size)
{
	register finode		*i;
	register databuf	*db;
	ui_io			uio;

	if ((! (i = get_finode (handle))) || (! (db = i -> priv)) || (db -> magic != DATABUF_MAGIC))
		return (Fail);
	if (size >= db -> siz) {
		db -> siz = size + 8192;
		if (! (db -> ptr = (unchar *) realloc (db -> ptr, (db -> siz + 16) * sizeof (unchar)))) {
			db -> siz = 0;
			db -> cnt = 0;
			return (Fail);
		}
	}
	uio.off = 0;
	uio.size = size;
	uio.i = i;
	inform_user (UI_Truncate, & uio);
	if (size != db -> cnt) {
		db -> cnt = size;
		set_filesize (i -> nr, db -> cnt);
	}
	return (Ok);
}

Status
def_fsync (void)
{
	inform_user (UI_Fsync, NULL);
	return (Ok);
}

Directory *
def_readdir (register ulong dir, register off_t off, register ulong ctok)
{
	static Directory	d;
	register finode		*i;
	register filenam	*f;

	if (! (i = get_finode (dir)))
		return (NULL);
	if (! i -> dir) {
		errno = ENOTDIR;
		return (NULL);
	}
	f = i -> dir;
	while (off-- > 0)
		if (f)
			f = f -> next;
		else
			break;
	if (f) {
		inform_user (UI_Readdir, f);
		d.handle = f -> i;
		d.off = 1;
		d.fname = f -> fname;
	} else {
		d.handle = 0;
		d.off = 0;
		d.fname = NULL;
	}
	return (& d);
}

Status
def_link (register ulong ohandle, register ulong dir, register char *fname)
{
	register filenam	*f;
	register finode		*i, *oi;

	if ((i = get_finode (dir)) && (f = i -> dir) && (oi = get_finode (ohandle))) {
		for (; f -> next; f = f -> next)
			;
		if (f -> next = alloc_filenam (fname, oi -> nr, True, NULL)) {
			i -> ino.size += namelen;
			inform_user (UI_Link, f -> next);
			return (Ok);
		} else
			errno = ENOMEM;
	}
	return (Fail);
}

Status
def_unlink (register ulong dir, register char *fname)
{
	register finode		*i, *tmp;
	register filenam	*f, *ftmp, *prev;

	if (i = get_finode (dir)) {
		prev = NULL;
		for (f = i -> dir; f; f = f -> next)
			if (! strcmp (f -> fname, fname))
				break;
			else
				prev = f;
		if (f) {
			inform_user (UI_Unlink, f);
			if (tmp = get_finode (f -> i)) {
				if (S_ISDIR (tmp -> ino.mode) && (ftmp = tmp -> dir) && ftmp -> next && ftmp -> next -> next)
					errno = ENOTEMPTY;
				else {
					i -> ino.size -= namelen;
					if (tmp -> ino.nlink)
						tmp -> ino.nlink--;
					if (prev)
						prev -> next = f -> next;
					else
						i -> dir = i -> dir -> next;
					free_filenam (f);
					if ((! tmp -> ino.nlink) && tmp -> slink) {
						free (tmp -> slink);
						tmp -> slink = NULL;
					}
					if (S_ISDIR (tmp -> ino.mode) && (tmp -> ino.nlink == 1) && (f = tmp -> dir))
						if (f -> next && (! f -> next -> next)) {
							tmp -> ino.nlink--;
							tmp = get_finode (f -> next -> i);
							free_filenam (f -> next);
							if (tmp -> ino.nlink)
								tmp -> ino.nlink--;
							free_filenam (f);
						}
					return (Ok);
				}
			}
		} else
			errno = ENOENT;
	}
	return (Fail);
}

Status
def_symlink (ulong dir, char *name, char *symname, Permission *perm)
{
	finode	*di, *i;
	filenam	*prev;
	filenam	*f;

	if ((! (di = get_finode (dir))) || (! di -> dir))
		return (Fail);
	if (! (f = alloc_filenam (name, 0, False, NULL)))
		return (Fail);
	if (! (i = get_finode (f -> i))) {
		free_filenam (f);
		return (Fail);
	}
	i -> ino.mode |= S_IFLNK | 0777;
	i -> ino.ctime =
	i -> ino.mtime =
	i -> ino.atime = time (NULL);
	i -> ino.size = strlen (symname);
	set_filesize (i -> nr, i -> ino.size);
	i -> slink = strdup (symname);
	di -> ino.size += namelen;
	prev = di -> dir;
	while (prev -> next)
		prev = prev -> next;
	prev -> next = f;
	inform_user (UI_Symlink, f);
	return (Ok);
}

char *
def_readlink (ulong handle)
{
	finode	*i;

	if (i = get_finode (handle)) {
		inform_user (UI_Readlink, i);
		return (i -> slink);
	}
	return (NULL);
}

char *
def_followlink (ulong dir, ulong handle, long flag, long mode)
{
	finode	*i;

	if (i = get_finode (handle)) {
		inform_user (UI_Followlink, i);
		return (i -> slink);
	}
	return (NULL);
}

Inode *
def_iread (register ulong handle)
{
	register finode	*i;

	if (! (i = get_finode (handle)))
		return (NULL);
	inform_user (UI_Iread, i);
	return (& i -> ino);
}

Status
def_iwrite (ulong handle, Inode *ino)
{
	finode	*i;

	if (! (i = get_finode (handle)))
		return (Fail);
	i -> ino = *ino;
	inform_user (UI_Iwrite, i);
	return (Ok);
}

FS_Status *
def_statfs (void)
{
	static FS_Status	fs;
	register int		n;
	register ulong		cnt;

	fs.bsize = blksize;
	cnt = 0;
	for (n = 0; n < icnt; ++n)
		cnt += (ilst[n].ino.size + blksize - 1) / blksize;
	fs.blocks = cnt;
	fs.bfree = 0;
	fs.bavail = 0;
	fs.files = 0;
	fs.ffree = 0;
	fs.fsid.val[0] = fs.fsid.val[1] = 0;
	fs.namelen = namelen - 1;
	inform_user (UI_Statfs, & fs);
	return (& fs);
}

Status
def_iput (ulong handle)
{
	inform_user (UI_Iput, & handle);
	return ((handle >= OFFSET) && (handle < icnt + OFFSET) ? Ok : Fail);
}

ulong
def_open (register ulong handle, register Permission *perm)
{
	register finode	*i;

	if (i = get_finode (handle)) {
		i -> linked++;
		inform_user (UI_Open, i);
		return (1);
	}
	return (0);
}

Status
def_permission (register ulong handle, register long mask, 
	register Permission *perm)
{
	return (Ok);
}

Status
def_rename (register ulong odir, register char *oname, 
	register ulong ndir, register char *nname)
{
	errno = ENOSYS;
	return (Fail);
}

Directory *
def_multireaddir (register ulong dir, register off_t off, register ulong ctok,
	register ulong *count)
{
	static Directory	d[16];
	register finode		*i;
	register filenam	*f;
	register int		n;

	if (! (i = get_finode (dir)))
		return (NULL);
	f = i -> dir;
	while (off-- > 0)
		if (f)
			f = f -> next;
		else
			break;
	for (n = 0; n < 16; ++n)
		if (f) {
			inform_user (UI_Multireaddir, f);
			d[n].handle = f -> i;
			d[n].off = 1;
			d[n].fname = f -> fname;
			f = f -> next;
		} else
			break;
	*count = n;
	errno = 0;
	return (d);
}

Status
def_notify_change (register ulong handle, register Inode *ino,
	register long flags)
{
	register finode	*i;

	if (! (i = get_finode (handle)))
		return (Fail);
	i -> ino = *ino;
	return (Ok);
}
