// -*- C++ -*-

// FTP filesystem

// (C) Copyright 1994 Jeremy Fitzhardinge <jeremy@sw.oz.au>
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#pragma implementation

#include <GetOpt.h>
#include <stdio.h>

#include <ThreadComm.h>
#include "ftpfs.h"
#include "topdir.h"

#include <sys/param.h>
#include <unistd.h>

ftpfs::ftpfs(const char *mpoint, const String &cache)
      : cache(cache), Filesystem(mpoint), uid(0), gid(0)
{
	hcount = 0;
	root = new topdir(*this, cache);
}

int
ftpfs::do_mount(const up_preamble &, upp_repl &,
		upp_mount_r &ret)
{
	ret.root.handle = root->gethandle();
	return 0;
}

ftpfs::~ftpfs()
{
}

int
ftpfs::Enquire(up_ops op)
{
	switch(op)
	{
	case up_mount:
	case up_iread:
	case up_multireaddir:
	case up_lookup:
	case up_notify_change:
	case up_readlink:
	case up_create:
	case up_read:
	case up_iput:
		return 0;
	default:
		break;
	}
	
	return Filesystem::Enquire(op);
}

// Globals for controlling nonblocking IO (nonblk_io.cc)
extern CommBase *selector;
extern int do_nonblock;

// Parse the arguments from muserfs and construct the filesystem
// and communications.
int
main(int argc, char **argv)
{
	GetOpt getopt(argc, argv, "i:o:m:c:");
	int optch, err = 0;
	int infd, outfd;
	const char *mpoint = "";
	
	infd = outfd = -1;
	while((optch = getopt()) != EOF)
		switch(optch)
		{
		case 'i':
			infd = atoi(getopt.optarg);
			break;
		case 'o':
			outfd = atoi(getopt.optarg);
			break;
		case 'm':
			mpoint = getopt.optarg;
			break;
		default:
			err++;
		}

	if (err || infd == -1 || outfd == -1 || getopt.optind == argc)
	{
		cerr << "Usage: " << argv[0] << " -i infd -o outfd -m mpoint cachedir\n";
		exit(1);
	}

	char path[MAXPATHLEN];
	char *full = realpath(argv[getopt.optind], path);
	String cache(full ? full : argv[getopt.optind]);
	
#if 0 && DEBUG
	printf("pid=%d, return to cont\n", getpid());
	getchar();
#endif
	ftpfs ftp(mpoint, cache);
	ThreadComm comm(ftp, outfd, infd, 32*1024, COM_TPRI, OPS_TPRI);
	selector = &comm;
	do_nonblock = 1;
	if (comm.Run() == -1)
		cerr << argv[0] << ": Comm.Run() failed!\n";
	do_nonblock = 0;
	return 0;
}
