// -*- C++ -*-
//
// Example filesystem
//
// This program is distributed under the terms of the
// Free Software Foundation General Public Licence.
// See COPYING for details
// Copyright Jeremy Fitzhardinge <jeremy@sw.oz.au> 1994,1995

#pragma implementation

// Misc headers
#include <GetOpt.h>

// Filesystem related headers
#include <Comm.h>

#include "egfs.h"
#include "egdir.h"

// A simple constructor for the filesystem
// All this does is pass the arguments to the base class
// and initialize the inode handle counter.
// The counter is for generating unique handles as required.
// The mechanism used is very simple, because there is a bounded
// number of inodes.
egfs::egfs(const char *mpoint)
     : Filesystem(mpoint)
{
	hcount=2;
}

egfs::~egfs()
{
}

// Declare to the kernel which operations we will support.  Lots of
// this is entirely done by the base classes in libuserfs (for example,
// up_link is done by DirInode; we don't have to do anything explicitly).
int
egfs::Enquire(up_ops op)
{
	switch(op)
	{
	case up_mount:
	case up_iread:
	case up_notify_change:
	case up_readdir:
	case up_multireaddir:
	case up_lookup:
	case up_create:
	case up_unlink:
	case up_link:
	case up_rename:
	case up_truncate:
	case up_read:
	case up_write:
		return 0;
	default:
		break;
	}
	return Filesystem::Enquire(op);
}

// Do the mount.  This simply returns the handle of the root
// directory and creates the directory.
int
egfs::do_mount(const up_preamble &, upp_repl &,
	       upp_mount_r &ret)
{
	Inode *root = new egtopdir(*this, genhand(), NULL);
	ret.root.handle = root->gethandle();
	return 0;
}

// Parse the arguments from muserfs and construct the filesystem
// and communications.
int
main(int argc, char **argv)
{
	GetOpt getopt(argc, argv, "i:o:m:");
	int optch, err = 0;
	int infd, outfd;
	const char *mpoint = "";

	infd = outfd = -1;
	while((optch = getopt()) != EOF)
		switch(optch)
		{
		case 'i':
			infd = atoi(getopt.optarg);
			break;
		case 'o':
			outfd = atoi(getopt.optarg);
			break;
		case 'm':
			mpoint = getopt.optarg;
			break;
		default:
			err++;
		}

	if (err)
	{
		fprintf(stderr, "Usage: %s -i infd -o outfd -m mpoint\n",
			argv[0]);
		exit(1);
	}

#if 0 && DEBUG
	printf("pid=%d, return to cont\n", getpid());
	getchar();
#endif
	egfs efs(mpoint);
	Comm comm(efs, outfd, infd);

	if (comm.Run() == -1)
		fprintf(stderr, "%s: Comm.Run() failed!\n", argv[0]);
	
	return 0;
}
