/*
 *	ultrastor.c - Low-level SCSI driver for UltraStor 14F/34F
 *
 *      This is a total rewrite of the ultrastor.c module version 1.12
 *      included in linux.1.1.28 and written by the following people:
 *
 *	David B. Gentzel  (gentzel@nova.enet.dec.com)
 *      Scott Taylor (n217cg@tamuts.tamu.edu)
 *      John F. Carr (jfc@athena.mit.edu)
 *      Caleb Epstein (cae@jpmorgan.com)
 *      Eric Youngdale (ericy@cais.com)
 *
 *      16-Jul-1994 version 1.00 for linux 1.1.29
 *      20-Jul-1994 version 1.01 for linux 1.1.31 !!! 34F support untested !!!
 *
 *          This module should provide a better support for the 14F board.
 *          It is a total replacement of ultrastor.c, but it supports ONLY 
 *          the 14F and 34F boards (no 24F support).
 *
 *      Released by Dario Ballabio (Dario_Ballabio@milano.europe.dg.com)
 *
 */

/*
 *  The UltraStor 14F, 24F, and 34F are a family of intelligent, high
 *  performance SCSI-2 host adapters.  They all support command queueing
 *  and scatter/gather I/O.  Some of them can also emulate the standard
 *  WD1003 interface for use with OS's which don't support SCSI.  Here
 *  is the scoop on the various models:
 *  14F - ISA first-party DMA HA with floppy support and WD1003 emulation.
 *  24F - EISA Bus Master HA with floppy support and WD1003 emulation.
 *  34F - VL-Bus Bus Master HA with floppy support (no WD1003 emulation).
 *
 *  This code has been tested with an U14F board rev. 00D, default
 *  jumper setting and different values of AT bus-on time.
 *
 *  The following facts are based on real testing results (not on
 *  documentation) on the above U14F board.
 *  
 *  - The U14F board should be jumpered for bus on time less or equal to 7 
 *    microseconds, while the default is 11 microseconds. This is order to 
 *    get acceptable performance while using floppy drive and hard disk 
 *    together. The jumpering for 7 microseconds is: JP13 pin 15-16, 
 *    JP14 pin 7-8 and pin 9-10. It is useful to test different values of the 
 *    bus on time and see which value gives the best reliability.
 *    On my PC the optimal value is 4 microseconds.
 *    The reduction has a little impact on scsi performance.
 *  
 *  - If scsi bus length exceeds 3m., the scsi bus speed needs to be reduced
 *    from 10Mhz to 5Mhz (do this by inserting a jumper on JP13 pin 7-8).
 *
 *  - The U14F board is unable to provide reliable operations if the scsi 
 *    request length exceeds 16Kbyte. When this length is exceeded the
 *    behavior is: 
 *    - adapter_status equal 0x96 or 0xa3 or 0x93;
 *    - adapter_status equal 0 and target_status equal 2 on for all targets
 *      in the next operation following the reset.
 *    This sequence takes a long time (>3 seconds), so in the mantime
 *    the timeout in sd.c could expire giving rise to scsi aborts.
 *    Because of this I had to DISABLE_CLUSTERING and to work around the
 *    bus reset in the interrupt service routine, returning DID_BUS_BUSY
 *    so that the operations are retried without complains from the scsi.c
 *    code.
 *    To avoid scsi aborts it is strongly suggested to increase the timeout 
 *    in sd.c from 3 to 10 seconds.
 *    Any reset of the scsi bus is going to kill tape operations, since
 *    no retry is allowed for tapes. Bus resest are more likely when the
 *    scsi bus is under heavy load.
 *    Requests using scatter/gather have a maximum length of 16 x 1024 bytes 
 *    when DISABLE_CLUSTERING is in effect, but unscattered requests could be
 *    larger than 16Kbyte.
 *
 */

#include <linux/stddef.h>
#include <linux/string.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/delay.h>

#include <asm/io.h>
#include <asm/bitops.h>
#include <asm/system.h>
#include <asm/dma.h>

#include "../block/blk.h"
#include "scsi.h"
#include "hosts.h"
#include "sd.h"
#include "ultrastor.h"

/* Values for the PRODUCT_ID ports for the 14F */
#define US14F_PRODUCT_ID_0 0x56
#define US14F_PRODUCT_ID_1 0x40		/* NOTE: Only upper nibble is used */

/* Subversion values */
#define U14F 0
#define U34F 1

/* MSCP field values */

/* Opcode */
#define OP_HOST_ADAPTER 0x1
#define OP_SCSI 0x2
#define OP_RESET 0x4

/* Date Transfer Direction */
#define DTD_SCSI 0x0
#define DTD_IN 0x1
#define DTD_OUT 0x2
#define DTD_NONE 0x3

/* Host Adapter command subcodes */
#define HA_CMD_INQUIRY 0x1
#define HA_CMD_SELF_DIAG 0x2
#define HA_CMD_READ_BUFF 0x3
#define HA_CMD_WRITE_BUFF 0x4

#define FALSE 0
#define TRUE 1

/* The U14F has 16 mscp's, but only 7 are used (one for each target) */
#define U14F_MAX_MSCPS 7

#define ARRAY_SIZE(arr) (sizeof (arr) / sizeof (arr)[0])

#define PACKED		__attribute__((packed))
#define ALIGNED(x)	__attribute__((aligned(x)))

#define NO_DEBUG
#ifdef DEBUG
#define DEB(x) x
#else
#define DEB(x)
#endif

/* The 14F uses an array of 4-byte ints for its scatter/gather list.
   The data can be unaligned, but need not be.  It's easier to give
   the list normal alignment since it doesn't need to fit into a
   packed structure.  */

struct ultrastor_sg_list {
   unsigned int address;
   unsigned int num_bytes;
   };

/* MailBox SCSI Command Packet.  Basic command structure for communicating
   with controller. */
struct mscp {
   unsigned char opcode: 3;		/* type of command */
   unsigned char xdir: 2;		/* data transfer direction */
   unsigned char dcn: 1;		/* disable disconnect */
   unsigned char ca: 1;			/* use cache (if available) */
   unsigned char sg: 1;			/* scatter/gather operation */
   unsigned char target_id: 3;		/* target SCSI id */
   unsigned char ch_no: 2;		/* SCSI channel (always 0 for 14f) */
   unsigned char lun: 3;		/* logical unit number */
   unsigned int transfer_data PACKED;	/* transfer data pointer */
   unsigned int transfer_data_length PACKED;	/* length in bytes */
   unsigned int command_link PACKED;	/* for linking command chains */
   unsigned char scsi_command_link_id;	/* identifies command in chain */
   unsigned char number_of_sg_list;	/* (if sg is set) 8 bytes per list */
   unsigned char length_of_sense_byte;
   unsigned char length_of_scsi_cdbs;	/* 6, 10, or 12 */
   unsigned char scsi_cdbs[12];		/* SCSI commands */
   volatile unsigned char adapter_status; /* non-zero indicates HA error */
   volatile unsigned char target_status; /* non-zero indicates target error */
   unsigned int sense_data PACKED;
 
   /* The following fields are for software only.  They are included in
      the MSCP structure because they are associated with SCSI requests.  */
   void (*done)(Scsi_Cmnd *);
   Scsi_Cmnd *SCint;
   };

/* Port addresses (relative to the base address) */
#define U14F_PRODUCT_ID(port) ((port) + 0x4)
#define CONFIG(port) ((port) + 0x6)

/* Port addresses relative to the doorbell base address.  */
#define LCL_DOORBELL_MASK(port) ((port) + 0x0)
#define LCL_DOORBELL_INTR(port) ((port) + 0x1)
#define SYS_DOORBELL_MASK(port) ((port) + 0x2)
#define SYS_DOORBELL_INTR(port) ((port) + 0x3)

/* Used to store configuration info read from config i/o registers.  Most of
   this is not used yet, but might as well save it.
   This structure holds all data that must be duplicated to support multiple
   adapters. */

struct u14f_hostdata {
   unsigned short port_address;		/* base address of card */
   unsigned short doorbell_address;	/* base address of doorbell CSRs */
   unsigned short ogm_address;		/* base address of OGM */
   unsigned short icm_address;		/* base address of ICM */
   const void *bios_segment;
   unsigned char interrupt: 4;
   unsigned char dma_channel: 3;
   unsigned char bios_drive_number: 1;
   unsigned char heads;
   unsigned char sectors;
   unsigned char ha_scsi_id: 3;
   unsigned char subversion: 4;
   unsigned char revision;
 
   /* The slot number is used to distinguish the 24F (slot != 0) from
      the 14F and 34F (slot == 0). */
   unsigned char slot;
 
   /* A pool of MSCP structures for this adapter */
   struct mscp mscp[U14F_MAX_MSCPS];

   volatile int mscp_free[U14F_MAX_MSCPS];
   volatile int mscp_reset[U14F_MAX_MSCPS];
   };

#define HD(host) ((struct u14f_hostdata *) &host->hostdata)

static struct Scsi_Host * u14f = NULL;
static char* utype = "14F";
static char board_info[80];

/* Set this to 1 to reset the SCSI bus on error.  */
int ultrastor_bus_reset = 1;

/* Allowed BIOS base addresses (NULL indicates reserved) */
static const void *const bios_segment_table[8] = { 
   NULL, 
   (void *)0xC4000, (void *)0xC8000, (void *)0xCC000,
   (void *)0xD0000, (void *)0xD4000, (void *)0xD8000, 
   (void *)0xDC000
   };

/* Allowed IRQs for 14f */
static const unsigned char interrupt_table_14f[4] = { 15, 14, 11, 10 };

/* Allowed DMA channels for 14f (0 indicates reserved) */
static const unsigned char dma_channel_table_14f[4] = { 5, 6, 7, 0 };

/* Head/sector mappings allowed by 14f */
static const struct {
   unsigned char heads;
   unsigned char sectors;
   } mapping_table[4] = { { 16, 63 }, { 64, 32 }, { 64, 63 }, { 64, 32 } };

/* ??? A probe of address 0x310 screws up NE2000 cards */
static const unsigned short ultrastor_ports_14f[] = {
   0x330, 0x340, /*0x310,*/ 0x230, 0x240, 0x210, 0x130, 0x140,
   };

static void ultrastor_14f_interrupt(int cpl);
static inline void build_sg_list(struct mscp *, Scsi_Cmnd *SCpnt);

#ifdef DEBUG
static int max_sg_segment = 0;
static int max_transfer_length = 0;
#endif

int ultrastor_14f_detect(Scsi_Host_Template * tpnt) {
   size_t i;
   unsigned char in_byte, version_byte = 0;
   struct config_1 {
      unsigned char bios_segment: 3;
      unsigned char removable_disks_as_fixed: 1;
      unsigned char interrupt: 2;
      unsigned char dma_channel: 2;
      } config_1;
   struct config_2 {
      unsigned char ha_scsi_id: 3;
      unsigned char mapping_mode: 2;
      unsigned char bios_drive_number: 1;
      unsigned char tfr_port: 2;
      } config_2;

   for (i = 0; i < ARRAY_SIZE(ultrastor_ports_14f); i++) {

      if(check_region(ultrastor_ports_14f[i], 0x0c)) continue;

      in_byte = inb(U14F_PRODUCT_ID(ultrastor_ports_14f[i]));

      if (in_byte != US14F_PRODUCT_ID_0) continue;

      in_byte = inb(U14F_PRODUCT_ID(ultrastor_ports_14f[i]) + 1);

      /* Only upper nibble is significant for Product ID 1 */
      if ((in_byte & 0xF0) != US14F_PRODUCT_ID_1) continue;

      version_byte = in_byte;
      break;
      }

   if (i == ARRAY_SIZE(ultrastor_ports_14f)) return FALSE;

   u14f = scsi_register(tpnt, sizeof(struct u14f_hostdata));
   memset(HD(u14f), 0, sizeof(struct u14f_hostdata));
   HD(u14f)->port_address = ultrastor_ports_14f[i];

   /* Set local doorbell mask to disallow bus reset unless
   ultrastor_bus_reset is true.  */
   outb(ultrastor_bus_reset ? 0xc2 : 0x82, 
                      LCL_DOORBELL_MASK(HD(u14f)->port_address));

   /* All above tests passed, must be the right thing.  Get some useful
      info. */

   /* Register the I/O space that we use */
   snarf_region(HD(u14f)->port_address, 0x0c);

   *(char *)&config_1 = inb(CONFIG(HD(u14f)->port_address + 0));
   *(char *)&config_2 = inb(CONFIG(HD(u14f)->port_address + 1));
   HD(u14f)->bios_segment = bios_segment_table[config_1.bios_segment];
   HD(u14f)->doorbell_address = HD(u14f)->port_address;
   HD(u14f)->ogm_address = HD(u14f)->port_address + 0x8;
   HD(u14f)->icm_address = HD(u14f)->port_address + 0xC;
   HD(u14f)->interrupt = interrupt_table_14f[config_1.interrupt];
   HD(u14f)->ha_scsi_id = config_2.ha_scsi_id;
   HD(u14f)->heads = mapping_table[config_2.mapping_mode].heads;
   HD(u14f)->sectors = mapping_table[config_2.mapping_mode].sectors;
   HD(u14f)->bios_drive_number = config_2.bios_drive_number;
   HD(u14f)->subversion = (version_byte & 0x0F);

   if (HD(u14f)->subversion == U34F) {
      HD(u14f)->dma_channel = 0;
      tpnt->unchecked_isa_dma = 0;
      utype = "34F";
      }
   else {
      HD(u14f)->dma_channel = dma_channel_table_14f[config_1.dma_channel];
      tpnt->unchecked_isa_dma = 1;
      utype = "14F";
      }

   for (i = 0; i < U14F_MAX_MSCPS; i++) {
      HD(u14f)->mscp_free[i] = TRUE;
      HD(u14f)->mscp_reset[i] = FALSE;
      }

   if (request_irq(HD(u14f)->interrupt, ultrastor_14f_interrupt)) {
      printk("U%s: unable to allocate IRQ%u.\n", utype, HD(u14f)->interrupt);
      scsi_unregister(u14f);
      return FALSE;
      }

   if (HD(u14f)->dma_channel && request_dma(HD(u14f)->dma_channel)) {
      printk("U%s: unable to allocate DMA channel %u.\n", utype,
                HD(u14f)->dma_channel);
      free_irq(HD(u14f)->interrupt);
      scsi_unregister(u14f);
      return FALSE;
      }

   u14f->irq = HD(u14f)->interrupt;
   u14f->dma_channel = HD(u14f)->dma_channel;
   u14f->io_port = HD(u14f)->port_address;

   sprintf(board_info, 
              "UltraStor %s @ PORT 0x%03x, BIOS 0x%05x, IRQ %u, DMA %u.", 
              utype, HD(u14f)->port_address, (int)HD(u14f)->bios_segment,
	      HD(u14f)->interrupt, HD(u14f)->dma_channel);
   printk("%s\n", board_info);

   /* Reset the adapter and SCSI bus.  The SCSI bus reset can be
       inhibited by clearing ultrastor_bus_reset before probe.  */
   outb(0xc0, LCL_DOORBELL_INTR(HD(u14f)->doorbell_address));

   return TRUE;
}

const char *ultrastor_14f_info(void) {

   return board_info;
}

static inline void build_sg_list(struct mscp *mscp, Scsi_Cmnd *SCpnt) {
   long transfer_length = 0;
   struct scatterlist * sgpnt;
   struct ultrastor_sg_list * sglist;
   int i;

   SCpnt->host_scribble = (unsigned char *) scsi_malloc(512);
   sgpnt = (struct scatterlist *) SCpnt->request_buffer;

   if(((unsigned  int) sgpnt) & 0xff000000) 
         panic("U14F queuecommand, buffer address out of DMA range");

   sglist = (struct ultrastor_sg_list *) SCpnt->host_scribble;

   if (sglist == NULL) 
      panic("U14F queuecommand, unable to allocate DMA memory");

   for(i = 0; i < SCpnt->use_sg; i++) {
      if(sgpnt[i].length == 0 || SCpnt->use_sg > ULTRASTOR_14F_MAX_SG || 
            (((int)sgpnt[i].address) & 1) || (sgpnt[i].length & 1))
        	  panic("U14F queuecommand, bad sg list");

      sglist[i].address = (unsigned int) sgpnt[i].address;

      if (((unsigned  int) sgpnt[i].address) & 0xff000000) 
         panic("U14F queuecommand, sglist address out of DMA range");

      sglist[i].num_bytes = sgpnt[i].length;
      transfer_length += sgpnt[i].length;

      DEB(if (sgpnt[i].length > max_sg_segment) \
             max_sg_segment = sgpnt[i].length);
      }

   mscp->number_of_sg_list = SCpnt->use_sg;
   mscp->transfer_data = (unsigned int) sglist;
   mscp->transfer_data_length = transfer_length;

   DEB(if (transfer_length > max_transfer_length) \
          max_transfer_length = transfer_length);
}

int ultrastor_14f_queuecommand(Scsi_Cmnd *SCpnt, void (*done)(Scsi_Cmnd *)) {
   struct mscp *mscp;
   int mscp_index;
   unsigned int flags;
   int loop = 0;

   save_flags(flags);
   cli();
   mscp_index = SCpnt->target;

   if (!done) panic("U14F queuecommand, null done");

   if (!HD(u14f)->mscp_free[mscp_index]) 
      panic("U14F queuecommand, unexpected busy MSCP");

   if (mscp_index < 0 || mscp_index >= U14F_MAX_MSCPS) 
      panic("U14F queuecommand, invalid mscp_index");

   mscp = &HD(u14f)->mscp[mscp_index];
   mscp->opcode = OP_SCSI;
   mscp->xdir = DTD_SCSI;
   mscp->dcn = FALSE;
   mscp->ca = FALSE;
   mscp->target_id = SCpnt->target;
   mscp->ch_no = 0;
   mscp->lun = SCpnt->lun;

   if (SCpnt->use_sg) {
      /* Set scatter/gather flag in SCSI command packet */
      mscp->sg = TRUE;
      build_sg_list(mscp, SCpnt);
      }
   else {
      /* Unset scatter/gather flag in SCSI command packet */
      mscp->sg = FALSE;
      mscp->transfer_data = (unsigned int)SCpnt->request_buffer;
      mscp->transfer_data_length = SCpnt->request_bufflen;
      SCpnt->host_scribble = NULL;
      }

   mscp->command_link = 0;
   mscp->scsi_command_link_id = 0;
   mscp->length_of_sense_byte = sizeof SCpnt->sense_buffer;
   mscp->length_of_scsi_cdbs = COMMAND_SIZE(*(unsigned char *)SCpnt->cmnd);
   memcpy(mscp->scsi_cdbs, SCpnt->cmnd, mscp->length_of_scsi_cdbs);
   mscp->adapter_status = 0;
   mscp->target_status = 0;
   mscp->sense_data = (unsigned int)&SCpnt->sense_buffer;
   mscp->done = done;
   mscp->SCint = SCpnt;

   while (inb(LCL_DOORBELL_INTR(HD(u14f)->doorbell_address)) & 1) {

      if (loop++ < 1000) continue;

      if (SCpnt->host_scribble) scsi_free(SCpnt->host_scribble, 512);

      SCpnt->result = DID_BUS_BUSY << 16;
      restore_flags(flags);
      done(SCpnt);
      printk("U%s queuecommand, id %d, adapter  busy after %d tries\n", 
                  utype, mscp_index, loop - 1);
      return 0;
      }

   HD(u14f)->mscp_free[mscp_index] = FALSE;

   /* Store pointer in OGM address bytes */
   outl((unsigned int)mscp, HD(u14f)->ogm_address);

   /* Issue OGM interrupt */
   outb(0x1, LCL_DOORBELL_INTR(HD(u14f)->doorbell_address));

   DEB(if (loop) printk("Q0.%d.%d.", mscp_index, loop));

   restore_flags(flags);
   return 0;
}

int ultrastor_14f_abort(Scsi_Cmnd *SCpnt) {
   int mscp_index;
   int i;
   struct mscp *mscp;
   unsigned int flags;
   void (*done)(Scsi_Cmnd *);

   save_flags(flags);
   DEB(printk("A0.%p.", SCpnt));
   cli();

   mscp_index = SCpnt->target;

   if (mscp_index < 0 || mscp_index >= U14F_MAX_MSCPS) 
      panic("U14F abort, invalid MSCP pointer");

   mscp = &HD(u14f)->mscp[mscp_index];

   if (HD(u14f)->mscp_free[mscp_index]) {
      DEB(printk("A1.%d.", mscp_index));
      restore_flags(flags);
      return SCSI_ABORT_NOT_RUNNING;
      }

   DEB(printk("A2.%d.", mscp_index));
   cli();

   /* Set local doorbell mask to disallow bus reset unless
   ultrastor_bus_reset is true.  */
   outb(ultrastor_bus_reset ? 0xc2 : 0x82, 
             LCL_DOORBELL_MASK(HD(u14f)->port_address));
   udelay(1000);

   /* Reset the adapter and SCSI bus. */
   outb(0xc0, LCL_DOORBELL_INTR(HD(u14f)->doorbell_address));

   sti();

   for (i = 0; i < 500; i++) udelay(1000);

   while (inb(LCL_DOORBELL_INTR(HD(u14f)->doorbell_address)) & 1);
   cli();
   while (inb(LCL_DOORBELL_INTR(HD(u14f)->doorbell_address)) & 1);

   if (HD(u14f)->mscp_free[mscp_index]) {
      DEB(printk("A3.%d.", mscp_index));
      restore_flags(flags);
      return SCSI_ABORT_NOT_RUNNING;
      }

   if (mscp->SCint != SCpnt)
      panic("U14F abort, MSCP command pointer mismatch");

   done = mscp->done;

   if (!done) panic("U14F abort, null mscp->done");

   if (SCpnt->host_scribble) scsi_free(SCpnt->host_scribble, 512);

   SCpnt->result = DID_ABORT << 16;
   HD(u14f)->mscp_free[mscp_index] = TRUE;
   restore_flags(flags);
   done(SCpnt);
   DEB(printk("A4.%d.", mscp_index));
   return SCSI_ABORT_SUCCESS;
}

int ultrastor_14f_reset(Scsi_Cmnd * SCpnt) {
   unsigned int flags;
   int i, arg_done = FALSE;
   struct mscp *mscp;
   void (*done)(Scsi_Cmnd *);
   Scsi_Cmnd *SCtmp;

   save_flags(flags);
   DEB(printk("R0.%p.", SCpnt));
   cli();

   /* Set local doorbell mask to disallow bus reset unless
   ultrastor_bus_reset is true.  */
   outb(ultrastor_bus_reset ? 0xc2 : 0x82, 
             LCL_DOORBELL_MASK(HD(u14f)->port_address));
   udelay(1000);

   /* Reset the adapter and SCSI bus. */
   outb(0xc0, LCL_DOORBELL_INTR(HD(u14f)->doorbell_address));

   sti();

   for (i = 0; i < 500; i++) udelay(1000);

   while (inb(LCL_DOORBELL_INTR(HD(u14f)->doorbell_address)) & 1);
   cli();
   while (inb(LCL_DOORBELL_INTR(HD(u14f)->doorbell_address)) & 1);

   for (i = 0; i < U14F_MAX_MSCPS; i++) 
      if (!HD(u14f)->mscp_free[i]) {
         mscp = &HD(u14f)->mscp[i];
         done = mscp->done;
         SCtmp = mscp->SCint;
         if (!done) panic("U14F reset, null mscp->done");
         if (!SCtmp) panic("U14F reset, null mscp->SCint");
         if (SCtmp->target != i) panic("U14F reset, MSCP pointer mismatch");
         SCtmp->result = DID_RESET << 16;
         if (SCpnt == SCtmp) arg_done = TRUE;
         if (SCtmp->host_scribble) scsi_free(SCtmp->host_scribble, 512);
         HD(u14f)->mscp_free[i] = TRUE;
         restore_flags(flags);
         done(SCtmp);
         DEB(printk("R1.%d.%p.%p.", i, SCtmp, done));
         cli();
         }

   if (arg_done) {
      DEB(printk("R2."));
      restore_flags(flags);
      return SCSI_RESET_SUCCESS;
      }
   else {
      DEB(printk("R3."));
      restore_flags(flags);
      return SCSI_RESET_PUNT;
      }
}

int ultrastor_14f_biosparam(Disk * disk, int dev, int * dkinfo) {
   int size = disk->capacity;
   unsigned int s = HD(u14f)->heads * HD(u14f)->sectors;

   dkinfo[0] = HD(u14f)->heads;
   dkinfo[1] = HD(u14f)->sectors;
   dkinfo[2] = size / s;	/* Ignore partial cylinders */
   return 0;
}

static void ultrastor_14f_interrupt(int cpl) {
   unsigned int status;
   int i;
   int mscp_index;
   unsigned int flags;
   struct mscp *mscp;
   void (*done)(Scsi_Cmnd *);
   Scsi_Cmnd *SCtmp;

   save_flags(flags);
   cli();
   mscp = (struct mscp *)inl(HD(u14f)->icm_address);

   /* Clear interrupt pending flag */
   outb(0x1, SYS_DOORBELL_INTR(HD(u14f)->doorbell_address));

   mscp_index = mscp - HD(u14f)->mscp;

   if (mscp_index < 0 || mscp_index >= U14F_MAX_MSCPS)
      panic("U14F interrupt, invalid MSCP address");

   if (HD(u14f)->mscp_free[mscp_index]) {
      DEB(printk("I2.%d.", mscp_index)); /* mscp_index points to free MSCB */
      restore_flags(flags);
      return;
      }

   /* Save done locally.  This is needed as once we call done, 
   we may get another command queued before this interrupt service 
   routine can return, and mscp slot reused. */
   done = mscp->done;
   SCtmp = mscp->SCint;

   if (done == NULL || SCtmp == NULL) 
      panic("U14F interrupt, null mscp->done or mscp->SCint");

   if (SCtmp->target != mscp_index) 
      panic("U14F interrupt, MSCP index mismatch");

   switch (mscp->adapter_status) {
      case 0:

         if (HD(u14f)->mscp_reset[mscp_index] && mscp->target_status == 2  \
                                       &&SCtmp->device->type != TYPE_TAPE)
            status = DID_BUS_BUSY << 16;
         else
            status = DID_OK << 16;

         if (mscp->target_status == 0)
            HD(u14f)->mscp_reset[mscp_index] = FALSE;

         break;
      case 0x01:	/* Invalid command */
      case 0x02:	/* Invalid parameters */
      case 0x03:	/* Invalid data list */
      default:
         status = DID_ERROR << 16;
         break;
      case 0x84:	/* SCSI bus abort error */
         status = DID_ABORT << 16;
         break;
      case 0x91:	/* SCSI bus selection time out */
         status = DID_TIME_OUT << 16;
	 break;
      case 0x93:	/* Unexpected bus free */
      case 0x96:	/* Illegal SCSI command */
      case 0xa3:	/* SCSI bus reset error */

         if (SCtmp->device->type != TYPE_TAPE)
            status = DID_BUS_BUSY << 16;
         else
            status = DID_ERROR << 16;

         for (i = 0; i < U14F_MAX_MSCPS; i++) HD(u14f)->mscp_reset[i] = TRUE;

	 break;
         }

   if (SCtmp->host_scribble) scsi_free(SCtmp->host_scribble, 512);

   HD(u14f)->mscp_free[mscp_index] = TRUE;
   SCtmp->result = status | mscp->target_status;

#if defined (DEBUG)
   if (mscp->adapter_status != 0 || mscp->target_status != 0) {
      printk("I0.%d.%p.%x.%x.%d.%d.%d.", mscp_index, SCtmp, 
           mscp->adapter_status, mscp->target_status, 
           mscp->transfer_data_length, mscp->number_of_sg_list,
           HD(u14f)->mscp_reset[mscp_index]);
      printk("I1.%d.%d.", max_transfer_length, max_sg_segment);
      max_transfer_length = 0;
      max_sg_segment = 0;
      }
#else
   if (mscp->adapter_status != 0) 
      printk("U%s interrupt, status 0x%x, target %d\n", utype,
         mscp->adapter_status, mscp_index);
#endif

   restore_flags(flags);
   done(SCtmp);
   return;
}
