/*
 *  linux/fs/buffer.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

/*
 *  'buffer.c' implements the buffer-cache functions. Race-conditions have
 * been avoided by NEVER letting an interrupt change a buffer (except for the
 * data, of course), but instead letting the caller do it.
 */

/*
 * NOTE! There is one discordant note here: checking floppies for
 * disk change. This is where it fits best, I think, as it should
 * invalidate changed floppy-disk-caches.
 */

#include <stdarg.h>
 
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/string.h>
#include <linux/locks.h>
#include <linux/errno.h>
#include <linux/malloc.h>

#include <asm/system.h>
#include <asm/segment.h>
#include <asm/io.h>

#ifdef CONFIG_SCSI
#ifdef CONFIG_BLK_DEV_SR
extern int check_cdrom_media_change(int, int);
#endif
#ifdef CONFIG_BLK_DEV_SD
extern int check_scsidisk_media_change(int, int);
extern int revalidate_scsidisk(int, int);
#endif
#endif
#ifdef CONFIG_CDU31A
extern int check_cdu31a_media_change(int, int);
#endif
#ifdef CONFIG_MCD
extern int check_mcd_media_change(int, int);
#endif

#define NR_SIZES 4
static char buffersize_index[9] = {-1,  0,  1, -1,  2, -1, -1, -1, 3};

#define BUFSIZE_INDEX(X) (buffersize_index[(X)>>9])

static int grow_buffers(int pri, int size);

static int n_stat[16];

int nr_hash = 0;  /* Size of hash table */
struct buffer_head ** hash_table;
static struct buffer_head * lru_list = NULL;
static struct buffer_head * free_list[NR_SIZES] = {NULL, };
static struct buffer_head * unused_list = NULL;
static struct wait_queue * buffer_wait = NULL;

int nr_buffers = 0;
int buffermem = 0;
int nr_buffer_heads = 0;
static int min_free_pages = 20;	/* nr free pages needed before buffer grows */
extern int *blksize_size[];

/* Here is the parameter block for the bdflush process. */
static void wakeup_bdflush(int);

#define N_PARAM 7

static union bdflush_param{
  struct {
    int nfract;  /* Percentage of buffer cache to scan to search for clean blocks */
    int ndirty;  /* Maximum number of dirty blocks to write out per wake-cycle */
    int nrefill; /* Number of clean buffers to try and obtain each time we call refill */
    int nref_dirt; /* Dirty buffer threshold for activating bdflush when trying to refill
		      buffers. */
    int clu_nfract;  /* Percentage of buffer cache to scan to search for free clusters */
    int age_buffer;  /* Time for normal buffer to age before we flush it */
    int age_super;  /* Time for superblock to age before we flush it */
  } b_un;
  unsigned int data[N_PARAM];
} bdf_prm = {{25, 500, 64, 256, 15, 3000, 500}};

/* These are the min and max parameter values that we will allow to be assigned */
static bdflush_min[N_PARAM] = {  0,  10,    5,   25,  0,   100,   100};
static bdflush_max[N_PARAM] = {100,5000, 2000, 2000,100, 60000, 60000};

/*
 * Rewrote the wait-routines to use the "new" wait-queue functionality,
 * and getting rid of the cli-sti pairs. The wait-queue routines still
 * need cli-sti, but now it's just a couple of 386 instructions or so.
 *
 * Note that the real wait_on_buffer() is an inline function that checks
 * if 'b_wait' is set before calling this, so that the queues aren't set
 * up unnecessarily.
 */
void __wait_on_buffer(struct buffer_head * bh)
{
	struct wait_queue wait = { current, NULL };

	bh->b_count++;
	add_wait_queue(&bh->b_wait, &wait);
repeat:
	current->state = TASK_UNINTERRUPTIBLE;
	if (bh->b_lock) {
		schedule();
		goto repeat;
	}
	remove_wait_queue(&bh->b_wait, &wait);
	bh->b_count--;
	current->state = TASK_RUNNING;
}

/* Call sync_buffers with wait!=0 to ensure that the call does not
   return until all buffer writes have completed.  Sync() may return
   before the writes have finished; fsync() may not. */

static int sync_buffers(dev_t dev, int wait)
{
	int i, retry, pass = 0, err = 0;
	struct buffer_head * bh;

	/* One pass for no-wait, three for wait:
	   0) write out all dirty, unlocked buffers;
	   1) write out all dirty buffers, waiting if locked;
	   2) wait for completion by waiting for all buffers to unlock.
	 */
repeat:
	retry = 0;
	bh = lru_list;
	for (i = nr_buffers*2 ; i-- > 0 ; bh = bh->b_next_free) {
		if (dev && bh->b_dev != dev)
			continue;
#ifdef 0 /* Disable bad-block debugging code */
		if (bh->b_req && !bh->b_lock &&
		    !bh->b_dirt && !bh->b_uptodate)
			printk ("Warning (IO error) - orphaned block %08x on %04x\n",
				bh->b_blocknr, bh->b_dev);
#endif
		if (bh->b_lock)
		{
			/* Buffer is locked; skip it unless wait is
			   requested AND pass > 0. */
			if (!wait || !pass) {
				retry = 1;
				continue;
			}
			wait_on_buffer (bh);
		}
		/* If an unlocked buffer is not uptodate, there has been 
		   an IO error. Skip it. */
		if (wait && bh->b_req && !bh->b_lock &&
		    !bh->b_dirt && !bh->b_uptodate)
		{
			err = 1;
			continue;
		}
		/* Don't write clean buffers.  Don't write ANY buffers
		   on the third pass. */
		if (!bh->b_dirt || pass>=2)
			continue;
		bh->b_count++;
		bh->b_flushtime = 0;
		ll_rw_block(WRITE, 1, &bh);
		bh->b_count--;
		retry = 1;
	}
	/* If we are waiting for the sync to succeed, and if any dirty
	   blocks were written, then repeat; on the second pass, only
	   wait for buffers being written (do not pass to write any
	   more buffers on the second pass). */
	if (wait && retry && ++pass<=2)
		goto repeat;
	return err;
}

void sync_dev(dev_t dev)
{
	sync_buffers(dev, 0);
	sync_supers(dev);
	sync_inodes(dev);
	sync_buffers(dev, 0);
}

int fsync_dev(dev_t dev)
{
	sync_buffers(dev, 0);
	sync_supers(dev);
	sync_inodes(dev);
	return sync_buffers(dev, 1);
}

asmlinkage int sys_sync(void)
{
	sync_dev(0);
	return 0;
}

int file_fsync (struct inode *inode, struct file *filp)
{
	return fsync_dev(inode->i_dev);
}

asmlinkage int sys_fsync(unsigned int fd)
{
	struct file * file;
	struct inode * inode;

	if (fd>=NR_OPEN || !(file=current->filp[fd]) || !(inode=file->f_inode))
		return -EBADF;
	if (!file->f_op || !file->f_op->fsync)
		return -EINVAL;
	if (file->f_op->fsync(inode,file))
		return -EIO;
	return 0;
}

void invalidate_buffers(dev_t dev)
{
	int i;
	struct buffer_head * bh;

	bh = lru_list;
	for (i = nr_buffers*2 ; --i > 0 ; bh = bh->b_next_free) {
		if (bh->b_dev != dev)
			continue;
		wait_on_buffer(bh);
		if (bh->b_dev == dev)
			bh->b_flushtime = bh->b_uptodate = bh->b_dirt = bh->b_req = 0;
	}
}

/*
 * This routine checks whether a floppy has been changed, and
 * invalidates all buffer-cache-entries in that case. This
 * is a relatively slow routine, so we have to try to minimize using
 * it. Thus it is called only upon a 'mount' or 'open'. This
 * is the best way of combining speed and utility, I think.
 * People changing diskettes in the middle of an operation deserve
 * to loose :-)
 *
 * NOTE! Although currently this is only for floppies, the idea is
 * that any additional removable block-device will use this routine,
 * and that mount/open needn't know that floppies/whatever are
 * special.
 */
void check_disk_change(dev_t dev)
{
	int i;
	struct buffer_head * bh;

	switch(MAJOR(dev)){
	case FLOPPY_MAJOR:
		if (!(bh = getblk(dev,0,1024)))
			return;
		i = floppy_change(bh);
		brelse(bh);
		break;

#if defined(CONFIG_BLK_DEV_SD) && defined(CONFIG_SCSI)
         case SCSI_DISK_MAJOR:
		i = check_scsidisk_media_change(dev, 0);
		break;
#endif

#if defined(CONFIG_BLK_DEV_SR) && defined(CONFIG_SCSI)
	 case SCSI_CDROM_MAJOR:
		i = check_cdrom_media_change(dev, 0);
		break;
#endif

#if defined(CONFIG_CDU31A)
         case CDU31A_CDROM_MAJOR:
		i = check_cdu31a_media_change(dev, 0);
		break;
#endif

#if defined(CONFIG_MCD)
         case MITSUMI_CDROM_MAJOR:
		i = check_mcd_media_change(dev, 0);
		break;
#endif

         default:
		return;
	};

	if (!i)	return;

	printk("VFS: Disk change detected on device %d/%d\n",
					MAJOR(dev), MINOR(dev));
	for (i=0 ; i<NR_SUPER ; i++)
		if (super_blocks[i].s_dev == dev)
			put_super(super_blocks[i].s_dev);
	invalidate_inodes(dev);
	invalidate_buffers(dev);

#if defined(CONFIG_BLK_DEV_SD) && defined(CONFIG_SCSI)
/* This is trickier for a removable hardisk, because we have to invalidate
   all of the partitions that lie on the disk. */
	if (MAJOR(dev) == SCSI_DISK_MAJOR)
		revalidate_scsidisk(dev, 0);
#endif
}

#define _hashfn(dev,block) (((unsigned)(dev^block))%nr_hash)
#define hash(dev,block) hash_table[_hashfn(dev,block)]

static inline void remove_from_hash_queue(struct buffer_head * bh)
{
	if (bh->b_next)
		bh->b_next->b_prev = bh->b_prev;
	if (bh->b_prev)
		bh->b_prev->b_next = bh->b_next;
	if (hash(bh->b_dev,bh->b_blocknr) == bh)
		hash(bh->b_dev,bh->b_blocknr) = bh->b_next;
	bh->b_next = bh->b_prev = NULL;
}

static inline void remove_from_lru_list(struct buffer_head * bh)
{
	if (!(bh->b_prev_free) || !(bh->b_next_free))
		panic("VFS: LRU block list corrupted");
	if(bh->b_dev == 0xffff) panic("LRU list corrupted");
	bh->b_prev_free->b_next_free = bh->b_next_free;
	bh->b_next_free->b_prev_free = bh->b_prev_free;
	if (lru_list == bh)
		lru_list = bh->b_next_free;
	if(lru_list->b_next_free == lru_list)
	  lru_list = NULL;
	bh->b_next_free = bh->b_prev_free = NULL;
}

static inline void remove_from_free_list(struct buffer_head * bh)
{
        int isize = BUFSIZE_INDEX(bh->b_size);
	if (!(bh->b_prev_free) || !(bh->b_next_free))
		panic("VFS: Free block list corrupted");
	if(bh->b_dev != 0xffff) panic("Free list corrupted");
	if(!free_list[isize]){
#if 0
	  printk("BH %x %x %x %x %x %x\n",
		 bh->b_dev, bh->b_blocknr, bh->b_next, bh->b_prev,
		 bh->b_next_free, bh->b_prev_free);
#endif
	  panic("Free list empty");
	};
	if(bh->b_next_free == bh)
	  free_list[isize] = NULL;
	else {
	  bh->b_prev_free->b_next_free = bh->b_next_free;
	  bh->b_next_free->b_prev_free = bh->b_prev_free;
	  if (free_list[isize] == bh)
	    free_list[isize] = bh->b_next_free;
	};
	bh->b_next_free = bh->b_prev_free = NULL;
}

static inline void remove_from_queues(struct buffer_head * bh)
{
        if(bh->b_dev == 0xffff) {
	  remove_from_free_list(bh); /* Free list entries should not be in
					the hash queue */
	  return;
	};
	remove_from_hash_queue(bh);
	remove_from_lru_list(bh);
	
}

static inline void put_last_lru(struct buffer_head * bh)
{
	if (!bh)
		return;
	if (bh == lru_list) {
		lru_list = bh->b_next_free;
		return;
	}
	if(bh->b_dev == 0xffff) panic("Wrong block for lru list");
	remove_from_lru_list(bh);
/* add to back of free list */

	if(!lru_list) {
	  lru_list = bh;
	  lru_list->b_prev_free = bh;
	};

	bh->b_next_free = lru_list;
	bh->b_prev_free = lru_list->b_prev_free;
	lru_list->b_prev_free->b_next_free = bh;
	lru_list->b_prev_free = bh;
}

static inline void put_last_free(struct buffer_head * bh)
{
        int isize;
	if (!bh)
		return;

	isize = BUFSIZE_INDEX(bh->b_size);	
	bh->b_dev = 0xffff;  /* So it is obvious we are on the free list */
/* add to back of free list */

	if(!free_list[isize]) {
	  free_list[isize] = bh;
	  bh->b_prev_free = bh;
	};

	bh->b_next_free = free_list[isize];
	bh->b_prev_free = free_list[isize]->b_prev_free;
	free_list[isize]->b_prev_free->b_next_free = bh;
	free_list[isize]->b_prev_free = bh;
}

static inline void insert_into_queues(struct buffer_head * bh)
{
/* put at end of free list */

        if(bh->b_dev == 0xffff) {
	  put_last_free(bh);
	  return;
	};
	if(!lru_list) {
	  lru_list = bh;
	  bh->b_prev_free = bh;
	};
	bh->b_next_free = lru_list;
	bh->b_prev_free = lru_list->b_prev_free;
	lru_list->b_prev_free->b_next_free = bh;
	lru_list->b_prev_free = bh;
/* put the buffer in new hash-queue if it has a device */
	bh->b_prev = NULL;
	bh->b_next = NULL;
	if (!bh->b_dev)
		return;
	bh->b_next = hash(bh->b_dev,bh->b_blocknr);
	hash(bh->b_dev,bh->b_blocknr) = bh;
	if (bh->b_next)
		bh->b_next->b_prev = bh;
}

static struct buffer_head * find_buffer(dev_t dev, int block, int size)
{		
	struct buffer_head * tmp;

	for (tmp = hash(dev,block) ; tmp != NULL ; tmp = tmp->b_next)
		if (tmp->b_dev==dev && tmp->b_blocknr==block)
			if (tmp->b_size == size)
				return tmp;
			else {
				printk("VFS: Wrong blocksize on device %d/%d\n",
							MAJOR(dev), MINOR(dev));
				return NULL;
			}
	return NULL;
}

/*
 * Why like this, I hear you say... The reason is race-conditions.
 * As we don't lock buffers (unless we are readint them, that is),
 * something might happen to it while we sleep (ie a read-error
 * will force it bad). This shouldn't really happen currently, but
 * the code is ready.
 */
struct buffer_head * get_hash_table(dev_t dev, int block, int size)
{
	struct buffer_head * bh;

	for (;;) {
		if (!(bh=find_buffer(dev,block,size)))
			return NULL;
		bh->b_count++;
		wait_on_buffer(bh);
		if (bh->b_dev == dev && bh->b_blocknr == block && bh->b_size == size)
			return bh;
		bh->b_count--;
	}
}

void set_blocksize(dev_t dev, int size)
{
	int i;
	struct buffer_head * bh, *bhnext;

	if (!blksize_size[MAJOR(dev)])
		return;

	switch(size) {
		default: panic("Invalid blocksize passed to set_blocksize");
		case 512: case 1024: case 2048: case 4096:;
	}

	if (blksize_size[MAJOR(dev)][MINOR(dev)] == 0 && size == BLOCK_SIZE) {
		blksize_size[MAJOR(dev)][MINOR(dev)] = size;
		return;
	}
	if (blksize_size[MAJOR(dev)][MINOR(dev)] == size)
		return;
	sync_buffers(dev, 2);
	blksize_size[MAJOR(dev)][MINOR(dev)] = size;

  /* We need to be quite careful how we do this - we are moving entries
     around on the free list, and we can get in a loop if we are not careful.*/

	bh = lru_list;
	for (i = nr_buffers*2 ; --i > 0 ; bh = bhnext) {
	  if(!bh) break;
	  bhnext = bh->b_next_free; 
	  if (bh->b_dev != dev)
	    continue;
	  if (bh->b_size == size)
	    continue;
	    
	    wait_on_buffer(bh);
	    if (bh->b_dev == dev && bh->b_size != size) {
	      bh->b_uptodate = bh->b_dirt = bh->b_flushtime = 0;
	    };
	    remove_from_hash_queue(bh);
	  }
}

#define BADNESS(bh) (((bh)->b_dirt<<1)+(bh)->b_lock)

void refill_freelist(int size)
{
	struct buffer_head * bh, * tmp;
	int buffers;
	char skipped;
	int needed;
	int ndirty;

	/* If there are too many dirty buffers, we wake up the update process
	   now so as to ensure that there are still clean buffers available
	   for user processes to use (and dirty) */

	/* We are going to try and locate this much memory */
	needed =bdf_prm.b_un.nrefill * size;  

	while (nr_free_pages > min_free_pages && needed > 0 &&
	       grow_buffers(GFP_BUFFER, size)) {
		needed -= PAGE_SIZE;
	}
	/* OK, we cannot grow the buffer cache, now try and get some from
	   the lru list */

repeat:
	if(needed <= 0) return;
	buffers = nr_buffers;
	bh = NULL;

	skipped = 0;
	ndirty = 0;
	for (bh = lru_list; buffers-- > 0 && needed > 0; 
	     bh = tmp) {
	        if (!bh) break;
		tmp = bh->b_next_free;
		if (bh->b_count || bh->b_size != size)
			continue;
		if (mem_map[MAP_NR((unsigned long) bh->b_data)] != 1)
			continue;
		if (bh->b_dirt && !bh->b_lock) ndirty++;
		if(ndirty == bdf_prm.b_un.nref_dirt) wakeup_bdflush(0);
		if (BADNESS(bh)) continue;

		if(bh->b_dev == 0xffff) panic("Wrong list");
		remove_from_queues(bh);
		bh->b_dev = 0xffff;
		put_last_free(bh);
		needed -= bh->b_size;
	      }

	if(needed <= 0) return;

	/* Too bad, that was not enough. Try a little harder to grow some. */

	if (nr_free_pages > 5) {
	  if (grow_buffers(GFP_BUFFER, size)) {
	                needed -= PAGE_SIZE;
			goto repeat;
			};
	}
	
/* and repeat until we find something good */
	if (!grow_buffers(GFP_ATOMIC, size)) {
	  wakeup_bdflush(1);
	};
	needed -= PAGE_SIZE;
	goto repeat;
}

/*
 * Ok, this is getblk, and it isn't very clear, again to hinder
 * race-conditions. Most of the code is seldom used, (ie repeating),
 * so it should be much more efficient than it looks.
 *
 * The algoritm is changed: hopefully better, and an elusive bug removed.
 *
 * 14.02.92: changed it to sync dirty buffers a bit: better performance
 * when the filesystem starts to get full of dirty blocks (I hope).
 */
struct buffer_head * getblk(dev_t dev, int block, int size)
{
	struct buffer_head * bh;
        int isize = BUFSIZE_INDEX(size);


	/* If there are too many dirty buffers, we wake up the update process
	   now so as to ensure that there are still clean buffers available
	   for user processes to use (and dirty) */
repeat:
	bh = get_hash_table(dev, block, size);
	if (bh) {
		if (bh->b_uptodate && !bh->b_dirt)
			put_last_lru(bh);
		if(!bh->b_dirt) bh->b_flushtime = 0;
		return bh;
	}

	while(!free_list[isize]) refill_freelist(size);

	if (find_buffer(dev,block,size))
		goto repeat;

	bh = free_list[isize];
	remove_from_free_list(bh);

/* OK, FINALLY we know that this buffer is the only one of it's kind, */
/* and that it's unused (b_count=0), unlocked (b_lock=0), and clean */
	bh->b_count=1;
	bh->b_dirt=0;
	bh->b_uptodate=0;
	bh->b_flushtime = 0;
	bh->b_req=0;
	bh->b_dev=dev;
	bh->b_blocknr=block;
	insert_into_queues(bh);
	return bh;
}

void set_writetime(struct buffer_head * buf, int flag)
{
        int newtime;

	if (buf->b_dirt){
	  /* Move buffer to dirty list if jiffies is clear */
	    newtime = jiffies + (flag ? bdf_prm.b_un.age_super : bdf_prm.b_un.age_buffer);
	  if(!buf->b_flushtime || buf->b_flushtime > newtime)
	    buf->b_flushtime = newtime;
	} else {
	  buf->b_flushtime = 0;
	}
}

void brelse(struct buffer_head * buf)
{
	if (!buf)
		return;
	wait_on_buffer(buf);

	/* If dirty, mark the time this buffer should be written back */
	set_writetime(buf, 0);

	if (buf->b_count) {
		if (--buf->b_count)
			return;
		wake_up(&buffer_wait);
		return;
	}
	printk("VFS: brelse: Trying to free free buffer\n");
}

/*
 * bread() reads a specified block and returns the buffer that contains
 * it. It returns NULL if the block was unreadable.
 */
struct buffer_head * bread(dev_t dev, int block, int size)
{
	struct buffer_head * bh;

	if (!(bh = getblk(dev, block, size))) {
		printk("VFS: bread: READ error on device %d/%d\n",
						MAJOR(dev), MINOR(dev));
		return NULL;
	}
	if (bh->b_uptodate)
		return bh;
	ll_rw_block(READ, 1, &bh);
	wait_on_buffer(bh);
	if (bh->b_uptodate)
		return bh;
	brelse(bh);
	return NULL;
}

/*
 * Ok, breada can be used as bread, but additionally to mark other
 * blocks for reading as well. End the argument list with a negative
 * number.
 */

#define NBUF 16

struct buffer_head * breada(dev_t dev, int block, int bufsize, unsigned int pos,
				       unsigned int filesize)
{
	  struct buffer_head * bhlist[NBUF];
          unsigned int blocks;
	  struct buffer_head * bh;
	  int index;
	  int i, j;

	  if(pos >= filesize) return NULL;

	  if (!block || !(bh = getblk(dev,block,bufsize)))
	    return NULL;

	  index = BUFSIZE_INDEX(bh->b_size);

	  if(bh->b_uptodate) return bh;

	  blocks = ((filesize & (bufsize - 1)) - (pos & (bufsize - 1))) >> (9+index);

	  if(blocks > read_ahead[MAJOR(dev)] >> index)
	    blocks = read_ahead[MAJOR(dev)] >> index;
	  if(blocks > NBUF) blocks = NBUF;
	  
	  bhlist[0] = bh;
	  j = 1;
	  for(i=1; i<blocks; i++) {
	    bh = getblk(dev,block+i,bufsize);
	    if(bh->b_uptodate) {brelse(bh); break;};
	    bhlist[j++] = bh;
	  };

	  /* Request the read for these buffers, and then release them */
	  ll_rw_block(READ, j, bhlist);

	  for(i=1; i<j; i++)
	    brelse(bhlist[i]);

	  /* Wait for this buffer, and then continue on */
	  bh = bhlist[0];
	  wait_on_buffer(bh);
	  if(bh->b_uptodate) return bh;
	  brelse(bh);
	  return NULL;
}

/*
 * See fs/inode.c for the weird use of volatile..
 */
static void put_unused_buffer_head(struct buffer_head * bh)
{
	struct wait_queue * wait;

	wait = ((volatile struct buffer_head *) bh)->b_wait;
	memset((void *) bh,0,sizeof(*bh));
	((volatile struct buffer_head *) bh)->b_wait = wait;
	bh->b_next_free = unused_list;
	unused_list = bh;
}

static void get_more_buffer_heads(void)
{
	int i;
	struct buffer_head * bh;

	if (unused_list)
		return;

	if(! (bh = (struct buffer_head*) get_free_page(GFP_BUFFER)))
		return;

	for (nr_buffer_heads+=i=PAGE_SIZE/sizeof*bh ; i>0; i--) {
		bh->b_next_free = unused_list;	/* only make link */
		unused_list = bh++;
	}
}

static struct buffer_head * get_unused_buffer_head(void)
{
	struct buffer_head * bh;

	get_more_buffer_heads();
	if (!unused_list)
		return NULL;
	bh = unused_list;
	unused_list = bh->b_next_free;
	bh->b_next_free = NULL;
	bh->b_data = NULL;
	bh->b_size = 0;
	bh->b_req = 0;
	return bh;
}

/*
 * Create the appropriate buffers when given a page for data area and
 * the size of each buffer.. Use the bh->b_this_page linked list to
 * follow the buffers created.  Return NULL if unable to create more
 * buffers.
 */
static struct buffer_head * create_buffers(unsigned long page, unsigned long size)
{
	struct buffer_head *bh, *head;
	unsigned long offset;

	head = NULL;
	offset = PAGE_SIZE;
	while ((offset -= size) < PAGE_SIZE) {
		bh = get_unused_buffer_head();
		if (!bh)
			goto no_grow;
		bh->b_this_page = head;
		head = bh;
		bh->b_data = (char *) (page+offset);
		bh->b_size = size;
		bh->b_dev = 0xffff;  /* Flag as unused */
	}
	return head;
/*
 * In case anything failed, we just free everything we got.
 */
no_grow:
	bh = head;
	while (bh) {
		head = bh;
		bh = bh->b_this_page;
		put_unused_buffer_head(head);
	}
	return NULL;
}

static void read_buffers(struct buffer_head * bh[], int nrbuf)
{
	int i;
	int bhnum = 0;
	struct buffer_head * bhr[8];

	for (i = 0 ; i < nrbuf ; i++) {
		if (bh[i] && !bh[i]->b_uptodate)
			bhr[bhnum++] = bh[i];
	}
	if (bhnum)
		ll_rw_block(READ, bhnum, bhr);
	for (i = 0 ; i < nrbuf ; i++) {
		if (bh[i]) {
			wait_on_buffer(bh[i]);
		}
	}
}

static unsigned long check_aligned(struct buffer_head * first, unsigned long address,
	dev_t dev, int *b, int size)
{
	struct buffer_head * bh[8];
	unsigned long page;
	unsigned long offset;
	int block;
	int nrbuf;

	page = (unsigned long) first->b_data;
	if (page & ~PAGE_MASK) {
		brelse(first);
		return 0;
	}
	mem_map[MAP_NR(page)]++;
	bh[0] = first;
	nrbuf = 1;
	for (offset = size ; offset < PAGE_SIZE ; offset += size) {
		block = *++b;
		if (!block)
			goto no_go;
		first = get_hash_table(dev, block, size);
		if (!first)
			goto no_go;
		bh[nrbuf++] = first;
		if (page+offset != (unsigned long) first->b_data)
			goto no_go;
	}
	read_buffers(bh,nrbuf);		/* make sure they are actually read correctly */
	while (nrbuf-- > 0)
		brelse(bh[nrbuf]);
	free_page(address);
	++current->min_flt;
	return page;
no_go:
	while (nrbuf-- > 0)
		brelse(bh[nrbuf]);
	free_page(page);
	return 0;
}

static unsigned long try_to_load_aligned(unsigned long address,
	dev_t dev, int b[], int size)
{
	struct buffer_head * bh, * tmp, * arr[8];
	unsigned long offset;
	int * p;
	int block;

	bh = create_buffers(address, size);
	if (!bh)
		return 0;
	p = b;
	for (offset = 0 ; offset < PAGE_SIZE ; offset += size) {
		block = *(p++);
		if (!block)
			goto not_aligned;
		tmp = get_hash_table(dev, block, size);
		if (tmp) {
			brelse(tmp);
			goto not_aligned;
		}
	}
	tmp = bh;
	p = b;
	block = 0;
	while (1) {
		arr[block++] = bh;
		bh->b_count = 1;
		bh->b_dirt = 0;
		bh->b_flushtime = 0;
		bh->b_uptodate = 0;
		bh->b_dev = dev;
		bh->b_blocknr = *(p++);
		nr_buffers++;
		insert_into_queues(bh);
		if (bh->b_this_page)
			bh = bh->b_this_page;
		else
			break;
	}
	buffermem += PAGE_SIZE;
	bh->b_this_page = tmp;
	mem_map[MAP_NR(address)]++;
	read_buffers(arr,block);
	while (block-- > 0)
		brelse(arr[block]);
	++current->maj_flt;
	return address;
not_aligned:
	while ((tmp = bh) != NULL) {
		bh = bh->b_this_page;
		put_unused_buffer_head(tmp);
	}
	return 0;
}

/*
 * Try-to-share-buffers tries to minimize memory use by trying to keep
 * both code pages and the buffer area in the same page. This is done by
 * (a) checking if the buffers are already aligned correctly in memory and
 * (b) if none of the buffer heads are in memory at all, trying to load
 * them into memory the way we want them.
 *
 * This doesn't guarantee that the memory is shared, but should under most
 * circumstances work very well indeed (ie >90% sharing of code pages on
 * demand-loadable executables).
 */
static inline unsigned long try_to_share_buffers(unsigned long address,
	dev_t dev, int *b, int size)
{
	struct buffer_head * bh;
	int block;

	block = b[0];
	if (!block)
		return 0;
	bh = get_hash_table(dev, block, size);
	if (bh)
		return check_aligned(bh, address, dev, b, size);
	return try_to_load_aligned(address, dev, b, size);
}

#define COPYBLK(size,from,to) \
__asm__ __volatile__("rep ; movsl": \
	:"c" (((unsigned long) size) >> 2),"S" (from),"D" (to) \
	:"cx","di","si")

/*
 * bread_page reads four buffers into memory at the desired address. It's
 * a function of its own, as there is some speed to be got by reading them
 * all at the same time, not waiting for one to be read, and then another
 * etc. This also allows us to optimize memory usage by sharing code pages
 * and filesystem buffers..
 */
unsigned long bread_page(unsigned long address, dev_t dev, int b[], int size, int prot)
{
	struct buffer_head * bh[8];
	unsigned long where;
	int i, j;

	if (!(prot & PAGE_RW)) {
		where = try_to_share_buffers(address,dev,b,size);
		if (where)
			return where;
	}
	++current->maj_flt;
 	for (i=0, j=0; j<PAGE_SIZE ; i++, j+= size) {
		bh[i] = NULL;
		if (b[i])
			bh[i] = getblk(dev, b[i], size);
	}
	read_buffers(bh,i);
	where = address;
 	for (i=0, j=0; j<PAGE_SIZE ; i++, j += size,address += size) {
		if (bh[i]) {
			if (bh[i]->b_uptodate)
				COPYBLK(size, (unsigned long) bh[i]->b_data,address);
			brelse(bh[i]);
		}
	}
	return where;
}

/*
 * Try to increase the number of buffers available: the size argument
 * is used to determine what kind of buffers we want.
 */
static int grow_buffers(int pri, int size)
{
	unsigned long page;
	struct buffer_head *bh, *tmp, *tmp1;
        int isize;

	if ((size & 511) || (size > PAGE_SIZE)) {
		printk("VFS: grow_buffers: size = %d\n",size);
		return 0;
	}

        isize = BUFSIZE_INDEX(size);

	if(!(page = __get_free_page(pri)))
		return 0;
	bh = create_buffers(page, size);
	if (!bh) {
		free_page(page);
		return 0;
	}

	/* Reverse the order of the buffer headers before we stick them
	   on the free list. Clustering works better this way */

	tmp = NULL;
	while(bh){
	  tmp1 = bh;
	  bh = bh->b_this_page;
	  tmp1->b_this_page = tmp;
	  tmp = tmp1;
	};
	bh = tmp;

	tmp = bh;
	while (1) {
		if (free_list[isize]) {
			tmp->b_next_free = free_list[isize];
			tmp->b_prev_free = free_list[isize]->b_prev_free;
			free_list[isize]->b_prev_free->b_next_free = tmp;
			free_list[isize]->b_prev_free = tmp;
		} else {
			tmp->b_prev_free = tmp;
			tmp->b_next_free = tmp;
		}
		free_list[isize] = tmp;
		++nr_buffers;
		if (tmp->b_this_page)
			tmp = tmp->b_this_page;
		else
			break;
	}
	tmp->b_this_page = bh;
	wake_up(&buffer_wait);
	buffermem += PAGE_SIZE;
	return 1;
}

/*
 * try_to_free() checks if all the buffers on this particular page
 * are unused, and free's the page if so.
 */
static int try_to_free(struct buffer_head * bh, struct buffer_head ** bhp)
{
	unsigned long page;
	struct buffer_head * tmp, * p;

	*bhp = bh;
	page = (unsigned long) bh->b_data;
	page &= PAGE_MASK;
	tmp = bh;
	do {
		if (!tmp)
			return 0;
		if (tmp->b_count || tmp->b_dirt || tmp->b_lock)
			return 0;
		tmp = tmp->b_this_page;
	} while (tmp != bh);
	tmp = bh;
	do {
		p = tmp;
		tmp = tmp->b_this_page;
		nr_buffers--;
		if (p == *bhp)
			*bhp = p->b_prev_free;
		remove_from_queues(p);
		put_unused_buffer_head(p);
	} while (tmp != bh);
	buffermem -= PAGE_SIZE;
	free_page(page);
	return !mem_map[MAP_NR(page)];
}

/*
 * Try to free up some pages by shrinking the buffer-cache
 *
 * Priority tells the routine how hard to try to shrink the
 * buffers: 3 means "don't bother too much", while a value
 * of 0 means "we'd better get some free pages now".
 */
int shrink_buffers(unsigned int priority)
{
	struct buffer_head *bh;
	int i, isize;

	if (priority < 2) {
		sync_buffers(0,0);
	}

	if(priority == 2) {
	  wakeup_bdflush(1);
	};

	/* First try the free lists, and see if we can get a complete page
	   from here */
	for(isize = 0; isize<NR_SIZES; isize++){
	  bh = free_list[isize];
	  if(!bh) continue;
	  for (i=0 ; !i || bh != free_list[isize]; bh = bh->b_next_free, i++) {
	    if (bh->b_count || !bh->b_this_page)
	      continue;
	    if (try_to_free(bh, &bh))
	      return 1;
	  }
	}

	/* Not enough in the free lists, now try the lru list */

	bh = lru_list;
	i = nr_buffers >> priority;
	  for ( ; i-- > 0 ; bh = bh->b_next_free) {
	    if (bh->b_count || !bh->b_this_page)
	      continue;
	    if (bh->b_lock)
	      if (priority)
		continue;
	      else
		wait_on_buffer(bh);
	    if (bh->b_dirt) {
	      bh->b_count++;
	      bh->b_flushtime = 0;
	      ll_rw_block(WRITEA, 1, &bh);
	      bh->b_count--;
	      continue;
	    }
	    if (try_to_free(bh, &bh))
	      return 1;
	  }
	return 0;
}

/*
 * try_to_reassign() checks if all the buffers on this particular page
 * are unused, and reassign to a new cluster them if this is true.
 */
static inline int try_to_reassign(struct buffer_head * bh, struct buffer_head ** bhp,
			   dev_t dev, unsigned int starting_block)
{
	unsigned long page;
	struct buffer_head * tmp, * p;

	*bhp = bh;
	page = (unsigned long) bh->b_data;
	page &= PAGE_MASK;
	if(mem_map[MAP_NR(page)] != 1) return 0;
	tmp = bh;
	do {
		if (!tmp)
			return 0;
		if (tmp->b_count || tmp->b_dirt || tmp->b_lock)
			return 0;
		tmp = tmp->b_this_page;
	} while (tmp != bh);
	tmp = bh;

	while((unsigned int) tmp->b_data & (PAGE_SIZE - 1)) 
	  tmp = tmp->b_this_page;

	do {
		p = tmp;
		tmp = tmp->b_this_page;
		remove_from_queues(p);
		p->b_dev=dev;
		p->b_uptodate = 0;
		p->b_blocknr=starting_block++;
		insert_into_queues(p);
	} while (tmp != bh);
	return 1;
}

/*
 * Try to find a free cluster by locating a page where
 * all of the buffers are unused.  We would like this function
 * to be atomic, so we do not call anything that might cause
 * the process to sleep.  The priority is somewhat similar to
 * the priority used in shrink_buffers.
 * 
 * My thinking is that the kernel should end up using whole
 * pages for the buffer cache as much of the time as possible.
 * This way the other buffers on a particular page are likely
 * to be very near each other on the free list, and we will not
 * be expiring data prematurely.  For now we only canibalize buffers
 * of the same size to keep the code simpler.
 */
static int reassign_cluster(dev_t dev, 
		     unsigned int starting_block, int size)
{
	struct buffer_head *bh;
        int isize = BUFSIZE_INDEX(size);
	int ndirty;
	int i;

	while(!free_list[isize]) refill_freelist(size);
	n_stat[0]++;

	bh = free_list[isize];
	if(bh)
	  for (i=0 ; !i || bh != free_list[isize] ; bh = bh->b_next_free, i++) {
	    if (!bh->b_this_page)	continue;
	    if (try_to_reassign(bh, &bh, dev, starting_block)) {
	      n_stat[1]++;
/*	      printk("[%d] ", i); */
	      return 4;
	    };
	  }

	bh = lru_list;
	ndirty = 0;
	i = nr_buffers * bdf_prm.b_un.clu_nfract/100;
	for ( ; i-- > 0 ; bh = bh->b_next_free) {
	  if (bh->b_count || !bh->b_this_page || bh->b_lock || bh->b_dirt)
	    continue;
	  if (bh->b_size != size) continue;
	  if (bh->b_dirt && !bh->b_lock) ndirty++;
	  if(ndirty == bdf_prm.b_un.nref_dirt) wakeup_bdflush(0);
	  if (try_to_reassign(bh, &bh, dev, starting_block)) {
/*	    printk("{%d ", i); */
	    return 4;
	  };
	}
	n_stat[2]++;
/*	printk("@"); */
	return 0;
}

/* This function tries to generate a new cluster of buffers
 * from a new page in memory.  We should only do this if we have
 * not expanded the buffer cache to the maximum size that we allow.
 */
static unsigned long try_to_generate_cluster(dev_t dev, int block, int size)
{
	struct buffer_head * bh, * tmp, * arr[8];
	unsigned long offset;
	unsigned long page;
	int nblock;

	page = get_free_page(GFP_KERNEL);
	if(!page) return 0;

	n_stat[3]++;
	bh = create_buffers(page, size);
	if (!bh) {
	  free_page(page);
	  n_stat[4]++;
	  return 0;
	};
	nblock = block;
	for (offset = 0 ; offset < PAGE_SIZE ; offset += size) {
		tmp = get_hash_table(dev, nblock++, size);
		if (tmp) {
			brelse(tmp);
			goto not_aligned;
		}
	}
	tmp = bh;
	nblock = 0;
	while (1) {
		arr[nblock++] = bh;
		bh->b_count = 1;
		bh->b_dirt = 0;
		bh->b_flushtime = 0;
		bh->b_uptodate = 0;
		bh->b_dev = dev;
		bh->b_blocknr = block++;
		nr_buffers++;
		insert_into_queues(bh);
		if (bh->b_this_page)
			bh = bh->b_this_page;
		else
			break;
	}
	buffermem += PAGE_SIZE;
	bh->b_this_page = tmp;
	while (nblock-- > 0)
		brelse(arr[nblock]);
	n_stat[5]++;
	return 4;
not_aligned:
	n_stat[6]++;
	while ((tmp = bh) != NULL) {
		bh = bh->b_this_page;
		put_unused_buffer_head(tmp);
	}
	return 0;
}

unsigned long generate_cluster(dev_t dev, int b[], int size)
{
  int i, offset;
  
  for (i = 0, offset = 0 ; offset < PAGE_SIZE ; i++, offset += size) {
    if(i && b[i]-1 != b[i-1]) return 0;  /* No need to cluster */
    if(find_buffer(dev, b[i], size)) return 0;
  };

  /* OK, we have a candidate for a new cluster */

	n_stat[7]++;
	if (nr_free_pages > min_free_pages) 
	  return try_to_generate_cluster(dev, b[0], size);
	else
	  return reassign_cluster(dev, b[0], size);
}

/*
 * This initializes the initial buffer free list.  nr_buffers is set
 * to one less the actual number of buffers, as a sop to backwards
 * compatibility --- the old code did this (I think unintentionally,
 * but I'm not sure), and programs in the ps package expect it.
 * 					- TYT 8/30/92
 */
void buffer_init(void)
{
	int i;
        int isize = BUFSIZE_INDEX(BLOCK_SIZE);

	if (high_memory >= 4*1024*1024) {
		min_free_pages = 200;
		nr_hash = 4093;
	} else {
		min_free_pages = 20;
		nr_hash = 997;
	};

	hash_table = (struct buffer_head **) vmalloc(nr_hash * 
						     sizeof(struct buffer_head *));
	for (i = 0 ; i < nr_hash ; i++)
		hash_table[i] = NULL;
	lru_list = 0;
	grow_buffers(GFP_KERNEL, BLOCK_SIZE);
	if (!free_list[isize])
		panic("VFS: Unable to initialize buffer free list!");
	return;
}

/* This is a simple kernel daemon, whose job it is to provide a dynamicly
 * response to dirty buffers.  Once this process is activated, we write back
 * a limited number of buffers to the disks and then go back to sleep again.
 * In effect this is a process which never leaves kernel mode, and does not have
 * any user memory associated with it except for the stack.  There is also
 * a kernel stack page, which obviously must be separate from the user stack.
 */
struct wait_queue * bdflush_wait = NULL;
struct wait_queue * bdflush_done = NULL;

static int bdflush_running = 0;

static void wakeup_bdflush(int wait)
{
  if(!bdflush_running){
    printk("Warning - bdflush not running\n");
    sync_buffers(0,0);
    return;
  };
  wake_up(&bdflush_wait);
  if(wait) sleep_on(&bdflush_done);
}



/* 
 * Here we attempt to write back old buffers.  We also try and flush indoes and supers as well,
 * since this function is essentially "update", and otherwise there would be no way of ensuring
 * that these quantities ever get written back.  Ideally, we would have a timestamp on the inodes
 * and superblocks so that we could write back only the old ones as well
 */

asmlinkage int sync_old_buffers(void)
{
	int i;
	int ndirty, nwritten;
	struct buffer_head * bh;

	sync_supers(0);
	sync_inodes(0);

	bh = lru_list;
	ndirty = 0;
	nwritten = 0;
	if(bh) 
	  for (i = nr_buffers * 2 ; i-- > 0; bh = bh->b_next_free) {
	    if (bh->b_lock || !bh->b_dirt)
	      continue;
	    ndirty++;
	    if(bh->b_flushtime > jiffies) continue;
	    nwritten++;
	    bh->b_count++;
	    bh->b_flushtime = 0;
	    ll_rw_block(WRITE, 1, &bh);
	    bh->b_count--;
	  }
#ifdef DEBUG
	printk("Wrote %d/%d buffers\n", nwritten, ndirty);
#endif
	return 0;
}


/* This is the interface to bdflush.  As we get more sophisticated, we can
 * pass tuning parameters to this "process", to adjust how it behaves.  If you
 * invoke this again after you have done this once, you would simply modify the tuning
 * parameters.  We would want to verify each parameter, however, to make sure that it is
 * reasonable.
 */

asmlinkage int sys_bdflush(int func, int data)
{
	int i, error;
	int ndirty;
	struct buffer_head * bh;

	if(!suser()) return -EPERM;

	if(func == 1)
	  return sync_old_buffers();

	/* Basically func 0 means start, 1 means read param 1, 2 means write param 1, etc */
	if(func >= 2){
	  i = (func-2) >> 1;
	  if (i < 0 || i >= N_PARAM) return -EINVAL;
	  if((func & 1) == 0) {
	    error = verify_area(VERIFY_WRITE, (void *) data, sizeof(int));
	    if(error) return error;
	    put_fs_long(bdf_prm.data[i], data);
	    return 0;
	  };
	  if(data < bdflush_min[i] || data > bdflush_max[i]) return -EINVAL;
	  bdf_prm.data[i] = data;
	  return 0;
	};

	if(bdflush_running++) return -EBUSY; /* Only one copy of this running at one time */

	/* OK, from here on is the daemon */

	while(1==1){
#ifdef DEBUG
	  printk("bdflush() activated...");
#endif

	  bh = lru_list;
	  ndirty = 0;
	  if(bh) 
	    for (i = nr_buffers * bdf_prm.b_un.nfract/100 ; i-- > 0 && ndirty < bdf_prm.b_un.ndirty; 
		 bh = bh->b_next_free) {
	      if (bh->b_lock || !bh->b_dirt)
		continue;
	      /* Should we write back buffers that are shared or not??
		 currently dirty buffers are not shared, so it does not matter */
	      bh->b_count++;
	      ndirty++;
	      bh->b_flushtime = 0;
	      ll_rw_block(WRITE, 1, &bh);
	      bh->b_count--;
	    }
#ifdef DEBUG
	  printk("sleeping again.\n");
#endif
	  wake_up(&bdflush_done);
	  interruptible_sleep_on(&bdflush_wait);
	}
}




