/* 
   User mode program for timing-list-driver. Allows interactive adjust-
   ment of modes read from XF86Config.

   Copyright (C) 1995 Andreas Beck - becka@hp.rz.uni-duesseldorf.de

   If you do any modifications, I would like you to send diffs to me
   to allow for collecting a more and more complete set of drivers and
   to improve existing ones.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <linux/vt.h>
#include <string.h>
#include "graphlib.h"

struct ggi_MonParams mondesc={  
	{0,0},
	{0,0},
	{0,0},
	1024,786,
	 285,213,
	"NoName",
	"NoModel",
	VGA };

struct ggi_Timing prefmode[50]=
	{ { { 640,400,
	       80, 25,
	       GT_8BIT,
	       8, 16 },
	     25175000,	/* VGA Clock #1 */
	     640,680,776,800,
	     400,412,414,449,
	     1,1,
	     1,0
	  },
	};

int pref_num=1;

struct ggi_MonParams *mondptr=&mondesc ;
struct ggi_Timing trymode;
struct ggi_Timing    *prefptr=&prefmode[0];

static int verbose=2;

void toup(char *what)
{ do{*what=toupper(*what);}while(*what++); }

void beep(void)
{ fputs("\a",stderr); }

#define verb(a,b) do{ if (verbose>=a) printf(b,token);}while(0)

void parse_file(char *name)
{ char buffer[1024],*token;
  double hlp;
  int ignoreflag=1;
  FILE *infile;
  
  if ((infile=fopen(name,"r"))==NULL) 
    ggi_panic("Can't open input-file !");

  while(!feof(infile))
  { fgets(buffer,1024,infile);
    if (*buffer=='#'||*buffer=='\n') continue;
    toup(buffer);
    token=strtok(buffer," \t\n");
    if (strcmp(token,"SECTION")==0)
    { if (strcmp(strtok(NULL,"\" "),"MONITOR")==0) ignoreflag=0; }
    if (ignoreflag) continue;
    if (strcmp(token,"ENDSECTION")==0) break;
    else if (strcmp(token,"MODELINE")==0)
    { token=strtok(NULL,"\" ");verb(1,"Loading Mode: %s\n");
      sscanf(token=strtok(NULL," \t\n"),"%lf",&hlp);
      verb(2,"%s  ");
      prefmode[pref_num].clock     =1000000*hlp; /* MHz ! */
      prefmode[pref_num].tr.xvisible  =
      prefmode[pref_num].tr.xvirtual  =
      prefmode[pref_num].xwidth    =atoi(token=strtok(NULL," \t\n"));verb(2,"%s ");
      prefmode[pref_num].xsyncstart=atoi(token=strtok(NULL," \t\n"));verb(2,"%s ");
      prefmode[pref_num].xsyncend  =atoi(token=strtok(NULL," \t\n"));verb(2,"%s ");
      prefmode[pref_num].xend      =atoi(token=strtok(NULL," \t\n"));verb(2,"%s  ");
      prefmode[pref_num].tr.yvisible  =
      prefmode[pref_num].tr.yvirtual  =
      prefmode[pref_num].ywidth    =atoi(token=strtok(NULL," \t\n"));verb(2,"%s ");
      prefmode[pref_num].ysyncstart=atoi(token=strtok(NULL," \t\n"));verb(2,"%s ");
      prefmode[pref_num].ysyncend  =atoi(token=strtok(NULL," \t\n"));verb(2,"%s ");
      prefmode[pref_num].yend      =atoi(token=strtok(NULL," \t\n"));verb(2,"%s\n");
      prefmode[pref_num].tr.graphtype =GT_8BIT;
      if ((token=strtok(NULL," \t\n")))
      { puts("Warning: Flags are not yet handled !"); }
      pref_num++;
    }
    else if (strcmp(token,"VENDORNAME")==0)
    { token=strtok(NULL,"\" ");verb(1,"Manufacturer : %s\n");
      strncpy(mondesc.manufact,token,sizeof(mondesc.manufact));
      mondesc.manufact[sizeof(mondesc.manufact)-1]='\0'; }
    else if (strcmp(token,"MODELNAME")==0)
    { token=strtok(NULL,"\" ");verb(1,"Model : %s\n");
      strncpy(mondesc.model,token,sizeof(mondesc.model));
      mondesc.model[sizeof(mondesc.model)-1]='\0'; }
    else if (strcmp(token,"BANDWIDTH")==0)
    { puts(token=strtok(NULL,"-")); }
    else if (strcmp(token,"HORIZSYNC")==0)
    { sscanf(token=strtok(NULL,"-"),"%lf",&hlp);
      mondesc.hfreq.min=hlp*1000;verb(1,"HSync : %s - ");
      sscanf(token=strtok(NULL," \t\n"),"%lf",&hlp);
      mondesc.hfreq.max=hlp*1000;verb(1,"%s\n"); }
    else if (strcmp(token,"VERTREFRESH")==0)
    { mondesc.vfreq.min=atoi(token=strtok(NULL,"-"));
      verb(1,"VSync : %s - ");
      mondesc.vfreq.max=atoi(token=strtok(NULL," \t\n,"));
      verb(1,"%s\n"); }
  }
  if (ggi_ioctl(MONITOR_SETINFO,&mondptr)) { perror("ggi_ioctl : "); }
}

void testpic(struct ggi_Timing *pp)
{ int mx,my,x;  
  static struct ggi_Palentry pal[256]={{0x00,0x00,0x00},/* black */
  				       {0xff,0xff,0xff},/* white */
  				       {0x80,0x80,0x80},/* grey  */
  				       {0xff,0x80,0x80},/* ltred */
  				       {0xa0,0xa0,0xa0},/* ltgrey*/
  				       {0x50,0xff,0x50},/* green */
  				       };

  if (ggi_ioctl(MONITOR_CLEARPREF,NULL))
    ggi_panic("Clear preferred modes failed.Is the timelist-driver loaded ?");
  prefptr=&prefmode[0];
  if (ggi_ioctl(MONITOR_SETPREFMODE,&prefptr)) { perror("setprefmode : "); }
  if (ggi_ioctl(MONITOR_SETPREFMODE,&pp)) { perror("set mode : "); }

  if (ggi_graphmode(pp->tr.xvisible,pp->tr.yvisible,
  		    pp->tr.xvirtual,pp->tr.yvirtual,GT_8BIT)) return;

  ggi_setpalvec(0,6,pal);
  ggi_setcolor(1);
  ggi_drawhline(0,             0,pp->tr.xvisible);
  ggi_drawhline(0,pp->tr.yvisible-1,pp->tr.xvisible);
  ggi_drawvline(0,             0,pp->tr.yvisible);
  ggi_drawvline(pp->tr.xvisible-1,0,pp->tr.yvisible);

  ggi_setcolor(2);
  for(x=(mx=(pp->tr.xvisible/2))%50;x<pp->tr.xvisible-1;x+=50)
    if (x>0) ggi_drawvline(x,1,pp->tr.yvisible-2);
  for(x=(my=(pp->tr.yvisible/2))%50;x<pp->tr.yvisible-1;x+=50)
    if (x>0) ggi_drawhline(1,x,pp->tr.xvisible-2);

  ggi_setcolor(3);
  for(x=1;x<20;x++) 
  { ggi_drawpixel(               x,               x);
    ggi_drawpixel(pp->tr.xvisible-1-x,               x);
    ggi_drawpixel(pp->tr.xvisible-1-x,pp->tr.yvisible-1-x);
    ggi_drawpixel(               x,pp->tr.yvisible-1-x); }

  ggi_setcolor(4);
  ggi_drawcircle(mx,my,my-my%10-10);

  ggi_setcolor(5);
  ggi_printf(mx,my+ 1,"Clock : %4.3f",pp->clock/1000000.0);
  ggi_printf(mx,my+16,"Xsync : %5d %5d %5d %5d",
             pp->xwidth,pp->xsyncstart,
             pp->xsyncend,pp->xend);
  ggi_printf(mx,my+31,"Ysync : %5d %5d %5d %5d",
             pp->ywidth,pp->ysyncstart,
             pp->ysyncend,pp->yend);
  ggi_printf(mx,my+51,"Freq  : %5.2f %5.2f",
             0.001*pp->clock/pp->xend,
             1.0*pp->clock/pp->xend/pp->yend);
  ggi_printf(mx,my+66,"Press 'h' for help.");
}

void cls(void)
{ printf("\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n");}

void interactive(void)
{ int x,y;

  do{
    if (ggi_ioctl(MONITOR_CLEARPREF,NULL))
      ggi_panic("Clear preferred modes failed.Is the timelist-driver loaded ?");
    prefptr=&prefmode[0];
    if (ggi_ioctl(MONITOR_SETPREFMODE,&prefptr)) { perror("setprefmode : "); }

    ggi_textmode(80,25,8,16);
    cls();

    for(x=1;x<pref_num;x++)
      printf("%d. %4dx%4d \n",x,prefmode[x].tr.xvisible,prefmode[x].tr.yvisible);

    printf("Adjust which mode (0=save and exit) ?\n");
    scanf("%d",&x);
    if (x<1||x>=pref_num) return;

    trymode=prefmode[x];

    while(1)
    { if (ggi_ioctl(MONITOR_CLEARPREF,NULL))
      ggi_panic("Clear preferred modes failed.Is the timelist-driver loaded ?");
      prefptr=&prefmode[0];
      if (ggi_ioctl(MONITOR_SETPREFMODE,&prefptr)) { perror("setprefmode : "); }
      prefptr=&trymode;
      if (ggi_ioctl(MONITOR_SETPREFMODE,&prefptr)) { perror("set mode : "); }

      testpic(&trymode);
      switch(y=ggi_getch())
      { case 'l':if (trymode.xsyncend+8<trymode.xend) 
      		 { trymode.xsyncstart+=8;trymode.xsyncend+=8; }
      		 else beep();
      		 break;
        case 'r':if (trymode.xsyncstart-8>trymode.xwidth) 
        	 { trymode.xsyncstart-=8;trymode.xsyncend-=8; }
        	 else beep();
        	 break;
        case 'u':if (trymode.ysyncend+1<trymode.yend) 
        	 { trymode.ysyncstart++;trymode.ysyncend++; }
        	 else beep();
        	 break;
        case 'd':if (trymode.ysyncstart-1>trymode.ywidth) 
        	 { trymode.ysyncstart--;trymode.ysyncend--; }
        	 else beep();
        	 break;
        case 'U':if (trymode.yend-1>trymode.ysyncend)
        	   trymode.yend--;
        	 else beep();
        	 break;
        case 'D':trymode.yend++;break;
        case 'L':if (trymode.xend-8>trymode.xsyncend)
        	   trymode.xend-=8;
        	 else beep();
        	 break;
        case 'R':trymode.xend+=8;break;
	case 's':if (trymode.xsyncpol>0) trymode.xsyncpol=-1;
		 else trymode.xsyncpol=1;
		 break;
	case 'S':if (trymode.ysyncpol>0) trymode.ysyncpol=-1;
		 else trymode.ysyncpol=1;
		 break;
	case 'h':
	case 'H':ggi_textmode(80,25,8,16);cls();
		 puts(  "HELP :\n"
		 	"======\n"
		 	"Keys for adjusting a mode :\n"
		 	"u/d/l/r : reposition the image (will beep when max position)\n"
		 	"U/D/L/R : make image larger/smaller in y/x-direction\n"
		 	"s/S     : change h/v Sync polarity\n"
		 	"x/X,y/Y : change sync pulse width\n"
		 	"<ENTER> : save this values in the list\n"
		 	"<ESC>   : discard values\n");
		 ggi_getch();
		 break;
        case 'x':if (trymode.xsyncend-8>trymode.xsyncstart)
        	   trymode.xsyncend-=8;
        	 else beep();
        	 break;
        case 'X':if (trymode.xsyncend+8<trymode.xend)
        	   trymode.xsyncend+=8;
        	 else beep();
        	 break;
        case 'y':if (trymode.ysyncend-1>trymode.ysyncstart)
        	   trymode.ysyncend--;
        	 else beep();
        	 break;
        case 'Y':if (trymode.ysyncend+1<trymode.yend)
        	   trymode.ysyncend++;
        	 else beep();
        	 break;
	/* This is for testing the power-saving */
#if 0
	case 'p':trymode.ysyncstart=trymode.yend;break;
	case 'P':trymode.xsyncstart=trymode.xend;break;
#endif

        case '\n':prefmode[x]=trymode;break;
      }
      if (y=='\n'||y=='\x1b') break;
    }
  }while(x);
  ggi_textmode(80,25,8,16);
}

void download(void)
{ int x;

  if (ggi_ioctl(MONITOR_CLEARPREF,NULL)) 
    ggi_panic("Clear preferred modes failed.Is the timelist-driver loaded ?");

  for(x=0;x<pref_num;x++)
  { prefptr=&prefmode[x];
    if (ggi_ioctl(MONITOR_SETPREFMODE,&prefptr)) { perror("setprefmode : "); }
  }
}

void save(char *name)
{ int x;
  FILE *out;
  
  if ((out=fopen(name,"w"))==NULL) {perror("save : ");return;}

  fputs("SECTION \"MONITOR\"\n",out);
  fprintf(out,"VENDORNAME \"%s\"\n",mondesc.manufact);
  fprintf(out,"MODELNAME \"%s\"\n",mondesc.model);
  fprintf(out,"HORIZSYNC %d-%d\n",mondesc.hfreq.min/1000,
  	        mondesc.hfreq.max/1000);
  fprintf(out,"VERTREFRESH %d-%d\n",mondesc.vfreq.min,
  		mondesc.vfreq.max);
  for(x=1;x<pref_num;x++)
  { prefptr=&prefmode[x];
    fprintf(out,"ModeLine \"%dx%d\"\t %4.3f %5d %5d %5d %5d  %5d %5d %5d %5d\n",
            prefptr->tr.xvisible,prefptr->tr.yvisible,
            prefptr->clock/1000000.0,
            prefptr->xwidth,prefptr->xsyncstart,
             prefptr->xsyncend,prefptr->xend,
            prefptr->ywidth,prefptr->ysyncstart,
             prefptr->ysyncend,prefptr->yend);
  }
  fputs("ENDSECTION\n",out);

}

int main(int argc, char *argv[])
{ 
  if (argc<2) 
  { fprintf(stderr,"Usage : %s XF86Config [File_to_save_to]\n",argv[0]);
    exit(1); }
  ggi_init();

  parse_file(argv[1]);

  if (argc<3) download();
  else        {interactive();download();save(argv[2]);}
  
  ggi_textmode(80,25,8,16);
  ggi_exit();
  return(0);
}
