/* 
   Generic Monitor driver for modern Multi-Sync-type monitors.

   Copyright (C) 1995 Andreas Beck - becka@hp.rz.uni-duesseldorf.de

   If you do any modifications, I would like you to send diffs to me
   to allow for collecting a more and more complete set of drivers and
   to improve existing ones.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
 
#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/fcntl.h>
#include <linux/vt.h>
#include <sys/ioctl.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include "graphdev.h"
#include "kgi-module.h"
#include "multisync_config.h"
#include "monitor/monitor_generic.h"

static char *DISPTYPES[4]={ "B&W","EGA-8","EGA-16","VGA" };

int  kgi_CheckMonitorTiming  (struct ggi_Timing * MT,int cmd)
{ int flg,acfreq,h;
  int err=0;

  /* We ignore CMD right now and always want to be perfect ... FIXME */
  
  /* Lower clock if out of bounds ... */
  for(flg=0;MT->clock>MT->xwidth*MonPar.hfreq.min;MT->clock-=1000,flg++)

  { h=MT->clock/1000;			/* Pixels for 1 ms */

    MT->xend      =(MT->xwidth+
    		    (h*MonTimeConst.xafter_ns)/1000000+
    		    MonTimeConst.xafter_pix)
    		   &0xfffffff8;
    while(MT->xend<=MT->xwidth+16) MT->xend+=8;

    acfreq=MT->clock/MT->xend;			/* Lines for 1 us */

    if (acfreq>MonPar.hfreq.max) 
      if (err==MR_LOWEST_CLOCK) return(MR_X_TOO_SMALL);
      else continue;				/* Try lower clock */

    if (acfreq<MonPar.hfreq.min) 
      if (flg==0) return(MR_X_TOO_BIG);		/* Screen too wide even at highest clock */
      else        return(MR_Y_TOO_SMALL);	/* aspect ratio is bad ... */
      
    /* Here to increase performance - these do not matter for line frequency */
    MT->xsyncstart=(MT->xwidth+
    		    (h*MonTimeConst.xbefore_ns)/1000000+
    		    MonTimeConst.xbefore_pix)
    		   &0xfffffff8;		/* must be a multiple of 8 */
    while(MT->xsyncstart<=MT->xwidth) MT->xsyncstart+=8;

    MT->xsyncend  =(MT->xwidth+
                    (h*MonTimeConst.xlen_ns)/1000000+
                    MonTimeConst.xlen_pix)
                   &0xfffffff8;
    while(MT->xsyncend<=MT->xsyncstart) MT->xsyncend+=8;

    h=acfreq;					/* Lines for 1 s */
    
    MT->yend      =MT->ywidth+MonTimeConst.yafter_pix
    			     +(MonTimeConst.yafter_us*h)/1000000; /*flyback */
    while(MT->yend<=MT->ywidth+2) MT->yend++;

    acfreq=MT->clock/(MT->xend*MT->yend);
    
    if (acfreq>MonPar.vfreq.max)
      if (err==MR_LOWEST_CLOCK) return(MR_Y_TOO_SMALL);
      else continue;				/* Try lower clock */

    if (acfreq<MonPar.vfreq.min) return(MR_Y_TOO_BIG);		/* Screen too tall */

    MT->ysyncstart=MT->ywidth+MonTimeConst.ybefore_pix
    			     +(MonTimeConst.ybefore_us*h)/1000000;
    while(MT->ysyncstart<=MT->ywidth) MT->ysyncstart++;
    MT->ysyncend  =MT->ywidth+MonTimeConst.ylen_pix
    			     +(MonTimeConst.ylen_us*h)/1000000;
    while(MT->ysyncend<=MT->ysyncstart) MT->ysyncend++;

    MT->xsyncpol  =MT->ysyncpol		= 1;	/* No matter for this monitor */

    switch(err=kgi_CheckClockTiming(MT,CMD_CLK_LOWER))
    { case MR_LOWEST_CLOCK:
      case MR_LOWER_CLOCK :break;
      default: return err;
    }
  }
  return(-ENOMODESUP_MON);
}

int kgi_MonitorIoctl(struct inode *inode, struct file *file, \
        unsigned int cmd, unsigned long arg)
{ switch(cmd)
  { case MONITOR_GETINFO    :COPYPAR(sizeof(void *));
			     return(mon_getinfo());
    case MONITOR_SETINFO    :if (!suser()) return(-EPERM);
			     COPYPAR(sizeof(void *));
			     return(mon_setinfo());

    case MONITOR_SETPREFMODE:if (!suser()) return(-EPERM);
			     COPYPAR(sizeof(void *));
			     return(mon_setprefmode());
    case MONITOR_GETPREFMODE:COPYPAR(sizeof(int)+sizeof(void *));
			     return(mon_getprefmode());
    case MONITOR_CLEARPREF  :if (!suser()) return(-EPERM);
                             TMnum=0;return(0);

    case MONITOR_POWERSAVE  :/* if (!suser()) return(-EPERM);  Should this be suser only ??? */
    			     COPYPAR(sizeof(int));
    			     return(mon_vesa_pwr());
    default : return -ENODRVSUP_ALWAYS_CANT; }
  return -ENODRVSUP_ALWAYS_CANT;
}
          

/* Argument parsing and initialization */

int kgi_MonitorInit(void)
{ printk("Monitor driver V"VERSION" loading ...\n");
  printk("Manufacturer : %s\n",MonPar.manufact);
  printk("Model        : %s\n",MonPar.model);
  printk("Type         : %s - %s  [%dx%d]\n",
  	  DISPTYPES[MonPar.disptype/8],
  	  (MonPar.disptype&1) ? "Monochrome" : "Color",
  	  MonPar.xmaxdots,
  	  MonPar.ymaxdots);
  printk("Size         : %d mm x %d mm\n",
  	  MonPar.xsize,
  	  MonPar.ysize);
  printk("Frequencies  : %d-%d kHz , %d-%d Hz\n",
  	  MonPar.hfreq.min/1000 ,MonPar.hfreq.max/1000 ,
  	  MonPar.vfreq.min      ,MonPar.vfreq.max );
  return(0);	/* No chance to check if this is right ... */
}  				  

void kgi_MonitorCleanup(void) {}
