/* 
   Generic functionality. This does not depend on any particular hardware.

   Copyright (C) 1995 Andreas Beck - becka@hp.rz.uni-duesseldorf.de

   If you do any modifications, I would like you to send diffs to me
   to allow for collecting a more and more complete set of drivers and
   to improve existing ones.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/tty.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/fcntl.h>
#include <linux/vt.h>
#include <sys/ioctl.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include "graphdev.h"
#include "kgi-module.h"
#include "vga_generic.h"
#include "generic.h"
#include "generic-needs.h"

char OpenFont[8192];		/* We store the font of the textmode-console here */

struct ggi_StateInfo StateNow[MAX_CONS_NUM],*CurrState;

struct ggi_StateInfo OpenState
	= { {	80*8,25*16,
		1024,1024,
		GT_TEXT,
		8,16 },
		0,1,0,		/* Console is in FG-mode, available and "attached" to vt 0 */
	        NULL,		/* DrawInfo not present */
	    { 	0,0},{0,} };

char parameters[256];	/* parameter transfer buffer */


/*******************************************
*** Debugging routine for mode resolving ***
*******************************************/

#ifdef DEBUG
void dump_tm(struct ggi_Timing *tm,char *label,int x)
{ printk("%s : %x\n",label,x);
  printk("%6d: %5d %5d %5d %5d\n",
         tm->clock/1000,tm->xwidth,tm->xsyncstart,tm->xsyncend,tm->xend);
  printk("        %5d %5d %5d %5d\n",
         tm->ywidth,tm->ysyncstart,tm->ysyncend,tm->yend);
  printk("        %5dx%5d of %5dx%5d tg:%3dx%3d\n\n",
         tm->tr.xvisible ,tm->tr.yvisible ,
         tm->tr.xvirtual ,tm->tr.yvirtual ,
         tm->tr.xtextgrid,tm->tr.ytextgrid);
}
#endif

/******************************************/
/*** SET a mode - this will change soon ***/
/******************************************/
int _setmode(struct ggi_Timing *tm)
{ int rc;

  if ((rc=kgi_CheckChipsetTiming(tm,CMD_NONE))) return(rc);

  if (!CurrState->bgmode)
   { if (kgi_SetChipsetTiming  (tm)==-1) return -ENOMODESUP_CHIP; }
  return 0;
}

int setmode(void)
{ int i;struct ggi_Timing tm;

  if ((i=verify_area(VERIFY_READ , *(void **) parameters, sizeof(struct ggi_Timing_req)))||
      (i=verify_area(VERIFY_WRITE, *(void **) parameters, sizeof(struct ggi_Timing_req)))) 
         return(i);

  memcpy_fromfs(&tm.tr, *(void **) parameters, sizeof(struct ggi_Timing_req));

  if ((i=_setmode(&tm))) return i;

  memcpy_tofs  (*(void **) parameters,&tm.tr , sizeof(struct ggi_Timing_req));

  hw_setdispstart(0,0);
  CurrState->ModeNow=tm.tr;return(0);
}

int getmode(void)
{ int i;
  if ((i=verify_area(VERIFY_WRITE, *(void **) parameters, sizeof(struct ggi_Timing_req)))) 
         return(i);
  memcpy_tofs  (*(void **) parameters,&CurrState->ModeNow , sizeof(struct ggi_Timing_req));
  return(0);
}

int keepmode(void)
{ OpenState=*CurrState;return(0); }

int getdeffont(void)
{ int i;
  if ((i=verify_area(VERIFY_WRITE, *(void **) parameters, sizeof(OpenFont)))) 
         return(i);
  memcpy_tofs  (*(void **) parameters,OpenFont, sizeof(OpenFont));
  return(0);
}

int setdeffont(void)
{ int i;
  if ((i=verify_area(VERIFY_READ , *(void **) parameters, sizeof(OpenFont))))
         return(i);
  memcpy_fromfs(OpenFont, *(void **) parameters, sizeof(OpenFont));
  return(0);
}

int setdispstart(void)
{ CurrState->dispstart.x=*((int *)parameters);
  CurrState->dispstart.y=*((int *)parameters+1);
  hw_setdispstart(CurrState->dispstart.x,CurrState->dispstart.y);
  return 0; }

int getdispstart(void)
{ int i;
  if ((i=verify_area(VERIFY_WRITE, *(void **) parameters, sizeof(struct ggi_Coord)))) 
         return(i);
  memcpy_tofs  (*(void **) parameters,&CurrState->dispstart,sizeof(struct ggi_Coord));
  return(0);
}
                                                                                        
/* return size of state-record */
int getstatesize(void)
{ int i; 
  i = verify_area(VERIFY_WRITE, *(void **) parameters, sizeof(unsigned long));
  if (!i) put_fs_long(sizeof(struct ggi_StateInfo), *(unsigned long **)parameters);
  return i;
}

int savestate(void)
{ int i;
  if ((i=verify_area(VERIFY_WRITE, *(void **)parameters,sizeof(struct ggi_StateInfo)))) 
         return(i);
 memcpy_tofs(*(void **)parameters,CurrState,sizeof(struct ggi_StateInfo));
 return(0);
}

int _restorestate(struct ggi_StateInfo *State)
{ int i;
  struct ggi_Timing gt;

  gt.tr=State->ModeNow;
  if ((i=_setmode(&gt))) return(i);

  if ((i=hw_setpal (0,256,State->palette))) return(i);
  hw_setdispstart(State->dispstart.x,State->dispstart.y);

  if (State->ModeNow.graphtype==GT_TEXT) _getsetfont_hw(OpenFont,1);
 
  return(0);
}

int restorestate(void)
{ int i;
  if ((i=verify_area(VERIFY_READ, *(void **)parameters,sizeof(struct ggi_StateInfo)))) 
         return(i);
 memcpy_fromfs(CurrState,*(void **)parameters,sizeof(struct ggi_StateInfo));
 return(_restorestate(CurrState));
}

int bgmode(int onoff)
{ 
  CurrState->bgmode=0;			/* Really do it ! */
  if (onoff)
  { _restorestate(&OpenState);	/* Set state at open physically */
  } else
  { _restorestate(CurrState);	/* Reset to current state */
  }
  CurrState->bgmode=onoff;
  return(0);
}

int getvt(void)
{ int i;
  if ((i=verify_area(VERIFY_WRITE, *(void **)parameters,sizeof(int)))) 
         return(i);
 put_fs_long(CurrState->vt_num,*(void **)parameters);
 return(0);
}

/*****************************************************************************
****  Graphics  Device  Interface  Funtions                               ****
*****************************************************************************/

int kgi_Ioctl(struct inode *inode, struct file *file, \
        unsigned int cmd, unsigned long arg)
{ DEV_NUM_SETUP;

  switch(cmd&IOCTL_SUBSYSTEM_MASK)
  { case SUBSYSTEM_CARD	     :return(kgi_ChipsetIoctl(inode,file,cmd,arg));
    case SUBSYSTEM_MONITOR   :return(kgi_MonitorIoctl(inode,file,cmd,arg));
    case SUBSYSTEM_CLOCK     :return(kgi_ClockIoctl  (inode,file,cmd,arg));
    case SUBSYSTEM_RAMDAC    :return(kgi_RamdacIoctl (inode,file,cmd,arg));
    default		     :return -ENODRVSUP_ALWAYS_CANT;
   }
   /* We should never get here ... */
   return -ENODRVSUP_ALWAYS_CANT;
}                                                      

struct vm_operations_struct Graph_vmops = {
	NULL,			/* open */
	kgi_GraphmapClose,	/* close */
	NULL,			/* unmap */
	NULL,			/* protect */
	NULL,			/* sync */
	NULL,			/* advise */
	kgi_GraphmapNopage,	/* nopage */
	NULL,			/* wppage */
	NULL,			/* swapout */
	NULL,			/* swapin */
	};
                                                                                
int kgi_Mmap(struct inode * inode, struct file * file, struct vm_area_struct * vma)
{
	DEV_NUM_SETUP;

	if (vma->vm_offset & ~PAGE_MASK)
		return -ENXIO;
	switch(vma->vm_offset)
	{ default:return -ENXIO;	/* We don't support queer mappings right now ... */
	  case MMAP_LINEAR:
	  	  vma->vm_inode = inode;
	 	  vma->vm_ops   = &Graph_vmops;
		  vma->vm_offset= (unsigned long) CurrState;	/* This is ugly ... */
		  inode->i_count++;
		  /* Make sure paging tables are there ... */
	 	  zeromap_page_range(vma->vm_start,vma->vm_end - vma->vm_start,vma->vm_page_prot);
		  fast_unmap_page_range(vma->vm_task,vma->vm_start,vma->vm_end - vma->vm_start);
		  return 0;
	  case MMAP_FRAME:
	  	  vma->vm_inode = inode;
		  inode->i_count++;
		  /* Make sure paging tables are there ... */
	 	  zeromap_page_range(vma->vm_start,0x10000,vma->vm_page_prot);
		  fast_remap_page_range(vma->vm_task,vma->vm_start,0xa0000,0x10000,vma->vm_page_prot);
		  return 0;
	}
	return -ENXIO;
}

int kgi_Init(void)
{ int x;

  /* These have to fill in OpenState so we can revert to it */

  if ((x=kgi_ChipsetInit())) return(x);
  if ((x=kgi_ClockInit()  )) { kgi_ChipsetCleanup();return(x);}
  if ((x=kgi_RamdacInit() )) { kgi_ClockCleanup();
  			       kgi_ChipsetCleanup();return(x);}
  if ((x=kgi_MonitorInit())) { kgi_RamdacCleanup();
  			       kgi_ClockCleanup();
  			       kgi_ChipsetCleanup();return(x);}

  for(x=0;x<MAX_CONS_NUM;x++)
  { StateNow[x]=OpenState;}

  return(0);
}  				  

void kgi_Cleanup(void)
{ 
  kgi_MonitorCleanup();
  kgi_RamdacCleanup();
  kgi_ClockCleanup();
  kgi_ChipsetCleanup();
}  				  

static int kgi_Open(struct inode *inode, struct file *file)
{ int x;

  if (MINOR(inode->i_rdev) != 0xab)	/* MAGIC for ugly hack ... */
    return -ENODEV;
        
  if (!current->tty||
      (current->tty->device&0xff00)!=0x0400||
      (current->tty->device&0xff)>=64  )
    return -ENOTTY;			/* controlling tty is not a VT */

  for(x=0;x<MAX_CONS_NUM;x++)
  { if (StateNow[x].available) break;}

  if (x>=MAX_CONS_NUM)
    return -EBUSY;

  MOD_INC_USE_COUNT;

  /* Go to FORGROUND-MODE and mark console as unavailable */
  StateNow[x].bgmode=0;
  StateNow[x].available = 0;
  StateNow[x].vt_num=current->tty->device&0xff;
  file->private_data=&StateNow[x];
  
  return 0;     /* device open - graphics ready to accept commands */
        
}

static void kgi_Close(struct inode *inode, struct file *file)
{
  DEV_NUM_SETUP;
  
  /* Return to the standard OpenState */
  
  if  (CurrState->bgmode==0) 
    _restorestate(&OpenState);			/* Need to set old state or */
  else 
    CurrState->bgmode=0;			/* it is already set ... */

  *CurrState=OpenState;
   CurrState->available = 1;

  MOD_DEC_USE_COUNT;
}
                                  
/*****************************************************************************
*** Module loading stuff                                                   ***
*****************************************************************************/

static struct file_operations graphdev_fops =
{ NULL,		/* lseek */
  NULL,		/* read  */
  NULL,         /* write */
  NULL,         /* readdir */
  NULL,		/* select  */
  kgi_Ioctl,
  kgi_Mmap,
  kgi_Open,
  kgi_Close,
};
int init_module(void)
{ int x;
  printk("Graphics driver V"VERSION" loading ...\n");

  if (register_chrdev(GRAPHICS_MAJOR, "graphics", &graphdev_fops)) 
  { printk("graphdev.c: cannot register requested major number\n");
    return -EIO; }

  if ((x=kgi_Init())) unregister_chrdev(GRAPHICS_MAJOR, "graphics");
  return(x);
}  				  

void cleanup_module(void) 
{  printk("Removing graphics device\n");
   kgi_Cleanup();
   if (unregister_chrdev(GRAPHICS_MAJOR, "graphics"))
     printk("graphdev.c: unregister_chrdev() failed\n");
}
