/*\
|*| ---------------------------------------------------------------------------
|*|	ICS 2061 Clock setter
|*| ---------------------------------------------------------------------------
|*|
|*|	1st version by Andreas Beck   [becka@hp.rz.uni-duesseldorf.de]
|*|	   modified by Steffen Seeger [seeger@physik.tu-chemnitz.de]
|*|
|*|	Copyright (C) 1995 Andreas Beck, Steffen Seeger
|*|
|*|	This program is free software; you can redistribute it and/or modify
|*|	it under the terms of the GNU General Public License as published by
|*|	the Free Software Foundation; either version 2, or (at your option)
|*|	any later version.
|*|
|*|	This program is distributed in the hope that it will be useful,
|*|	but WITHOUT ANY WARRANTY; without even the implied warranty of
|*|	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|*|	GNU General Public License for more details.
|*|
|*|	You should have received a copy of the GNU General Public License
|*|	along with this program; see the file COPYING.  If not, write to
|*|	the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
|*| ---------------------------------------------------------------------------
|*|     This code was derived from the following sources of information:
|*|	[1]	VGADOC package by Finn Thoegersen
|*|     [2]     XFree86 3.0 package, files ICD2061.h,ICD2061Aalt.c
|*|             (partial Copyright (c) 1992 by Number Nine Corp.)
|*|
|*| Dependencies to the chipset driver:
|*|
|*|   *	The chipset driver has to take care of CRT I/O relocation in color/mono
|*|	modes, thus we import the CRTI, CRTD variables holding the actual port.
|*|   *	The chipset driver is responsible to give permission to the extended
|*|	CRT registers before calling the GGI_ClockSetTiming() function.
|*|	Registers modified:
|*|		Name	     #defined by	bits possibly modified
|*|		MISC		(VGA)		MISC_CLOCK_MASK
|*|		CRTI		(VGA)		all (0xFF)
|*|		CRT_EXTMODE1	(S3)		CR42_ICD
\*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <asm/io.h>

#include "graphdev.h"
#include "kgi-module.h"
#include "S3/s3.h"

/*\
|*| Chipset driver imports
\*/

extern unsigned short CRTI;
extern unsigned short CRTD;

/*\
|*| Clock driver dependend #define's
\*/

#define CLK_VERSION "0.00 ALPHA"

#define	REFERENCE_CLK	14318180			/* Hz */
static unsigned fref = REFERENCE_CLK * 2;		/* Hz */

#define MIN_FVCO  50000000				/* Hz */
#define MAX_FVCO 120000000				/* Hz */

static struct GGI_Range ClockPar[2]={ {10000000, 86000000}, {0,0} };

#define LAST_INDEX 14
static int range[LAST_INDEX+1]=		/* VCO clock ranges in Hz */
 {  50000000, 51000000, 53200000, 58500000,
    60700000, 64400000, 66800000, 73500000,
    75600000, 80900000, 83200000, 91500000,
   100000000, 120000000, 120000000 };

/* This calculates the DCLK frequency for a given ratio and VCO clock.      */

#define F(P,Q,VCO)   VCO*(P+3)/(Q+2)

/* These give clock commands, that ain't used by the icd2061.               */

#define INVALIDCLK   -1
#define VGA_25MHZ    -2
#define VGA_28MHZ    -3

#define INVALIDP    0xFFFF
#define INVALIDQ    0xFFFF


static int clk_last_freq = 0;
static int clk_last_cmd  = INVALIDCLK;


/*\
|*| The lower 4 bits in CR42 are reserved for clock setup. For the icd2061a
|*| connected in 'standard' fashion we have the following meaning.
\*/

#define CR42_BOTH 0x03	/* data and clock	      */
#define CR42_DATA 0x02	/* icd2061a data pin 	      */
#define CR42_CLK  0x01	/* icd2061a clock pin	      */
#define	CR42_NONE 0x00  /* neither DATA nor CLK       */
#define CR42_ICD  0x03  /* bits in CR42 used by clock */


static void
SetDCLK(int ClockCmd, int freq)
{
  switch(ClockCmd)
   {
     case VGA_25MHZ:

       outb((inb(MISCr)& ~MISC_CLOCK_MASK) | MISC_25MHZ_CLK, MISCw);
       TRACE(printk("%s: standard VGA %i MHz set.\n",__FILE__, freq/1000));
       break;

     case VGA_28MHZ:

       outb((inb(MISCr)& ~MISC_CLOCK_MASK) | MISC_28MHZ_CLK, MISCw);
       TRACE(printk("%s: standard VGA %i MHz set.\n",__FILE__, freq/1000));
       break;

     default:
       outb((inb(MISCr) & ~MISC_CLOCK_MASK) | MISC_EXT_CLK, MISCw);

       {
	 /*\
	 |*| program icd2061 command word into the clock chip
	 |*| a command word is as follows:
	 |*|
	 |*|  0 -  6		q
	 |*|  7 -  9		div
	 |*| 10 - 16		p
	 |*| 17 - 21		clock index selector
	 |*| 22 - 23		register number
	 |*| clock rate is (REFERENCE_CLK*n) / (m* 2^div)
	 \*/

	 register unsigned char old;
	 register unsigned i;
	 unsigned m;

	 #define SET(x)	outb((x|old),CRTD)

	 outb(CRT_EXTMODE1, CRTI);	/* select EXTMODE1 register           */
	 old = inb(CRTD) & ~CR42_ICD;	/* get old values, dont touch others  */

	 SET(CR42_DATA);		/* do setup as described in [1]       */
	 SET(CR42_NONE);
	 SET(CR42_DATA);

	 for (i=0; i<6; i++) { SET(CR42_BOTH); SET(CR42_DATA); };
	 for (i=0; i<2; i++) { SET(CR42_NONE); SET(CR42_CLK ); };

	 for (i=0, m=1; i<24; i++, m+=m)
	   if (ClockCmd & m)
		SET(CR42_CLK ), SET(CR42_NONE), SET(CR42_DATA), SET(CR42_BOTH);
	   else
		SET(CR42_BOTH), SET(CR42_DATA), SET(CR42_NONE), SET(CR42_CLK );

	 SET(CR42_BOTH);
	 SET(CR42_DATA);
	 SET(CR42_BOTH);
	 SET(CR42_DATA);

	 #undef SET

	 TRACE(printk("%s: set DCLK = %i kHz\n",__FILE__, freq/1000));
       }
   }
}


static unsigned int
bestratio(unsigned *p, unsigned *q, int fvco)
{
  int mindiff = 0x7FFFFFFF;
  int diff;

  unsigned n;

  *p = INVALIDP;			/* make p and q invalid               */
  *q = INVALIDQ;

  for(n = 15; n < 72; n++)		/* n = q must be in [15,71]           */
   {
     int M = MAX_FVCO*n / fref - 3;
     int m = MIN_FVCO*n / fref - 3;

     if (m <   0) m =   0;   /* p = m must be in [0,127] */
     if (M > 128) M = 128;
     
     for(; m < M; m++)
      {
        diff = fvco - F(m,n,fref);

        if ((diff >= 0) && (mindiff > diff))
	 {
           mindiff = diff;
           *p = m;
           *q = n;
         }
      }
   }
  return mindiff;
}

static int 
icd_calcfreq(int *freq, unsigned char clk)
{
  unsigned m;
  int fvco;
  struct {
     unsigned p,q,m,i;
     double   delta_vco;
     unsigned delta_f;
   } best = { INVALIDP, INVALIDQ, 0,0, 1e10, 0xFFFFFFFF };
  double delta_vco;
  unsigned delta_f = 0xFFFFFFFF;


  if(*freq == clk_last_freq)
    return clk_last_cmd | (clk << 22);


  fvco = *freq;

  for(m=0; m < 8; m++)
   {
     int i,p,q;

     for(i = LAST_INDEX; (i > -1) && (fvco < range[i]); i--);

     if (i < 0)
      {
        fvco += fvco;
        continue;
      }

     if (i == LAST_INDEX)
       break;

     delta_vco = abs((fvco - (range[i]+range[i+1])/2.0) / (double)fvco);
     delta_f   = bestratio(&p, &q, fvco);

     if((delta_vco < best.delta_vco) || (delta_f < best.delta_f))
      {
        best.delta_f   = delta_f;
        best.delta_vco = delta_vco;
        best.p = p;
        best.q = q;
        best.m = m;
        best.i = i;
      }
   }

  if(best.p == INVALIDP)
    return INVALIDCLK;

  fvco = fref >> best.m;
  clk_last_freq = *freq = F(best.p, best.q, fvco);

  clk_last_cmd  = (best.i << 17) |	/* clock index ???                    */
		  (best.p << 10) |
                  (best.m <<  7) |	/* divider                            */
                  (best.q      );

  return clk_last_cmd | (clk << 22);
}

static int
calcfreq(int *freq)
{
  int newfreq  = *freq;
  int ClockCmd = INVALIDCLK;

  if (newfreq < ClockPar[0].min)
   {
     TRACE(printk("%s: requested DCLK too low (%i kHz)\n",__FILE__, newfreq/1000));
     return INVALIDCLK;
   }

  if (newfreq > ClockPar[0].max)
    newfreq = ClockPar[0].max;		/* meet limits                        */

  if (abs(25175000-newfreq) < 25175)	/* 25 MHz VGA frequency ?             */
    {
      newfreq  = clk_last_freq = 25175000;
      ClockCmd = clk_last_cmd  = VGA_25MHZ;
    }
   else
    if (abs(28322000-newfreq) < 28322)	/* 28 MHz VGA frequency ?             */
      {
        newfreq  = clk_last_freq = 28322000;
        ClockCmd = clk_last_cmd  = VGA_28MHZ;
      }
     else
      ClockCmd = icd_calcfreq(&newfreq, 1); /* 1 means clock generator 2 ??? */

  #ifdef DEBUG
   if(ClockCmd == INVALIDCLK)
     TRACE(printk("%s: invalid DCLK request (%i kHz)\n",__FILE__, *freq/1000));
    else
     TRACE(printk("%s: request for DCLK = %i kHz (did %i kHz)\n",__FILE__,\
	   *freq/1000, newfreq/1000));
  #endif

  if (ClockCmd != INVALIDCLK)
    *freq = newfreq;

  return ClockCmd;
}

static unsigned char MISC_save;

/*\
|*|     GGI clock driver interface
\*/

int
kgi_ClockInit(void)
{
  printk("IC Designs ICD 2061a clockchip driver v"CLK_VERSION".\n");
  MISC_save = inb(MISCr);
  switch(MISC_save & MISC_CLOCK_MASK)
   {
     case MISC_25MHZ_CLK:
     case MISC_28MHZ_CLK:
        break;  
     case MISC_EXT_CLK:
     default:
        printk("%s: Sorry, don't know a way to save the initial clock "\
               "value.\n%s: Please reconfigure your system to startup "\
               "with 80x25.\n",__FILE__,__FILE__);
        return FALSE; 
   } 
  TRACE(printk("%s: standard VGA mode, clocks saved.\n",__FILE__));
  return TRUE;
}

void
kgi_ClockDone(void)
{
  TRACE(printk("ICD 2061a clock driver removed.\n"));
}

void
kgi_ClockGetParameters(struct GGI_Range *params)
{
  *params = ClockPar[0];
}

int
kgi_ClockCheckTiming(struct GGI_Timing *TM)
{
  return calcfreq(&(TM->clock)) != INVALIDCLK;
}

int
kgi_ClockSetTiming(struct GGI_Timing *TM)
{
  int ClockCmd = calcfreq(&TM->clock);

  if (ClockCmd != INVALIDCLK)
    {
      SetDCLK(ClockCmd, TM->clock);
      return TRUE;
    }
   else
    {
      TRACE(printk("%s: failed to set DCLK (%i Hz)\n", __FILE__, TM->clock));
      return FALSE;
    }
}

int
kgi_ClockIoctl(struct inode *inode, struct file *file, unsigned int cmd,
	       unsigned long arg)
{
  return -1;
}
