/***
 *** VGA chip programming functions for SVGATextMode
 *** Written by Koen Gadeyne (kmg@barco.be)
 ***
 *** version : 0.2
 ***/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include "../include/vga_prg.h"

/*
 * Some functions to make life easier (?!)
 */

void Outbit_SEQ (int index, int bitno, int data)
{
  int bitmask = (1 << bitno);
  int onebit = ((data == 0) ? 0 : bitmask);
  Outb_SEQ (index, (Inb_SEQ(index) & ~bitmask) | onebit);
}

void Outbit_CRTC (int index, int bitno, int data)
{
  int bitmask = (1 << bitno);
  int onebit = ((data == 0) ? 0 : bitmask);
  Outb_CRTC (index, (Inb_CRTC(index) & ~bitmask) | onebit);
}

void Outbit_GR_CTL(int index, int bitno, int data)
{ 
  int bitmask = (1 << bitno);
  int onebit = ((data == 0) ? 0 : bitmask);
  Outb_GR_CTL(index, (Inb_GR_CTL(index) & ~bitmask) | onebit);
}

void Outb_ATR_CTL (int index, int data)
{
  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb (index & 0x1f, ATR_CTL_INDEX);
  outb ( data, ATR_CTL_DATA_W);
  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb ( (index & 0x1f) | 0x20, ATR_CTL_INDEX);
  outb ( data, ATR_CTL_DATA_W);
}

int Inb_ATR_CTL (int index)
{
  int res;

  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb (index & 0x1f, ATR_CTL_INDEX);
  res=inb(ATR_CTL_DATA_R);
  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb ( (index & 0x1f) | 0x20, ATR_CTL_INDEX);
  inb(ATR_CTL_DATA_R);
  return res;
}

void Outb_VGA_mem(int register_set, int data)
{
   switch(register_set)
  {
    case REGSET_MISC  : outb_p(data, VGA_MISC_W); break;
    default: printk("outb_VGA_indexed: unknown register set %d",register_set);
  }
}

int inb_VGA_mem(int register_set)
{
   switch(register_set)
  {
    case REGSET_MISC  : return(inb(VGA_MISC_R)); break;
    default: printk("inb_VGA_indexed: unknown register set %d",register_set);
  }
  return(0);
}

void Outb_VGA_indexed(int register_set, int reg_index, int data)
{
   switch(register_set)
  {
    case REGSET_CRTC  : Outb_CRTC(reg_index,data); break;
    case REGSET_SEQ   : Outb_SEQ(reg_index,data); break;
    case REGSET_ATRCTL: Outb_ATR_CTL(reg_index,data); break;
    case REGSET_GRCTL : Outb_GR_CTL(reg_index,data); break;
    default: printk("Outb_VGA_indexed: unknown register set %d",register_set);
  }
}

int inb_VGA_indexed(int register_set, int reg_index)
{
   switch(register_set)
  {
    case REGSET_CRTC  : return(Inb_CRTC(reg_index)); break;
    case REGSET_SEQ   : return(Inb_SEQ(reg_index)); break;
    case REGSET_ATRCTL: return(Inb_ATR_CTL(reg_index)); break;
    case REGSET_GRCTL : return(Inb_GR_CTL(reg_index)); break;
    default: printk("inb_VGA_indexed: unknown register set %d",register_set);
  }
  return(0);
}

/*****************************************************************************************************************************/

/*
 * Some more general functions (less low-level than separate register access)
 */

void unlock_CRTC(void)
{ Outbit_CRTC(0x11,7,0); }

void set_VERT_TOTAL (int vt)
{
  vt = vt - 2; /* must program actual value - 2 */
  Outb_CRTC (0x6, vt & 0xff);       /* bits 0..7 */
  Outbit_CRTC(0x7, 0, vt & 0x100);  /* bit 8 */ 
  Outbit_CRTC(0x7, 5, vt & 0x200);  /* bit 9 */
}

void set_MAX_SCANLINE (int msl)
{
  msl = msl - 1;
  Outb_CRTC (0x9, (Inb_CRTC (0x9) & 0xe0) | (msl & 0x1f));
}

void set_VRETRACE (int start, int end)
{
  Outb_CRTC (0x10, start & 0xff);                           /* start bits 0..7 */
  Outbit_CRTC (0x7, 2, start & 0x100);                      /* start bit 8 */
  Outbit_CRTC (0x7, 7, start & 0x200);                      /* start bit 9 */
  Outb_CRTC (0x11, (Inb_CRTC (0x11) & 0xf0) | (end & 0xf)); /* end */
}

void set_VDISPL_END (int vde)
{
  vde = vde - 1;
  Outb_CRTC (0x12, vde & 0xff);      /* bits 0..7 */
  Outbit_CRTC(0x7, 1, vde & 0x100);  /* bit 8 */
  Outbit_CRTC(0x7, 6, vde & 0x200);  /* bit 9 */
}

void set_VBLANK (int start, int end)
{
  start = start - 1;
  Outb_CRTC (0x15, start & 0xff);                       /* start bits 0..7 */
  Outbit_CRTC(0x7, 3, start & 0x100);                   /* start bit 8 */
  Outbit_CRTC(0x9, 5, start & 0x200);                   /* start bit 9 */
  Outb_CRTC (0x16, (start & 0xFF) - 1 + (end - start)); /* end */
}

void set_CURSOR (int start, int end)
{
  Outb_CRTC (0x0A,  (Inb_CRTC(0x0a) & 0xe0) | (start & 0x1f) );
  Outb_CRTC (0x0B,  (Inb_CRTC(0x0b) & 0xe0) | (end   & 0x1f) );
}

void set_HOR_TOTAL (int htot)
{
  Outb_CRTC(0, htot - 5);
}

void set_HOR_DISPL_END (int hend)
{
  Outb_CRTC(1, hend - 1);
}

void set_HSYNC (int start, int end)
{
  Outb_CRTC(4, start);
  Outb_CRTC(5, (Inb_CRTC(5) & 0xe0) | (end & 0x1f));
}

void set_HBLANK (int start, int end)
{
  Outb_CRTC(2, start);                                 /* start */
  Outb_CRTC(3, (Inb_CRTC(3) & 0xe0) | (end & 0x1f));   /* end bits 0..4 */
  Outbit_CRTC(5, 7, end & 0x20);                       /* end bit 5 */  
}

void set_SYNC_POLARITY(int hpol, int vpol)
{
   outb((inb(VGA_MISC_R) & 0xBF) | ((hpol < 0) ? 0x40 : 0x00) , VGA_MISC_W);
   outb((inb(VGA_MISC_R) & 0x7F) | ((vpol < 0) ? 0x80 : 0x00) , VGA_MISC_W);
}

void set_LOG_SCREEN_WIDTH(int width)
{
   Outb_CRTC(0x13, width);
}

void set_textmode()
{
   Outb_GR_CTL(6,Inb_GR_CTL(6) & 0xFE);
   Outb_ATR_CTL(16,Inb_ATR_CTL(16) & 0xFE);
}

void set_graphmode()
{
   Outb_GR_CTL(6,Inb_GR_CTL(6) | 0x01);
   Outb_ATR_CTL(16,Inb_ATR_CTL(16) | 0x01);
}

int set_charwidth(int width)
{
   switch(width)
   {
      case 8: Outbit_SEQ(1, 0, 1);
              SET_PIXELPAN(0);
              break;
      case 9: Outbit_SEQ(1, 0, 0);
              SET_PIXELPAN(8);
              break;
      default: return(1);
   }
   return(0);
}
