/* 
   Generic VGA functionality.

   Copyright (C) 1995 Andreas Beck - becka@hp.rz.uni-duesseldorf.de

   If you do any modifications, I would like you to send diffs to me
   to allow for collecting a more and more complete set of drivers and
   to improve existing ones.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/fcntl.h>
#include <linux/vt.h>
#include <sys/ioctl.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include "graphdev.h"
#include "kgi-module.h"
#include "vga_prg.h"
#include "vga_generic.h"

/*****************************************************************************
*** Save/Restore several chipset-registers                                 ***
*** This is used to set defaults for the modes,too.                        ***
*****************************************************************************/

char OpenFont[8192];

/**********/
/** CRTC **/
/**********/
void SaveCRTC(char *crtc,int anz)		/* anz was 128 */
{ int x;for(x=0;x<anz;x++) crtc[x]=Inb_CRTC(x); }

void RestoreCRTC(char *crtc,int anz)
{ int x;for(x=0;x<anz;x++) Outb_CRTC(x,crtc[x]); }

/***************/
/** Sequencer **/
/***************/
void SaveSEQ(char *seqregs,int anz)		/* anz was   5 */
{ int x;for(x=0;x<anz;x++) seqregs[x]=Inb_SEQ(x);}

void RestoreSEQ(char *seqregs,int anz)
{ int x;
  Outb_SEQ(0,1);
  for(x=1;x<anz;x++) Outb_SEQ(x,seqregs[x]);
  Outb_SEQ(0,3);
}

/*************************/
/** Graphics Controller **/
/*************************/
void SaveGR(char *gregs,int anz)			/* anz was   9 */
{ int x;for(x=0;x<anz;x++) gregs[x]=Inb_GR_CTL(x);}

void RestoreGR(char *gregs,int anz)
{ int x;for(x=0;x<anz;x++) Outb_GR_CTL (x,gregs[x]);}

/**************************/
/** Attribute Controller **/
/**************************/
void SaveATR(char *atregs,int anz)		/* anz was  21 */
{ int x;for(x=0;x<anz;x++) atregs[x]=Inb_ATR_CTL(x);}

void RestoreATR(char *atregs,int anz)
{ int x;for(x=0;x<anz;x++) Outb_ATR_CTL(x,atregs[x]);}


/*****************************************************************************
*** Setup standard VGA registers for a given timing.                       ***
*** Used as a general setup for all modes.                                 ***
*****************************************************************************/
void setup_VGA_timing(struct ggi_Timing *TM) 
{ int h; 

  unlock_CRTC();

  h=(inb(VGA_MISC_R)&0x0c)|0x23;	/* Leave ClockSelect */
  if (TM->xsyncpol<0) h|=0x40;
  if (TM->ysyncpol<0) h|=0x80;
  outb(h,VGA_MISC_W);		/* Set sync polarity */

  /*
   * CRTC Controller
   */
   Outb_CRTC(0,  (TM->xend       >> 3) - 5);
   Outb_CRTC(1,  (TM->xwidth     >> 3) - 1);

   Outb_CRTC(2,  (TM->xsyncstart >> 3) -1);
   Outb_CRTC(3, ((TM->xsyncend   >> 3) & 0x1F) | 0x80); /* SKEW ??? */

   Outb_CRTC(4,  (TM->xsyncstart >> 3));
   Outb_CRTC(5,(((TM->xsyncend   >> 3) & 0x20 ) << 2 )
     | (((TM->xsyncend >> 3)) & 0x1F));

   Outb_CRTC(6,  (TM->yend - 2) & 0xFF);
   Outb_CRTC(7,(((TM->yend - 2) & 0x100) >> 8 )
     | (((TM->ywidth - 1) & 0x100) >> 7 )
       | ((TM->ysyncstart & 0x100) >> 6 )
         | (((TM->ysyncstart) & 0x100) >> 5 )
           | 0x10
             | (((TM->yend - 2) & 0x200)   >> 4 )
               | (((TM->ywidth -1) & 0x200) >> 3 )
                 | ((TM->ysyncstart & 0x200) >> 2 ));

   Outb_CRTC(8,0x00);
   Outb_CRTC(9,((TM->ysyncstart & 0x200) >>4) | 0x40 |
                ((TM->ymagnify-1)&0x1f) );
   /* if (mode->Flags & V_DBLSCAN) new->CRTC[9] |= 0x80;*/

   Outb_CRTC(10,0x00);
   Outb_CRTC(11,0x00);
   Outb_CRTC(12,0x00);
   Outb_CRTC(13,0x00);
   Outb_CRTC(14,0x00);
   Outb_CRTC(15,0x00);

   Outb_CRTC(16, TM->ysyncstart & 0xFF);
   Outb_CRTC(17,(TM->ysyncend   & 0x0F) | 0x20);
   Outb_CRTC(18,(TM->ywidth -1) & 0xFF);

   /* logical screen width - THIS MIGHT NEED OVERRIDE after return ! */
   Outb_CRTC(19,TM->tr.xvirtual>>2);  

   Outb_CRTC(20,0x1f);		/* underline position */
   Outb_CRTC(21, TM->ysyncstart & 0xFF);
   Outb_CRTC(22,(TM->ysyncstart +1) & 0xFF);
   /* 23 Must be set according to mode ! */
   Outb_CRTC(24,0xFF);
}

/* This checks generic requirements any standard VGA textmode must meet */
int vga_CheckText(struct ggi_Timing *TM)
{ int x;

  if (TM->tr.xtextgrid< 8||TM->tr.xtextgrid> 9) 
    { TM->tr.xtextgrid= 8;return(-ENOMODESUP_CHIP);}
  if (TM->tr.ytextgrid< 1||TM->tr.ytextgrid>32) 
    { TM->tr.ytextgrid=16;return(-ENOMODESUP_CHIP);}	/* Resonable default */

  if (TM->tr.xvisible &         0x0f)   
    { TM->tr.xvisible&=0xfffffff0; }	/* This must be a multiple of 8*2 */
  if ((x=TM->tr.yvisible%TM->tr.ytextgrid))  
    { TM->tr.yvisible-=x; }		/* This must be a multiple of ytextgrid */

  TM->tr.xvirtual=TM->tr.xvisible/4;
  TM->tr.yvirtual=TM->tr.yvisible/TM->tr.ytextgrid;

  return(0);
}

void vga_SetText(struct ggi_Timing *TM)
{ 
  setup_VGA_timing(TM);

  Outb_CRTC(23,0xA3);

  set_MAX_SCANLINE (TM->tr.ytextgrid);
  set_CURSOR(TM->tr.ytextgrid-2,TM->tr.ytextgrid-1); /* cursor start/end */
  set_charwidth(TM->tr.xtextgrid);

  { short int *ptr;int x;       /* erase text ram */
    for(x=0x8000,ptr=(short int *)0xb8000;x>0;x--) *ptr++=0x0720; }
}

void vga_Set4bit(struct ggi_Timing *TM)
{ char SEQ [5]={0x03,0x01,0x0f,0x00,0x06};
  char GR  [9]={0x00,0x0f,0x00,0x00,0x00,0x03,0x05,0x00,0xFF};
  char ATR[21]={0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
                0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
                0x01,0x00,0x0f,0x00,0x00};

  RestoreSEQ(SEQ,5);
  RestoreGR (GR , 9);
  RestoreATR(ATR,21);

  setup_VGA_timing(TM);
  Outb_CRTC(19,TM->tr.xvirtual>>4);
  Outb_CRTC(23,0xE3);
  memset((void *)0xa0000,0xff,0x10000);  /* erase graphics RAM */
}

int _getsetfont_hw(char *buffer,int mode)
{ char SEQ [5]={0x03,0x01,0x04,0x00,0x07};
  char GR  [9]={0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x0F,0xFF};
  char oldSEQ[5],oldGR[9];

  SaveSEQ(oldSEQ,5);SaveGR(oldGR,9);
  RestoreSEQ(SEQ,5);RestoreGR(GR,9);

  if (mode) memcpy((void *)0xa0000,buffer,8192);
  else      memcpy(buffer,(void *)0xa0000,8192);

  RestoreSEQ(oldSEQ,5);RestoreGR(oldGR,9);
  return(0);
}

/* FIXME - this is a dummy right now ... */
int vesa_pwrsave(int mode)
{ int i;
  struct ggi_Timing gt;

  gt.tr=CurrState->ModeNow;

  if ((i=kgi_CheckChipsetTiming(&gt,CMD_NONE))) return(i);
  
  if (!CurrState->bgmode)
  { switch(mode)
    { case PWR_OFF    :gt.xsyncstart=gt.xend;
      case PWR_SUSPEND:gt.ysyncstart=gt.yend;break;
      default	     :break;
    }
    if (kgi_SetChipsetTiming  (&gt)==-1) return -ENOMODESUP_CHIP; }

  return 0; }
