/* 
   Generic VGA Chipset-Driver.
   Copyright (C) 1995 Andreas Beck - becka@hp.rz.uni-duesseldorf.de

   If you do any modifications, I would like you to send diffs to me
   to allow for collecting a more and more complete set of drivers and
   to improve existing ones.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>

/* THIS IS FOR TESTING PURPOSES ! COMMENT OUT FOR RELEASES ! */
#undef suser
#define suser() 1

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/fcntl.h>
#include <linux/vt.h>
#include <sys/ioctl.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include "graphdev.h"
#include "kgi-module.h"
#include "vga_prg.h"
#include "vga_generic.h"
#include "generic.h"

char kernel_version[]= UTS_RELEASE;

/*****************************************************************************
*** Chipset characteristics                                                ***
*****************************************************************************/

struct ggi_ChipParams ChipPar = 
	{	256*1024,		/* 256KB RAM */
		 2048,
		 1024,
		"Generic",
		"VGA",
		GT_TEXT|GT_4BIT_PLANES|GT_8BIT,	/* all that VGA has */
	};

/*****************************************************************************
*** Enable S3 extended register access - left in for testing               ***
*****************************************************************************/
void unlock_s3(void)
{ Outb_CRTC(0x38,0x48);			/* unlock CR30-CR3F */
  Outb_CRTC(0x39,0xA5);			/* unlock CR40-CRFF */
  Outb_CRTC(0x35,Inb_CRTC(0x35)&0xcf);	/* & 0xf0 ??? SVS3  */
}

/*****************************
*** Check if this is a VGA ***
*****************************/
static int Detect_VGA(void)
{ int ov,nv;
  ov=Inb_ATR_CTL(0x14);		/* Only VGA has this register */
  Outb_ATR_CTL(0x14,ov^0x0f);	/* Change it */
  nv=Inb_ATR_CTL(0x14);
  Outb_ATR_CTL(0x14,ov);	/* Restore it */
  
  if (nv!=(ov^0x0f)) return(0);	/* No VGA ! */

  return 1;
}


/*****************************************************************************
*** Check/Setup a Text-Mode of the given size                              ***
*****************************************************************************/
int CheckText(struct ggi_Timing *TM,int command)
{ int x;

  if ((x=vga_CheckText(TM))) return(x);

  TM->clock=75000000;/* Max for text - normal VGA will have 28.3 MHz Clock only */
  TM->xwidth=TM->tr.xvisible;
  TM->ywidth=TM->tr.yvisible;
  
  TM->ymagnify=1;
  
  /* We can do nothing against RAMDAC objections */
  if ((x=kgi_CheckRamdacTiming(TM,CMD_NONE))) return(x); 

  /* We can do nothing against Monitor objections except asking for
     a less perfect mode */
  if ((x=kgi_CheckMonitorTiming(TM,CMD_MON_PERFECT)))
  x=kgi_CheckMonitorTiming(TM,CMD_MON_SOME);

  return x;
}  

void ResetCard(void)
{ static char SEQ [5]={0x03,0x01,0x03,0x00,0x02};
  static char GR  [9]={0x00,0x00,0x00,0x00,0x00,0x10,0x0e,0x00,0xFF};
  static char ATR[21]={0x00,0x01,0x02,0x03,0x04,0x05,0x14,0x07,
  			0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
	  		0x0c,0x00,0x0f,0x00,0x00};
  static char CRTC[25]= { 0x82,0x63,0x69,0x9d,0x6a,0x9d,0x88,0xf0
			  ,0x00,0x6f,0x0e,0x0f,0x00,0x00,0x00,0x64
			  ,0x51,0x22,0x4f,0x32,0x1f,0x51,0x52,0xa3
 			  ,0xff };
  unlock_CRTC();
  unlock_s3();			/* FIXME */
  RestoreSEQ(SEQ,5);		/* Reset CRTC and Sequencer to defined state */
  RestoreCRTC(CRTC,25);
  RestoreGR (GR , 9);
  RestoreATR(ATR,21);

}

int SetText(struct ggi_Timing *TM)
{ ResetCard();
  kgi_SetClockTiming(TM);
  vga_SetText(TM);
  return(0);
}  

/*****************************************************************************
*** Check/Setup a 4bit graphics-mode of the given size                     ***
*****************************************************************************/
int Check4bit(struct ggi_Timing *TM,int olderr)
{ int x,clksave;

  if (TM->tr.xvirtual*TM->tr.yvirtual/8>64*1024) 
  { TM->tr.xvirtual=800;TM->tr.yvirtual=600;return(-ENOMODESUP_CHIP); }

  if (TM->tr.xvisible&0x07) 
    TM->tr.xvisible&=0xfffffff8;	/* This must be a multiple of 8 */

  TM->clock=86000000;	/* FIXME ... */
  TM->ymagnify=1;

  /* We can do nothing against RAMDAC objections */
  if ((x=kgi_CheckRamdacTiming(TM,CMD_NONE))) return(x); 

  clksave=TM->clock;	/* RAMDAC might have limited it */

  while(1)
  { TM->xwidth=TM->tr.xvisible;
    TM->ywidth=TM->tr.yvisible*TM->ymagnify;
    switch((x=kgi_CheckMonitorTiming(TM,CMD_MON_PERFECT)))
    { case MR_Y_TOO_SMALL:TM->clock=clksave;
    			  if (TM->ymagnify++<32) break;	/* Retry with more magn. */
    			  TM->ymagnify=1;
    			  x=kgi_CheckMonitorTiming(TM,CMD_MON_SOME);
    			  return x;
      default:return x;	/* We can do nothing about it */
    } 
  }  
  return(-ENOMODESUP_CHIP);
}  

int Set4bit(struct ggi_Timing *TM)
{ ResetCard();
  kgi_SetClockTiming(TM);
  vga_Set4bit(TM);
  return(0);
}  


/***************************************************************************/
/* *** ***        I n t e r f a c e   d e f i n i t i o n s        *** *** */
/***************************************************************************/
void kgi_GetChipsetParameters(struct ggi_ChipParams *CP)
{ *CP=ChipPar; }

int kgi_CheckChipsetTiming(struct ggi_Timing *TM,int olderr)
{ if ((ChipPar.graphtypes&TM->tr.graphtype)==0) return(-1);
  switch(TM->tr.graphtype)
  { case GT_TEXT	: return(CheckText(TM,olderr));
    case GT_4BIT_PLANES	: return(Check4bit(TM,olderr));
    default     	: return(-1);
  }
}  

int kgi_SetChipsetTiming(struct ggi_Timing *TM)
{ if ((ChipPar.graphtypes&TM->tr.graphtype)==0) return(-1);
  switch(TM->tr.graphtype)
  { case GT_TEXT	: return(SetText(TM));
    case GT_4BIT_PLANES	: return(Set4bit(TM));
    default   		: return(-1);
  }
}

/****************************/
/*** Set start of display ***/
/****************************/
void hw_setdispstart(int dx,int dy)
{ int x;
  if (CurrState->bgmode) return;
    
  x=dx+dy*CurrState->ModeNow.xvirtual;
  
  Outb_CRTC(0x0d,(x >>  2) & 0xff);    /*  sa2-sa9 */
  Outb_CRTC(0x0c,(x >> 10) & 0xff);    /* sa10-sa17 */

  Outb_ATR_CTL(0x13,(Inb_ATR_CTL(0x13)&0xf0)|((x & 3)<<1));
}

int setinforec(void)
{ CurrState->drawinfo=*(void **)parameters;return(0); }


/*****************************************************************************
****  Graphics  Device  Interface  Funtions                               ****
*****************************************************************************/

int kgi_ChipsetIoctl(struct inode *inode, struct file *file, \
        unsigned int cmd, unsigned long arg)
{ DEV_NUM_SETUP;

  switch(cmd) { 
	case GRAPH_GETVT       :COPYPAR(sizeof(void *));
				return(getvt());
        case GRAPH_SETMODE     :COPYPAR(sizeof(void *));
    		  		return(setmode());
        case GRAPH_GETMODE     :COPYPAR(sizeof(void *));
    				return(getmode());
        case GRAPH_KEEPMODE    :if (suser()) return(keepmode());
    				else return(-EPERM);
        case GRAPH_SETDRAWINFO :COPYPAR(sizeof(void *));
        			return(setinforec());

        case GRAPH_SETVISWIN   :COPYPAR(2*sizeof(int));
    				return(setdispstart());
        case GRAPH_GETVISWIN   :COPYPAR(2*sizeof(int));
    				return(getdispstart());

        case GRAPH_GETSTATESIZE:COPYPAR(sizeof(int *));
    			        return(getstatesize());
        case GRAPH_SAVESTATE   :COPYPAR(sizeof(void *));
    			        return(savestate());
        case GRAPH_RESTORESTATE:COPYPAR(sizeof(void *));
    			        return(restorestate());

        case GRAPH_BGMODE      :return(bgmode(1));
        case GRAPH_FGMODE      :return(bgmode(0));

        case GRAPH_SETDEFFONT  :if (!suser()) return(-EPERM);
        			COPYPAR(sizeof(void *));
        			return(setdeffont());
        case GRAPH_GETDEFFONT  :COPYPAR(sizeof(void *));
				return(getdeffont());

	case GRAPH_TEST	       :return(0);	/* For timing tests */

	case GRAPH_DRAWPIXEL   : /* Faster than doing it here */
	case GRAPH_GETPIXEL    :
	case GRAPH_PUTPIXEL    :
	case GRAPH_DRAWHLINE   :
	case GRAPH_GETHLINE    :
	case GRAPH_PUTHLINE    :
        case GRAPH_DRAWBOX     :
	case GRAPH_GETBOX      :
	case GRAPH_PUTBOX      :
        case GRAPH_FILLSCREEN  :return(-ENODRVSUP_ALWAYS_MMAP);

        default		       :return -ENODRVSUP_ALWAYS_LOWER;
   }
   /* We should never get here ... */
   return -ENODRVSUP_ALWAYS_CANT;
}                                                      

static int nopage_deadlock_check=0;

#define MAXMP 100

static struct Map { 
		struct task_struct *tsk;
		unsigned long lastmap; } map[MAXMP],*mp;

static int mapcnt=0;		

void kgi_GraphmapClose(struct vm_area_struct * area)
{ nopage_deadlock_check=0;
  for(;mapcnt;mapcnt--) /* FIXME ... delete only those mappings that belong */
  { mp=&map[mapcnt-1];	/* to the closing task - not too critical ... */
    fast_unmap_page_range(mp->tsk,mp->lastmap, 0x10000); }
}

unsigned long kgi_GraphmapNopage(struct vm_area_struct * area, unsigned long address,
	unsigned long page, int no_share)
{
	static pte_t *last_pte=NULL;
	static struct task_struct *tsk;
	unsigned long mappedpage=-1;
	
	VM_DEV_NUM_SETUP;
        	
	if (CurrState->bgmode) {	/* A backgrounded process accesses graphics mem ! */
	  area->vm_task->blocked &= ~(1<<(SIGSEGV-1));
	  send_sig(SIGSEGV, area->vm_task, 1);
        }

	if (nopage_deadlock_check && !pte_young(*last_pte) && tsk==area->vm_task) {
	  area->vm_task->blocked &= ~(1<<(SIGBUS-1));
	  send_sig(SIGBUS, area->vm_task, 1);
        }

        address -= area->vm_start;
	nopage_deadlock_check=!(address&0xffff);	/* fault at first byte of VGA page ? */
	address &= 0xffff0000;				/* which page to map */

	if (mappedpage!=address)			/* delete all mapping for old page */
 	  for(;mapcnt;mapcnt--) 
	  { mp=&map[mapcnt-1];
	    fast_unmap_page_range(mp->tsk,mp->lastmap, 0x10000); }

	Outb_GR_CTL(4,(address>>16)&3);	/* Read Map select */
	Outb_GR_CTL(0,address>>16);	/* Color select */

	address+= area->vm_start;
        map[mapcnt].tsk=area->vm_task;
        map[mapcnt].lastmap=address;
        if (mapcnt++>=MAXMP) {mapcnt--;printk("Whee - to many mappings - going mad !\n");}
        
	/* we want to map the graphics-area thus 0xa0000,64K */
        fast_remap_page_range(area->vm_task,address, 0xa0000, 0x10000, area->vm_page_prot);

        if (nopage_deadlock_check)
          last_pte = pte_offset( pmd_offset( pgd_offset(tsk=area->vm_task, address), address), address);

	invalidate();

        ++area->vm_task->mm->min_flt;	/* do some statistics */
        
	return((unsigned long)NULL);
}

int kgi_ChipsetInit(void)
{ 
  if (Detect_VGA()==0)
  { printk("No VGA found !\n");return(-EIO); }
  
  unlock_s3();	/* FIXME - not needed for release */
  
  printk("Manufacturer : %s\n",ChipPar.manufact);
  printk("Model        : %s\n",ChipPar.model);
  printk("RAM / Maxsize: %d Bytes (%d x %d) \n",
         ChipPar.ramsize,
         ChipPar.xmaxdots,
         ChipPar.ymaxdots);

  _getsetfont_hw(OpenFont,0);
  return(0);				/* All went OK - install */
}

void kgi_ChipsetCleanup(void) 
{ }