/*
 * This file is part of fb, the frame buffer device, a grafics card driver for
 *                                linux.
 *
 *      Copyright (C) 1995 Michael Weller (eowmob@exp-math.uni-essen.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Michael Weller (eowmob@exp-math.uni-essen.de or
 * mat42b@aixrs1.hrz.uni-essen.de) Heiderhoefen 116b,
 * D 46049 Oberhausen, Germany.
 */

/*
 * needs /dev/fb w/o dead lock detection.
 */

#include "fb.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/vt.h>
#include <linux/kd.h>
#include <sys/mman.h>
#include <signal.h>
#include <sys/resource.h>
#include <time.h>
#include <asm/signal.h>


#define SYS_ERROR(str) { perror(str); exit(2); }

int tty_fd;

struct rusage start_usage;
struct timeval start_time;

long long time2ll(struct timeval time) {
	return ((long long)time.tv_sec) * 1000000LL + time.tv_usec;
}

void sigint(void) {
	struct rusage cur_usage;
	struct timeval cur_time;
	double overall;
	unsigned long num_faults;

	gettimeofday(&cur_time, NULL);
	getrusage(RUSAGE_SELF, &cur_usage);

	printf("Faults: %ld (%ld before entering page dead lock)\n", cur_usage.ru_minflt - start_usage.ru_minflt,
			start_usage.ru_minflt);
	num_faults = cur_usage.ru_minflt - start_usage.ru_minflt;

	printf("User time: %g (%g before entering page dead lock)\n",
				(time2ll(cur_usage.ru_utime) - 
						time2ll(start_usage.ru_utime)) / (double) 1e6,
						time2ll(start_usage.ru_utime) / (double) 1e6);

	printf("System time: %g (%g before entering page dead lock)\n",
				 (time2ll(cur_usage.ru_stime) - 
						time2ll(start_usage.ru_stime)) / (double) 1e6,
						time2ll(start_usage.ru_stime) / (double) 1e6);

	puts("The before values were already subtracted.");
	overall = (time2ll(cur_time) - time2ll(start_time)) / (double) 1e6;
	printf("Real time used: %g\n", overall);

	printf("Efficiency: %g usecs per fault or %g faults per sec.\n",
		(overall / num_faults) * (double)1e6, num_faults / overall);

	printf("This is about %d machine cycles per fault on my 486/50\n",
		 (int) ((overall / num_faults) * (double)50e6));
		
		

	fflush(stdout);

	exit(0);
}

void beep(unsigned short freq, unsigned short ms)
{
	static err = 0;

	if(err) return;
		
	if (ioctl(tty_fd, KDMKTONE, ((unsigned long)ms<<16) + freq)) {
		perror("KDMKTONE on tty failed");
		err++;
	}
}

void pr_context(struct sigcontext_struct *context) {
	int i;
	char *ptr;

	printf("cs  =     %04x\t", context->cs);
	printf("ds  =     %04x\t", context->ds);
	printf("es  =     %04x\n", context->es);
	printf("fs  =     %04x\t", context->fs);
	printf("gs  =     %04x\t", context->gs);
	printf("ss  =     %04x\n", context->ss);

	printf("edi = %08lx\t", context->edi);
	printf("esi = %08lx\t", context->esi);
	printf("ebp = %08lx\t", context->ebp);
	printf("esp = %08lx\n", context->esp);
	printf("ebx = %08lx\t", context->ebx);
	printf("edx = %08lx\t", context->edx);
	printf("ecx = %08lx\t", context->ecx);
	printf("eax = %08lx\n", context->eax);
	printf("err = %08lx\t", context->err);
	printf("eip = %08lx\n", context->eip);

	printf("eflags = %08lx\t", context->eflags);
	printf("trapno = %08lx\t", context->trapno);
	printf("esp_at_signal = %08lx\n", context->esp_at_signal);

	printf("i387 = %08lx\t", context->i387);
	printf("oldmask = %08lx\t", context->oldmask);
	printf("cr2 = %08lx\n", context->cr2);

	puts("code sequence:");
	for(ptr = (char *)context->eip, i = 0; i < 26; i++)
		printf("%02x ", (*ptr++) & 0xff);
	puts("");
}

void sigsegv(int i, struct sigcontext_struct context) {
	printf("SIGSEGV:\n");
	pr_context(&context);
}

void sigbus(int i, struct sigcontext_struct context) {
	printf("SIGBUS:\n");
	pr_context(&context);
}

int main(int argc, char *argv[]) {
	int fd;
	unsigned char *base;

	struct sigaction action;

	action.sa_handler = (void *)sigsegv;
	sigemptyset(&action.sa_mask);
	action.sa_flags = SA_ONESHOT;

	sigaction(SIGSEGV, &action, NULL);

	action.sa_handler = (void *)sigbus;
	sigemptyset(&action.sa_mask);
	action.sa_flags = SA_ONESHOT;

	sigaction(SIGBUS, &action, NULL);

	action.sa_handler = (void *)sigint;
	sigemptyset(&action.sa_mask);
	action.sa_flags = SA_ONESHOT;

	sigaction(SIGINT, &action, NULL);

	if ((tty_fd = open("/dev/tty", O_RDWR)) < 0)
		SYS_ERROR("Opening: /dev/tty");

	if ((fd = open("/dev/fb", O_RDWR)) < 0)
		SYS_ERROR("Opening: /dev/fb");

	base = mmap(NULL, 1024*1024, PROT_READ|PROT_WRITE, MAP_SHARED, fd, (off_t)0);
	if (base == (void *)-1)
		SYS_ERROR("mmap");

	fflush(stdout);

	getrusage(RUSAGE_SELF, &start_usage);
	gettimeofday(&start_time, NULL);

	/* cause dead lock: */
	*(long *)(void *)(base + 0xffff) = 32;
	return 0;
}
