/*
 * This file is part of fb, the frame buffer device, a grafics card driver for
 *                                linux.
 *
 *     Copyright (C) 1995 Pascal Haible (haible@IZFM.Uni-Stuttgart.DE)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Dipl.-Ing. Pascal Haible (haible@IZFM.Uni-Stuttgart.DE)
 *
 */

/* visa.c */

#include <linux/autoconf.h>
#include <linux/module.h>
#include <linux/types.h>	/* NULL */
#include <linux/kernel.h>	/* printk() */
#include <linux/mm.h>
#include <asm/io.h>
#include <linux/errno.h>
#include "fb.h"
#include "visa.h"
#include "vga.h"
#include "chipset.h"

#define VGA_MAPS 5

#define VISA_INDEX	0x3B4
#define VISA_DATA	0x3B5
#define VISA_STATUS	0x3BA
#define VISA_MODE	0x3B8
#define VISA_CONF	0x3BF

/* #define DEBUG */

#ifdef DEBUG
#define DEB(__a) __a
#else
#define DEB(__a)
#endif

/* variables */

/* this kind of initialization is gcc specific and not portable, but       */
/* who would want to compile a linux kernel module with an other compiler? */

fb_info_t chip_info = {
driver_name:	"visa",
chipset_name:	"visa",
copyright:	"(c) 1994 Pascal Haible",
info:		"VISA high resolution text/graphics display subsystem",
rev_major:	0,
rev_minor:	1,
rev_tiny:	0,
totalmem:	256*1024,
memtype:	0,			/* 0: DRAM, 1: VRAM */
number_modes:	1,
default_mode:	0,
};

fb_mode_t __chip_modesinfo[1] = {
{
num:			0,
xsize:			1280,
ysize:			1024,
xvirt:			1280,
yvirt:			1024,
xdim:			2048,
ydim:			1024,
scanlinelength:		2048,
vidmem:			256*1024,
vidmem2mmap:		256*1024,
mmapmodel:		1,
bitordering:		76543210,
byteordering:		1234,
memorymodel:		0,
numwindows:		1,
#ifdef VISA_32K
windowsize:		(1<<15),
windowshift:		15,
numbanks:		8,
#else
windowsize:		(1<<16),
windowshift:		16,
numbanks:		4,
#endif
colormodel:		0,
bpp:			1,
bpp_used:		1,
bitsRGB:		0,
sizered:		0,
posred:			0,
sizegreen:		0,
posgreen:		0,
sizeblue:		0,
posblue:		0,
sizeres:		0,
posres:			0,
black:			0,
white:			1,
pixelclock:		0,
horizontalrefresh:	0,
verticalfrequency:	0,
} };

fb_mode_t *chip_modesinfo=__chip_modesinfo;

unsigned long chip_mode = 0;


void chip_do_bank0r(int nr);
void chip_do_bank0w(int nr);
void chip_do_bank0rw(int nr);

/* We rely on these being inited with zeros  */
static struct mapping_desc maps_0r[VGA_MAPS], maps_0w[VGA_MAPS];

struct chip_bank_desc chip_banks[2] = {
	/* VGA bank 0, read only, */
	{0xB0000UL, CHIP_BANK_SIZE / PAGE_SIZE, 16, 0, chip_do_bank0r, chip_do_bank0rw,
		maps_0r, maps_0r + VGA_MAPS, maps_0r},
	/* VGA bank 0, write only, */
	{0xB0000UL, CHIP_BANK_SIZE / PAGE_SIZE, 16, 0, chip_do_bank0w, NULL,
		maps_0w, maps_0w + VGA_MAPS, maps_0w},
	};

short chip_num_banks = 2;

/* assume 80x25 text mode with
 - text mode,
 - screen enabled,
 - text blinking enabled,
 - page 0 selected,
 - graphics setting disabled,
 - page 1 disabled.
*/
static struct chip_regs_s current_regs =
	{ 0x50, /*R1*/
	  0x00, /*R12*/
	  0x00, /*R13*/
	  0x00, /*R14*/
	  0x00, /*R15*/
	  0x28, /*mode*/
	  0x00  /*conf*/};

unsigned char *chip_test_chipset(void)
{
	return "VISA";
}

int chip_init(void)
{
	init_mapdesc(maps_0r, VGA_MAPS);
	init_mapdesc(maps_0w, VGA_MAPS);

	return 0;
}

void chip_cleanup(void)
{
}

void chip_leave(void)
{
}

unsigned long chip_savestate_getsize(void)
{
	return sizeof(struct chip_regs_s);
}

static void chip_saveregs(struct chip_regs_s *regs) {
	DEB(printk(KERN_DEBUG "chip_saveregs()\n");)
	regs->R1 = current_regs.R1;
	regs->R12 = current_regs.R12;
	regs->R13 = current_regs.R13;
	regs->R14 = current_regs.R14;
	regs->R15 = current_regs.R15;
	regs->conf = current_regs.conf;
	regs->mode = current_regs.mode;
}

static void chip_restoreregs(struct chip_regs_s *regs) {
	DEB(printk(KERN_DEBUG "chip_restoreregs()\n");)
	OUTB(VISA_INDEX,1);
	OUTB(VISA_DATA,current_regs.R1=regs->R1);
	OUTB(VISA_INDEX,12);
	OUTB(VISA_DATA,current_regs.R12=regs->R12);
	OUTB(VISA_INDEX,13);
	OUTB(VISA_DATA,current_regs.R13=regs->R13);
	OUTB(VISA_INDEX,14);
	OUTB(VISA_DATA,current_regs.R14=regs->R14);
	OUTB(VISA_INDEX,15);
	OUTB(VISA_DATA,current_regs.R15=regs->R15);

	OUTB(VISA_CONF,current_regs.conf=regs->conf);
	OUTB(VISA_MODE,current_regs.mode=regs->mode);
}

void chip_savestate(void *state, void *font)
{
	chip_saveregs((struct chip_regs_s *)state);
	if (font) {
		vga_savetextfontdata(font);
		chip_restoreregs((struct chip_regs_s *)state);
	}
}

void chip_restorestate(void *state, void *font)
{
	chip_restoreregs((struct chip_regs_s *)state);
	if (font) {
		vga_savetextfontdata(font);
		chip_restoreregs((struct chip_regs_s *)state);
	}
}

int chip_graphics(void)
{
	struct chip_regs_s graphics_regs =
	{ 0x50, /*R1*/
	  0x00, /*R12*/
	  0x00, /*R13*/
	  0x00, /*R14*/
	  0x00, /*R15*/
	  0x02, /*mode*/
#ifdef VISA_32K
	  0x01  /*conf*/};
#else
	  0x03  /*conf*/};
#endif
	DEB(printk(KERN_DEBUG "chip_graphics()\n");)
	chip_restorestate(&graphics_regs, NULL);
	return 0;
}

static int chip_bank0r = 0;
static int chip_bank0w = 0;

void chip_do_bank0rw(int nr)
{
	DEB(printk(KERN_DEBUG "chip_do_bank0rw(%i)\n",nr);)
	chip_bank0r = nr;
	chip_bank0w = nr;
#ifdef VISA_32K
	OUTB(VISA_CONF, current_regs.conf=0x01|(nr<<4));
#else
	OUTB(VISA_CONF, current_regs.conf=0x03|(nr<<5));
#endif
}

void chip_do_bank0r(int nr)
{
	DEB(printk(KERN_DEBUG "chip_do_bank0r(%i)\n",nr);)
	chip_bank0r = nr;
	chip_bank0w = nr;
#ifdef VISA_32K
	OUTB(VISA_CONF, current_regs.conf=0x01|(nr<<4));
#else
	OUTB(VISA_CONF, current_regs.conf=0x03|(nr<<5));
#endif
}

void chip_do_bank0w(int nr)
{
	DEB(printk(KERN_DEBUG "chip_do_bank0w(%i)\n",nr);)
	chip_bank0r = nr;
	chip_bank0w = nr;
#ifdef VISA_32K
	OUTB(VISA_CONF, current_regs.conf=0x01|(nr<<4));
#else
	OUTB(VISA_CONF, current_regs.conf=0x03|(nr<<5));
#endif
}

int chip_blank(void)
{
	DEB(printk(KERN_DEBUG "chip_blank()\n");)
	OUTB(VISA_MODE, current_regs.mode&=~0x08);
	return 0;
}

int chip_unblank(void)
{
	DEB(printk(KERN_DEBUG "chip_unblank()\n");)
	OUTB(VISA_MODE, current_regs.mode|=0x08);
	return 0;
}

int chip_setorigin(fb_origin_t *origin)
{
	return -ENXIO;
}

int chip_setcolormap(fb_cmap_t *cmap)
{
	return -ENXIO;
}

int chip_getcolormap(fb_cmap_t *cmap)
{
	return -ENXIO;
}

int chip_choose_bank(int mode, int address) {
	if(mode & VM_WRITE) {
		if(mode & VM_READ)
			return 0 | NEEDS_NEXT_BANK;
		else    return 1;
	} else {
		return 0;
	}
}

void chip_test_test(void)
{
}
