/*  double/dbck.c	checks consistency of DouBle file  20 Dec 1993
 *
 *  Copyright (C) 1994 Jean-Marc Verbavatz  <verbavatz@dsvidf.cea.fr>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <errno.h>
#include <linux/fs.h>
#include <linux/double.h>

struct dble_device db;
extern struct db_header dbh;

unsigned char memory[65536];	/* for block check */

int options = 0;

void read_opt(char *name, char *s)
{
	while(*s) {
		switch(*s) {
			case 'a': options |= 1;
				  break;
			case 'v': options |= 2;
				  break;
			case 't': options |= 4;
				  break;
			case 'r': options |= 8;
				  break;
			case 0:	  return;
			default:  usage(name);
		}
		s++;
	}
}

/* Check consistency of block n */
int get_block(struct dble_device *db, u_long n, int options)
{
u_long *p;
int i,j;
struct BAT *b;
u_char buffer[DB_MAXSZ], buffer1[DB_MAXSZ], *s;

if(!get_BAT(db, n)) return -1;
b = (struct BAT *)(db->BAT_block + db->BATsize*(n%(db->isize/db->BATsize)));

p=&b->iblock;
s = buffer;
for(i=j=0; i<db->ratio; i++, j++, p++) {
	if(*p==0) { /* Empty; check other iblocks */
		while(++i<db->ratio)
			if(*(++p)!=0) return -2;	/* Error */
		if(j==0)
			return 0;			/* Unused */
		break;
	}
	if(get_bit(db, *p)) return -3;		/* Error; not in bitmap */
	db_rw_iblock(db, READ, *p, s);
	s += db->isize;
}
/* BAT entry not empty */
if(options & 4)	{	/* Check blocks */
	switch(b->type) { /* Check data	*/
		case 0:	if(i<db->ratio) return -4;
			i = db->osize;
			break;
		case 1: i = RLZW((code *) buffer, buffer1, db->osize);
			break;
		case 10: i = rpred(buffer, buffer1, db->osize);
			 break;
		case 101:
			 break;
		case 11:
		case 102: i = rLZV(buffer, buffer1, db->osize);
			 break;
		case 12:
		case 212: i = rlzrw2(buffer, buffer1, db->osize);
			 break;
		case 13:
		case 312: i = rlzrw3a(buffer, buffer1, db->osize);
			 break;
		default: fprintf(stderr, "Type %d unknown\n", b->type);
			 return -6;
	}
	/* Is i compatible with j ? */
	if(i <= (j-1)*db->isize || i > j*db->isize) /* We have a problem ! */
		return -5;
}
return j;					/* j iblocks in use, OK */
}

main(argc, argv)
int argc;
char **argv;
{
int	i;
u_long	n, bitmaps, errors;
u_long	empty, total, blocks;
float	factor;
char *file = NULL;

for(i = 1; i < argc; i++)
	switch(*argv[i]) {
		case '-': read_opt(argv[0], argv[i]+1);
			  break;
		default: if(file != NULL) usage(argv[0]);
			 file = argv[i];
	}
if(file == NULL || !db_open(&db, file)) usage(argv[0]);

if(options & 4) { /* We'll need the decomp functions */
	i = iLZW(memory);
	if(i > 65536) options &= ~4;
	i = iLZV(memory);
	if(i > 65536) options &= ~4;
	i = ilzrw2(memory);
	if(i > 65536) options &= ~4;
	i = ilzrw3a(memory);
	if(i > 65536) options &= ~4;
	i = ipred(memory);
	if(i > 65536) options &= ~4;
}

printf("DouBle device version %d.%d\n", dbh.version>>8, dbh.version&0xff);

if(dbh.version < 3)
	printf("Old DouBle device version - check documentation\n");

if(options & 2) printf("Compression algorithm: %d\n", db.code);

total = blocks = empty = 0;
if(options & 2) printf("Checking blocks ...\n");
for(n = errors = 0; n<db.oblocks; n++) {
	i = get_block(&db, n, options);
	if((options & 2) && !(n&0x7f)) {
		putchar('.');
		fflush(stdout);
	}
	switch(i) {
		case  0: empty++; break;
		case -1: printf("\nCannot read cluster map");
			 break;
		case -2: printf("\nError in cluster map");
			 break;
		case -3: printf("\nBlock used set clear in bitmap");
			 break;
		case -4: printf("\nCluster truncated");
			 break;
		case -5: printf("\nDecompression error");
			 break;
		case -6: printf("\nCompression algorithm unknown");
			 break;
		default: if(i > 0) total += i, blocks++;
			 else printf("\nError %d unknown", i);
	}
	if(i < 0) {
		errors++;
		printf(", cluster %ld ", n);
		fflush(stdout);
	}
}
if(errors && (options & 2)) printf("%ld Error(s) found\n", errors);
total += db.addr_blocks;

if(options & 2) {
	printf("\nChecking bitmap ... ");
	fflush(stdout);
}
for(bitmaps = n = 0; n < db.iblocks; n++)
	if(!get_bit(&db, n)) bitmaps++;
if(bitmaps == total) {
	if(options & 2) printf("OK\n");
}
else printf("%ld blocks set used are clear\n", bitmaps-total);

if(total > db.addr_blocks) {
	factor = (float)db.ratio/((float)(total-db.addr_blocks)/blocks);
	printf("\nAverage compression factor (excluding headers) = %2.2f\n", factor);
	printf("Approximately %.0fK still available\n", factor*db.isize*(db.iblocks-total)/1024);
} else {
	factor = 0;
	if(options & 2) printf("\nDevice is empty\n");
}
printf("%ld/%ld (%2d%%) clusters (%d) used, %ld/%ld (%2d%%) blocks (%d) used.\n",
	blocks, db.oblocks, (int)(100*blocks/db.oblocks), db.osize, total, db.iblocks,
	(int)(100*total/db.iblocks), db.isize);
db_close();
if((options & 2) && factor > 0) {
	printf("\nConclusion: ");
	if(db.iblocks > total) {
		factor = empty*db.ratio/factor/(db.iblocks-total);
		if(factor < 0.75)
			printf("Compression ratio better than expected !\n\n");
		else if(factor > 1.3)
			printf("Poor compression ratio; could run out of space (beware) !\n\n");
		else printf("So far, the compression ratio looks OK to me.\n\n");
	}
	else printf("!! This thing is FULL !! Action required immediately!\n\n");
}
if(errors) exit(4);
exit(0);
}

usage(char *s)
{
if(errno) perror(s);
printf("usage: %s [options] device_file\n", s);
printf("\tcheck consistency of \"DouBle\" device_file\n");
printf("\toptions:\n\t\t-v:\tverbose.\n\t\t-t:\tcheck blocks (slooow).\n");
printf("\t\t-a:\tautomatic repair.\n\t\t-r:\tinteractive repair.\n");
if(errno) exit(8);
exit(16);
}
