/* sa.c */

/* Copyright (C) 1993 Free Software Foundation, Inc.

This file is part of the GNU Accounting Utilities

The GNU Accounting Utilities are free software; you can redistribute
them and/or modify them under the terms of the GNU General Public
License as published by the Free Software Foundation; either version
2, or (at your option) any later version.

The GNU Accounting Utilities are distributed in the hope that they will
be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with the GNU Accounting Utilities; see the file COPYING.  If
not, write to the Free Software Foundation, 675 Mass Ave, Cambridge,
MA 02139, USA.  */

#include "config.h"
#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
/* #include <sys/acct.h> -- already included in common.h */
#include <time.h>

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#include <pwd.h>
#include <math.h>
#include "sa.h"
#include "common.h"
#include "getopt.h"
#include "uid_hash.h"

#ifndef HAVE_GETPAGESIZE
#define getpagesize() 4096
#endif

static char rcsid[] = "$Id: sa.c,v 1.5 1994/03/23 03:03:24 noel Exp $";

/* globals */

#ifndef ACCT_FILE
/* not defined, so let's try and guess... */
#ifdef sun
#define ACCT_FILE "/var/adm/pacct"
#else
#ifdef sgi
#define ACCT_FILE "/usr/adm/pacct"
#else
#define ACCT_FILE "/usr/adm/acct"
#endif /* sgi */
#endif /* sun */
#endif /* top */

#ifndef SAVACCT_FILE
/* not defined, so let's try and guess... */
#ifdef sun
#define SAVACCT_FILE "/var/adm/savacct"
#else
#ifdef sgi
#define SAVACCT_FILE "/var/adm/savacct"
#else
#define SAVACCT_FILE "/usr/adm/savacct"
#endif /* sgi */
#endif /* sun */
#endif /* top */

struct name_entry *name_list;	/* lookup table for ttys */
struct sum_entry *command_list[CMD_TABLE_SIZE];	/* list of cmds */

struct sum_entry sum_totals;	/* total of all commands */

int debugging_enabled;		/* print internal information relating
				   to entries read, data structures,
				   etc. */
double system_page_size;	/* size in kbytes of each page in
				   memory */

#define PAGES_TO_KB(x) ((double) (x) / system_page_size)

int print_seconds;		/* print seconds instead of minutes */
int separate_times;		/* print separate user and sys times */
int dont_read_summary_file;	/* guess! */
int print_ksec;			/* ^^^ */
int print_ratio;		/* ^^ */
int print_users;		/* ^ */
int total_io;			/* print total disk io operations */
int percentages;		/* include percentages in printout */

char *program_name;		/* name of the program, for usage & errs */
FILE *Afile;			/* pointer to the acct file */
int sort_type;			/* which field to sort by */
int reverse_sort;		/* nonzero if the sort is reversed */

char default_acct_file[] = ACCT_FILE;
char default_savacct_file[] = SAVACCT_FILE;

char *acct_file_name = default_acct_file; /* the file name */
char *savacct_file_name = default_savacct_file;

long line_number = 0;		/* the line number of the file,
				 * for error reporting purposes
				 */


/* code */

void
main (int argc, char *argv[])
{
  int c;

  init_flags_and_data ();
  program_name = argv[0];
  
  while (1)
    {
      int option_index = 0;
      
      static struct option long_options[] = {
	{ "debug", no_argument, NULL, 1 },
	{ "version", no_argument, NULL, 2 },
	{ "help", no_argument, NULL, 3 },
	{ "other-acct-file", 1, NULL, 4 },
	{ "print-seconds", no_argument, NULL, 5 },
	{ "dont-read-summary-files", no_argument, NULL, 6 },
	{ "list-all-names", no_argument, NULL, 7 },
	{ "separate-times", no_argument, NULL, 8 },
	{ "other-savacct-file", 1, NULL, 9 },
	{ "sort-ksec", no_argument, NULL, 10 },
	{ "print-ratio", no_argument, NULL, 11 },
	{ "print-users", no_argument, NULL, 12 },
	{ "sort-tio", no_argument, NULL, 13 },
	{ "percentages", no_argument, NULL, 14 },
	{ "sort-sys-user-div-calls", no_argument, NULL, 15 },
	{ "sort-avio", no_argument, NULL, 16 },
	{ "sort-cpy-avmem", no_argument, NULL, 17 },
	{ "sort-num-calls", no_argument, NULL, 18 },
	{ "not-interactive", no_argument, NULL, 19 },
	{ "user-summary", no_argument, NULL, 20 },
	{ "reverse-sort", no_argument, NULL, 21 },
	{ "merge", no_argument, NULL, 22 },
	{ "threshold", required_argument, NULL, 23 },
	{ 0, 0, 0, 0 }
      };
      
      c = getopt_long (argc, argv, "abcdfiljkmnrstuvDK",
		       long_options, &option_index);
      
      if (c == EOF)
	break;

      switch (c)
	{
	case 1:
	  debugging_enabled = 1;
	  break;
	case 2:
	  fprintf (stderr, "%s\n", rcsid);
	  break;
	case 3:
	  give_usage ();
	  exit (1);
	  break;
	case 4:
	  acct_file_name = optarg;
	  break;
	case 'j':
	case 5:
	  print_seconds = 1;
	  break;
	case 'i':
	case 6:
	  dont_read_summary_file = 1;
	  break;
	case 'l':
	case 8:
	  separate_times = 1;
	  break;
	case 9:
	  savacct_file_name = optarg;
	  break;
	case 'K':
	case 10:
	  sort_type = sort_cpu_storage;
	  print_ksec = 1;
	  break;
	case 't':
	case 11:
	  print_ratio = 1;
	  break;
	case 'u':
	case 12:
	  print_users = 1;
	  break;
	case 'D':
	case 13:
	  sort_type = sort_tio;
	  total_io = 1;
	  break;
	case 'c':
	case 14:
	  percentages = 1;
	  break;
	case 'b':
	case 15:
	  sort_type = sort_sys_plus_user_div_calls;
	  break;
	case 'd':
	case 16:
	  sort_type = sort_avio;
	  break;
	case 'k':
	case 17:
	  sort_type = sort_cpu_mem_average;
	  break;
	case 'n':
	case 18:
	  sort_type = sort_num_calls;
	  break;
	case 'r':
	case 21:
	  reverse_sort = 1;
	  break;
	case 'a':
	case 7:
	case 'f':
	case 19:
	case 'm':
	case 20:
	case 's':
	case 22:
	case 'v':
	case 23:
	  {
	    fatal ("flags 'afmsv' not implemented yet");
	    give_usage ();
	    exit (1);
	  }
	  break;
	default:
	  {
	    /* char ts[255];
	    sprintf (ts, "?? getopt returned char code 0%o ??", c);
	    fatal (ts); */
	    give_usage ();
	    exit (1);
	  }
	  break;
	}
    }

  if (optind == (argc - 1))
    {
      /* if we get here, we're expecting a filename */
      acct_file_name = argv[optind++];
    }
  else if (optind < argc)
    {
      fprintf (stderr, "%s: extra arguments ", program_name);

      while (optind < argc)
	{
	  fprintf (stderr, "'%s'", argv[optind++]);
	  if (optind < argc)
	    fprintf (stderr, ", ");
	}

      fprintf (stderr, " are being ignored\n");
    }
  
  /* see if the summary file is there -- if so, load it and parse it */
  if (!dont_read_summary_file)
    {
      struct stat st;
      
      if (stat (savacct_file_name, &st) == 0)
	{
	  Afile = open_binary (savacct_file_name);
	  parse_savacct_entries ();
	  fclose (Afile);
	}
    }
  
  /* now, do the acct file */
  
  Afile = open_binary (acct_file_name);
  parse_entries ();
  fclose (Afile);
  
  print_command_list ();

  exit (0);			/* guarantee the proper return value */
}


/* guess what this does... */
void
give_usage (void)
{
  char *usage = "\
Usage: %s [ options ] [ file ]\n\
\n\
       options: [-bcdiljknrtuDK] [--debug] [--version] [--help]\n\
       [--other-acct-file] [--print-seconds] [--dont-read-summary-files]\n\
       [--separate-times] [--other-savacct-file]\n\
       [--sort-ksec] [--print-ratio] [--print-users] [--sort-tio]\n\
       [--percentages] [--sort-sys-user-div-calls] [--sort-avio]\n\
       [--sort-cpy-avmem] [--sort-num-calls]\n\
\n\
       not implemented yet: [-afmsv] [--list-all-names] [--merge]\n\
       [--not-interactive] [--user-summary] [--threshold <num>]\n\
";
  
  fprintf (stderr, usage, program_name);
}


/* parse the entries in a savacct file */
void
parse_savacct_entries (void)
{
  struct savacct *rec;
  
  if (debugging_enabled)
    {
      fprintf (stddebug, "savacct entries\n---------------\n");
    }

  rec = (struct savacct *) xmalloc (sizeof (struct savacct));
  
  while (fread (rec, sizeof (struct savacct), 1, Afile))
    {
      char exec_flag = 0x00;
      int a;
      
      /* search the name of the command from the end to the beginning -- if
       * there's an asterisk there, replace it and set the executable flag */

      for (a = CMD_LEN; a > 0; a--)
	{
	  if (rec->name[a] == '\0')
	    {
	      continue;
	    }
	  else if (rec->name[a] == '*')
	    {
	      rec->name[a] = '\0';
	      exec_flag = 0xFF & AFORK;
	      break;
	    }
	  else
	    {
	      break;
	    }
	}

      update_command_list (rec->name, rec->num_calls,
			   rec->user_time / AHZ,
			   rec->sys_time / AHZ,
			   rec->elapsed_time / AHZ,
			   rec->disk_io,
			   rec->mem_usage / AHZ,
			   exec_flag);
    }
  
  free (rec);
}


/* parse the entries in an acct file */
void
parse_entries (void)
{
  struct acct *rec;             /* the current record */
  
  if (debugging_enabled)
    fprintf (stddebug, "\nacct entries\n------------\n");

  rec = (struct acct *) xmalloc (sizeof (struct acct));
  
  /* loop while there are entries to be had */
  while (fread (rec, sizeof (struct acct), 1, Afile))
    {
#ifdef HAVE_COMP_T
      double ut = comp_t_2_double (rec->ac_utime) / (double) AHZ;
      double st = comp_t_2_double (rec->ac_stime) / (double) AHZ;
      double et = comp_t_2_double (rec->ac_etime) / (double) AHZ;
      double di = comp_t_2_double (rec->ac_io) / (double) AHZ;
#else
      double ut = (double) rec->ac_utime / (double) AHZ;
      double st = (double) rec->ac_stime / (double) AHZ;
      double et = (double) rec->ac_etime / (double) AHZ;
      double di = (double) rec->ac_io / (double) AHZ;
#endif

      /* Christoph Badura (bad@flatlin.ka.sub.org) says that ac_mem IS
	 NOT a comp_t */
#ifdef 0			/* OLD CODE */
      double mu = (ut + st) * PAGES_TO_KB (comp_t_2_double (rec->ac_mem));
#endif
      double mu = PAGES_TO_KB (rec->ac_mem);

      /* if (debugging_enabled)
	fprintf (stddebug,
		 "%-10.8s ac_mem:%6.2f corr:%6.2f pgs:%6.2f\n",
		 rec->ac_comm, (double) rec->ac_mem,
		 comp_t_2_double (rec->ac_mem), mu); */

      if (print_users)
	{
	  printf ("%-8.8s %6.2f cpu %8.0fk mem %6.0f io %-10.*s%s\n",
		  uid_name (rec->ac_uid), ut + st, mu / (ut + st), di,
		  CMD_LEN, rec->ac_comm,
		  (rec->ac_flag & AFORK) ? "*" : "");
	}
      else
	{
	  update_command_list (rec->ac_comm, 1, ut, st, et, di, mu,
			       (rec->ac_flag & AFORK));
	}
    }

  free (rec);

  if (print_users) exit (0);
}


int
hash_name (char *s)
{
  unsigned h = 0, g, q;

  for (q = 0; (s[q] != '\0') && (q < CMD_LEN); q++)
    {
      h = (h << 4) + s[q];
      if ((g = h & 0xf0000000))
	{
	  h = h ^ (g >> 24);
	  h = h ^ g;
	}
    }
  
  return h % CMD_TABLE_SIZE;
}
  

void
update_command_list (char *name, long num_calls, double user_time,
		     double sys_time, double elapsed_time, double disk_io,
		     double mem_usage, char exec_flag)
{
  /* Look for the command in the list.  If found, add the stats.  If
     not found, create a new entry and add the stats. */

  struct sum_entry *ap;
  int hashval;

  if (debugging_enabled)
    {
      fprintf (stddebug,
       "+:%-10.10s%s %10ld %10.2fu %10.2fs %10.2fe %10.2fio %10.2fmem\n",
	       name, (exec_flag) ? "*" : " ", num_calls,
	       user_time, sys_time, elapsed_time, disk_io, mem_usage);
    }
  
  hashval = hash_name (name);

  for (ap = command_list[hashval]; ap != NULL; ap = ap->next)
    {
      if ((strncmp (ap->name, name, CMD_LEN) == 0)
	  && (exec_flag == ap->exec_flag))
	/* this is the one! */
	break;
    }
  
  /* If we didn't find it, AP will be null (and therefore we know
   * when to create a new record
   */
  if (ap == NULL)
    {
      ap = (struct sum_entry *) xmalloc (sizeof (struct sum_entry));
      strncpy (ap->name, name, CMD_LEN);
      ap->next = command_list[hashval];
      command_list[hashval] = ap;
      ap->exec_flag = exec_flag;
      ap->user_time = 0.0;
      ap->sys_time = 0.0;
      ap->elapsed_time = 0.0;
      ap->mem_usage = 0.0;
      ap->disk_io = 0.0;
      ap->num_calls = 0;
    }

#define ADDIT(thing) \
  { \
    ap->thing += thing; \
    sum_totals.thing += thing; \
  }

  ADDIT (user_time);
  ADDIT (sys_time);
  ADDIT (elapsed_time);
  ADDIT (disk_io);
  ADDIT (mem_usage);
  ADDIT (num_calls);
}

/* compare two entries, returning an int less than, equal to, or
 * greater than zero reflecting whether s1 is less than, equal to, or
 * greater than s2. */

int
compare_sum_entry (struct sum_entry **s1, struct sum_entry **s2)
{
  double v1, v2;

  switch (sort_type)
    {
    case sort_sys_plus_user:
      v1 = (*s1)->user_time + (*s1)->sys_time;
      v2 = (*s2)->user_time + (*s2)->sys_time;
      break;
    case sort_sys_plus_user_div_calls:
      v1 = ((*s1)->user_time + (*s1)->sys_time) / (double) (*s1)->num_calls;
      v2 = ((*s2)->user_time + (*s2)->sys_time) / (double) (*s2)->num_calls;
      break;
    case sort_avio:
      v1 = (*s1)->disk_io / (double) (*s1)->num_calls;
      v2 = (*s2)->disk_io / (double) (*s2)->num_calls;
      break;
    case sort_tio:
      v1 = (*s1)->disk_io;
      v2 = (*s2)->disk_io;
      break;
    case sort_cpu_mem_average:
      v1 = (*s1)->mem_usage / (double) (*s1)->num_calls;
      v2 = (*s2)->mem_usage / (double) (*s2)->num_calls;
      break;
    case sort_cpu_storage:
      v1 = 1;
      v2 = 0;
      break;
    case sort_num_calls:
      v1 = (double) (*s1)->num_calls;
      v2 = (double) (*s2)->num_calls;
      break;
    }
  if (v1 < v2) return ((reverse_sort) ? -1 : 1);
  if (v1 > v2) return ((reverse_sort) ? 1 : -1);

#define compareit(x); \
  if ((*s1)->x < (*s2)->x) return ((reverse_sort) ? -1 : 1); \
  if ((*s1)->x > (*s2)->x) return ((reverse_sort) ? 1 : -1);
  
  compareit (num_calls);
  compareit (user_time);
  compareit (sys_time);
  compareit (disk_io);
  compareit (elapsed_time);
  compareit (mem_usage);
  return 0;
}

void
print_command_list (void)
{
  struct sum_entry *ap;
  struct sum_entry **entry_array;
  long num_commands, temp, which;

  /* sum_totals.next = command_list; */

  /* start num_commands at one because of the summary entry */

  for (temp = 0, num_commands = 1; temp < CMD_TABLE_SIZE; temp++)
    {
      for (ap = command_list[temp]; ap != NULL; ap = ap->next)
	num_commands++;
    }

  entry_array = (struct sum_entry **)
    xmalloc (sizeof (struct sum_entry *) * num_commands);

  which = 0;
  entry_array[which++] = &sum_totals;

  for (temp = 0; temp < CMD_TABLE_SIZE; temp++)
    {
      for (ap = command_list[temp]; ap != NULL; ap = ap->next)
	entry_array[which++] = ap;
    }
  
  /* the summary entry should always be first, so don't sort it --
     therefore, pass location 1 of the array to qsort and one less
     than the number of commands */

  qsort (entry_array + 1, (size_t) num_commands - 1,
	 sizeof (struct sum_entry *), (int (*)()) compare_sum_entry);

  for (temp = 0; temp < num_commands; temp++)
    {
      ap = entry_array[temp];

#define NC ap->num_calls
#define DNC ((double) NC)
#define RE ap->elapsed_time
#define U ap->user_time
#define S ap->sys_time
#define CP (U + S)
#define IO ap->disk_io
#define K ap->mem_usage

      if (debugging_enabled)
	{
	  fprintf (stddebug,
	   "t:%-10.10s%s %10ld %10.2fu %10.2fs %10.2fe %10.2fio %10.2fmem\n",
		   ap->name, (ap->exec_flag) ? "*" : " ", ap->num_calls,
		   ap->user_time, ap->sys_time, ap->elapsed_time,
		   ap->disk_io, ap->mem_usage);
	}
      
      /* print it out */
      
      printf ("%8ld ", NC);
      
      if (percentages)
	printf ("%7.2f%% ", ((sum_totals.num_calls)
			     ? (DNC / (double) (sum_totals.num_calls)) * 100.0
			     : 0.0));
		
      
      printf ("%10.2fre ", (((DNC && print_seconds) || !print_seconds)
			    ? RE / ((print_seconds) ? DNC : 60.0)
			    : 0.0));
      
      if (percentages)
	printf ("%7.2f%% ", ((sum_totals.elapsed_time)
			     ? (RE / (sum_totals.elapsed_time / AHZ)) * 100.0
			     : 0.0));
      
      if (separate_times)
	{
	  printf ("%10.2fu ", (((DNC && print_seconds) || !print_seconds)
			       ? U / ((print_seconds) ? DNC : 60.0)
			       : 0.0));
	  
	  if (percentages)
	    printf ("%7.2f%% ", ((sum_totals.user_time)
				 ? (U / sum_totals.user_time) * 100.0
				 : 0.0));
	  
	  printf ("%10.2fs", (((DNC && print_seconds) || !print_seconds)
			      ? S / ((print_seconds) ? DNC : 60.0)
			      : 0.0));
	  
	  if (percentages)
	    printf (" %7.2f%%", ((sum_totals.sys_time)
				 ? (S / sum_totals.sys_time) * 100.0
				 : 0.0));
	}
      else
	{
	  printf ("%10.2fcp", (((DNC && print_seconds) || !print_seconds)
			       ? CP / ((print_seconds) ? DNC : 60.0)
			       : 0.0));
	  
	  if (percentages)
	    printf (" %7.2f%%", ((sum_totals.sys_time + sum_totals.user_time)
				 ? (CP /
				    ((sum_totals.sys_time
				      + sum_totals.user_time) / AHZ))* 100.0
				 : 0.0));
	}
      
      if (print_ratio)
	{
	  if (CP == 0.0)
	    printf ("*ignore*");
	  else
	    printf ("%8.1f", ((CP) ? RE / CP : 0.0));
	  
	  fputs ("re/cp", stdout);
	}
      
      if (total_io)
	printf ("%10.0ftio ", IO);
      else
	printf ("%10.0favio ", ((DNC) ? IO / DNC : 0.0));
      
#if 0
      if (print_ksec)
	printf ("%9.0fk*sec", K);
      else
	printf ("%9ldk", (long) ((CP) ? (K / CP) : 0.0));
#endif

      /* Christoph says: */

      if (print_ksec)
 	printf ("%9.0fk*sec", (CP) ? K * CP : K);
      else
 	printf ("%9ldk", (long) (K / NC));

      if (ap->name[0] != '\0')
	{
	  printf ("   %-.10s", ap->name);
	  
	  if (ap->exec_flag)
	    putchar ('*');
	}

      putchar ('\n');
    }
}


/* clear all global flags & data */
void
init_flags_and_data (void)
{
  int a;

  debugging_enabled = 0;
  print_seconds = 0;
  separate_times = 0;
  dont_read_summary_file = 0;
  print_ksec = 0;
  print_ratio = 0;
  print_users = 0;
  total_io = 0;
  percentages = 0;
  sort_type = sort_sys_plus_user;
  name_list = NULL;
  reverse_sort = 0;

  /* get the page size of the machine for the PAGES_TO_KB macro */
  system_page_size = (double) getpagesize () / 1024.0;

  /* init the summary record */
  sum_totals.next = NULL;
  sum_totals.name[0] = '\0';
  sum_totals.exec_flag = 0;
  sum_totals.user_time = 0.0;
  sum_totals.sys_time = 0.0;
  sum_totals.elapsed_time = 0.0;
  sum_totals.mem_usage = 0.0;
  sum_totals.disk_io = 0.0;
  sum_totals.num_calls = 0;

  clear_uid_table ();

  /* clear the hash tables */
  for (a = 0; a < CMD_TABLE_SIZE; a++)
    {
      command_list[a] = NULL;
    }
}

