/* ac.c */

/* Copyright (C) 1993 Free Software Foundation, Inc.

This file is part of the GNU Accounting Utilities

The GNU Accounting Utilities are free software; you can redistribute
them and/or modify them under the terms of the GNU General Public
License as published by the Free Software Foundation; either version
2, or (at your option) any later version.

The GNU Accounting Utilities are distributed in the hope that they will
be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with the GNU Accounting Utilities; see the file COPYING.  If
not, write to the Free Software Foundation, 675 Mass Ave, Cambridge,
MA 02139, USA.  */

#include "config.h"
#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <string.h>
#include <sys/types.h>
#include <utmp.h>
#include <time.h>

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#include "ac.h"
#include "common.h"
#include "getopt.h"

#ifndef HAVE_MKTIME
#undef HAVE_CONFIG_H
#undef CONFIG_BROKETS
#include "mktime.c"
#endif

static char rcsid[] = "$Id: ac.c,v 1.5 1994/03/23 02:53:27 noel Exp $";

/* globals */

int debugging_enabled;           /* print internal information
				  * relating to entries read, data
				  * structures, etc. */

long time_warp_leniency;         /* number of seconds that an entry
				  * can be "before" the previous one
				  * -- to deal with simultaneous
				  * init(8) writes, or to overlook
				  * problems with sun's wtmp files.
				  * Defaults to 60. */

int nasty_reboot;                /* most ac's count the time that the
				  * user was logged in before a reboot
				  * against them.  Off by default. */

int nasty_supplant;              /* if the same tty appears for a
				  * login without an intervening
				  * logout, there's a problem.  Most
				  * ac's count the time between the
				  * two logins against the user.  Off
				  * by default. */

int nasty_time_warp;             /* if the running date in the wtmp
				  * file jumps backwards, charge the
				  * user until midnight of the day
				  * they logged in.  Off by default. */

int print_file_inconsistencies;  /* when there's a problem with the
				  * file's format, scream to stderr! */

int print_individual_totals;     /* -p flag */
int print_midnight_totals;       /* -d flag */
int dont_print_zero_totals;      /* -z flag */

#ifndef WTMP_FILE
/* not defined, so let's try and guess... */
#ifdef sun
#define WTMP_FILE "/var/adm/wtmp"
#else
#define WTMP_FILE "/usr/adm/wtmp"
#endif
#endif

/* system V machines have a specific variable in their utmp structure
 * that tells us what type of record it is.  bsd machines don't... */

#ifdef SYSTEM_V_STYLE_RECORDS

#define RECORD_TYPE ut_type

#else

#define RECORD_TYPE ut_line[0]
#define OLD_TIME '|'
#define NEW_TIME '{'
#define BOOT_TIME '~'

#endif

struct list_entry *waiting_list;	/* lookup tables for ttys, ids, names */
struct user_entry *user_list;
struct name_entry *name_list;

char *program_name;              /* name of the program, for usage & errs */
FILE *Wfile;                     /* pointer to the wtmp file */
char default_wtmp_file[] = WTMP_FILE;  /* self-explanatory */
char *wtmp_file_name = default_wtmp_file;    /* the file name */
long line_number = 0;            /* the line number of the file,
				  * for error reporting purposes
				  */


/* code */

void
main (int argc, char *argv[])
{
  int c;

  init_flags_and_data ();
  program_name = argv[0];
  
  while (1)
    {
      int option_index = 0;
      
      static struct option long_options[] = {
	{ "complain", 0, 0, 1 },
	{ "reboots", 0, 0, 2 },
	{ "supplants", 0, 0, 3 },
	{ "timewarps", 0, 0, 4 },
	{ "dont-print-zeros", 0, 0, 5 },
	{ "debug", 0, 0, 6 },
	{ "timewarp-value", 1, 0, 7 },
	{ "version", 0, 0, 8 },
	{ "help", 0, 0, 9 },
	{ "daily-totals", 0, 0, 10 },
	{ "individual-totals", 0, 0, 11 },
	{ "other-file", 1, 0, 12 },
	{ 0, 0, 0, 0 }
      };
      
#ifdef SYSTEM_V_STYLE_RECORDS
      c = getopt_long (argc, argv, "adpw:", long_options, &option_index);
#else
      c = getopt_long (argc, argv, "dpw:", long_options, &option_index);
#endif
      
      if (c == EOF)
	break;

      switch (c)
	{
	case 1:
	  print_file_inconsistencies = 1;
	  break;
	case 2:
	  nasty_reboot = 1;
	  break;
	case 3:
	  nasty_supplant = 1;
	  break;
	case 4:
	  nasty_time_warp = 1;
	  break;
	case 5:
	  dont_print_zero_totals = 1;
	  break;
	case 6:
	  debugging_enabled = 1;
	  print_file_inconsistencies = 1;
	  break;
	case 7:
	  time_warp_leniency = atol (optarg);
	  if (time_warp_leniency < 0)
	    fatal ("time warp leniency value has to be non-negative");
	  break;
	case 8:
	  fprintf (stderr, "%s\n", rcsid);
	  break;
	case 9:
	  give_usage ();
	  exit (1);
	  break;
	case 10:
	case 'd':
	  print_midnight_totals = 1;
	  break;
	case 11:
	case 'p':
	  print_individual_totals = 1;
	  break;
	case 12:
	case 'w':
	  wtmp_file_name = optarg;
	  break;
	default:
	  {
	    /* char ts[255];
	    sprintf (ts, "?? getopt returned char code 0%o ??", c);
	    fatal (ts); */
	    give_usage ();
	    exit (1);
	  }
	  break;
	}
    }

  while (optind < argc)
    {
      /* if we get here, we expect everything else to be a username */
      struct name_entry *node;
      
      node = (struct name_entry *) xmalloc (sizeof (struct name_entry));
      node->name = argv[optind++];
      node->next = name_list;
      name_list = node;
    }
  
  if (debugging_enabled)
    {
      struct name_entry *np;
      for (np = name_list; np != NULL; np = np->next)
	fprintf (stddebug, "%s\n", np->name);
    }
  
  Wfile = open_binary (wtmp_file_name);
  
  parse_entries ();
  
  fclose (Wfile);

  exit (0);			/* guarantee the proper return value */
}


/* guess what this does... */
void
give_usage (void)
{
  char *usage = "\
Usage: %s [-dp] [-w <file>] [people] ...\n\
       [--daily-totals] [--individual-totals] [--other-file <file>]\n\
       [--complain] [--reboots] [--supplants] [--timewarps]\n\
       [--dont-print-zeros] [--debug] [--timewarp-value <value>]\n\
       [--version] [--help]\n";
  
  fprintf (stderr, usage, program_name);
}


/* parse the entries in a wtmp file */
void
parse_entries (void)
{
  struct utmp rec;            /* the current record */
  long next_midnight;
  int expecting_clock = 0;    /* set to a nonzero value if the next entry
			       * is expected to be the second half of a
			       * clock pair
			       */
  long clock_before = 0;      /* the clock before the date cmd */
  long last_time = 0;         /* time of last event (error checking) */
  int first_time = 1;         /* init stuff first time through */
  FILE *fp = Wfile;           /* pointer to the file */

  /* loop while there are still entries to read... */
  while ( (fread ( (void *)&rec, sizeof (rec), 1, fp )) != 0 )
    {	  
      if (debugging_enabled)
	{
	  fprintf (stddebug, "---------------------------------------------------------------------------\n");
#ifdef SYSTEM_V_STYLE_RECORDS
	  fprintf (stddebug, "%-8.8s %-8.8s %ld (%d) %s",
		   rec.ut_name, rec.ut_line, (long) rec.ut_time,
		   rec.RECORD_TYPE, ctime ((time_t *) &(rec.ut_time)));
#else
	  fprintf (stddebug, "%-8.8s %-8.8s %ld %s",
		   rec.ut_name, rec.ut_line, (long) rec.ut_time,
		   ctime ((time_t *) &(rec.ut_time)));
#endif
	}
      
      /*
       * if this is the first time, we need to set up next_midnight
       * correctly, so that do_totals will function correctly
       */
      if (first_time)
	{
	  first_time = 0;
	  next_midnight = midnight_after_me (&rec.ut_time);
	}
      
      /*
       * if the time of this record occurs *before* the time of the last
       * record, there's something wrong.  To deal with it, log everyone
       * out at the next midnight, doing the totals at that time.
       * Then reset NEXT_MIDNIGHT so the next do_totals call is correct.
       * Also, reset LAST_TIME so this check can be made correctly.
       * NOTE: the '{' entry is a special case, so ignore it.
       */
      if ((rec.ut_time + time_warp_leniency < last_time)
	  && (rec.RECORD_TYPE != NEW_TIME))
	{
	  long temp_time;
	  struct tm *temp_tm;
	  char month_day_string[30];
	  
	  if (print_file_inconsistencies)
	    fprintf (stddebug, "%s:%ld: time warp backwards (%ld/%ld)\n",
		     wtmp_file_name, line_number, last_time, rec.ut_time);
	  
	  /* We need the proper label for do_statistics: we can get it
	   * relative to next_midnight.
	   */
	  temp_time = next_midnight - 10;
	  temp_tm = localtime ((time_t *) &temp_time);
	  sprintf (month_day_string, "%s %2d",
		   months[temp_tm->tm_mon], temp_tm->tm_mday);
	  
	  log_everyone_out (next_midnight, nasty_time_warp, FALSE, 
			    "time warp");
	  
	  if (print_midnight_totals)
	    do_statistics (month_day_string);
	  
	  next_midnight = midnight_after_me (&rec.ut_time);
	}
      last_time = rec.ut_time; /* so we're ready to run next time */
      
      /* if we're doing totals every midnight, we have to check and see
       * if the current record occurs past the next midnight.  If it does,
       * print out the totals for the evening and increment the
       * NEXT_MIDNIGHT variable.  Continue until the record is before our
       * midnight.
       */
      if (rec.RECORD_TYPE != NEW_TIME)
	do_totals (&next_midnight, &rec.ut_time, print_midnight_totals);
      
      /* now, process each record, depeding on its type */
      switch (rec.RECORD_TYPE)
	{
	  
	case NEW_TIME:		/* time after a date command */
	  if (expecting_clock)
	    {
	      update_system_time (rec.ut_time - clock_before);
	      expecting_clock = 0;
	      next_midnight = midnight_after_me (&rec.ut_time);
	    }
	  else
	    if (print_file_inconsistencies)
	      fprintf (stddebug, "%s:%ld: expecting new time record \n",
		       wtmp_file_name, line_number);
	  break;
	  
	case OLD_TIME:		/* time before a date command */
	  expecting_clock = 1;
	  clock_before = rec.ut_time;
	  break;
	  
	case BOOT_TIME:		/* reboot or shutdown */
	  if (strncmp (rec.ut_name, "shutdown", 8) == 0)
	    {
	      /* at the given time, the system was shut down
	       * so log everybody out
	       */
	      log_everyone_out (rec.ut_time, TRUE, FALSE,
				"shutdown");
	    }
	  else
	    {
	      /* the system was rebooted at the given time, so we don't
	       * know when it really went down.  Let's be nice to the
	       * users, and don't count the time against their total
	       * time.  Just clear the current logins!
	       * The NASTY_REBOOT flag is put in for compatability...
	       */
	      log_everyone_out (rec.ut_time, nasty_reboot, FALSE,
				"reboot");
	    }
	  break;
	  
#ifdef SYSTEM_V_STYLE_RECORDS
	  
	case USER_PROCESS:
	  /* our meat & potatoes */
	  log_in (&rec);
	  break;

#ifdef DEAD_PROCESS
	case DEAD_PROCESS:
	  log_out (&rec);
	  break;
#endif
	  
	  /* FIXME: here we assume that any other types are just
             superfluous records (like RUN_LVL, INIT_PROCESS,
             LOGIN_PROCESS, whatever) and they need to be ignored.
             This may not hold true for every arch, however.  So if
             there's a special kind of record that needs to be
             handled, put it (with ifdefs) above. */

	default:
	  break;

#else /* ! SYSTEM_V_STYLE_RECORDS */

	default:                 /* login or logout */
	  if (strlen (rec.ut_name) == 0)
	    log_out (&rec);
	  else
	    log_in (&rec);

#endif /* SYSTEM_V_STYLE_RECORDS */

	}

      if (debugging_enabled)
	{
	  /* print out list */
	  struct list_entry *np;
	  for (np = waiting_list; np != NULL; np = np->next)
	    {
	      fprintf (stddebug, "**\t%.8s %.8s ", np->ut_line, np->ut_name);
	      fprintf (stddebug, "%s", ctime ((time_t *) &(np->time)));
	    }
	}
      line_number++;
    }
  
  /* at this point, there are no more entries in the wtmp file.  We
   * need to do statistics for everything up to now, so update all of the
   * people who are logged in presently and print today's statistics
   */
  {
    long now = (long) time ((time_t *)0);
    
    do_totals (&next_midnight, &now, print_midnight_totals);
    
    log_everyone_out (now, TRUE, FALSE, "catch-up");
    
    if (print_midnight_totals)
      do_statistics ("Today");
  }
  
  if (!print_midnight_totals)
    do_statistics ("");
}



/* since the sys clock has changed, each entry's login time has to be
 * adjusted...
 */
void
update_system_time (long the_time)
{
  struct list_entry *np;
  
  if (debugging_enabled)
    fprintf (stddebug, "\t\t\t\t\tupdate by %ld\n", the_time);
  
  for (np=waiting_list; np!=NULL; np=np->next)
    {
      np->time += the_time;
      
      if (debugging_enabled)
	fprintf (stddebug, "\t\t\t\t\t%ld %ld %-8.8s (clock_chg)",
		 np->time, np->time + the_time, np->ut_name);
    }
}


/* fills in the days between entries, calculating how much time a user
 * has racked up by midnight of each period
 *
 * side-effect (desireable) -- changes the value passed in NEXT_MIDNIGHT
 * to catch up with CURRENT_TIME
 */
void
do_totals (long *next_midnight, long *current_time, int print_flag)
{
  struct tm *temp_tm;
  char month_day_string[255];
  long temp_time;
  
  while (*next_midnight < *current_time)
    {
      /* We need the proper label for do_statistics: we can get it
       * relative to next_midnight.
       */
      temp_time = *next_midnight - 10;
      temp_tm = localtime ((time_t *) &temp_time);
      sprintf (month_day_string, "%s %2d",
	       months[temp_tm->tm_mon], temp_tm->tm_mday);
      
      if (debugging_enabled)
	fprintf (stddebug, "midnight @ %s", ctime ((time_t *)next_midnight));
      
      log_everyone_out (*next_midnight, TRUE, TRUE, "midnight logout");
      
      if (print_flag)
	do_statistics (month_day_string);
      
      /* get the NEXT day */
      temp_time = *next_midnight + 93600;    /* 26 hours to be sure */
      *next_midnight = midnight_before_me (&temp_time);
    }
}



/* print out statistics and clear the user totals
 * if !PRINT_INDIVIDUAL_TOTALS && !PRINT_MIDNIGHT_TOTALS
 *   don't print anything until the end
 * if PRINT_MIDNIGHT_TOTALS && !PRINT_INDIVIDUAL_TOTALS
 *   print totals for each day, & clear user times
 * if PRINT_MIDNIGHT_TOTALS && PRINT_INDIVIDUAL_TOTALS
 *   print totals, individ user times, & clear times
 */
void
do_statistics (char *date_string)
{
  struct user_entry *np, *temp;
  long total;
  float float_total;
  
  total = 0;
  float_total = 0.0;
  
  for (np = user_list; np != NULL; )
    {
      /* print if the user table is null or the name is on the command line */
      if ((name_list == NULL) || (desired_name (np->name)))
	{
	  total += np->time;

	  if ((print_individual_totals)
	      && !((np->time == 0.0) && dont_print_zero_totals))
	    printf ("\t%-8.8s%6.2f\n", np->name, (float)np->time/3600.0);
	} 
      
      temp = np->next;
      free (np);
      np = temp;
    }
  
  user_list = NULL;
  float_total = (float) total / 3600.0;
  
  if (print_midnight_totals)
    {
      if (!((float_total == 0.0) && dont_print_zero_totals))
	printf ("%s\ttotal%9.2f\n", date_string, float_total);
    }
  else
    printf ("\ttotal%9.2f\n", float_total);
}


/* log all entries out at THE_TIME.  Update statistics if UPDATE_TIME_FLAG
 * is non-zero, and print out the entries preceded by DEBUG_STR.  If
 * CHANGE_LOGIN_FLAG is non-zero, reset the login times to THE_TIME.
 */
void
log_everyone_out (long the_time, int update_time_flag,
		  int change_login_flag, char *debug_str)
{
  struct list_entry *np, *temp;
  
  for (np = waiting_list; np != NULL; )
    {
      /* sometimes, when there is a change to/from daylight savings time,
       * the login time is later than the logout time.  KLUDGE: when this
       * happens, subtract an hour from the login time.
       */
      if (np->time > the_time)
	{
	  if (print_file_inconsistencies)
	    fprintf (stddebug, "%s:%ld: logout (daylight savings time) %.8s\n",
		     wtmp_file_name, line_number, np->ut_name);
	  np->time -= 3600;
	}
      
      if (update_time_flag)
	update_user_time (np->ut_name, the_time - np->time, debug_str);
      if (debugging_enabled)
	fprintf (stddebug, "\t\t\t\t\t\t%-8.8s (%s) leo\n",
		 np->ut_name, debug_str);
      
      temp = np->next;
      
      if (change_login_flag)
	np->time = the_time;
      else
	free (np);
      
      np = temp;
    }
  
  if (!change_login_flag)
    waiting_list = NULL;
}


/* put a terminal into the waiting list */
void
log_in (struct utmp *entry)
{
  struct list_entry *np;
  
  for (np = waiting_list; np != NULL; np = np->next)
    {
      if (strncmp (np->ut_line, entry->ut_line, 8) == 0)
	{
	  if (print_file_inconsistencies)
	    fprintf (stddebug, "%s:%ld: duplicate tty record\n",
		     wtmp_file_name, line_number);
	  
	  /* we should just write over the old one -- nasty ac's
	   * charge the user being for being logged in until this new
	   * tty entry appears, so the flag NASTY_SUPPLANT is included
	   */
	  if (nasty_supplant)
	    update_user_time (np->ut_name, entry->ut_time - np->time,
			      "supplant");
	  
	  strncpy (np->ut_name, entry->ut_name, 8);
	  np->time = entry->ut_time;
	  return;
	}
    }
  
  /* if we get here, we didn't find the entry in the list */
  np = (struct list_entry *) xmalloc (sizeof (struct list_entry));
  
  /* copy the info */
  strncpy (np->ut_line, entry->ut_line, 8);
  strncpy (np->ut_name, entry->ut_name, 8);
  np->time = entry->ut_time;
  
  /* insert at the beginning of the list */
  np->next = waiting_list;
  waiting_list = np;
}


/* log somebody out & update the user's time */
void
log_out (struct utmp *entry)
{
  struct list_entry *np, *last;
  
  /* step thru the entries and find it, keeping track of the one
     previous, so we can efficiently update the linked list.  NOTE:
     since we can't look for user names here, we have to assume that
     the most recent login on the terminal is the one we want. */
  
  for (np = waiting_list, last = NULL; np != NULL; last = np, np = np->next)
    {
      if (strncmp (entry->ut_line, np->ut_line, 8) == 0)
	{
	  /* this is the one! */
	  
	  if (np->time > entry->ut_time)
	    {
	      /* this happens when there is a change to/from daylight
	       savings time -- result = +/- 1 hour's worth of ac'd
	       time on those days; we have to kludge here because it's
	       possible for someone to log out before they've logged
	       in (according to localtime(3)) -- solution is to add
	       one hour to the logout time... */

	      entry->ut_time += 3600;
	      if (print_file_inconsistencies)
		fprintf (stddebug,
			 "%s:%ld: logout (daylight savings time) %.8s\n",
			 wtmp_file_name, line_number, np->ut_name);
	    }
	  
	  update_user_time (np->ut_name, entry->ut_time - np->time, "logout");
	  
	  if (last == NULL)
	    waiting_list = np->next;
	  else
	    last->next = np->next;
	  
	  free (np);
	  return;
	}
    }
  /* if we got here, there's a problem with the wtmp file */
  if (print_file_inconsistencies)
    fprintf (stddebug, "%s:%ld: couldn't logout '%.8s'\n", 
	     wtmp_file_name, line_number, entry->ut_line);
}



/* put something into the user table */
void
update_user_time (char *name, long the_time, char *debug_label)
{
  struct user_entry *np;
  long temp;
  int print_flag;
  
  if (name_list == NULL)
    print_flag = debugging_enabled;
  else
    print_flag = debugging_enabled && desired_name (name);
  
  if (print_flag)
    fprintf (stddebug, "\t\t\t\t\t%8.2f %-8.8s (%s) uut\n",
	     (float)the_time / 3600.0, name, debug_label);
  
  for (np = user_list; np != NULL; np = np->next)
    {
      if (strncmp (name, np->name, 8) == 0)  /* it's there! */
	{
	  temp = np->time;
	  np->time += the_time;
	  if (np->time < temp)
	    {
	      char temp_str[255];
	      
	      /* this is an unrecoverable error - quit! */
	      sprintf (temp_str, "%s:%ld: possible number overflow\n"
		       "%.8s b4:%ld aft:%ld add:%ld\n",
		       wtmp_file_name, line_number,
		       np->name, temp, np->time, the_time);
	      fatal (temp_str);
	    }
	  return;
	}
    }
  
  /* if we've reached this point, we didn't find the name in the
     waiting list, so make a new one! */
  
  np = (struct user_entry *) xmalloc (sizeof (struct user_entry));
  
  /* copy the info */
  strncpy (np->name, name, 8);
  np->time = the_time;
  
  /* insert at beginning of list */
  np->next = user_list;
  user_list = np;
}



/* clear all global flags & data */
void
init_flags_and_data (void)
{
  debugging_enabled = 0;
  print_midnight_totals = 0;
  print_individual_totals = 0;
  dont_print_zero_totals = 0;
  nasty_reboot = 0;
  nasty_supplant = 0;
  nasty_time_warp = 0;
  
  time_warp_leniency = 60;
  
  waiting_list = NULL;
  user_list = NULL;
  name_list = NULL;

}


/* return the time of midnight that happened before our time */
long
midnight_before_me (long *the_time)
{
  struct tm *tm_ptr, temp_tm;
  
  tm_ptr = localtime ((time_t *) the_time);
  memcpy ((void *)&temp_tm, (void *)tm_ptr, sizeof (struct tm));
  temp_tm.tm_sec = 0;
  temp_tm.tm_min = 0;
  temp_tm.tm_hour = 0;
  
  return (long) mktime (&temp_tm);
}


/* return the time of midnight that will happen after our time */
long
midnight_after_me (long *the_time)
{
  long temp;
  
  temp = midnight_before_me (the_time) + 93600;     /* add 26 hrs to be safe */
  
  return midnight_before_me (&temp);
}


