#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>

#include "ncd.h"

/*************************************************************************/

void cleanUp( void )
{
	if (_cursesOn) 
		ioTerminate(); 
}

/*************************************************************************/

volatile sig_atomic_t _signaling = 0;

void signalCleanUp (int sig)
{
	if (_signaling) 
		raise (sig);
	_signaling = 1;

	cleanUp();
	
	raise (sig);
}

/*************************************************************************/
/* returns !=0 if error */

int parseOpts(int argc, char *argv[])
{
	int c;
	char hlpmsg[] =
	"Ninux Change Directory 0.9.8 (" __DATE__ ")\n"
	"Copyright (C) 1995  Borja Etxebarria\n"
	"Basque Country University\n"
	"\n"
	"usage: %s [-rRfhHaAvVdDtTxlL?i] [dir]\n"
	" -r -R     force rebuild: on/off(*)\n"
	" -f -h -H  scope area: full/home/auto(*)\n"
	" -a -A     rebuild mode: auto(*)/manual\n"
	" -v -V     verbose: on/off(*)\n"
	" -d -D     dump directory tree: on/interactive(*)\n"
	" -t -T -x  normal text instead of lineart: on/off/auto(*)\n"
	" -l -L     link format: (ldir -> dir)/(ldir@)(*)\n"
	" -? -i     this help\n"
	" dir       direct jump to <dir>\n";

	while ((c = getopt(argc, argv, "rRfhHaAvVdDtTxlL?i")) != -1) {
		switch (c) {
		case 'r':
			_rebuild = 1;
			break;
		case 'R':
			_rebuild = 0;
			break;
		case 'f':
			_scope = 1;
			break;
		case 'h':
			_scope = -1;
			break;
		case 'H':
			_scope = 0;
			break;
		case 'v':
			_verbose = 1;
			break;
		case 'V':
			_verbose = 0;
			break;
		case 'a':
			_rebauto = 1;
			break;
		case 'A':
			_rebauto = 0;
			break;
		case 'd':
			_justdump = 1;
			break;
		case 'D':
			_justdump = 0;
			break;
		case 'T':
			_lineart = 1;
			break;
		case 't':
			_lineart = 0;
			break;
		case 'x':
			_lineart = 2;
			break;
		case 'l':
			_showlink = 1;
			break;
		case 'L':
			_showlink = 0;
			break;
		case '?':
		case 'i':
		default:
			fprintf(stderr, hlpmsg, argv[0]);
			return 1;
		}
	}
	return 0;
}

/*************************************************************************/
/* returns !=0 if error. Parses NCD_OPTS environment switches, and then
   command line switches */

int parseArguments(int argc, char *argv[])
{
	extern int optind;
	int i, ret;
	char *ss[2];

	ss[1] = getenv("NCD_OPTS");
	if (ss[1]) {
		i = optind;
		optind = 0;
		if (parseOpts(2, ss))
			return 1;
		optind = i;
	}
	ret = parseOpts(argc, argv);
	
	if ((ret==0)&&(optind<argc)) {
		strcpy(_argumentDir,argv[optind]);
		if (_argumentDir[0]=='/')  {
			if (_scope==0)
				_scope = 1;
			if (_argumentDir[1]!='\0')
				strcpy(_argumentDir, _argumentDir+1);
		}
		
		if (_verbose) 
			fprintf(stderr,"command-line dir: %s\n",_argumentDir);
	}	                   
	else
		_argumentDir[0]='\0';
	                        
	return ret;
}

/*************************************************************************/

DirNode *rebuildTree(void)
{
	DirNode *rootNode;
	DirNode *dn;
	DirNode homeNode;

	if (_cursesOn) 
		cursRebuildMsg();
	else
		fprintf(stderr, "Rebuilding directory tree...");
	if ((_verbose)&&(!_cursesOn))
		fprintf(stderr, "\n");

	rootNode = readDirsInNodes(_root, NULL);

	if (_scope == 1) {
		dn = searchNodeForDir(rootNode, _root, _xhome);
		if (dn == NULL) {
			cleanUp();
			fprintf(stderr, "error: invalid HOME directory\n");
			exit(1);
		}
		homeNode = *dn;
		homeNode.left = homeNode.up = homeNode.down = NULL;
		homeNode.name = getDirName(_home);
		homeNode.lname = NULL;

		writeNodesToFile(_fullFile, rootNode);
		writeNodesToFile(_homeFile, &homeNode);
	}
	else {
		writeNodesToFile(_homeFile, rootNode);
	}
	if (!_cursesOn)
		fprintf(stderr, " done!\n");

	return rootNode;
}

/*************************************************************************/

DirNode *readTreeFiles(void)
{
	DirNode *homeNode, *rootNode;
	DirNode *dn;

	if (_rebuild) {
		_rebuild = 0;
		return rebuildTree();
	}

	if ((_verbose)&&(!_cursesOn))
		fprintf(stderr, "reading %s...\n", _homeFile);
	rootNode = readNodesFromFile(_homeFile);

	if (_scope == 1) {
		homeNode = rootNode;
		if ((_verbose)&&(!_cursesOn))
			fprintf(stderr, "reading %s...\n", _fullFile);
		rootNode = readNodesFromFile(_fullFile);
		dn = searchNodeForDir(rootNode, _root, _xhome);
		if (dn == NULL) {
			if (_rebauto == 0) {
				cleanUp();
				fprintf(stderr, "error: %s unusable, please rebuild\n", _fullFile);
				exit(1);
			}
			else {
				_rebauto = 0;
				delNodesFromMemory(homeNode);
				delNodesFromMemory(rootNode);
				return rebuildTree();
			}
		}
		delNodesFromMemory(dn->right);
		dn->right = homeNode->right;
		free(homeNode->name);
		if (homeNode->lname)
			free(homeNode->lname);
		*homeNode = *dn;
		if (homeNode->down != NULL)
			(homeNode->down)->up = homeNode;
		if (homeNode->up != NULL)
			(homeNode->up)->down = homeNode;
		else if (homeNode->left != NULL)
			(homeNode->left)->right = homeNode;
		free(dn);
	}

	dn = searchNodeForDir(rootNode, _xroot, _cwd);
	if ((dn == NULL) && (_rebauto != 0)) {
		delNodesFromMemory(rootNode);
		return rebuildTree();
	}

	return rootNode;
}

/*************************************************************************/
/* 1 if cancel */

int getFinalPath( DirNode * node )
{
	char * s;
	
	if (node==NULL) 
		return 1;

	s = getNodeFullPath(node, 0, 0, NULL, 0);
	if ((_verbose)&&(!_cursesOn))
		fprintf(stderr,"selected directory: %s\n",s);
	strcpy(_finalDir, s);

	return 0;
}

/*************************************************************************/

void changeToFinalDir( void )
{
	FILE * f;
	int len;
	
	len = strlen(_finalDir);

	f = fopen(_selDirFile,"w");
	if (f==NULL) {
		cleanUp();
		fprintf(stderr,"error: can't open file %s\n",_selDirFile);
		exit(1);
	}
	if (fprintf(f,"%s",_finalDir)<len) {
		cleanUp();
		fprintf(stderr,"error: can't write file %s\n",_selDirFile);
		exit(1);
	}
	if (fclose(f)!=0) {
		cleanUp();
		fprintf(stderr,"error: can't close file %s\n",_selDirFile);
		exit(1);
	}
}

/*************************************************************************/

void initializeFinalDir( void )
{
	FILE * f;
	
	strcpy(_finalDir,".");

	f = fopen(_selDirFile,"w");
	fprintf(f,"%s",_finalDir);
	fclose(f);
}

/*************************************************************************/

void getGlobals( int init_cwd )
{
	char *s;

	if (init_cwd)
		_cursesOn = 0;

	s = getenv("HOME");
	if (s == NULL) {
		cleanUp();
		fprintf(stderr, "error: can't get HOME directory\n");
		exit(1);
	}
	
	strcpy(_home, s);
	addSlash(_home);
	strcpy(_xhome, _home);

	if (trueDir(_xhome) == NULL) {
		cleanUp();
		fprintf(stderr, "error: invalid HOME directory\n");
		exit(1);
	}
	if ((_verbose)&&(!_cursesOn))
		fprintf(stderr, "HOME=%s (%s)\n", _home, _xhome);

	if (init_cwd) {
		s = getcwd(_cwd, PATH_MAX);
		if (s == NULL) {
			cleanUp();
			fprintf(stderr, "error: can't get current directory\n");
			exit(1);
		}
		addSlash(_cwd);
		if ((_verbose)&&(!_cursesOn))
			fprintf(stderr, "cwd=%s\n", _cwd);

		if (_scope == 0) {
			if (strstr(_cwd, _xhome) == _cwd)
				_scope = -1;
			else
				_scope = 1;
		}
	}

	if (_scope == -1) {
		strcpy(_root, _home);
		strcpy(_xroot, _xhome);
		if ((_verbose)&&(!_cursesOn))
			fprintf(stderr, "scope area: HOME\n");
	}
	else {
		strcpy(_root, "/");
		strcpy(_xroot, _root);
		if ((_verbose)&&(!_cursesOn))
			fprintf(stderr, "scope area: FULL\n");
	}
	strcpy(_homeFile, _home);
	strcpy(_fullFile, _home);
	strcpy(_selDirFile, _home);
	strcat(_homeFile, ".ncd_htree");
	strcat(_fullFile, ".ncd_ftree");
	strcat(_selDirFile, ".ncd_sdir");
	initializeFinalDir();
	
	signal(SIGHUP,signalCleanUp);
	signal(SIGINT,signalCleanUp);
	signal(SIGQUIT,signalCleanUp);
	signal(SIGTERM,signalCleanUp);
	signal(SIGKILL,signalCleanUp);
}

/*************************************************************************/

int main(int argc, char *argv[])
{
	int quit;
	DirNode * dn;
	int ret;

	ret = 1;

	if (parseArguments(argc, argv))
		return 1;

	getGlobals(1);

	_rootNode = readTreeFiles();

	if (_rootNode != NULL) {
		numerateNodeTree(_rootNode, 0, 0);
		_lastNode = getLastDescendant(_rootNode);

		if (_justdump) {
			showTree(_rootNode);
			ret = 2;
		}
		else if (strlen(_argumentDir)>0) {
			dn = directSelectANode();
			ret = getFinalPath(dn);
		}
		else { 
			do {
				dn = selectANode(&quit);
				switch (quit) {
				case 3:  /* rescan */
					_rebuild = 1;
					_scope = -_scope;
				case 4:  /* scope toggle */
					_scope = -_scope;
					if (_scope==-1)
						_rebauto = 0;
					delNodesFromMemory(_rootNode);
					getGlobals(0);
					_rootNode = readTreeFiles();
					quit = 0;
					if (_rootNode != NULL) {
						numerateNodeTree(_rootNode, 0, 0);
						_lastNode = getLastDescendant(_rootNode);
					}
					else
						quit = 1;
					break;
				case 1:
					if ((_verbose)&&(!_cursesOn))
						fprintf(stderr,"operation cancelled by the user!\n");
				case 2:
				default:
					ret = getFinalPath(dn);
					quit = 1;
					break;					
				}
			} while (!quit);
		}

		delNodesFromMemory(_rootNode);
	}
	else
		ret = 1;
	
	changeToFinalDir();

	return ret;
}

/*************************************************************************/
