/* TFileWindow.cc  ---  The file window class implementation
   
   Copyright (C) 1993  A.Matthias
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */

#include <iostream.h>
#include <fstream.h>
extern "C" {
#include <ncurses/ncurses.h>
}
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>

#include "globals.h"  // Global definitions
#include "config.h"   // Site-dependent setup options
#include "TDir.h"     // Directory manager class
#include "TList.h"    // Linked list class
#include "TElem.h"    // Directory entry definition 
#include "TInfo.h"    // The "long filename"-line
#include "TFileWindow.h" // Header for file windows

extern "C" {
#include "trim.h"     // Truncate pathnames to a given length
}


void TFileWindow::fillbuffer( void )  // fills Fname with the directory listing
{
  char dummy[PATH_MAX];
  TElem Elem;            // One directory element
  bool isok;             // Should the processed element be shown?
  
  ifstream dirfile( dirfilename );
  
  if ( !dirfile )
  { 
    endwin(); 
    system( "clear" );
    cout << "Dirfile: " << dirfilename << " not accessible\n"; 
    exit(1); 
  }
  
  List.wipe();   // clear list
  maxnum = 0;
  totalbytes=0;
  werase( wptr );
  
  dirfile >> dummy;  // "Total"-line
  dirfile >> dummy;
  
  while ( dirfile  )
  {
    dirfile >> Elem.fmode;
    dirfile >> dummy;
    dirfile >> Elem.uname;
    dirfile >> Elem.gname;
    dirfile >> Elem.fsize;
    dirfile >> dummy;
    dirfile >> dummy;
    dirfile >> dummy;
    dirfile >> Elem.fname;
    // Handle links correctly
    if ( Elem.fmode[0] == 'l' )
    {
      dirfile >> dummy;    // throw "->" away
      dirfile >> Elem.truename;    // target of link
    }
    
    isok = TRUE;
    if ( (execonly == TRUE) && (strchr( Elem.fmode, 'x')==NULL ) )
      isok = FALSE;
    if ( Elem.fmode[0]=='d' ) 
      isok = TRUE; 
    // don't show "."-entry
    if ( (strcmp( Elem.fname, "." )==0) || 
	(strcmp( Elem.fname, ""  )==0) ) 
      isok = FALSE;
    
    if ( isok==TRUE ) 
    {
      maxnum++;
      totalbytes += Elem.fsize;
      List.add( &Elem );
    }
  }      
  // We must add an empty element at the end of list
  strcpy( Elem.fname, "" );
  strcpy( Elem.fmode, "" );
  strcpy( Elem.uname, "" );
  strcpy( Elem.gname, "" );
  strcpy( Elem.truename, "" );
  Elem.fsize = 0;
  List.add( &Elem );
  maxnum-=1;
  dirfile.close();
}

// insert() returns TRUE if the window is already filled and no more names
// are needed. It returns FALSE if there are still names needed.

bool TFileWindow::insert( TElem *Elem, int num )
{
  char s[PATH_MAX], tmp[PATH_MAX], formatstring[128], fhead[32], ftail[32];
  
  if ( num < firstshown )
    return FALSE;
  if ( num >= firstshown+(y2-y1) )
    return TRUE;
  
  // fname could be longer than we can display, so we truncate
  // after the first FNAME_LEN characters
  strcpy( tmp, Elem->fname  );
  tmp[FNAME_LEN]='\0';
  if ( strcmp( tmp, Elem->fname ) != 0 ) 
    strcat( tmp, TRUNCATE_INDICATOR );
  
  strcpy( fhead, "%-" );
  strcpy( ftail, "s%s %9d" );

  if ( Elem->status==NONE )
    sprintf( formatstring, " %s%d%s", fhead, FNAME_LEN+1, ftail );
  else
    sprintf( formatstring, "*%s%d%s", fhead, FNAME_LEN+1, ftail );

  sprintf( s, formatstring, tmp, Elem->fmode, Elem->fsize );  
  strcpy( onscreen[num-firstshown], s );
  
  if ( hascolor )
    wattron( wptr, COLOR_PAIR( 1 ) );
  if ( num==hilight && active ) wstandout( wptr );
  mvwaddstr( wptr, num-firstshown, 0, s );
  wclrtoeol( wptr );
  if ( num==hilight && active ) wstandend( wptr );
  if ( hascolor )
    wattron( wptr, COLOR_PAIR( 1 ) );
  return FALSE;
}

void TFileWindow::changehilight( int oldhi, int newhi )
{
  wstandend( wptr );
  if ( hascolor )
    wattron( wptr, COLOR_PAIR( 1 ) );
  mvwaddstr( wptr, oldhi-firstshown, 0, onscreen[oldhi-firstshown] );
  wclrtoeol( wptr );
  wstandout( wptr );
  mvwaddstr( wptr, newhi-firstshown, 0, onscreen[newhi-firstshown] );
  wclrtoeol( wptr );
  wstandend( wptr );
  if ( hascolor )
    wattron( wptr, COLOR_PAIR( 1 ) );
  wrefresh( wptr );
}

TFileWindow::TFileWindow( WINDOW *scr, WHICH which ) : Dir( NULL )
{ 
  std=scr;
  scrx=1; scry=1;
  scrw=SCR_X; scrh=SCR_Y;
  TFileWindow::which=which;  // Is this one correct ?
  hilight=0;
  maxnum = 0;
  firstshown = 0;
  totalbytes=0;
  execonly = FALSE;
  strcpy( hilightname, "." );
  for ( int i=0; i<255; i++ )
    onscreen[i][0] = '\0';
  
  if ( which==LEFT ) 
  {
    Dir.setdirfile( DIRFILE_L );
    sprintf( dirfilename, "%s%s", localtmpdir, DIRFILE_L );
    x1=1; x2=scrw/2;
    y1=1; y2=scrh/2+scrh/4;
    active=TRUE;
  }
  else 
  {
    Dir.setdirfile( DIRFILE_R );
    sprintf( dirfilename, "%s%s", localtmpdir, DIRFILE_R );
    x1=(scrw/2); x2=scrw;
    y1=1; y2=scrh/2+scrh/4;
    active=FALSE;
  }
  frameptr=newwin( y2-y1, x2-x1, y1, x1 );
  box( frameptr, FRAME_CHAR_V, FRAME_CHAR_H );
  y1++; x1++; y2--; x2--;
  wptr=newwin( y2-y1, x2-x1, y1, x1 );
  hascolor = has_colors();
  if ( hascolor )
  {
    start_color();
    init_pair( 1, COLOR_WHITE, COLOR_BLUE );
    wattron( wptr, COLOR_PAIR( 1 ) );
    touchwin( wptr );
  }      
  leaveok( wptr, TRUE );
  leaveok( frameptr, TRUE );
  show();
} /* end of constructor */


void TFileWindow::remote( char *host )
{
  Dir.remote( host );
  fill( TRUE );
}

void TFileWindow::local( void )
{
  Dir.local();
  fill( TRUE );
}


void TFileWindow::header( void )
{ 
  char tmp[2048];
  
  box( frameptr, FRAME_CHAR_V, FRAME_CHAR_H );
  if ( active ) wstandout( frameptr );
  mvwaddstr( frameptr, 0, 0, trim(Dir.telldir( cwd ), tmp, 37) ); 
  if ( active ) wstandend( frameptr );
}

void TFileWindow::summary( void )
{ 
  char line[1024], tmp[1024], host[PATH_MAX];
  
  if ( execonly == TRUE )
    sprintf( line, "%ld bytes in %d files (E)", totalbytes, maxnum+1 );
  else
    sprintf( line, "%ld bytes in %d files", totalbytes, maxnum+1 );
  
  sprintf( tmp, "%s [%s]", line, Dir.tellhost( host ) );
  if ( strlen( tmp ) > x2-x1+1 )
    tmp[x2-x1+1] = '\0';
  strcpy( line, tmp );
  
  if ( active ) wstandout( frameptr );
  mvwaddstr( frameptr, y2-1, 0, line ); 
  if ( active ) wstandend( frameptr );
}

void TFileWindow::show( void )
{ 
  header(); 
  summary();
  wrefresh( frameptr ); wrefresh( wptr ); 
}

void TFileWindow::showinside( void )
{
  wrefresh( wptr );
}


void TFileWindow::repaint( void )
{
  touchwin( frameptr );
  touchwin( wptr );
  show();
}


bool TFileWindow::chdir( char *newdir )
{
  char olddir[PATH_MAX], searchdir[PATH_MAX], *start;
  Dir.telldir( olddir );
  if ( Dir.changedir( newdir ) == TRUE )
  {
    Dir.telldir( cwd );
    start = strrchr( olddir, '/' );
    strcpy( searchdir, ++start );
    
    firstshown = 0;
    fillbuffer();
    if ( strlen( newdir ) < strlen( olddir ) ) // => cd ..
      strcpy( hilightname, searchdir );
    
    totalbytes=0;
    return TRUE;
  }
  else
    return FALSE;
}


char* TFileWindow::telldir( char *dir )
{
  return( Dir.telldir( dir ) );
}


// fill() returns TRUE if OK, FALSE if a directory read error occured

bool TFileWindow::fill( bool readdir )
{ 
  TElem *Elem;
  bool ret=TRUE;
  
  for ( int i=0; i<255; i++ )
    onscreen[i][0] = '\0';
  
  if ( readdir == TRUE )
  {
    // Generate a directory listing file
    ret=Dir.writedir();

    fillbuffer();    // and read it into the list Fname
    sethilightname( hilightname );
    fillbuffer();    // and read it into the list Fname
    strcpy( hilightname, gethilightedfname() );
    touchwin( frameptr );
  }
  
  List.rewind();
  
  bool full;               // Is window already full ?
  do
  {
    Elem = List.getelem();   // Get the filenames from list
    // and print them
    if ( Elem != NULL ) full=insert( Elem, List.getnum() ); 
    
    List.next();
  } while ( (Elem != NULL) && ( full != TRUE ) );
  
  if ( readdir == TRUE )
    show();
  else
    showinside();

  return( ret );
}

char* TFileWindow::gethilightedfname( void )
{
  return( List.getfname( hilight ) );
}

// The difference between the following two functions is that
// gethilightedfmode(), if applied to a link, returns the modes
// of the link; gethilightedtruemode() returns the modes of the
// file or directory the link points to. If gethilightedtruemode()
// is applied to a non-link, it returns the file modes as usual.
// NOTE: 
// get...fmode() returns a string representation of permission
// flags, as they appear in an "ls -la" listing !!! 
// get...truemode() returns the st_mode field of a stat structure,
// as defined in sys/stat.h !!!

char* TFileWindow::gethilightedfmode( void )
{
  return( List.getfmode( hilight ) );
}

int TFileWindow::gethilightedtruemode( void )
{
  struct stat sbuf;
  
  stat( List.getfname( hilight ), &sbuf );
  return (sbuf.st_mode);
}


int TFileWindow::gethilight( void )
{ return( hilight ); }


int TFileWindow::getnroflines( void )
{ return( y2-y1 ); }



void TFileWindow::pageup( void )
{
  int oldfirstshown = firstshown;
  int oldhilight = hilight;
  
  if ( firstshown>=getnroflines() )
    firstshown-=getnroflines();
  else
    firstshown=0;
  
  hilight=firstshown;
  
  if ( oldfirstshown!=firstshown ) 
  {
    werase( wptr );
    fill( FALSE );
  }
  
  if ( oldhilight != hilight )
    changehilight( oldhilight, hilight );
  
  if ( LONGNAMES == 1 )
  {
    char tmp[PATH_MAX];
    sprintf( tmp, "%s\t%s\t%s", gethilightedfname(), 
	    List.getuname( hilight ), List.getgname( hilight ) );
    tmp[scrw-2]='\0';  // truncate the line
    Info.show( tmp );
  }
  
  strcpy( hilightname, gethilightedfname() );
}



void TFileWindow::pagedown( void )
{
  int oldfirstshown = firstshown;
  int oldhilight = hilight;
  
  if ( firstshown<=getmaxnum()-getnroflines() )
  {
    firstshown+=getnroflines();
    hilight=firstshown;
  }
  else
    hilight=getmaxnum();
  
  if ( oldfirstshown!=firstshown ) 
  {
    werase( wptr );
    fill( FALSE );
  }
  
  if ( oldhilight != hilight )
    changehilight( oldhilight, hilight );
  
  if ( LONGNAMES == 1 )
  {
    char tmp[PATH_MAX];
    sprintf( tmp, "%s\t%s\t%s", gethilightedfname(), 
	    List.getuname( hilight ), List.getgname( hilight ) );
    tmp[scrw-2]='\0';  // truncate the line
    Info.show( tmp );
  }
  
  strcpy( hilightname, gethilightedfname() );
}



void TFileWindow::sethilight( int value )
{ 
  int oldfirstshown, oldhilight;
  
  oldfirstshown = firstshown;
  oldhilight = hilight;
  
  if ( value >= 0 && value <= getmaxnum() ) hilight = value; 
  if ( hilight >= firstshown+(y2-y1) ) 
  {
    if ( hilight < firstshown+(y2-y1)+2 )
      if ( firstshown+((y2-y1)/2) <= getmaxnum() )
	firstshown+=(y2-y1)/2;
      else ;
    else
      firstshown = hilight - ((y2-y1)/2);
    
  }
  if ( hilight < firstshown ) 
  {
    if ( firstshown-((y2-y1)/2) >= 0 )
      firstshown-=(y2-y1)/2;
    else
      firstshown = 0;
  }
  if ( hilight==0 ) 
    firstshown=0;
  if ( hilight==getmaxnum() )
    firstshown=getmaxnum()-((y2-y1)/2);
  if ( firstshown < 0 ) 
    firstshown=0;
  
  if ( oldfirstshown!=firstshown ) 
  {
    werase( wptr );
    fill( FALSE );
  }
  else
  {
    if ( oldhilight != hilight )
      changehilight( oldhilight, hilight );
    
  }
  if ( LONGNAMES == 1 )
    {
      char tmp[PATH_MAX];
      sprintf( tmp, "%s\t%s\t%s", gethilightedfname(), 
	      List.getuname( hilight ), List.getgname( hilight ) );
      tmp[scrw-2]='\0';  // truncate the line
      Info.show( tmp );
    }

  strcpy( hilightname, gethilightedfname() );
}

// sethilightname() tries to hilight an entry denoted by its name
// If the entry doesn't exist, the first entry is hilighted instead
void TFileWindow::sethilightname( char *name )
{
  TElem *Elem;
  bool found = FALSE;
  int  number = 0;
  
  List.rewind();
  
  do
  {
    Elem = List.getelem();   // Get the filenames from list
    // and print them
    if ( Elem != NULL ) 
    {
      if ( strcmp( Elem->fname, name ) == 0 )
      {
	found = TRUE;
	number = List.getnum();
      }
      List.next();
    }
  } while ( (Elem != NULL) && ( found==FALSE ) );
  //      if ( hilight!=number) 
  sethilight( number ); 
  // fprintf( stderr, "wanted:%s found:%s %d\n", name, Elem->fname, number );
}

WINDOW* TFileWindow::getwptr( void )
{
  return( wptr );
}

int TFileWindow::getmaxnum( void )
{ return ( maxnum ); }

bool TFileWindow::isactive( void )
{
  if ( active )
    return ( TRUE );
  else
    return ( FALSE );
}

void TFileWindow::activate( void )
{ active = TRUE; Dir.activate(); /*sethilight( 0 )*/; fill( TRUE ); }

void TFileWindow::inactivate( void )
{ active = FALSE; fill( TRUE ); }


// Bring window to top 
void TFileWindow::top( void )
{ 
  touchwin( frameptr ); 
  wrefresh( frameptr ); 
  touchwin( wptr ); 
  wrefresh( wptr ); 
}

void TFileWindow::toggleentry( void )
{
  if ( *gethilightedfmode() != 'd' )  // don't mark directories
    if ( List.getstatus( hilight ) != MARK )
      List.mark( hilight );
    else
      List.unmark( hilight );
  sethilight( hilight+1 );
  fill( FALSE );
}  


void TFileWindow::markpattern( char *pattern )
{
  List.markpattern( pattern );
  fill( FALSE );
}


void TFileWindow::unmarkpattern( char *pattern )
{
  List.unmarkpattern( pattern );
  fill( FALSE );
}


void TFileWindow::postprocess( char *lastcommand )
     // This function sets the cwd and the environment of the current shell
     // to the values generated by the subshell which executed the last
     // user command and sets the name to be hilighted to the name
     // of the last executed command
{
  // First we evaluate the cwd
  char line[PATH_MAX], cwdfilename[PATH_MAX];
  sprintf( cwdfilename, "%s%s", localtmpdir, CWDFILE );
  ifstream f( cwdfilename );
  if ( !f )
    return;
  f >> line;
  f.close();
  this->chdir( line );
  if ( strcmp( lastcommand, "" ) != 0 )
  {
    strcpy( hilightname, lastcommand );
    // sethilightname( lastcommand );
  }
}

void TFileWindow::makepfile( char *what, char *param )
{
  char pfilename[PATH_MAX];
  TElem *Elem;  
  
  sprintf( pfilename, "%s%s", localtmpdir, PARAMFILE );
  ofstream pfile( pfilename );
  if ( !pfile )
    return;
  switch( what[0] )
  {
  case 'h':
    pfile << gethilightedfname();
    break;
  case 'p':
    pfile << param;
    break;
  case 'm':
  case 'u':
    bool marked_exist = FALSE;
    List.rewind();
    do
    {
      Elem = List.getelem();
      if ( ( (Elem->status==MARK) && (what[0]=='m') ) ||
	  ( (Elem->status==NONE) && (what[0]=='u') ) )
      {
	pfile << Elem->fname << ' ';
	marked_exist = TRUE;
      }
      List.next();
    }
    while ( Elem != NULL );
    
    // If nothing is marked, return the hilighted filename
    if ( ( marked_exist == FALSE ) && ( what[0]=='m' ) )
      pfile << gethilightedfname();
    
    break;
  }
  
  pfile.close();
}

void TFileWindow::setexeconly( bool value )
{
  execonly = value;
}

bool TFileWindow::getexeconly( void )
{
  return( execonly );
}


// ***********************************************************************
// Here are all the remote operations
// ***********************************************************************

// Remote Get
void TFileWindow::RemGet( char *remotefile, char *localfile )
{
  Dir.RemGet( remotefile, localfile );
}

// Remote Put
void TFileWindow::RemPut( char *localfile, char *remotefile )
{
  Dir.RemPut( localfile, remotefile );
}
