/*
 * Memory management debugging routines
 */

#include <stdio.h>

#ifdef __MSDOS__
#include <alloc.h>
#define PTR_TO_LONG(p)	((((unsigned long)(p)) >> 16l)*16l +\
				(unsigned long)((unsigned short)(p)))
#else
#include <malloc.h>
/* added the above line */
#include <stdlib.h>
#define PTR_TO_LONG(p)	((unsigned long)p)
#endif
/*
************************* MEMORY MANAGEMENT *****************************

	MEM_DEBUG - tells the system to include code to keep track of up
			to MEMTABLESIZE pointers, and check all attempts
			to free blocks for legality. Mem_Check can be
			called to report the location and size of all
			currently allocated memory chunks.

 The functions are:
	Mem_Init : Allocate and initialise the memory system. If neither
		of the conditional compiles are being used, this does
		nothing.
	Mem_Check : If MEM_DEBUG is defined, print details of all currently
		allocated chunks. If the Boolean parameter is true, the
		check table is freed up as well.
	Mem_Calloc : A `calloc' substitute. An extra parameter is used to
		identify the caller. This means that if MEM_DEBUG is set
		and an error occursin Mem_Free, the original allocation
		call can be identified.
	Mem_Malloc : As above, but a `malloc' workalike.
	Mem_Free: A `free' workalike, including error checking if MEM_DEBUG
		is defined.
*/

static struct MemTableEntry {
	long p;
	int size, who;
} *MemTable=NULL;

#ifdef MEM_DEBUG

#define MEMTABLESIZE	2500

#if __STDC__
static int findPointer(char *p) {
#else
static int findPointer(p)
	char *p;
{
#endif
	long pv = PTR_TO_LONG(p);
	int i;
	for (i=0;i<MEMTABLESIZE;i++) {
		if (MemTable[i].p == pv) break;
	}
	return i;
}
#endif

#if __STDC__
void Mem_Init(void) {
#else
void Mem_Init() {
#endif
#ifdef MEM_DEBUG
	int i;
	/* Allocate pointer tracking table and initialise it */
	if (MemTable==NULL)
		MemTable = (struct MemTableEntry *)calloc(MEMTABLESIZE,sizeof(struct MemTableEntry));
	/* else print warning */
	if (MemTable==NULL) {
		fprintf(stderr,"Cannot allocate memory table!\n");
		exit(0);
	} else for (i=0;i<MEMTABLESIZE;i++) {
		MemTable[i].p = 0l;
		MemTable[i].size = 0;
	}
#endif
}

#if __STDC__
void Mem_Check(int free_table) {
#else
void Mem_Check(free_table)
	int free_table;
{
#endif
#ifdef MEM_DEBUG
	if (MemTable) {
		int i;
		for (i=0;i<MEMTABLESIZE;i++)
			if (MemTable[i].p != 0l)
				fprintf(stderr,"Memory block of %d bytes currently allocated at %08lX by %d\n",
					MemTable[i].size,MemTable[i].p, MemTable[i].who);
		if (free_table) {
			free(MemTable);
			MemTable = NULL;
		}
	}
#endif
}

#if __STDC__
char *Mem_Calloc(int nelts, int size, int who) {
#else
char *Mem_Calloc(nelts, size, who)
	int nelts, size, who;
{
#endif
	char *tmp;
	tmp = (char *)calloc(nelts,size);
	if (tmp==NULL) {
		fprintf(stderr,"Calloc failure! Bytes %d ID %d\n",
				nelts*size, who);
		return NULL;
	}
#ifdef MEM_DEBUG
	if (MemTable) {
		int i;
		for (i=0;i<MEMTABLESIZE;i++) {
		     	if (MemTable[i].p==0l) {
				MemTable[i].p = PTR_TO_LONG(tmp);
				MemTable[i].size = nelts*size;
				MemTable[i].who = who;
				break;
			}
		}
		if (i>=MEMTABLESIZE)
			fprintf(stderr,"Memory table full!\n");
	}
#endif
	return tmp;
}

#if __STDC__
char *Mem_Malloc(int nbytes, int who) {
#else
char *Mem_Malloc(nbytes, who)
	int nbytes, who;
{
#endif
	return Mem_Calloc(nbytes,1,who);
}

#if __STDC__
char *Mem_Realloc(char *p, int nbytes, int who) {
#else
char *Mem_Realloc(p, nbytes, who)
	char *p;
	int nbytes, who;
{
#endif
	char *newp = (char *)realloc(p,nbytes);
#ifdef MEM_DEBUG
	int i;
	if (p && MemTable) {
		i = findPointer(p);
		if (i>=MEMTABLESIZE)
			fprintf(stderr,"Attempting realloc of unknown pointer, ID %d\n",who);
		else {
			MemTable[i].p = PTR_TO_LONG(newp);
			MemTable[i].size = nbytes;
		}
	}
#endif
	return newp;
}

#if __STDC__
void Mem_Free(char *p, int who)
#else
void Mem_Free(p, who)
	char *p;
	int who;
#endif
{
#ifdef MEM_DEBUG
	if (p && MemTable) {
		int i = findPointer(p);
		if (i>=MEMTABLESIZE)
			fprintf(stderr,"Illegal attempt to free memory, ID %d\n",who);
		else {
			MemTable[i].p = 0l;
			MemTable[i].size = 0;
		}
	}
#endif
#if __STDC__
	if (p) free((void *)p);
#else
	if (p) free(p);
#endif
}

