/*
 *       open.c open a vt to run a new command (or shell).
 *       
 *	 Copyright (c) 1994 by Jon Tombs <jon@gtex02.us.es>
 *
 *       This program is free software; you can redistribute it and/or
 *       modify it under the terms of the GNU General Public License
 *       as published by the Free Software Foundation; either version
 *       2 of the License, or (at your option) any later version.
 */

#include "open.h"

const char *OPENversion = "open: 1.1 (c) Jon Tombs 1994";

#ifndef VTNAME
#error vt device name must be defined in open.h
#endif


int 
main(int argc, char *argv[])
{

   int fd = 0;
   int opt, pid;
   struct vt_stat vt;
   int vtno     = -1;
   char show    = FALSE;
   char login   = FALSE;
   char verbose = FALSE;
   char do_wait	= FALSE;
   char vtname[sizeof VTNAME + 2]; /* allow 999 possible VTs */
   char *cmd, *def_cmd = NULL;

   /*
    * I don't like using getopt for this, but otherwise this gets messy.
    * POSIX/Gnu getopt forces the use of -- to separate child/program
    * options. RTFM.
    */
   while ((opt = getopt(argc, argv, "c:lsvw")) != -1) {
      switch (opt) {
	case 'c':
	  vtno = (int) atol(optarg);
	  if (vtno < 0 || vtno > 99) {	  
	    fprintf(stderr, "open: %s illegal vt number\n", optarg); 
	    return 5;
	  }
	  /* close security holes - until we can do this safely */
	  (void) setuid(getuid());
	  break;
	case 'l':
	  login = TRUE;
	  break;
	case 's':
	  show = TRUE;
	  break;
	case 'v':
	  verbose = TRUE;
	  break;	    	    
	case 'w':
	  do_wait = TRUE;
	  break;
	default:
	  usage(1);
	
      }
   }
	    

   if (!(argc > optind)) {
      def_cmd = getenv("SHELL");
      if (def_cmd == NULL)
	usage(0);
   }


   if (vtno == -1) {
     if ((fd = open("/dev/console",O_WRONLY,0)) < 0) {
	perror("open: Failed to open /dev/console\n");	
	return(2);
     }

     
     if ((ioctl(fd, VT_OPENQRY, &vtno) < 0) || (vtno == -1)) {
        perror("open: Cannot find a free VT\n");
        close(fd);
        return(3);
     }

     if (ioctl(fd, VT_GETSTATE, &vt) < 0) {
	perror("open: can't get VTstate\n");
        close(fd);
        return(4);
     }
   }

   sprintf(vtname, VTNAME, vtno);

   /* We assume getty has made any in use VT non accessable */
   if (access(vtname, R_OK | W_OK) < 0) {
      fprintf(stderr, "open: Failed to open %s read/write (%s)\n", vtname,
	     strerror(errno));
      return (5);
   }

   if (!geteuid()) {
      uid_t uid = getuid();
      chown(vtname, uid, getgid());
      setuid(uid);
   }

   if (verbose)
	fprintf(stderr,	"open: using VT %s\n", vtname);
	
   cmd = malloc(strlen(argv[optind] + 2));

   if (login)
      strcpy(cmd, "-");
   else
      cmd[0] = '\0';

   if (def_cmd)
      strcat(cmd, def_cmd);
   else
      strcat(cmd, argv[optind]);

   if (login) 
      argv[optind] = cmd++;
   	

   if((pid=fork()) == 0) {
      /* leave current vt */
#ifdef   ESIX_5_3_2_D
      if (setpgrp() < 0) {
#else
      if (setsid() < 0) {
#endif      
        fprintf(stderr, "open: Unable to set new session (%s)\n",
	strerror(errno));
      }
      close(0);
      close(1);
      close(2);
      close(fd);	 

      /* and grab new one */
      if ((fd = open(vtname, O_RDWR)) == -1) { /* Shouldn't happen */
        _exit (4); /* silently die */
      }
      dup(fd); dup(fd);

      if (show) {
	/* 
         * Can't tell anyone if any of these fail, so throw away
	 * the return values 
         */
        (void) ioctl(fd, VT_ACTIVATE, vtno);
        /* wait to be really sure we have switched */
	(void) ioctl(fd, VT_WAITACTIVE, vtno);
      }
      if (def_cmd)
         execlp(cmd, def_cmd, NULL);
      else
	 execvp(cmd, &argv[optind]);
   }
   if ( pid < 0 ) {
      perror("open: fork() error");
      return(6);
   }


   if ( do_wait ) {
      wait(NULL);
      if (show) { /* Switch back... */
	 (void) ioctl(fd, VT_ACTIVATE, vt.v_active);
	 /* wait to be really sure we have switched */
	 (void) ioctl(fd, VT_WAITACTIVE, vt.v_active);
      }
   }

   close (fd);
   return 0;
}
      

void usage(int stat)
{
   fprintf(stderr,
      "Usage: open [-c vtnumer ][-l] [-s] [-v] -- command_line\n");
   exit (stat);
}


