/*
 * cmdline.c	- get command line arguments / environment
 *
 * Copyright (c) 1992 Branko Lankester
 *
 */
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <limits.h>
#include <string.h>
#include <linux/sched.h>
#include "ps.h"

#define	MAX_SWAPFILES	8

#define	STKBUF	2048	/* must be bigger than MAXCMD */

int maxcmd = 132;

static int vmread(void *buf, unsigned long pdir, unsigned long addr, int n);
static int swapread(void *buf, int swap_nr, off_t pos, int n);

/*
 * return string of command and arguments of task
 */
char *
cmd_args(struct task_struct *task)
{
    static char buf[STKBUF]; 
    int n;
    char *p, *commline;

    if (task == NULL)
	return("");

    if (kern_comm) {
	buf[0] = 0;
	return(strncat((char *) buf, task->comm, 
		sizeof task->comm));
    }
    if (task->state == TASK_ZOMBIE)
	return("<defunct>");
    if (task->pid == 0)
	return("scheduler");
    
    if ((n = vmread(buf, task->tss.cr3, task->mm->arg_start, STKBUF)) <= 0)
	goto bad;

    if (show_env)
	n = task->mm->env_end - task->mm->arg_start;
    else
	n = task->mm->arg_end - task->mm->arg_start;
    p = &buf[0];
    if (maxcmd < n)
	n = maxcmd;
    commline = p;
    p[n] = '\0';
    while (1) {
	for (; *p != '\0'; ++p)
	    if (*p < ' ' || *p > '\176')
		*p = ' ';
	if (p < commline + n)
	    *p = ' ';
	else
	    break;
    }
    while (*--p == ' ')
	;
    p[1] = '\0';
    /* if cmd empty or dash (login shell): also give real shell name */
    if (!show_env && (*commline == '-' || *commline == '\0')) {
	strcat(commline, " (");
	strncat(commline, task->comm, sizeof task->comm);
	strcat(commline, ")");
	commline[maxcmd] = '\0';
    }
    return(commline);

bad:
    strcpy((char *) buf, " (");
    strncat((char *) buf, task->comm, sizeof task->comm);
    strcat((char *) buf, ")");
    return((char *) buf);
}

static int
vmread(void *buf, unsigned long pdir, unsigned long addr, int n)
{
    
	unsigned long pde, pte, tmp;
	int cnt, count = n;
	static unsigned long high_mem;
	int rd;

	if (count <= 0)
		return(0);
	
	if (!high_mem) {
		high_mem = get_kword(k_addr("_high_memory"));
	}
/* rep: */
	pde = addr >> 20 & 0xffc;
	if (((pte = get_kword(pdir + pde)) & 1) == 0)
		return(n - count);	/* page table not present */
	pte &= 0xfffff000;
	pte += addr >> 10 & 0xffc;

	cnt = 0x1000 - (addr & 0xfff);
	while (1) {
		if (cnt > count)
			cnt = count;
		if ((tmp = get_kword(pte)) == 0)
			break;
		if (tmp & 1) {
			tmp = (tmp & 0xfffff000) + (addr & 0xfff);
			if (tmp >= high_mem)
				break;
			rd = kmemread(buf, tmp, cnt);
		} else {
			rd = swapread(buf, (tmp&0xffe) >> 1, (tmp & 0xfffff000) + (addr & 0xfff), cnt);
		}
		if (rd != cnt) {
			if (rd > 0)
				count -= rd;
			break;
		}
		if ((count -= cnt) == 0)
			break;
		buf += cnt;
		addr += cnt;
		cnt = 0x1000;
		if (((pte += 4) & 0xfff) == 0)
			break; /* goto rep; */
	}
	return(n - count);
}

static int
swapread(void *buf, int swap_nr, off_t pos, int n)
{
    static int swapfd[MAX_SWAPFILES] = {0,};

    /**
    static int nr_swapfiles = 0;

    if (!nr_swapfiles)
	nr_swapfiles = get_kword(k_addr("_nr_swapfiles"));
    if (swap_nr >= nr_swapfiles)
	return(0);
	**/
    if (!swapfd[swap_nr])
	swapfd[swap_nr] = open(swappath[swap_nr], 0);

    if (swapfd[swap_nr] == -1)
	return(0);

    lseek(swapfd[swap_nr], pos, 0);
    return(read(swapfd[swap_nr], buf, n));
}
