#include <stdio.h>
#include "printer.h"
#include "decoder.h"

extern char *myname;
int rawmode, lines, columns, maxcol, threshold, color;
static int pbm_pixel(FILE *fp);
static void ppm_pixel(FILE *fp,
		      char *cyan, char *magenta, char *yellow, char *black);
static void pbm_decode(FILE *fp);
static void ppm_decode(FILE *fp);
static unsigned char *pb_gets(FILE *fp);
static char dots[4];
#define comefrom(x) x##:	/* ;-) */

void pnm_decode (char *fname, int printer) {
  char *buf;
  FILE *fp;

  if (!scale_x) scale_x = 1000;
  if (!scale_y) scale_y = 1000;
  
  if (fname) {
    if ((fp = fopen (fname, "r")) == NULL) {
      fprintf (stderr, "%s: can't open `%s' for reading\n", myname, fname);
      exit (1);
    }
  } else {
    fname = "stdin";
    fp = stdin;
  }

  if (!(buf=pb_gets(fp))) goto error;
  if (buf[0]!='P') goto nomagic;
  switch(buf[1]) {
  case '1':		/* pbmascii */
    rawmode=0;
    color=0;
    break;
  case '3':		/* ppmascii */
    rawmode=0;
    color=1;
    break;
  case '4':		/* pbmraw */
    rawmode=1;
    color=0;
    break;
  case '6':		/* ppmraw */
    rawmode=1;
    color=1;
    break;
  case '2':
  case '5':
    fprintf(stderr, "%s: graymaps not supported, use pgmtopbm first\n",
	    myname);
    exit(1);
  default:
    comefrom(nomagic) {
      fprintf(stderr, "%s: bad magic number - not a pbm or ppm file\n",myname);
      exit(1);
    }
  }

  if (!(buf=pb_gets(fp))) goto error;
  columns=atoi(buf);
  if (columns<=0) {
    fprintf(stderr, "%s: junk in file where an integer should be\n", myname);
    exit(1);
  }

  if (!(buf=pb_gets(fp))) goto error;
  lines=atoi(buf);
  if (lines<=0) {
    fprintf(stderr, "%s: junk in file where an integer should be\n", myname);
    exit(1);
  }

  if (color) {
    if (!(buf=pb_gets(fp))) goto error;
    maxcol=atoi(buf);
    if (maxcol<=0) {
      fprintf(stderr, "%s: junk in file where an integer should be\n", myname);
      exit(1);
    }
    if (maxcol>1)
      fprintf(stderr, "%s: Warning: input contains more than eight colors\n",
	      myname);
    threshold=maxcol>>1;
  }

  init_printer(printer, color);
  if (color) ppm_decode(fp); else pbm_decode(fp);
  fclose (fp);
  prt_feedup();
  return;

  comefrom(error) {
    fprintf(stderr,"%s: EOF / read error\n",myname);
    exit(1);
  }
}

/*
 * Return the next pixel from a pbm file
 *
 */
static int pbm_pixel(FILE *fp) {
  char *buf;

  if (rawmode) {
    static int byte = 0;
    static int mask = 0;
    int result;

    if (mask) {
      result = byte & mask;
      mask=mask>>1;
      return result;
    } else {
      if ((byte=getc(fp))==EOF) goto error;
      mask=64;
      return byte & 128;
    }
  } else {
    if ((buf=pb_gets(fp))==0) goto error;
    return atoi(buf);
  }

  comefrom(error) {
    fprintf(stderr, "%s: EOF / read error\n", myname);
    exit(1);
  }
}

/*
 * Return the next pixel, convert RGB into CMYK on the way.
 */
static void ppm_pixel(FILE *fp,
		     char *cyan, char *magenta, char *yellow, char *black) {
  unsigned char *buf;
  int red, green, blue;

  if (rawmode) {
    if ((red=getc(fp))==EOF) goto error;
    if ((green=getc(fp))==EOF) goto error;
    if ((blue=getc(fp))==EOF) goto error;
  } else {
    if ((buf=pb_gets(fp))==0) goto error;
    red=atoi(buf);
    if ((buf=pb_gets(fp))==0) goto error;
    green=atoi(buf);
    if ((buf=pb_gets(fp))==0) goto error;
    blue=atoi(buf);
  }

  *cyan=(red<=threshold);
  *magenta=(green<=threshold);
  *yellow=(blue<=threshold);

  if (*cyan && *magenta && *yellow) {
    *black=1; *cyan=0; *magenta=0; *yellow=0;
  } else {
    *black   = 0;
  }
  return;

  comefrom(error) {
    fprintf(stderr, "%s: EOF / read error\n", myname);
    exit(1);
  }
}

/*
 * Read the next field from the input
 *
 */
static unsigned char *pb_gets(FILE *fp) {
  static unsigned char buf[80];
  int c;
  int p=0;

  for(;;) {
    while (((c=getc(fp))==32) || (c==9) || (c==13) || (c==10));
    if (c=='#') {
      while (((c=getc(fp))!=13) && (c!=10));
    } else {
      break;
    }
  }

  if (c==EOF) return 0;

  buf[p++]=c;

  while (((c=getc(fp))!=32) && (c!=9) && (c!=13) && (c!=10) && (p<79))
    buf[p++]=c;

  if (p>=79) {
    fprintf(stderr, "%s: input line too long\n", myname);
    exit(1);
  }

  buf[p]=0;
  return buf;
}


static void pbm_decode(FILE *fp) {
  int x,y,from,to;
  
  /* We search for a straight run of 1's in the input and pass them
   * to the printer driver at once.
   * Passing each single pixel would lead to holes in the output if
   * output scaling is used.
   */

  for (y=0;y<lines;y++) {
    from=to=0;
    for (x=0;x<columns;x++) {
      if (pbm_pixel(fp)) {
	to=x+1;
      } else {
	if (to>from) prt_draw(0, from, to-from);
	from=to=x+1;
      }
    }
    if (to>from) prt_draw(0, from, to-from);
    prt_next();
    while (rawmode && (x&7)) { pbm_pixel(fp); x++; }
  }
}

static void ppm_decode (FILE *fp) {
  int x,y,c,from[4],to[4];

  /* We search for a straight run of 1's in the input for each color
   * and pass them to the printer driver at once.
   * Passing each single pixel would lead to holes in the output if
   * output scaling is used.
   */
  for (y=0;y<lines;y++) {
    for (c=0; c<4; c++) from[c]=to[c]=0;
    for (x=0;x<columns;x++) {
      ppm_pixel(fp, &dots[CYAN], &dots[MAGENTA], &dots[YELLOW], &dots[BLACK]);
      for (c=0; c<4; c++) {
	if (dots[c]) {
	  to[c]=x+1;
	} else {
	  if (to[c]>from[c]) prt_draw(c, from[c], to[c]-from[c]);
	  from[c]=to[c]=x+1;
	}
      }
    }
    for (c=0; c<4; c++)
      if (to[c]>from[c]) prt_draw(c, from[c], to[c]-from[c]);
    prt_next();
  }
}

