/*
 * Copyright (c) 1982, 1986, 1988, 1990, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)udp_usrreq.c	8.4 (Berkeley) 1/21/94
 * udp_usrreq.c,v 1.4 1994/10/02 17:48:45 phk Exp
 */

/*
 * Changes and additions relating to SLiRP
 * Copyright (c) 1995 Danny Gasparovski.
 * 
 * Please read the file COPYRIGHT for the 
 * terms and conditions of the copyright.
 */

#include "h/common.h"
#include <sys/socket.h>

#ifdef HAVE_SYS_FILIO_H
#include <sys/filio.h>
#endif

#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif

#include "h/ip.h"
#include "h/udp.h"
#include "h/socket.h"
#include "h/main.h"

struct udpstat udpstat;

struct socket udb;

/*
 * UDP protocol implementation.
 * Per RFC 768, August, 1980.
 */
#ifndef	COMPAT_42
int	udpcksum = 1;
#else
int	udpcksum = 0;		/* XXX */
#endif

struct	socket *udp_last_so = &udb;

void
udp_init()
{
	udb.so_next = udb.so_prev = &udb;
}

void
udp_input(m, iphlen)
	register struct mbuf *m;
	int iphlen;
{
	register struct ip *ip;
	register struct udphdr *uh;
	struct mbuf *opts = 0;
	int len;
/*	struct ip save_ip; */
	struct socket *so;

#ifdef DEBUG
	debug_call(dfd,"udp_input(m,iphlen) called ...\n");
	debug_call(dfd,"    m = %d\n    iphlen = %d\n", (int)m, iphlen);
	fflush_call(dfd);
#endif

	udpstat.udps_ipackets++;

	/*
	 * Strip IP options, if any; should skip this,
	 * make available to user, and use on returned packets,
	 * but we don't yet have a way to check the checksum
	 * with options still present.
	 */
	if (iphlen > sizeof (struct ip)) {
		ip_stripoptions(m, (struct mbuf *)0);
		iphlen = sizeof(struct ip);
	}

	/*
	 * Get IP and UDP header together in first mbuf.
	 */
	ip = mtod(m, struct ip *);
	uh = (struct udphdr *)((caddr_t)ip + iphlen);

	/*
	 * Make mbuf data length reflect UDP length.
	 * If not enough data to reflect UDP length, drop.
	 */
	len = ntohs((u_int16_t)uh->uh_ulen);

	if (ip->ip_len != len) {
		if (len > ip->ip_len) {
			udpstat.udps_badlen++;
			goto bad;
		}
		m_adj(m, len - ip->ip_len);
		ip->ip_len = len;
	}
	
	/*
	 * Save a copy of the IP header in case we want restore it
	 * for sending an ICMP error message in response.
	 */
/*	save_ip = *ip; */

	/*
	 * Checksum extended UDP header and data.
	 */
	if (udpcksum && uh->uh_sum) {
		((struct ipovly *)ip)->ih_next = 0;
		((struct ipovly *)ip)->ih_prev = 0;
		((struct ipovly *)ip)->ih_x1 = 0;
		((struct ipovly *)ip)->ih_len = uh->uh_ulen;
		uh->uh_sum = cksum(m, len + sizeof (struct ip));
		if (uh->uh_sum) {
			udpstat.udps_badsum++;
			m_freem(m);
			return;
		}
	}

	/*
	 * Locate pcb for datagram.
	 */
	so = udp_last_so;
	if (so->so_lport != uh->uh_sport ||
	    so->so_laddr.s_addr != ip->ip_src.s_addr) {
		struct socket *tmp;
		
		for (tmp = udb.so_next; tmp != &udb; tmp = tmp->so_next) {
			if (tmp->so_lport == uh->uh_sport &&
			    tmp->so_laddr.s_addr == ip->ip_src.s_addr)  {
				tmp->so_faddr.s_addr = ip->ip_dst.s_addr;
				tmp->so_fport = uh->uh_dport;
				so = tmp;
				break;
			}
		}
		if (tmp == &udb)
		   so = NULL;
	}
	
	if (so == NULL) {
		/*
		 * If there's no socket for this packet,
		 * create one
		 */
		if ((so = socreate()) == NULL)
			goto bad;
		udp_attach(so);
		
		/*
		 * Setup fields
		 */
/*		udp_last_so = so; */
		so->so_laddr = ip->ip_src;
		so->so_lport = uh->uh_sport;
		so->so_faddr = ip->ip_dst; /* XXX */
		so->so_fport = uh->uh_dport; /* XXX */

		if ((so->so_iptos = udp_tos(so)) == 0)
			so->so_iptos = ip->ip_tos;
		
	}

	iphlen += sizeof(struct udphdr);
	m->m_len -= iphlen;
	m->m_data += iphlen;

	/*
	 * Now we sendto() the packet.
	 */
	if (so->so_iptos & IPTOS_EMU)
	   udp_emu(so, m);
	sosendto(so,m);
	m_free(m);

	return;
bad:
	m_freem(m);
	if (opts)
		m_freem(opts);
}

/*
 * Create a "control" mbuf containing the specified data
 * with the specified type for presentation with a datagram.
 */

#ifdef notdef

struct mbuf *
udp_saveopt(p, size, type)
	caddr_t p;
	register int size;
	int type;
{
	register struct cmsghdr *cp;
	struct mbuf *m;

	if ((m = m_get(M_DONTWAIT, MT_CONTROL)) == NULL)
		return ((struct mbuf *) NULL);
	cp = (struct cmsghdr *) mtod(m, struct cmsghdr *);
	memcpy(CMSG_DATA(cp), p, size);
	size += sizeof(*cp);
	m->m_len = size;
	cp->cmsg_len = size;
	cp->cmsg_level = IPPROTO_IP;
	cp->cmsg_type = type;
	return (m);
}

/*
 * Notify a udp user of an asynchronous error;
 * just wake up so that he can collect error status.
 */

static void
udp_notify(inp, errno)
	register struct inpcb *inp;
	int errno;
{
	inp->inp_socket->so_error = errno;
	sorwakeup(inp->inp_socket);
	sowwakeup(inp->inp_socket);
}

void
udp_ctlinput(cmd, sa, ip)
	int cmd;
	struct sockaddr *sa;
	register struct ip *ip;
{
	register struct udphdr *uh;
	extern struct in_addr zeroin_addr;
	extern u_char inetctlerrmap[];

	if (!PRC_IS_REDIRECT(cmd) &&
	    ((unsigned)cmd >= PRC_NCMDS || inetctlerrmap[cmd] == 0))
		return;
	if (ip) {
		uh = (struct udphdr *)((caddr_t)ip + (ip->ip_hl << 2));
		in_pcbnotify(&udb, sa, uh->uh_dport, ip->ip_src, uh->uh_sport,
			cmd, udp_notify);
	} else
		in_pcbnotify(&udb, sa, 0, zeroin_addr, 0, cmd, udp_notify);
}

#endif /* notdef */

int
udp_output(so, m, addr)
	struct socket *so;
	struct mbuf *m;
	struct sockaddr_in *addr;
{
	register struct udpiphdr *ui;
	int error = 0;

#ifdef DEBUG
	debug_call(dfd,"udp_output(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif

	/*
	 * Adjust for header
	 */
	m->m_data -= sizeof(struct udpiphdr);
	m->m_len += sizeof(struct udpiphdr);
	
	/*
	 * Fill in mbuf with extended UDP header
	 * and addresses and length put into network format.
	 */
	ui = mtod(m, struct udpiphdr *);
	ui->ui_next = ui->ui_prev = 0;
	ui->ui_x1 = 0;
	ui->ui_pr = IPPROTO_UDP;
	ui->ui_len = htons(m->m_len - sizeof(struct ip)); /* + sizeof (struct udphdr)); */
	ui->ui_src = addr->sin_addr;
	ui->ui_dst = so->so_laddr;
	ui->ui_sport = addr->sin_port;
	ui->ui_dport = so->so_lport;
	ui->ui_ulen = ui->ui_len;

	/*
	 * Stuff checksum and output datagram.
	 */
	ui->ui_sum = 0;
	if (udpcksum) {
	    if ((ui->ui_sum = cksum(m, /* sizeof (struct udpiphdr) + */ m->m_len)) == 0)
		ui->ui_sum = 0xffff;
	}
	((struct ip *)ui)->ip_len = m->m_len;

	((struct ip *)ui)->ip_ttl = ip_defttl;
	((struct ip *)ui)->ip_tos = so->so_iptos;
	
	udpstat.udps_opackets++;
	
	error = ip_output(so, m);
	
	return (error);
}

int
udp_attach(so)
	struct socket *so;
{
	int n = 1;
	
	so->s = socket(AF_INET,SOCK_DGRAM,0);
	ioctl(so->s, FIONREAD, &n);
	so->so_expire = curtime + SO_EXPIRE;
	insque(so,&udb);
	return so->s;
}

void
udp_detach(so)
	struct socket *so;
{
	close(so->s);
	sofree(so);
}

struct tos_t {
	u_int16_t lport;
	u_int16_t fport;
	u_int8_t tos;
};

struct tos_t udptos[] = {
	{0, 53, IPTOS_LOWDELAY},			/* DNS */
	{0, 517, (IPTOS_LOWDELAY|IPTOS_EMU)},	/* talk */
	{0, 518, (IPTOS_LOWDELAY|IPTOS_EMU)},	/* ntalk */
	{517, 0, (IPTOS_LOWDELAY|IPTOS_EMU)},     /* talk response */
	{518, 0, (IPTOS_LOWDELAY|IPTOS_EMU)},     /* ntalk response */
	{0, 0, 0}
};

u_int8_t
udp_tos(so)
	struct socket *so;
{
	int i = 0;
	
#ifdef DEBUG
	debug_call(dfd,"udp_tos(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	
	while(udptos[i].tos)  {
		if (ntohs(so->so_fport) == udptos[i].fport ||
		    ntohs(so->so_lport) == udptos[i].lport)
		   return udptos[i].tos;
		i++;
	}
	return 0;
}

void
udp_bind(so)
	struct socket *so;
{
	struct sockaddr_in addr;
	int addrlen = sizeof(addr);
	int n = 1;
	
	so->s = socket(AF_INET, SOCK_DGRAM, 0);
	ioctl(so->s, FIONREAD, &n);
	
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = 0;
	
	bind(so->s,(struct sockaddr *)&addr, addrlen); /* XXX */
	
	getsockname(so->s, (struct sockaddr *)&addr, &addrlen); /* XXX */
	
	so->so_fport = addr.sin_port;
	so->so_faddr = addr.sin_addr;
	so->so_expire = curtime + SO_EXPIRE;
}

#include "h/talkd.h"

/*
 * Here, talk/ytalk/ntalk requests must be emulated
 */
void
udp_emu(so, m)
	struct socket *so;
	struct mbuf *m;
{
	CTL_MSG *msg;
	struct socket *tmpso;
	
	switch(ntohs(so->so_fport)) {
	 case 517:
	 case 518:
		/*
		 * Emulate talk/ntalk request so we get the response
		 */
		msg = mtod(m, CTL_MSG *);
		
		/*
		 * Emulate all CTL_MSG types
		 * 
		 * This is a kludge...
		 * When an announce arrives, we sent it with a simulated 
		 * addr and ctl_addr, and we also send a LEAVE_INVITE to
		 * the local talk daemon...
		 * 
		 * XXX Not finished yet
		 */
		if (msg->type != ANNOUNCE)
		   return;
		
		if ((tmpso = socreate()) == NULL)
		   return;
		
		/*
		 * Redirect the UDP response
		 */
		udp_attach(tmpso);
		udp_bind(tmpso);
		
		tmpso->so_lport = ((struct sockaddr_in *)(&msg->ctl_addr))->sin_port;
		tmpso->so_laddr = ((struct sockaddr_in *)(&msg->ctl_addr))->sin_addr;
		((struct sockaddr_in *)&msg->ctl_addr)->sin_port = tmpso->so_fport;
		((struct sockaddr_in *)&msg->ctl_addr)->sin_addr = tmpso->so_faddr;
		
		return;
		
	}
}

struct socket *
udp_listen(port, laddr, lport, flags)
	u_int port;
	u_int32_t laddr;
	u_int lport;
	int flags;
{
	struct sockaddr_in addr;
	struct socket *so;
	int addrlen = sizeof(struct sockaddr_in), opt = 1;
	
#ifdef DEBUG
        debug_call(dfd,"udp_listen(port,laddr,lport,flags) called ...\n");
	debug_call(dfd,"    port = %d\n    laddr = %x\n    lport = %d\n    flags = %d\n",
		   port, (int)laddr, lport, flags);
	fflush_call(dfd);
#endif
	
	if ((so = socreate()) == NULL) {
		free(so);
		return NULL;
	}
	udp_attach(so);
	
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = port;
	
	if (bind(so->s,(struct sockaddr *)&addr, addrlen) < 0) {
		udp_detach(so);
		return NULL;
	}
	setsockopt(so->s,SOL_SOCKET,SO_REUSEADDR,(char *)&opt,sizeof(int));
/*	setsockopt(so->s,SOL_SOCKET,SO_OOBINLINE,(char *)&opt,sizeof(int)); */
	
	getsockname(so->s,(struct sockaddr *)&addr,&addrlen);
	so->so_fport = addr.sin_port;
	if (addr.sin_addr.s_addr == 0 || addr.sin_addr.s_addr == inet_addr("127.0.0.1"))
	   so->so_faddr.s_addr = our_addr;
	else
	   so->so_faddr = addr.sin_addr;
	
	so->so_lport = lport;
	so->so_laddr.s_addr = laddr;
	if (flags != SS_FACCEPTONCE)
	   so->so_expire = 0;
	
	so->so_state = SS_ISFCONNECTED;
	
	return so;
}
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
