/*
 * slirppp.c - interface between SLiRP and ppp-2.1.2b package
 *
 * Copyright (c) 1995 Juha Pirkola 
 *
 * This file adds PPP functionality to SLiRP, a free SLIP emulator
 * by Danny Gasparovski, using the free PPP package ppp-2.1.2b
 * maintained by Paul Mackerras.
 *
 */

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <syslog.h>
#include <netdb.h>
#include <utmp.h>
#include <pwd.h>
#include <sys/wait.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif


#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <net/if.h>
#include <termios.h>

#include "callout.h"
#include "ppp.h"
#include "magic.h"
#include "fsm.h"
#include "lcp.h"
#include "ipcp.h"
#include "upap.h"
#include "chap.h"

#include "pppd.h"
#include "pathnames.h"
#include "h/common.h"
#include "h/main.h"
#include "h/if.h"
#include "h/mbuf.h"
#include "h/ip.h"
#include "h/slcompress.h"
#include "h/ctl.h"



extern void alrm __ARGS((int));	   /* Alarm signal handler in pppdfncs.c */
 
extern char pppdevname[];			/* for pppd */
extern char *progname;			/* for pppd */
extern int ttyfd;			/* for slirp */

extern struct protent {			/* the protocol table in	*/
    u_short protocol;			/* pppdfncs.c			*/
    void (*init)();
    void (*input)();
    void (*protrej)();
    int  (*printpkt)();
    char *name;
} prottbl[]; 

#define N_PROTO         4		/* Nr of protocol entries	*/
					/* in prottbl			*/



/* Lookup table for checksum calculation. From RFC-1662			*/

static unsigned short fcstab[256] = {
  0x0000, 0x1189, 0x2312, 0x329b, 0x4624, 0x57ad, 0x6536, 0x74bf,
  0x8c48, 0x9dc1, 0xaf5a, 0xbed3, 0xca6c, 0xdbe5, 0xe97e, 0xf8f7,
  0x1081, 0x0108, 0x3393, 0x221a, 0x56a5, 0x472c, 0x75b7, 0x643e,
  0x9cc9, 0x8d40, 0xbfdb, 0xae52, 0xdaed, 0xcb64, 0xf9ff, 0xe876,
  0x2102, 0x308b, 0x0210, 0x1399, 0x6726, 0x76af, 0x4434, 0x55bd,
  0xad4a, 0xbcc3, 0x8e58, 0x9fd1, 0xeb6e, 0xfae7, 0xc87c, 0xd9f5,
  0x3183, 0x200a, 0x1291, 0x0318, 0x77a7, 0x662e, 0x54b5, 0x453c,
  0xbdcb, 0xac42, 0x9ed9, 0x8f50, 0xfbef, 0xea66, 0xd8fd, 0xc974,
  0x4204, 0x538d, 0x6116, 0x709f, 0x0420, 0x15a9, 0x2732, 0x36bb,
  0xce4c, 0xdfc5, 0xed5e, 0xfcd7, 0x8868, 0x99e1, 0xab7a, 0xbaf3,
  0x5285, 0x430c, 0x7197, 0x601e, 0x14a1, 0x0528, 0x37b3, 0x263a,
  0xdecd, 0xcf44, 0xfddf, 0xec56, 0x98e9, 0x8960, 0xbbfb, 0xaa72,
  0x6306, 0x728f, 0x4014, 0x519d, 0x2522, 0x34ab, 0x0630, 0x17b9,
  0xef4e, 0xfec7, 0xcc5c, 0xddd5, 0xa96a, 0xb8e3, 0x8a78, 0x9bf1,
  0x7387, 0x620e, 0x5095, 0x411c, 0x35a3, 0x242a, 0x16b1, 0x0738,
  0xffcf, 0xee46, 0xdcdd, 0xcd54, 0xb9eb, 0xa862, 0x9af9, 0x8b70,
  0x8408, 0x9581, 0xa71a, 0xb693, 0xc22c, 0xd3a5, 0xe13e, 0xf0b7,
  0x0840, 0x19c9, 0x2b52, 0x3adb, 0x4e64, 0x5fed, 0x6d76, 0x7cff,
  0x9489, 0x8500, 0xb79b, 0xa612, 0xd2ad, 0xc324, 0xf1bf, 0xe036,
  0x18c1, 0x0948, 0x3bd3, 0x2a5a, 0x5ee5, 0x4f6c, 0x7df7, 0x6c7e,
  0xa50a, 0xb483, 0x8618, 0x9791, 0xe32e, 0xf2a7, 0xc03c, 0xd1b5,
  0x2942, 0x38cb, 0x0a50, 0x1bd9, 0x6f66, 0x7eef, 0x4c74, 0x5dfd,
  0xb58b, 0xa402, 0x9699, 0x8710, 0xf3af, 0xe226, 0xd0bd, 0xc134,
  0x39c3, 0x284a, 0x1ad1, 0x0b58, 0x7fe7, 0x6e6e, 0x5cf5, 0x4d7c,
  0xc60c, 0xd785, 0xe51e, 0xf497, 0x8028, 0x91a1, 0xa33a, 0xb2b3,
  0x4a44, 0x5bcd, 0x6956, 0x78df, 0x0c60, 0x1de9, 0x2f72, 0x3efb,
  0xd68d, 0xc704, 0xf59f, 0xe416, 0x90a9, 0x8120, 0xb3bb, 0xa232,
  0x5ac5, 0x4b4c, 0x79d7, 0x685e, 0x1ce1, 0x0d68, 0x3ff3, 0x2e7a,
  0xe70e, 0xf687, 0xc41c, 0xd595, 0xa12a, 0xb0a3, 0x8238, 0x93b1,
  0x6b46, 0x7acf, 0x4854, 0x59dd, 0x2d62, 0x3ceb, 0x0e70, 0x1ff9,
  0xf78f, 0xe606, 0xd49d, 0xc514, 0xb1ab, 0xa022, 0x92b9, 0x8330,
  0x7bc7, 0x6a4e, 0x58d5, 0x495c, 0x3de3, 0x2c6a, 0x1ef1, 0x0f78
};


u_long recv_async_map;		/* which received characters to ignore	*/
u_long xmit_async_map[8];	/* which xmitted characters to escape	*/
int proto_field_comp;		/* compressing the protocol field ?	*/
int addr_field_comp;		/* compressing the address field ?	*/
FILE *logfile;			/* where to log events			*/
int ppp_up = 0;			/* is the PPP link up ?			*/



/*
 * stuff character in the output buffer, escaped if mentioned 
 * in xmit_async_map, and update the check sum				
 */
void
stuff_char(c, outp)
	u_char c;
	struct ppp_out *outp;
{
       	if  (in_xmap(c)){
		*outp->head++ = PPP_ESC;
                *outp->head++ = c^PPP_TRANS;
       	} else
             	*outp->head++ = c;
       	outp->fcs = (outp->fcs >> 8) ^ fcstab[(outp->fcs ^ c) & 0xff];
}

/*
 * Add a two byte check sum to the end of the outgoing packet
 */
void
add_fcs(outp)
	struct ppp_out *outp;
{
       u_short s = outp->fcs;
    
       s ^= 0xffff;
       stuff_char(s & 0x00ff, outp);
       stuff_char((s & 0xff00) >> 8, outp);
}

/*
 * check the check sum of an incoming frame
 */
int check_fcs(buff, len)
	u_char *buff;
	int len;
{
        u_short fcs = PPP_FCS_INIT;
        u_char *c = buff;
        int i;
                       
        for (i = 0; i < len ; i++, c++)
        fcs = (fcs >> 8) ^ fcstab[(fcs ^ *c) & 0xff];
                                
        if (fcs == PPP_FCS_GOOD)
        	return 1;
        else{
		syslog(0, "check_fcs: Checksum error, packet length = %d", len);
		return 0;
	}       	
}


/*
 * IPCP tells us that the link is broken, we're not allowed 
 * pass IP packets
 */
int sifdown(unit)
	int unit;
{
	ppp_up = 0;		/* we allways have unit = 0 in slirp	*/
	syslog(0, "slirppp: PPP is down now");
	return 1;
}
	
/*
 * IPCP says link is open, we can pass IP packets
 */
int sifup (u)
	int u;
{
	ppp_up = 1;
	syslog(0, "slirppp: PPP is up now");
	return 1;
}
	

/* 
 * configure receive characteristics after negotiations
 * we don't really care about pcomp and accomp, because compression
 * can be directly detected from the incoming packet
 */
void ppp_recv_config (unit, mru, asyncmap, pcomp, accomp)
	int unit, mru;
	u_long asyncmap;
	int pcomp, accomp;
{
	recv_async_map = asyncmap;
	syslog(0, "ppp_recv_config: (recv) asyncmap set to %08lx", asyncmap);
	syslog(0, "               :  mru set to %d", mru);
}	

/*
 * set the transmit asyncmap, in other words the characters to
 * be escaped when transmitted
 */

void
ppp_set_xaccm(unit, accm)
int unit;
u_long     accm[8];
{
	memcpy(xmit_async_map, accm, sizeof(accm) * 8);
	syslog(0,
   "ppp_set_xaccm: extended xmit asyncmap set to %08lx%08lx%08lx%08lx%08lx%08lx%08lx%08lx",
	       accm[7], accm[6], 
	       accm[5], accm[4], accm[3], accm[2], accm[1], accm[0]);
}        

/*
 * configure our receive characteristic after negotiations
 */
void ppp_send_config (unit, mtu, asyncmap, pcomp, accomp)
	int unit, mtu;
	u_long asyncmap;
	int pcomp, accomp;
{
	ifp.ifp_mtu = MIN(mtu, ifp.ifp_mtu);
	syslog(0, "ppp_send_config: mtu set to %d", ifp.ifp_mtu);
	xmit_async_map[0] = asyncmap;
	syslog(0, "ppp_send_config: (xmit) asyncmap set to %08lx", asyncmap);
	proto_field_comp = pcomp;
	if (pcomp) 
		syslog(0, "ppp_send_config: compressing the protocol field");
	addr_field_comp = accomp;
	if (accomp)
		syslog(0, "ppp_send_config: compressing the address field");

}

/*
 * set TCP/IP header compression mode according to what
 * has been negotiated
 * I don't know what to do with cidcomp and maxcid
 * must fix later
 */
void sifvjcomp (u, vjcomp, cidcomp, maxcid)
	int u, vjcomp, cidcomp, maxcid;
{
	if (vjcomp){
		ifp.ifp_flags = IF_COMPRESS;
		syslog(0, "sifvjcomp: Using VJ header compression");
	} else {
		ifp.ifp_flags = IF_NOCOMPRESS;	        
		syslog(0, "sifvjcomp: Not using VJ header compression");
	}
}

/*
 * now we have a frame from ppp_input
 * if the protocol field says that it is an IP packet, copy
 * the it to a mbuf and deliver to slirps ip_input function
 */

void
doframe(rbuff, count)
	u_char *rbuff;
	int count;
{
	u_char *c = rbuff;
	u_short proto;
	int i;

	if (!check_fcs(c, count))
		return;

	count -= 2;			/* don't count checksum field	*/		
	
	if ((c[0] == ALLSTATIONS) && (c[1] == UI)) {
		c = c+2;
		count -= 2;		/*don't count address field	*/
	}

	proto = (u_short) *c++;
	if (proto &1) {
		count--;
	} else {
		proto = (proto << 8 | (u_short) *c++);
		count -= 2;		/*don't count protocol field	*/
	}

	syslog(0, "Received a packet of %d bytes, protocol = 0x%04x",
							count, proto);

	for (i =0; i < N_PROTO; i++)
		if (prottbl[i].protocol == proto) {
			(*prottbl[i].input)(0, c, count);
			return;
		}
		
	if (ppp_up) {
		if_m = m_get();
		if_m->m_data += ifp.ifp_maxlinkhdr; /*space for uncompress*/
		if_m->m_len = count;
		memcpy(if_m->m_data, c, count);
		switch (proto) {
			case PROTO_IP:	
				ip_input(if_m);
				break;
			case PROTO_VJCOMP:
				if_m->m_len =
					sl_uncompress_tcp((u_char **)&if_m->m_data, count,
			     			(u_int) TYPE_COMPRESSED_TCP, comp_s);
				if (if_m->m_len >0)
					ip_input(if_m);
				else 
					m_free(if_m);
				break;
			case PROTO_VJUNCOMP:
				if_m->m_len = 
					sl_uncompress_tcp((u_char **)&if_m->m_data, if_m->m_len,
			      			(u_int) TYPE_UNCOMPRESSED_TCP, comp_s);
				if (if_m->m_len > 0)
					ip_input(if_m);
				else
					m_free(if_m);
				break;
			default:		/* Unknown protocol */
				m_free(if_m);
				break;
		}	
	}		
	return;
}
	
/*
 * the main input routine corresponding to sl_input
 * I tried to make this similar to sl_input, but it was not 
 * possible to write the unescaped data directly to a mbuf,
 * and therefore this is a bit different
 */
void ppp_input()
{
	u_char if_inbuff[2*2048 + 14];
	u_char rbuff[2048 + 6];
	int i;
	static int zeros = 0, rcount = 0;
	static u_char escape = 0;
	int bad = 0;

	if_n = read(ttyfd, if_inbuff, 2*ifp.ifp_mtu+2);
	if (if_n <= 0)
		slirp_exit(0);
		
	if_bptr = if_inbuff;
	
	for(i = 0; i < if_n; if_bptr++, i++){
		 if ((if_n == 1) && (*if_bptr == '0')){
                         zeros++;
                         if (zeros > 4) {
				syslog(0, "Received five zeros, exiting...");
				slirp_exit(1);
			 }
                 } 
                 else
                 	zeros = 0;
		
			
		if (*if_bptr == PPP_FLAG){
			if (if_inpkt == 0)
				continue;
			if (escape)
				bad = 1;
			if ((bad == 0) && (rcount > 0))
				doframe(rbuff, rcount);
			else {
				syslog(0,
				       "ppp_input: Got a bad frame of %d bytes",rcount);
			}
			if_inpkt = 0;
			continue;
		}

		/* We fall here if it was not PPP_FLAG */
		if (if_inpkt == 0){		/* new frame  starting	*/
			if_inpkt = 1;			
			if_mptr = rbuff;
			escape = 0;
			bad = 0;
			rcount = 0;
		}
		if (*if_bptr == PPP_ESC){
			escape = PPP_TRANS;
			continue;
		}	
		if (!in_rmap(*if_bptr)){
			if (rcount >= sizeof(rbuff))
				bad = 1;             /* frame too long */
			if (bad == 0) {
				*if_mptr++ = *if_bptr ^ escape;
				rcount++;
			}
			escape = 0;
		}
	}
}

/*
 * this is the output function SLiRP uses, correspondig to sl_encap. 
 * data from a mbuf is encapsulated according to the HDLC-like 
 * framing scheme (RFC-1662) and put to the buffer pointed by inbptr 
 */
int ppp_encap(inbptr, m)
	char *inbptr;
	struct mbuf *m;
{
	int i;
	u_short proto;
	u_char c;	
	struct ppp_out out;

	out.buff = out.head = (u_char *) inbptr;
	out.fcs = PPP_FCS_INIT;
	if (towrite == TOWRITEMAX) {
		*out.head = PPP_FLAG;
		out.head++;
	}
	if (!addr_field_comp) {
		stuff_char(ALLSTATIONS, &out);    
		stuff_char(UI, &out);
	}
	
	proto = PROTO_IP;
/*
 *	Try to find out if the packet is compressed and in that case
 *	change protocol field to PROTO_VJCOMP or PROTO_VJUNCOMP
 */
	if ((c = *mtod(m, u_char *) & 0xf0) != (IPVERSION << 4)){
		if (c & TYPE_COMPRESSED_TCP) {
			proto = PROTO_VJCOMP;
		}
		else if (c == TYPE_UNCOMPRESSED_TCP){
			proto = PROTO_VJUNCOMP;
/*			*mtod(if_m, u_char *) = (*mtod(if_m, u_char *) & 0x0f) | 0x40; */
			*mtod(m, u_char *) = (*mtod(m, u_char *) & 0x0f) | 0x40;
		}
	}
	if (!proto_field_comp)
		stuff_char((proto & 0xff00) >> 8, &out);
	stuff_char(proto & 0xff, &out);

	for(i = 0; i < m->m_len; i++)
		stuff_char(*(m->m_data + i), &out);
	add_fcs(&out);
	*out.head = PPP_FLAG;
	return (out.head - out.buff) + 1;

}

/*
 * this is the output routine used by the link level protocols
 * it writes directly to the tty
 */
void output (unit, p, len)
	int unit;
	u_char *p;
	int len;
{
        u_char outgoing[2*2048 + 14];
        int i;
        struct ppp_out out;

        outgoing[0] = PPP_FLAG;
        out.buff = out.head = outgoing + 1;
        out.fcs = PPP_FCS_INIT;

        for (i = 0; i <len; i++)
                stuff_char(p[i], &out);
        add_fcs(&out);
        *out.head = PPP_FLAG;
        write(ttyfd, (char *) outgoing, (out.head - out.buff) + 2);
}

/*
 * we are exiting, but we don't know if we should go through
 * slirp's exit routine or just exit. therefore we examine
 * the terminal mode to see if it has allready been modified
 * by slirp
 */ 
void die(status)
	int status;
{
	struct termios tempio;

	tcgetattr(ttyfd, &tempio);
	if ((tempio.c_iflag | tempio.c_oflag | tempio.c_lflag) == 0)
		slirp_exit(status);
	else
		exit (status);
}


/*
 *  we seem to be using PPP - initialise a few things
 */

void ppp_init(argc, argv)
	int argc;
	char **argv;
{
	int  i;
	char *p;
        sigset_t mask;
        struct sigaction sa;

	p = (char *) ttyname(0);
	if (p) {
		strcpy(pppdevname, p);
	}

	if (gethostname(hostname, MAXNAMELEN) < 0 ) {
		perror("couldn't get hostname");
		die(1);
	}

	hostname[MAXNAMELEN-1] = 0;

	for (i = 0; i < N_PROTO; i++)
		(*prottbl[i].init)(0);

	progname = "SLIRP/PPP";
			
	if (!options_from_user() ||
	    !parse_args(argc-1, argv+1))
	      die(1);
	check_auth_options();
	setipdefault();

        sigemptyset(&mask);
        sigaddset(&mask, SIGALRM);
        sa.sa_mask = mask;
        sa.sa_flags = 0;
        sa.sa_handler = alrm; 
        sigaction(SIGALRM, &sa, NULL);
	
	magic_init();
	ipcp_wantoptions[0].hisaddr = inet_addr(CTL_LOCAL);
	
	ifp.ifp_input = ppp_input;
	ifp.ifp_output = if_output;
	ifp.ifp_start = if_start;
	ifp.ifp_encap = ppp_encap;
	
	ifp.ifp_maxlinkhdr = 40;
	sl_compress_init(comp_s);
	if_batchq.ifq_next = if_batchq.ifq_prev = &if_batchq;
	if_fastq.ifq_next = if_fastq.ifq_prev = &if_fastq;
	next_m = &if_batchq;

}

/*
 * tell the user what options we are going to negotiate
 * and then start the negotiation process 
 */
void ppp_start()
{	
	int i;
	lcp_options *wo = &lcp_wantoptions[0];
	
	fprintf(stderr, "Going to negotiate for ");
	fprintf(stderr, "mru: %4d, asyncmap: %08lx \n\r",
				wo->mru, wo->asyncmap);
	fprintf(stderr, "Escaping at least the following characters (hex):");
	for (i = 0; i < 256; i++){
		if (xmit_accm[0][i >> 5] & (1 << (i & 0x1f)))
			fprintf(stderr, " %02x", i);
	}
	fprintf(stderr, "\n\r");
	if (debug) fprintf(stderr, "Logging PPP events to file: %s\n\r", PPP_LOGFILE);
	fflush(stderr);

	lcp_lowerup(0);
	lcp_open(0);
}

/* 
 * The following functions (random, srandom, index, bcmp, gethostid)
 * are defined here for systems that don't have them. The pppd pack-
 * age was written for BSD systems that have these non-ANSI funcs
 */

#ifndef HAVE_RANDOM
long
random ()
{
	return rand();
}

void
srandom (seed)
	int seed;
{
	srand(seed);
}
#endif

#ifndef HAVE_INDEX
char *
index(s, c)
	const char *s;
	int c;
{
	return strchr(s, c);
}
#endif

#ifndef HAVE_BCMP
int
bcmp(s1, s2, n)
	const void *s1, *s2;
	int n;
{
	return memcmp(s1, s2, n);
}
#endif

/*
 * we'll define gethostid() allways, because I had a problem with
 * it in a HP-UX system, where configure reported it to exist, but
 * it didn't
 */
long 
gethostid()
{
	return 12345678;
}



/*
 * We define our own syslog, because we are not running as a
 * system daemon but a normal user program. This will raise
 * compile warning, because in some environments syslog is
 * defined to return int, and in some others it is void
 */  

void
#ifdef __STDC__
do_syslog(int priority, const char *format, ...)
#else
do_syslog(va_alist) va_dcl
#endif
{
	va_list argp;

#ifdef __STDC__
	va_start(argp, format);
#else
	int priority;
	char *format;
	
	va_start(argp);
	priority = va_arg(argp, int);
	format = va_arg(argp, char *);
#endif	
	vfprintf(logfile, format, argp);
	va_end(argp);
	fprintf(logfile, "\n");
	fflush(logfile);
}


/*
 * The rest of the functions are there just to satisfy
 * lcp.c, ipcp.c etc. They don't really do anything but
 * return an acceptable value
 */

int sifaddr (unit, our_adr, his_adr, net_mask)
	int unit, our_adr, his_adr, net_mask;
{
        return 1;
}

int cifaddr (unit, our_adr, his_adr)
	int unit, our_adr, his_adr;
{
        return 1;
}

int sifdefaultroute (unit, gateway)
	int unit, gateway;
{
        return 1;
}

int cifdefaultroute (unit, gateway)
	int unit, gateway;
{
        return 1;
}

int logwtmp(line, name, host)
	char *line, *name, *host;
{
        return 1;
}

int cifproxyarp (unit, his_adr)
	int unit;
	u_long his_adr;
{
        return 1;

}

int sifproxyarp (unit, his_adr)
	int unit;
	u_long his_adr;
{
        return 1;
}


int
run_program(prog, args, must_exist)
	char *prog, **args;
	int must_exist;
{
        return 0;
}


void
print_string(p, len, printer, arg)
    char *p;
    int len;
    void (*printer) __ARGS((void *, char *, ...));
    void *arg;
{

}
