/*
 * Copyright (c) 1995 Danny Gasparovski.
 *
 * Please read the file COPYRIGHT for the
 * terms and conditions of the copyright.
 */

#include "h/common.h"
#include "h/ip.h"
#include "h/tcp.h"
#include "h/sl.h"
#include "h/slcompress.h"
#include "h/main.h"
#include "h/socket.h"

int     if_inpkt = 0;                   /* Flag to indicate we're still reading a packet */
int     if_n;                           /* Bytes left in interface input buffer */
int     if_esc;                         /* Flag that we received an FRAME_ESC */
int     esc;                            /* Number of times '0' has been hit (for quitting slirp) */
struct  mbuf *if_m;                     /* mbuf we're transferring the packet to */
int     if_msize;                       /* Size of mbuf we're transferring the packet to */
u_char  *if_mptr;                       /* Pointer into the mbuf we're writing to */
u_char  *if_bptr;                       /* Pointer to the if_inbuff we're reading from */
int     if_queued = 0;                  /* Number of packets queued so far */
int     if_thresh = 10;                 /* Number of packets queued before we start sending
					 * (to prevent allocing too many mbufs) */

struct  mbuf if_fastq;                  /* fast queue (for interactive data) */
struct  mbuf if_batchq;                 /* queue for non-interactive data */
struct	mbuf *next_m;			/* Pointer to next mbuf to output */

#define CHECK_MSIZE  {if_msize--; if (if_msize < 0) sl_outofroom(); }
#define ifs_init(ifm) ((ifm)->ifs_next = (ifm)->ifs_prev = (ifm))

void
ifs_insque(ifm, ifmhead)
	struct mbuf *ifm, *ifmhead;
{
	ifm->ifs_next = ifmhead->ifs_next;
	ifmhead->ifs_next = ifm;
	ifm->ifs_prev = ifmhead;
	ifm->ifs_next->ifs_prev = ifm;
}

void
ifs_remque(ifm)
	struct mbuf *ifm;
{
	ifm->ifs_prev->ifs_next = ifm->ifs_next;
	ifm->ifs_next->ifs_prev = ifm->ifs_prev;
}

int writen _P((int, char *, int));

/*
 * This shouldn't be needed since the modem is blocking and
 * we don't expect any signals, but what the hell..
 * just to make sure...
 */
int
writen(fd, ptr, n)
	int fd;
	char *ptr;
	int n;
{
	int ret;
	int total;
	
	/* This should succeed most of the time */
	ret = write(fd, ptr, n);
	if (ret == n || ret <= 0)
	   return ret;
	
	/* Didn't write everything, go into the loop */
	total = ret;
	while (n > total)  {
		ret = write(fd, ptr+total, n-total);
		if (ret <= 0)
		   return ret;
		total += ret;
	}
	return total;
}


/*
 * if_output: Queue packet into an output queue.
 * There are 2 output queue's, if_fastq and if_batchq. 
 * Each output queue is a doubly linked list of double linked lists
 * if mbufs, each list belonging to one "session" (socket).  This
 * way, we can output packets fairly by sending one packet from each
 * session, instead of all the packets from one session, then all packets
 * from the next session, etc.  Packets on the if_fastq get absolute 
 * priority, but if one session hogs the link, it gets "downgraded"
 * to the batchq until it runs out of packets, then it'll return
 * to the fastq (eg. if the user does an ls -alR in a telnet session,
 * it'll temporarily get downgraded to the batchq)
 */
void
if_output(so, ifm)
	struct socket *so;
	struct mbuf *ifm;
{
	struct mbuf *ifq;
	int on_fastq = 1;
	struct ip *ip;
	
#ifdef DEBUG
	debug_call(dfd,"if_output(so, ifm) called ...\n");
	debug_call(dfd,"    so = %d\n    ifm = %d\n", (int)so, (int)ifm);
	fflush_call(dfd);
#endif
	
	/*
	 * First remove the mbuf from m_usedlist,
	 * since we're gonna use m_next and m_prev ourselves
	 * XXX Shouldn't need this, gotta change dtom() etc.
	 */
	if (ifm->m_flags & M_USEDLIST) {
		remque(ifm);
		ifm->m_flags &= ~M_USEDLIST;
	}
	
	/*
	 * See if there's already a batchq list for this session.  
	 * This can include an interactive session, which should go on fastq,
	 * but gets too greedy... hence it'll be downgraded from fastq to batchq.
	 * We mustn't put this packet back on the fastq (or we'll send it out of order)
	 */
	for (ifq = if_batchq.ifq_prev; ifq != &if_batchq; ifq = ifq->ifq_prev) {
		if (so != ifq->ifq_so)
		   continue;
		/* A match! */
		ifm->ifq_so = so;
		ifs_insque(ifm, ifq->ifs_prev);
		goto diddit; /* Boy, do I love spaghetti code! :) */
	}
	
	/* No match, check which queue to put it on */
	if (so && (so->so_iptos & IPTOS_LOWDELAY)) {
		ifq = if_fastq.ifq_prev;
		on_fastq = 1;
		/*
		 * Check if this packet is a part of the last
		 * packet's session
		 */
		if (ifq->ifq_so == so) {
			ifm->ifq_so = so;
			ifs_insque(ifm, ifq->ifs_prev);
			goto diddit;
		}
	} else
		ifq = if_batchq.ifq_prev;
	
	/* Create a new doubly linked list for this session */
	ifm->ifq_so = so;
	ifs_init(ifm);
	insque(ifm, ifq);
	
diddit:
	/*
	 * Compress now so we have a higher "cache" hit rate
	 * in sl_compress_tcp()
	 */
	ip = mtod(ifm, struct ip *);
	if (ifp.ifp_flags == IF_COMPRESS && ip->ip_p == IPPROTO_TCP)
	   *mtod(ifm, u_char *) |= sl_compress_tcp(ifm, ip, comp_s, 1);
	
	++if_queued;
	
	if (so) {
		/* Update *_queued */
		so->so_queued++;
		so->so_nqueued++;
		/*
		 * Check if the interactive session should be downgraded to
		 * the batchq.  A session is downgraded if it has queued 16
		 * packets without pausing, and at least 8 of those packets
		 * have been sent over the link
		 * (XXX These are arbitrary numbers, probably not optimal..)
		 */
		if (on_fastq && ((so->so_nqueued >= 16) && 
				 (so->so_nqueued - so->so_queued) >= 8)) {
			
			/* Remove from current queue... */
			remque(ifm->ifs_next);
			
			/* ...And insert in the new.  That'll teach ya! */
			insque(ifm->ifs_next, &if_batchq);
		}
	}
		
	/*
	 * This prevents us from malloc()ing too many mbufs
	 */
/*	if (if_queued > if_thresh && towrite >= 0) */
	if (towrite >= 0)
	   (*ifp.ifp_start)(); /* XXX */
	
	return;
}

/*
 * Send a packet
 * We choose a packet based on it's position in the output queues;
 * If there are packets on the fastq, they are sent FIFO, before
 * everything else.  Otherwise we choose the first packet from the
 * batchq and send it.  the next packet chosen will be from the session
 * after this one, then the session after that one, and so on..  So,
 * for example, if there are 3 ftp session's fighting for bandwidth,
 * one packet will be sent from the first session, then one packet
 * from the second session, then one packet from the third, then back
 * to the first, etc. etc.
 */
void
if_start()
{
	struct mbuf *ifm, *ifqt;
	int n;
#ifdef __GNUC__
	char if_outbuff[2*ifp.ifp_mtu+2];
#else
	char if_outbuff[2*2048+2]; /* XXX */
#endif
	
#ifdef DEBUG
	debug_call(dfd,"if_start(void) called ...\n");
	fflush_call(dfd);
#endif
	
again:
	if (if_queued == 0)
	   return; /* Nothing to do */
	
	/*
	 * See which queue to get next packet from
	 * If there's something in the fastq, select it immediately
	 */
	if (if_fastq.ifq_next != &if_fastq)
		ifm = if_fastq.ifq_next;
	else {
		/* Nothing on fastq, see if next_m is valid */
		if (next_m != &if_batchq)
		   ifm = next_m;
		else
		   ifm = if_batchq.ifq_next;
		
		/* Set which packet to send on next iteration */
		next_m = ifm->ifq_next;
	}
	/* Remove it from the queue */
	ifqt = ifm->ifq_prev;
	remque(ifm);
	--if_queued;
	
	/* If there are more packets for this session, re-queue them */
	if (ifm->ifs_next != /* ifm->ifs_prev != */ ifm)  {
		insque(ifm->ifs_next, ifqt);
		ifs_remque(ifm);
	}
	
	/* Update so_queued */
	if (ifm->ifq_so) {
		if (--ifm->ifq_so->so_queued == 0)
		   /* If there's no more queued, reset nqueued */
		   ifm->ifq_so->so_nqueued = 0;
	}
	
	/* Encapsulate the packet for sending */
	n = (*ifp.ifp_encap)(if_outbuff, ifm);
	m_free(ifm);
	
	/* Send it */
	writen(ttyfd, if_outbuff, n);
	
	/* Not idle anymore */
	idle_sec = 0;
	
	towrite -= n;
	if (towrite >= 0)
	   goto again;
}
