/*
 * Copyright (c) 1995 Danny Gasparovski.
 * 
 * Please read the file COPYRIGHT for the 
 * terms and conditions of the copyright.
 */

#include "h/common.h"
#include "h/ip.h"
#include "h/tcp.h"
#include "h/udp.h"
#include "h/socket.h"
#include "h/main.h"
#include "h/terminal.h"
#include <fcntl.h>
#include <sys/un.h>
#include <signal.h>
#ifdef HAVE_SYS_SIGNAL_H
#include <sys/signal.h>
#endif
#include <sys/socket.h>
#include <sys/termios.h>

FILE *dfd = NULL;
int slirp_errors = 0;
int dostats = 0;
int slirp_debug = 0;
#ifdef DO_RESTART
pid_t remote_pid = 0;
int slirp_forked = 0;
#endif

extern char *strerror _P((int));

void
debug_init(file, dbg)
	char *file;
	int dbg;
{
	dfd = fopen(file,"w");
	if (dfd != NULL) {
		fprintf(dfd,"SLiRP %s - Debugging Started.\n", VERSION);
		fflush(dfd);
		slirp_debug = dbg;
	} else {
		fprintf(stderr,"ERROR: Debugging file \"%s\" could not be opened: %s\n",
			file, strerror(errno));
	}
}

/*
 * Dump a packet in the same format as tcpdump -x
 */
#ifdef DEBUG
void
dump_packet(dat, n)
	void *dat;
	int n;
{
	u_short *ptr = (u_short *)dat;
	int j,k;
	
	n /= 16;
	n++;
	debug_misc(dfd,"PACKET DUMPED: \n");
	for(j = 0; j < n; j++)   {
		for(k = 0; k < 8; k++)
			debug_misc(dfd,"%04x ", *ptr++);
		debug_misc(dfd,"\n");
		fflush(dfd);
	}
}
#endif

/*
 * Statistic routines
 * 
 * These will print statistics to the screen, the debug file (dfd), or
 * a buffer, depending on "type", so that the stats can be sent over
 * the link as well.
 */

/* 
 * This is common to all the stats functions
 */
#define STAT_INIT int (*print) _P((void *, const char *format, ...)); \
	char *ptr; \
	char *ptr2; \
        char **arg; \
\
	switch (type) { \
	 case PRN_SPRINTF: \
		print = (int (*) _P((void *, const char *, ...)))sprintf; \
		ptr = sb->sb_wptr; \
		arg = (char **)&ptr; \
		break; \
	 case PRN_DFD: \
		print = (int (*) _P((void *, const char *, ...)))fprintf; \
		arg = (char **)&dfd; \
		break; \
	 case PRN_STDERR: \
	 default: \
		print = (int (*) _P((void *, const char *, ...)))fprintf; \
		ptr2 = (char *)stderr; \
		arg = (char **)&ptr2; \
		break; \
	}

int
ipstats(sb, type)
	struct sbuf *sb;
	int type;
{
	STAT_INIT;

	ptr += (*print)(*arg," \r\n");	

	ptr += (*print)(*arg,"IP stats:\r\n");
	ptr += (*print)(*arg,"Total packets received:			%d\r\n", ipstat.ips_total);
	ptr += (*print)(*arg,"    Bad version:			%d\r\n", ipstat.ips_badvers);
	ptr += (*print)(*arg,"    Bad checksum:			%d\r\n", ipstat.ips_badsum);
	ptr += (*print)(*arg,"    Too short:				%d\r\n", ipstat.ips_tooshort);
	ptr += (*print)(*arg,"    Too small:				%d\r\n", ipstat.ips_toosmall);
	ptr += (*print)(*arg,"    Bad header length:			%d\r\n", ipstat.ips_badhlen);
	ptr += (*print)(*arg,"    Bad length:				%d\r\n", ipstat.ips_badlen);
	ptr += (*print)(*arg,"    Fragments:				%d\r\n", ipstat.ips_fragments);
	ptr += (*print)(*arg,"    Fragments dropped:			%d\r\n", ipstat.ips_fragdropped);
	ptr += (*print)(*arg,"    Fragments timed out:		%d\r\n", ipstat.ips_fragtimeout);
	ptr += (*print)(*arg,"    Total fragments reassembled:	%d\r\n", ipstat.ips_reassembled);
	ptr += (*print)(*arg,"    Total outgoing packets fragmented:	%d\r\n", ipstat.ips_fragmented);
	ptr += (*print)(*arg,"    Total outgoing fragments:		%d\r\n", ipstat.ips_ofragments);
	ptr += (*print)(*arg,"    Bad protocol field:			%d\r\n", ipstat.ips_noproto);
	ptr += (*print)(*arg,"Total packets delivered:		%d\r\n", ipstat.ips_delivered);
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 0;
}

int
tcpstats(sb, type)
	struct sbuf *sb;
	int type;
{
	STAT_INIT;

	ptr += (*print)(*arg," \r\n");

	ptr += (*print)(*arg,"TCP stats:\r\n");
	ptr += (*print)(*arg,"    Connections initiated: 		%d\r\n", tcpstat.tcps_connattempt);
	ptr += (*print)(*arg,"    Connections accepted:		%d\r\n", tcpstat.tcps_accepts);
	ptr += (*print)(*arg,"    Connections established: 		%d\r\n", tcpstat.tcps_connects);
	ptr += (*print)(*arg,"    Connections dropped:		%d\r\n", tcpstat.tcps_drops);
	ptr += (*print)(*arg,"    Embryonic connections dropped: 	%d\r\n", tcpstat.tcps_conndrops);
	ptr += (*print)(*arg,"    Conn. closed (includes drops): 	%d\r\n", tcpstat.tcps_closed);
	ptr += (*print)(*arg,"    Segs where we tried to get rtt: 	%d\r\n", tcpstat.tcps_segstimed);
	ptr += (*print)(*arg,"    Times we succeeded:			%d\r\n", tcpstat.tcps_rttupdated);
	ptr += (*print)(*arg,"    Delayed acks sent: 			%d\r\n", tcpstat.tcps_delack);
	ptr += (*print)(*arg,"    Conn. dropped in rxmt timeout: 	%d\r\n", tcpstat.tcps_timeoutdrop);
	ptr += (*print)(*arg,"    Retransmit timeouts:		%d\r\n", tcpstat.tcps_rexmttimeo);
	ptr += (*print)(*arg,"    Persist timeouts: 			%d\r\n", tcpstat.tcps_persisttimeo);
	ptr += (*print)(*arg,"    Keepalive timeouts:			%d\r\n", tcpstat.tcps_keeptimeo);
	ptr += (*print)(*arg,"    Keepalive probes sent: 		%d\r\n", tcpstat.tcps_keepprobe);
	ptr += (*print)(*arg,"    Connections dropped in keepalive: 	%d\r\n", tcpstat.tcps_keepdrops);
	
	ptr += (*print)(*arg,"Total TCP packets sent:			%d\r\n", tcpstat.tcps_sndtotal);
	ptr += (*print)(*arg,"    Data packets sent: 			%d\r\n", tcpstat.tcps_sndpack);
	ptr += (*print)(*arg,"    Data bytes sent: 			%d\r\n", tcpstat.tcps_sndbyte);
	ptr += (*print)(*arg,"    Data packets retransmitted:		%d\r\n", tcpstat.tcps_sndrexmitpack);
	ptr += (*print)(*arg,"    Data bytes retransmitted: 		%d\r\n", tcpstat.tcps_sndrexmitbyte);
	ptr += (*print)(*arg,"    ACK-only packets sent: 		%d\r\n", tcpstat.tcps_sndacks);
	ptr += (*print)(*arg,"    Window probes sent:			%d\r\n", tcpstat.tcps_sndprobe);
	ptr += (*print)(*arg,"    Packets sent with URG only:		%d\r\n", tcpstat.tcps_sndurg);
	ptr += (*print)(*arg,"    Window update-only packets sent: 	%d\r\n", tcpstat.tcps_sndwinup);
	ptr += (*print)(*arg,"    Control (SYN|FIN|RST) packets sent: %d\r\n", tcpstat.tcps_sndctrl);
	ptr += (*print)(*arg,"    Times tcp_output did nothing:	%d\r\n", tcpstat.tcps_didnuttin);
	
	ptr += (*print)(*arg,"Total TCP packets received:		%d\r\n", tcpstat.tcps_rcvtotal);       
	ptr += (*print)(*arg,"    Packets received in sequence:	%d\r\n", tcpstat.tcps_rcvpack);       
	ptr += (*print)(*arg,"    Bytes received in sequence:		%d\r\n", tcpstat.tcps_rcvbyte);
	ptr += (*print)(*arg,"    Packets received with cksum errs: 	%d\r\n", tcpstat.tcps_rcvbadsum);
	ptr += (*print)(*arg,"    Packets received with bad offset: 	%d\r\n", tcpstat.tcps_rcvbadoff);      
/*	ptr += (*print)(*arg,"    Packets received too short:		%d\r\n", tcpstat.tcps_rcvshort); */
        ptr += (*print)(*arg,"    Duplicate-only packets received: 	%d\r\n", tcpstat.tcps_rcvduppack);
	ptr += (*print)(*arg,"    Duplicate-only bytes received: 	%d\r\n", tcpstat.tcps_rcvdupbyte);
	ptr += (*print)(*arg,"    Packets with some duplicate data: 	%d\r\n", tcpstat.tcps_rcvpartduppack);
	ptr += (*print)(*arg,"    Dup. bytes in part-dup. packets: 	%d\r\n", tcpstat.tcps_rcvpartdupbyte);
	ptr += (*print)(*arg,"    Out-of-order packets received: 	%d\r\n", tcpstat.tcps_rcvoopack);
	ptr += (*print)(*arg,"    Out-of-order bytes received:	%d\r\n", tcpstat.tcps_rcvoobyte);
	ptr += (*print)(*arg,"    Packets with data after window: 	%d\r\n", tcpstat.tcps_rcvpackafterwin);
	ptr += (*print)(*arg,"    Bytes rcvd after window: 		%d\r\n", tcpstat.tcps_rcvbyteafterwin);
	ptr += (*print)(*arg,"    Packets rcvd after \"close\":		%d\r\n", tcpstat.tcps_rcvafterclose);
	ptr += (*print)(*arg,"    Rcvd window probe packets: 		%d\r\n", tcpstat.tcps_rcvwinprobe);
	ptr += (*print)(*arg,"    Rcvd duplicate acks:		%d\r\n", tcpstat.tcps_rcvdupack);
	ptr += (*print)(*arg,"    Rcvd acks for unsent data: 		%d\r\n", tcpstat.tcps_rcvacktoomuch);
	ptr += (*print)(*arg,"    Rcvd ack packets: 			%d\r\n", tcpstat.tcps_rcvackpack);
	ptr += (*print)(*arg,"    Bytes acked by rcvd acks: 		%d\r\n", tcpstat.tcps_rcvackbyte);
	ptr += (*print)(*arg,"    Rcvd window update packets:		%d\r\n", tcpstat.tcps_rcvwinupd);
/*	ptr += (*print)(*arg,"    Segments dropped due to PAWS:	%d\r\n", tcpstat.tcps_pawsdrop); */
	ptr += (*print)(*arg,"    Times hdr predict ok for acks:	%d\r\n", tcpstat.tcps_predack);
	ptr += (*print)(*arg,"    Times hdr predict ok for data pkts:	%d\r\n", tcpstat.tcps_preddat);
	ptr += (*print)(*arg,"    TCP ptr cache misses:		%d\r\n", tcpstat.tcps_socachemiss);
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 0;
}

int
udpstats(sb, type)
	struct sbuf *sb;
	int type;
{
	STAT_INIT;
	
        ptr += (*print)(*arg," \r\n");

	ptr += (*print)(*arg,"UDP stats:\r\n");
	ptr += (*print)(*arg,"Total UDP packets received:		%d\r\n", udpstat.udps_ipackets);
	ptr += (*print)(*arg,"    Packets shorter than header:	%d\r\n", udpstat.udps_hdrops);
	ptr += (*print)(*arg,"    Bad checksum:			%d\r\n", udpstat.udps_badsum);
	ptr += (*print)(*arg,"    Data length larger than packet:	%d\r\n", udpstat.udps_badlen);
	ptr += (*print)(*arg,"    Socket cache misses:		%d\r\n", udpstat.udpps_pcbcachemiss);
	ptr += (*print)(*arg,"Total UDP output packets:		%d\r\n", udpstat.udps_opackets);
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 0;
}

int
mbufstats(sb, type)
	struct sbuf *sb;
	int type;
{
	struct mbuf *m;
	int i;
	STAT_INIT;
	
        ptr += (*print)(*arg," \r\n");

	ptr += (*print)(*arg,"Number of mbufs allocated: %d\r\n", mbuf_alloced);
	ptr += (*print)(*arg,"Max number of mbufs allocated: %d\r\n", mbuf_max);
	
	i = 0;
	for (m = m_freelist.m_next; m != &m_freelist; m = m->m_next)
		i++;
	ptr += (*print)(*arg,"Number of mbufs on free list: %d\r\n",  i);
	
	i = 0;
	for (m = m_usedlist.m_next; m != &m_usedlist; m = m->m_next)
		i++;
	ptr += (*print)(*arg,"Number of mbufs on used list: %d\r\n",  i);
        ptr += (*print)(*arg,"Number of packets queued: %d\r\n\r\n", if_queued);
	
	if (sb) 
	   return ptr - sb->sb_data;
	else
	   return 0;
}

int
tcpsockstats(sb, type)
	struct sbuf *sb;
	int type;
{
	struct socket *so;
	STAT_INIT;

        ptr += (*print)(*arg," \r\n");
	
	ptr += (*print)(*arg,"Active TCP Sockets:\r\n");

	for (so = tcb.so_next; so != &tcb; so = so->so_next) {
		
		/* XXX 1024? */
		if (type == 3 && (ptr - sb->sb_wptr) >= (sb->sb_datalen - 1024)) {
			int deltaw = sb->sb_wptr - sb->sb_data;
			int deltar = sb->sb_rptr - sb->sb_data;
			int deltap = ptr -         sb->sb_data;
			
			sb->sb_data = (char *)realloc(sb->sb_data, sb->sb_datalen + 8096);
			
			/* Adjust all values */
			sb->sb_wptr = sb->sb_data + deltaw;
			sb->sb_rptr = sb->sb_data + deltar;
			ptr = sb->sb_data + deltap;
			
			sb->sb_datalen += 8096;
		}
		
		
		ptr += (*print)(*arg,"    Socket no.: %d\r\n", so->s);
		ptr += (*print)(*arg,"    Local address (your end of the link): %s\r\n",
			inet_ntoa(so->so_laddr));
		ptr += (*print)(*arg,"    Local port: %d\r\n", ntohs(so->so_lport));
		ptr += (*print)(*arg,"    Foreign address (host on internet): %s\r\n",
			inet_ntoa(so->so_faddr));
		ptr += (*print)(*arg,"    Foreign port: %d\r\n", ntohs(so->so_fport));
		ptr += (*print)(*arg,"    Type of service: %d\r\n", so->so_iptos);
		ptr += (*print)(*arg,"    Socket state (see source): %x\r\n", so->so_state);
		if (so->so_tcpcb) {
			ptr += (*print)(*arg,"    TCP state: %s\r\n", tcpstates[so->so_tcpcb->t_state]);
			ptr += (*print)(*arg,"    Bytes left in receive buffer: %d\r\n", so->so_rcv.sb_cc);
			ptr += (*print)(*arg,"    Bytes left in send buffer: %d\r\n", so->so_snd.sb_cc);
		}
		ptr += (*print)(*arg,"\r\n");
	}
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 9;
}

int
udpsockstats(sb, type)
	struct sbuf *sb;
	int type;
{
	struct socket *so;
	STAT_INIT;
	
        ptr += (*print)(*arg," \r\n");

	ptr += (*print)(*arg,"Active UDP Sockets:\r\n");
	
	for (so = udb.so_next; so != &udb; so = so->so_next) {
		
		/* XXX 1024? */
		if (type == 3 && (ptr - sb->sb_wptr) >= (sb->sb_datalen - 1024))  {
			int deltaw = sb->sb_wptr - sb->sb_data;
			int deltar = sb->sb_rptr - sb->sb_data;
			int deltap = ptr -         sb->sb_data;
			
			sb->sb_data = (char *)realloc(sb->sb_data, sb->sb_datalen + 8096);
			
			/* Adjust all values */
			sb->sb_wptr = sb->sb_data + deltaw;
			sb->sb_rptr = sb->sb_data + deltar;
			ptr = sb->sb_data + deltap;
			
			sb->sb_datalen += 8096;
		}
		
		
		ptr += (*print)(*arg,"    Socket fd: %d\r\n", so->s);

		ptr += (*print)(*arg,"    Local address (your end of the link): %s\r\n",
				  inet_ntoa(so->so_laddr));
		ptr += (*print)(*arg,"    Local port: %d\r\n", ntohs(so->so_lport));
		ptr += (*print)(*arg,"    Foreign address (host on internet): %s\r\n",
				  inet_ntoa(so->so_faddr));
		ptr += (*print)(*arg,"    Foreign port: %d\r\n", ntohs(so->so_fport));
		ptr += (*print)(*arg,"    Type of service: %d\r\n", so->so_iptos);
		ptr += (*print)(*arg,"    State (see source): %d\r\n", so->so_state);
		if (so->so_expire)
		   ptr += (*print)(*arg,"    Expires in: %u seconds\r\n", 
				     (so->so_expire - curtime) / 1000);
		ptr += (*print)(*arg,"    Number of packets queued: %d\r\n\r\n", so->so_queued);
	}
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 0;
}

void
slirp_exit(exit_status)
	int exit_status;
{
	int type = PRN_DFD;
	
	term_restore(ttyfd);

	if (dostats)
	   type = PRN_STDERR;

#define SBNULL (struct sbuf *)NULL
	
	if (type == PRN_STDERR || dfd != NULL) {
		ipstats(SBNULL, type);
		tcpstats(SBNULL, type);
		udpstats(SBNULL, type);
		mbufstats(SBNULL, type);
		
		tcpsockstats(SBNULL, type);
		udpsockstats(SBNULL, type);
	}

#undef SBNULL

	/*
	 * Report "regular" errors that occurred during the running of slirp,
	 * which might have perplexed the user
	 */
	if (slirp_errors & ERR_REDUCEMTU) {
		fprintf(stderr,"Error:\r\n");
		fprintf(stderr,"    SLiRP received (and discarded) packets that were too large.\n");
		fprintf(stderr,"    Please check your SLIP software and make sure your\n");
		fprintf(stderr,"    MTU matches SLiRP's MTU (currently %d).\n", ifp.ifp_mtu);
		fprintf(stderr,"    You can change SLiRP's MTU via the -m command-line option.\n\n");
		fprintf(stderr,"    If your MTU setting is OK then ignore this message,\n");
		fprintf(stderr,"    it could have been caused by line noise or something.\n\n");
	}
	if (slirp_errors & ERR_RCVICMPECHOREQ) {
		fprintf(stderr,"Warning:\n");
		fprintf(stderr,"    SLiRP received some ICMP ECHO-REQUEST packets.\n");
		fprintf(stderr,"    SLiRP does not yet emulate ICMP ECHO-REQUEST packets, so programs\n");
		fprintf(stderr,"    like \"ping\" will not work.  If you used \"ping\" to test whether\n");
		fprintf(stderr,"    the link was up, then it will fail, even though the link may have\n");
		fprintf(stderr,"    been up.  Use \"telnet\" to test the link instead.\n");
	}

#ifdef DO_RESTART

	if (slirp_forked) {
#ifdef DEBUG
		debug_verbose(dfd, "Sending SIGINT to parent %ld\n",
		    (long) getppid());
		fflush_verbose(dfd);
#endif
		if (kill(getppid(), SIGINT) < 0)
			fprintf(stderr, "Couldn't kill parent process %ld!\n",
			    (long) getppid());
    	}

	if (remote_pid > 0) {
#ifdef DEBUG
		debug_verbose(dfd, "Sending SIGINT to guardian %ld\n",
		    (long) remote_pid);
		fflush_verbose(dfd);
#endif
		signal(SIGHUP, SIG_IGN);	/* ignore guardian kill() */
		kill(remote_pid, SIGINT);   /* we may just have timed out */
    	}

#endif /* DO_RESTART */

	exit(exit_status);
}

/* Wait ten minutes for a new connection from this user */
#define RECONNECT_WAIT (10*60)

/*
 * This function is called when SLiRP gets a SIGHUP, presumably
 * because the line has gone down.  It builds a Unix socket called
 * /tmp/slirp_$USER, disconnects from its control tty, closes its tty
 * file descriptors, and waits for someone to connect to it and pass
 * their pid and a new tty to use.  When that happens, we pass our pid
 * back, open the new tty, set the new tty to have the right terminal
 * modes, and continue.  The other slirp process hangs around and HUPs
 * this process if it gets any signals.  It would probably be better
 * to pass all our state and fd's back to the other process, then exit,
 * but that would be harder.
 */
void
slirp_hup(sig)
	int sig;	/* arg not used */
{
#ifndef DO_RESTART
	slirp_exit(1);;
#else
	char buf[1024], path[256], msg[256], *tty;
	struct sockaddr_un sock_un;
	struct timeval timeout;
	struct socket *so;
	int s, n, i = -1;
	int maxfd, msglen, retval, tcp_count, udp_count;
	int din = sizeof(struct sockaddr_un);
	fd_set mask;

	signal(SIGHUP, SIG_IGN);		/* don't re-interrupt */

#ifdef DEBUG
	debug_call(dfd,"slirp_hup(sig) called ...\n");
	fflush_call(dfd);
#endif

	if (username == NULL) {
#ifdef DEBUG
		debug_misc(dfd, "Don't know your name; exiting.\r\n");
#endif
		slirp_exit(1);
	}
	sprintf(path, "/tmp/slirp_%s", username);

	/* If there's a guardian, make sure it's dead -- if we somehow got a
	   HUP but it wasn't from the guardian, the guardian had better go. */
	if (remote_pid) {
		(void) kill(remote_pid, SIGINT);
		remote_pid = 0;
	}

	/* Create a Unix domain socket */
	if ((s = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
#ifdef DEBUG
		debug_misc(dfd, "Exiting; can't create Unix socket: %s\r\n",
		    strerror(errno));
#endif
		slirp_exit(1);
	}
	sock_un.sun_family = AF_UNIX;
	strcpy(sock_un.sun_path, path);
	(void) unlink(path);
	if (bind(s, (struct sockaddr *) &sock_un, strlen(path) + 1) < 0) {
#ifdef DEBUG
		debug_misc(dfd,"Exiting; can't bind Unix socket at %s: %s\r\n",
		    path, strerror(errno));
#endif
		close(s);
		slirp_exit(1);
	}
	if (listen(s, 1) < 0) {
#ifdef DEBUG
		fprintf(stderr, "Exiting; can't listen on Unix socket: %s\r\n",
		    strerror(errno));
#endif
bail_out:
		if (i >= 0)
			close(i);
		close(s);
		unlink(path);
		slirp_exit(1);
	}

	/*
	 * Song and dance to detach from ctty but not be orphaned.  This
	 * madness is because we don't want to stay attached to the old 
	 * tty (thus potentially blocking it, or getting random signals 
	 * from it), but if we detach from it with setsid(), we end up 
	 * as an "orphaned process group".  As such we can't write to 
	 * another terminal, so we fork once and have the child start a
	 * new process group, which makes the child not an orphan, but
	 * clutters up the process table with yet a third slirp process.
	 * Better ways to do this would be most appreciated.
	 */
	if (!slirp_forked) {
		(void) setsid();	/* new session */
		retval = fork();
		if (retval < 0)
			goto bail_out;
		if (retval) {		/* parent idles... sigh */
			while (1)
				pause();
		}
		retval = setpgid(0, 0);	/* child in new process group */
		if (retval < 0) {
#ifdef DEBUG
			debug_error(dfd,"setpgid(): %s\n", strerror(errno));
			fflush_error(dfd);
#endif
			goto bail_out;
		}
		slirp_forked = 1;
	}

	/* Nuke stdin et al to get off old, useless tty */
	n = open("/dev/null", O_RDWR);
	if (n < 0) {
#ifdef DEBUG
		debug_misc(dfd, "Exiting; can't open /dev/null: %s\r\n",
		    strerror(errno));
#endif
		goto bail_out;
	}
	dup2(n, 0);
	dup2(n, 1);
	dup2(n, 2);
	if (n > 2)
		close(n);

#ifdef DEBUG
	debug_verbose(dfd,"slirp_hup: entering select wait loop\n");
	fflush_verbose(dfd);
#endif

	/* Wait for a while for a reconnect, then exit. */
	for (;;) {
		timeout.tv_sec = RECONNECT_WAIT;
		timeout.tv_usec = 0;
		FD_ZERO(&mask);
		if (i >= 0) {
			FD_SET(i, &mask);
			maxfd = i+1;
		} else {
			FD_SET(s, &mask);
			maxfd = s+1;
		}
		if (select(maxfd, &mask, NULL, NULL, &timeout) < 0) {
			if (errno == EINTR)	/* Odd. */
				continue;
			goto bail_out;
		}

		/* If we have data on a connection, we're done here. */
		if (i >= 0 && FD_ISSET(i, &mask))
			break;

		/* If we have data on the socket, we have the connection. */
		if (FD_ISSET(s, &mask)) {
			i = accept(s, (struct sockaddr *) &sock_un, &din);
			if (i < 0)
				goto bail_out;	/* Didn't work too well. */
			continue;
		}

		/* Must have been a timeout.  We're done. */
		slirp_exit(0);
	}

	/* We got some data.  Hopefully, it's the remote pid and new tty. */
	retval = read(i, buf, sizeof(buf)-1);
	if (retval < sizeof("000000quit"))
		goto bail_out;	/* No data present? Curious. */
	buf[retval] = '\0';	/* in case of bogus data */

#ifdef DEBUG
	debug_verbose(dfd,"slirp_hup: got new slirp info: %s\n", buf);
	fflush_verbose(dfd);
#endif

	/* Read %06d%s format to get remote pid and tty; sanity-check it. */
	remote_pid = atoi(&buf[0]);
	tty = &buf[6];
	sprintf(msg, "%06ld", (long) getpid());	/* build pid into our reply */
	if (kill(remote_pid, 0) < 0) {
	    	sprintf(msg+6, "guardian slirp pid (%.6s) invalid: %s\n",
	    	    buf, strerror(errno));
bail_msg:
#ifdef DEBUG
		debug_error(dfd, "%s", msg+6);
	    	fflush_error(dfd);
#endif
		msglen = strlen(msg)+1;
	    	if (write(i, msg, msglen) != msglen) {
#ifdef DEBUG
			debug_error(dfd, "slirp_hup: bad write %d:%s\n",
			    msglen, msg);
			fflush_error(dfd);
#endif
	    	}
	    	sleep(1);
	    	goto bail_out;
	}
	if (strcmp(tty, "quit") == 0) {
		remote_pid = 0;
		(void) close(i);
		(void) close(s);
		(void) unlink(path);
		slirp_exit(0);
	}
	if (strncmp(tty, "/dev", 4) != 0) {
	    	sprintf(msg+6, "new tty doesn't start with /dev: %s\n", tty);
	    	goto bail_msg;
	}

	/* Try to open the new tty and make it the new stdout, etc. */
	n = open(tty, O_RDWR);
	if (n < 0) {
	    	sprintf(msg+6, "couldn't open new tty (%s): %s\n",
	    	    tty, strerror(errno));
	    	goto bail_msg;
	}
	if (dup2(n, 0) < 0 || dup2(n, 1) < 0 || dup2(n, 2) < 0) {
	    	sprintf(msg+6, "couldn't dup new tty to all stdio fds: %s\n",
	    	    strerror(errno));
	    	goto bail_msg;
	}
	if (n > 2)
		close(n);
	
	/* Now condition the tty */
	term_save(0);	/* might as well reload the cooked settings */
	if ((ifp.ifp_flags == IF_COMPRESS ?
	    term_raw(0, 3, 2, baud) :
	    term_raw(0, 40, 2, baud)) < 0) {
#ifdef DEBUG
		debug_error(dfd,"ERROR: term_raw: %s\n", strerror(errno));
		fflush_error(dfd);
#endif
		slirp_exit(1);
	}

	/* Write message to the tty, and go-ahead message to the guardian. */
	for (tcp_count = 0, so = tcb.so_next; so != &tcb; so = so->so_next)
		++tcp_count;
	for (udp_count = 0, so = udb.so_next; so != &udb; so = so->so_next)
		++udp_count;
	fprintf(stderr,
"SLiRP continuing on tty %s with %d tcp sockets, %d udp sockets...\r\n",
	    tty, tcp_count, udp_count);
	if (write(i, msg, 7) != 7) {	/* null message means OK */
#ifdef DEBUG
		debug_error(dfd, "slirp_hup: bad OK write %s\n", msg);
	    	fflush_error(dfd);
#endif
		goto bail_out;
    	}
    	if (read(i, buf, sizeof(buf)-1) != 3 || strcmp(buf, "OK") != 0)
    		goto bail_out;
    	(void) close(i);
    	(void) close(s);
    	(void) unlink(path);

	/* Pick up where we left off as if nothing had happened :-) */
	signal(SIGHUP, slirp_hup);
#ifdef EASY_RENEGOTIATE_PPP_HACK
	if (ppp_in_use)
		ppp_start();
#endif
#endif /* DO_RESTART */
}

#ifdef DO_RESTART
static int old_pid;

/*
 * Once the guardian slirp has finished chatting with the old slirp, it
 * sets up this function as a signal handler for everything, and sleeps.
 */
void
hup_and_die(sig)
	int sig;
{
	signal(sig, SIG_IGN);
	(void) kill(old_pid, SIGHUP);
    	exit(0);
}
#endif /* DO_RESTART */

/* Value of largest signal number (see, e.g., /usr/include/sys/signal.h) */
#define MAX_SIG 40

/*
 * This routine tries to open /tmp/slirp_$USER, if present, and pass our
 * pid and current tty name to it so it can continue.  If it succeeds,
 * it reads the old process's pid and just hangs around and waits for a
 * SIGHUP signal when the connection closes.  At that point it SIGHUPs
 * the old process and exits.
 *
 * If kill_old is true, we kill the old process by sending it our pid and
 * the magic string "quit".
 */
void
slirp_restart(kill_old)
	int kill_old;
{
#ifdef DO_RESTART
	char buf[1024], path[100];
	struct sockaddr_un sock_un;
	int s, i, len, retval;
	char *tn;

#ifdef DEBUG
	debug_call(dfd,"slirp_restart() called ...\n");
	fflush_call(dfd);
#endif

	if (username == NULL) {
		fprintf(stderr, "Don't know your name; can't restart.\n");
		return;
	}
	sprintf(path, "/tmp/slirp_%s", username);

	/* Open the Unix domain socket, if present. */
	if ((s = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
		fprintf(stderr,
		    "Can't create Unix socket (%s); can't restart.\n",
		    strerror(errno));
		return;
	}
	sock_un.sun_family = AF_UNIX;
	strcpy(sock_un.sun_path, path);
	if (connect(s, (struct sockaddr *) &sock_un, strlen(path) + 1) < 0) {
		/* No socket, so we are on our own. */
bail_restart:
	    	(void) close(s);
	    	return;
	}

	/* Pass pid and new tty name to it. */
	if (kill_old)
		tn = "quit";
	else {
		tn = ttyname(0);
		if (tn == NULL) {
			fprintf(stderr,
			    "Can't get name of this tty; can't restart.\n");
			goto bail_restart;
		}
	}
	len = sprintf(buf, "%06ld%s", (long) getpid(), tn)+1;
	if (write(s, buf, len) != len) {
		fprintf(stderr,
		    "Couldn't write \"%s\" to socket; can't restart.\n", tn);
	    	goto bail_restart;
	}
	if (kill_old) {
		close(s);
		return;
	}
	retval = read(s, buf, sizeof(buf)-1);
	if (retval < 0) {
		fprintf(stderr,
		   "Got an error from reading old slirp (%s); can't restart\n",
		   strerror(errno));
	    	return;
	}
	if (retval == 0) {
		fprintf(stderr,
		    "Got an empty message from old slirp; can't restart\n");
	    	return;
	}
	buf[retval] = '\0';
	old_pid = atoi(buf);
	if (kill(old_pid, 0) < 0) {
		fprintf(stderr,
		    "Got a bad pid from old slirp (%s); can't restart\n", buf);
	    	return;
	}
	if (buf[6]) {
		fprintf(stderr, "%s", buf+6);
		exit(1);
	}

	/* Acknowledge write so old slirp can close and unlink socket. */
	if (write(s, "OK", 3) != 3) {
		fprintf(stderr,
	       "Couldn't write ACK message back to old slirp (%d); exiting.\n",
		    old_pid);
	    	kill(old_pid, SIGINT);
	    	exit(1);
	}

	/* Now we just wait until we get a signal and die gracefully. */
	for (i = 1; i <= MAX_SIG; ++i)
		signal(i, hup_and_die);
	while (1)
		pause();
#endif /* DO_RESTART */
}
