/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.2   JRR     28-Feb-1992	Added Cosmic V2 changes.
*	V01.3	JRR	17-Jun-1992	Added header.
*	V01.4	JRR	07-Dec-1992	Modified for C prototypes.
*	V01.5	JRR	21-Apr-1994	Ranking compute servers.
*/
/*++ sizedb.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.36/lib/RCS/sizedb.c,v $
 *
 * DESCRIPTION:
 *
 *	Return the number of bytes required to represent the specified
 *	generic descriptor structure which must be completely initialized
 *	with the exception of the size field.  The size value returned
 *	does NOT include the bytes required to store the "size" field
 *	of a generic descriptor.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.5 $ $Date: 1994/09/02 17:37:59 $ $State: Exp $)
 * $Log: sizedb.c,v $
 * Revision 1.5  1994/09/02  17:37:59  jrroma
 * Version 3.36
 *
 * Revision 1.4  1992/12/22  15:46:34  jrroma
 * Version 3.30
 *
 * Revision 1.3  92/06/18  13:24:52  jrroma
 * Added gnu header
 * 
 * Revision 1.2  92/02/28  15:35:18  jrroma
 * Added Cosmic V2 changes.
 * 
 * Revision 1.1  92/02/28  15:33:38  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include <string.h>
#include <errno.h>

/*** sizedb
 *
 *
 *	int sizedb():
 *
 *	Return the number of bytes required to represent the specified
 *	generic descriptor structure which must be completely initialized
 *	with the exception of the size field.  The size value returned
 *	does NOT include the bytes required to store the "size" field
 *	of a generic descriptor.
 */
int sizedb (descr, descrtype)
register struct gendescr *descr;	/* Descriptor */
register int descrtype;			/* Descriptor type:  DSC_ */
{
	register int size;

	switch (descrtype) {
        case DSC_QCOMPLEX:
                size = sizeof (descr->v.qcom);
                break;
	case DSC_QUEUE:
		size = sizeof (descr->v.que);
		break;
	case DSC_DEVICE:
		size = sizeof (descr->v.dev) - MAX_PATHNAME
		       + strlen (descr->v.dev.fullname);
		break;
	case DSC_QMAP:
		size = sizeof (descr->v.map);
		break;
	case DSC_PIPEQDEST:
		size = sizeof (descr->v.dest);
		break;
	case DSC_PARAM:
		switch (descr->v.par.paramtype) {
		case PRM_GENPARAMS:
			size = sizeof (descr->v.par.v.genparams);
			break;
		case PRM_LOGFILE:
			size = sizeof (descr->v.par.v.logfile);
			break;
		case PRM_NETPROCS:
			size = sizeof (descr->v.par.v.netprocs);
			break;
		case PRM_NETPEER:
			size = sizeof (descr->v.par.v.netpeer);
			break;
		default:
			printf ("F$Invalid NQS general parameters file ");
			printf ("descriptor type\n");
			printf ("F$specified in call to sizedb().\n");
			printf ("I$Type was %1d.\n", descr->v.par.paramtype);
			fflush (stdout);
			errno = 0;		/* Not a system call error */
			nqs_abort();		/* Abort execution */
		}
		/*
		 *  Add in the space for the paramtype field.
		 */
		size += (((char *) &descr->v.par.v) -
			 ((char *) &descr->v.par.paramtype));
		break;
	case DSC_MGR:
		size = sizeof (descr->v.mgr);
		break;
	case DSC_FORM:
		size = sizeof (descr->v.form);
		break;
	case DSC_SERVER:
		size = sizeof (descr->v.cserver);
		break;
	default:
		printf ("F$Invalid NQS database descriptor type.\n");
		printf ("I$Type was %1d.\n", descrtype);
	}
	return (size);
}
