/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*	V01.3	JRR	17-Jun-1992	Added header.
*	V01.4	JRR	07-Dec-1992	Modified for C prototypes.
*	V01.5	JRR			Placeholder
*	V01.6	JRR	03-Mar-1994	Fixed up NMAP.
*/
/*++ pipeqdest.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/lib/RCS/pipeqdest.c,v $
 *
 * DESCRIPTION:
 *
 *
 *	Return the machine-id, queue name, and retry count values for
 *	the specified pipe queue destination (as defined in the environment
 *	of a pipe queue server).
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	May 8, 1986.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.6 $ $Date: 1994/03/30 20:32:38 $ $State: Exp $)
 * $Log: pipeqdest.c,v $
 * Revision 1.6  1994/03/30  20:32:38  jrroma
 * Version 3.35.6
 *
 * Revision 1.5  92/12/22  15:46:20  jrroma
 * Version 3.30
 * 
 * Revision 1.4  92/06/18  13:24:25  jrroma
 * Added gnu header
 * 
 * Revision 1.3  92/02/28  15:20:40  jrroma
 * Added Cosmic V2 changes.
 * 
 * Revision 1.2  92/01/16  17:03:54  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/16  17:03:15  jrroma
 * Initial revision
 * 
 *
 */

#include "nmap.h"			/* Network mapping stuff (mid_t) */
#include <stdlib.h>

/*** pipeqdest
 *
 *
 *	char *pipeqdest():
 *
 *	Return the machine-id, time spent in retry state, and queue name
 *	values for the specified pipe queue destination (as defined in the
 *	environment of a pipe queue server).
 *
 *	Returns:
 *		A pointer to the name of the destination queue (if
 *		the specified queue destination exists).  Otherwise,
 *		(char *) 0 is returned.
 */
char *pipeqdest (destno, destmid, retrytime)
int destno;				/* Destination number [1..n] */
Mid_t *destmid;				/* Machine-id of destination */
long *retrytime;			/* Time spent in retry state */
{
	char envvarname [5];		/* Environment var name */
	register char *dest;		/* Ptr to dest environment var */
	register unsigned long ulong;	/* For scanning */
	register char ch;		/* Scan character */
        register short neg;

	if (destno < 1 || destno > 1000) {
		return ((char *) 0);	/* No such destination */
	}
	sprintf (envvarname, "D%03d", destno-1);
	dest = getenv (envvarname);	/* Get the environment var for dest */
	if (dest == (char *) 0) {
		return ((char *) 0);	/* No such destination */
	}
	ch = *dest;			/* Get scan character */
	while (ch == ' ') {		/* Scan any whitespace */
		ch = *++dest;
	}
        if (ch == '-') {                /* Scan any for negative */
                ch = *++dest;
                neg = -1;
        }
        else
                neg =1;

	ulong = 0;
	while (ch >= '0' && ch <= '9') {/* Scan destination machine-id */
		ulong *= 10;
		ulong += ch - '0';
		ch = *++dest;
	}
	*destmid = ulong * neg;
	while (ch == ' ') {		/* Scan any whitespace */
		ch = *++dest;
	}
	ulong = 0;
	while (ch >= '0' && ch <= '9') {/* Scan destination machine-id */
		ulong *= 10;
		ulong += ch - '0';
		ch = *++dest;
	}
	*retrytime = ulong;
	while (ch == ' ') {		/* Scan any whitespace */
		ch = *++dest;
	}
	return (dest);			/* Return pointer to queue name */
}
